/**********************************************************************/
/* Bibliotheque de filtrage d'images au format TrueColor 32bitd       */
/* image_filter.c                                                     */
/*                                                                    */
/* Ecrit par : Daniel Lacroix (all rights reserved)                   */
/*                                                                    */
/**********************************************************************/

#include <image_filter.h>
#include <image_draw_func.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

/**************************************************************/
/* Fait un flou gaussien de l'image psrc et place le resultat */
/* dans pdest. pdest et psrc doivent avoir la mme taille.    */
void image_blur(image *pdest, image *psrc)
{ int32 vx,vy;
  int i,j;
  float res_r, res_g, res_b;
  pix vpix;
  /* dfinition du filtre utilis (en forme de cloche => gaussien) */
  float filter[3][3] = {{1.0,2.0,1.0},
                        {2.0,4.0,2.0},
		        {1.0,2.0,1.0}};
  float filter_div = 16.0;

  /* si les prconditions ne sont pas remplies, on quitte */
  if((pdest == NULL) || (psrc == NULL) ||
     (psrc->width != pdest->width) || (psrc->height != pdest->height))
    return;

  for(vy = 0;vy < psrc->height; vy++)
  {
    for(vx = 0;vx < psrc->width; vx++)
    {
      res_r = 0;
      res_g = 0;
      res_b = 0;
      for(j=0;j<3;j++)
        for(i=0;i<3;i++)
	{
  	  vpix = get_pix(psrc, vx-(3>>1)+i, vy-(3>>1)+j);
	  res_r += (float)COL_RED  (vpix) * filter[j][i];
	  res_g += (float)COL_GREEN(vpix) * filter[j][i];
	  res_b += (float)COL_BLUE (vpix) * filter[j][i];
	}
      res_r /= filter_div;
      res_g /= filter_div;
      res_b /= filter_div;

      /* on sature le rsultat */
      res_r = (res_r > 255.0)? 255.0 : ((res_r < 0.0)? 0.0:res_r);
      res_g = (res_g > 255.0)? 255.0 : ((res_g < 0.0)? 0.0:res_g);
      res_b = (res_b > 255.0)? 255.0 : ((res_b < 0.0)? 0.0:res_b);

      /* on place le rsultat dans l'image destination */      
      put_pix_alpha_replace(pdest, vx, vy, COL(
        (uint8)res_r,(uint8)res_g,(uint8)res_b));
    }
  }
}
/**************************************************************/

/**************************************************************/
/* Ce filtre est un filtre qui prserve les contours en       */
/* s'adaptant aux drives des points autour du point en      */
/* traitement. pdest et psrc doivent avoir la mme taille.    */
void image_edge_protect(image *pdest, image *psrc)
{ int32 vx,vy;
  int i,j;
  float res_r, res_g, res_b;
  pix vpix,cpix;
  /* definition di filtre utilise (en forme de cloche => gaussien) */
  float filter_red  [3][3];
  float filter_green[3][3];
  float filter_blue [3][3];
  float filter_red_sum, filter_green_sum, filter_blue_sum;

  /* si les prconditions ne sont pas remplies, on quitte */
  if((pdest == NULL) || (psrc == NULL) ||
     (psrc->width != pdest->width) || (psrc->height != pdest->height))
    return;

  for(vy = 0;vy < psrc->height; vy++)
  {
    for(vx = 0;vx < psrc->width; vx++)
    {
      filter_red_sum = filter_green_sum = filter_blue_sum = 0.0;
      /* recupere le point que l'on traite */
      cpix = get_pix(psrc, vx, vy);
      /* fabrique la matrice de convolution en fonction des variations */
      /* de l'intensit (fabrique le filtre local)                     */
      for(j=0;j<3;j++)
        for(i=0;i<3;i++)
	{
          if((j == 1) && (i == 1))
          {
            filter_red  [1][1] = 0.5;
            filter_green[1][1] = 0.5;
            filter_blue [1][1] = 0.5;
          } else {
            vpix = get_pix(psrc, vx-(3>>1)+i, vy-(3>>1)+j);
                  
	    res_r = ((float)COL_RED  (cpix)) - ((float)COL_RED  (vpix));
            if(res_r < 0.0) res_r = -res_r;
            if(res_r != 0)
              filter_red  [j][i] = 1.0/res_r;
            else
              filter_red  [j][i] = 1.0;
            
	    res_g = ((float)COL_GREEN(cpix)) - ((float)COL_GREEN(vpix));
            if(res_g < 0.0) res_g = -res_g;
            if(res_g != 0)
              filter_green[j][i] = 1.0/res_g;
            else
              filter_green[j][i] = 1.0;
            
	    res_b = ((float)COL_BLUE (cpix)) - ((float)COL_BLUE (vpix));
            if(res_b < 0.0) res_b = -res_b;
            if(res_b != 0)
              filter_blue [j][i] = 1.0/res_b;
            else
              filter_blue [j][i] = 1.0;
          }
          filter_red_sum   += filter_red  [j][i];
          filter_green_sum += filter_green[j][i];
          filter_blue_sum  += filter_blue [j][i];
	}
      for(j=0;j<3;j++)
        for(i=0;i<3;i++)
	{
          if(filter_red_sum != 0.0)
            filter_red  [j][i] /= filter_red_sum;
          if(filter_green_sum != 0.0)
            filter_green[j][i] /= filter_green_sum;
          if(filter_blue_sum != 0.0)
            filter_blue [j][i] /= filter_blue_sum;
        }
      /* on applique le filtre calcul */
      res_r = 0.0;
      res_g = 0.0;
      res_b = 0.0;
      for(j=0;j<3;j++)
        for(i=0;i<3;i++)
	{
  	  vpix = get_pix(psrc, vx-(3>>1)+i, vy-(3>>1)+j);
	  res_r += (float)COL_RED  (vpix) * filter_red  [j][i];
	  res_g += (float)COL_GREEN(vpix) * filter_green[j][i];
	  res_b += (float)COL_BLUE (vpix) * filter_blue [j][i];
	}

      /* on sature le rsultat */
      res_r = (res_r > 255.0)? 255.0 : ((res_r < 0.0)? 0.0:res_r);
      res_g = (res_g > 255.0)? 255.0 : ((res_g < 0.0)? 0.0:res_g);
      res_b = (res_b > 255.0)? 255.0 : ((res_b < 0.0)? 0.0:res_b);

      /* on place le rsultat dans l'image destination */      
      put_pix_alpha_replace(pdest, vx, vy, COL(
        (uint8)res_r,(uint8)res_g,(uint8)res_b));
    }
  }
}
/**************************************************************/

/*******************************************************************/
/* Applique un filtre mdian sur l'image psrc et place le resultat */
/* dans pdest. pdest et psrc doivent avoir la mme taille.         */
void image_median(image *pdest, image *psrc)
{ int32 vx,vy;
  int i,j;
  pix vpix;
  pix   pix_tab[9];
  float lum_tab[9];
  float lum_swap;
  pix   pix_swap;
  int pos;

  /* si les prconditions ne sont pas remplies, on quitte */
  if((pdest == NULL) || (psrc == NULL) ||
     (psrc->width != pdest->width) || (psrc->height != pdest->height))
    return;

  for(vy = 0;vy < psrc->height; vy++)
  {
    for(vx = 0;vx < psrc->width; vx++)
    {
      pos = 0;
      for(j=0;j<3;j++)
        for(i=0;i<3;i++)
	{
  	  vpix = get_pix(psrc, vx-(3>>1)+i, vy-(3>>1)+j);
          
          pix_tab[pos] = vpix;
          /* on calcule la luminance */
          lum_tab[pos] = 0.299*(float)COL_RED  (vpix) +
                         0.587*(float)COL_GREEN(vpix) +
                         0.114*(float)COL_BLUE (vpix);
          pos++;
	}

      /* on tri le tableau des pixels en fonction de la luminance */
      for(j=0;j<8;j++)
      {
        pos = j;
        for(i=j+1;i<9;i++)
        {
          if(lum_tab[i] < lum_tab[pos])
            pos = i;
        }
        lum_swap = lum_tab[j];
        pix_swap = pix_tab[j];
        lum_tab[j] = lum_tab[pos];
        pix_tab[j] = pix_tab[pos];
        lum_tab[pos] = lum_swap;
        pix_tab[pos] = pix_swap;
      }

      /* on place le point avec l'intensit moyenne dans l'image destination */
      put_pix_alpha_replace(pdest, vx, vy, pix_tab[4]);
    }
  }
}
/*******************************************************************/

/********************************************************/
/* Calcul la drive x,y en chaque points  l'aide du   */
/* filtre de Sobel. La driv est calcul pour chaque   */
/* couleur. C'est la norme du gradient qui est utilise */
/* pour dterminer le niveau de chaque couleur dans     */
/* l'image destination. psrc est l'image d'origine et   */
/* pdest est l'image destination. psrc et pdest doivent */
/* avoir la mme taille.                                */
void image_sobel(image *pdest, image *psrc)
{ int32 vx,vy;
  int i,j;
  float res_r_x, res_g_x, res_b_x;
  float res_r_y, res_g_y, res_b_y;
  float res_r, res_g, res_b;
  pix vpix;
  /* filtre de calcul de la drive horizontale */
  float filter_sobel_x[3][3] = {{-1.0,0.0,1.0},
                                {-2.0,0.0,2.0},
		                {-1.0,0.0,1.0}};
  /* filtre de calcul de la drive verticale */
  float filter_sobel_y[3][3] = {{-1.0,-2.0,-1.0},
                                { 0.0, 0.0, 0.0},
		                { 1.0, 2.0, 1.0}};

  /* si les prconditions ne sont pas remplies, on quitte */
  if((pdest == NULL) || (psrc == NULL) ||
     (psrc->width != pdest->width) || (psrc->height != pdest->height))
    return;

  for(vy = 0;vy < psrc->height; vy++)
  {
    for(vx = 0;vx < psrc->width; vx++)
    {
      res_r_x = 0;
      res_g_x = 0;
      res_b_x = 0;
      res_r_y = 0;
      res_g_y = 0;
      res_b_y = 0;
      for(j=0;j<3;j++)
        for(i=0;i<3;i++)
	{
  	  vpix = get_pix(psrc, vx-(3>>1)+i, vy-(3>>1)+j);
	  res_r_x += (float)COL_RED(vpix) * filter_sobel_x[j][i];
	  res_g_x += (float)COL_GREEN(vpix) * filter_sobel_x[j][i];
	  res_b_x += (float)COL_BLUE(vpix) * filter_sobel_x[j][i];

	  res_r_y += (float)COL_RED(vpix) * filter_sobel_y[j][i];
	  res_g_y += (float)COL_GREEN(vpix) * filter_sobel_y[j][i];
	  res_b_y += (float)COL_BLUE(vpix) * filter_sobel_y[j][i];
	}
      res_r_x /= 4.0;
      res_g_x /= 4.0;
      res_b_x /= 4.0;

      res_r_y /= 4.0;
      res_g_y /= 4.0;
      res_b_y /= 4.0;

      /* calcul la norme du vecteur gradient */
      res_r = sqrt(res_r_x*res_r_x + res_r_y*res_r_y);
      res_g = sqrt(res_g_x*res_g_x + res_g_y*res_g_y);
      res_b = sqrt(res_b_x*res_b_x + res_b_y*res_b_y);

      /* on sature les valeurs */
      res_r = (res_r > 255.0)? 255.0 : ((res_r < 0.0)? 0.0:res_r);
      res_g = (res_g > 255.0)? 255.0 : ((res_g < 0.0)? 0.0:res_g);
      res_b = (res_b > 255.0)? 255.0 : ((res_b < 0.0)? 0.0:res_b);

      /* on place le rsultat dans pdest */      
      put_pix_alpha_replace(pdest, vx, vy, COL(
        (uint8)res_r,(uint8)res_g,(uint8)res_b));
    }
  }
}
/********************************************************/

/********************************************************************/
/* Ce filtre est un filtre de Blur qui preserve les contours en     */
/* utilisant le calcul de drives de Sobel. Si il s'agit d'un      */
/* contour, on conserve le point d'origine proportionnellement  la */
/* drive. pdest et psrc doivent avoir la mme taille.             */
void image_blur_edge_protect(image *pdest, image *psrc)
{ int32 vx,vy;
  int i,j;
  float res_r_x, res_g_x, res_b_x;
  float res_r_y, res_g_y, res_b_y;
  float res_r, res_g, res_b;
  float res2_r, res2_g, res2_b;
  pix vpix;
  /* filtre de calcul de la drive horizontale */
  float filter_sobel_x[3][3] = {{-1.0,0.0,1.0},
                                {-2.0,0.0,2.0},
		                {-1.0,0.0,1.0}};
  /* filtre de calcul de la drive verticale */
  float filter_sobel_y[3][3] = {{-1.0,-2.0,-1.0},
                                { 0.0, 0.0, 0.0},
		                { 1.0, 2.0, 1.0}};
  /* definition di filtre utilise (en forme de cloche => gaussien) */
  float filter[3][3] = {{1.0,2.0,1.0},
                        {2.0,4.0,2.0},
		        {1.0,2.0,1.0}};
  float filter_div = 16.0;

  /* si les prconditions ne sont pas remplies, on quitte */
  if((pdest == NULL) || (psrc == NULL) ||
     (psrc->width != pdest->width) || (psrc->height != pdest->height))
    return;

  for(vy = 0;vy < psrc->height; vy++)
  {
    for(vx = 0;vx < psrc->width; vx++)
    {
      res_r_x = 0;
      res_g_x = 0;
      res_b_x = 0;
      res_r_y = 0;
      res_g_y = 0;
      res_b_y = 0;
      for(j=0;j<3;j++)
        for(i=0;i<3;i++)
	{
  	  vpix = get_pix(psrc, vx-(3>>1)+i, vy-(3>>1)+j);
	  res_r_x += (float)COL_RED(vpix) * filter_sobel_x[j][i];
	  res_g_x += (float)COL_GREEN(vpix) * filter_sobel_x[j][i];
	  res_b_x += (float)COL_BLUE(vpix) * filter_sobel_x[j][i];

	  res_r_y += (float)COL_RED(vpix) * filter_sobel_y[j][i];
	  res_g_y += (float)COL_GREEN(vpix) * filter_sobel_y[j][i];
	  res_b_y += (float)COL_BLUE(vpix) * filter_sobel_y[j][i];
	}
      res_r_x /= 4.0;
      res_g_x /= 4.0;
      res_b_x /= 4.0;

      res_r_y /= 4.0;
      res_g_y /= 4.0;
      res_b_y /= 4.0;

      /* calcul la norme du vecteur gradient */
      res_r = sqrt(res_r_x*res_r_x + res_r_y*res_r_y);
      res_g = sqrt(res_g_x*res_g_x + res_g_y*res_g_y);
      res_b = sqrt(res_b_x*res_b_x + res_b_y*res_b_y);

      /* on sature les valeurs */
      res_r = (res_r > 255.0)? 255.0 : ((res_r < 0.0)? 0.0:res_r);
      res_g = (res_g > 255.0)? 255.0 : ((res_g < 0.0)? 0.0:res_g);
      res_b = (res_b > 255.0)? 255.0 : ((res_b < 0.0)? 0.0:res_b);


      res2_r = 0;
      res2_g = 0;
      res2_b = 0;
      for(j=0;j<3;j++)
        for(i=0;i<3;i++)
	{
  	  vpix = get_pix(psrc, vx-(3>>1)+i, vy-(3>>1)+j);
	  res2_r += (float)COL_RED  (vpix) * filter[j][i];
	  res2_g += (float)COL_GREEN(vpix) * filter[j][i];
	  res2_b += (float)COL_BLUE (vpix) * filter[j][i];
	}
      res2_r /= filter_div;
      res2_g /= filter_div;
      res2_b /= filter_div;

      vpix = get_pix(psrc, vx, vy);
      res_r = res2_r*(1.0-(res_r/255.0)) + (float)COL_RED  (vpix)*(res_r/255.0);
      res_g = res2_g*(1.0-(res_g/255.0)) + (float)COL_GREEN(vpix)*(res_g/255.0);
      res_b = res2_b*(1.0-(res_b/255.0)) + (float)COL_BLUE (vpix)*(res_b/255.0);

      /* on sature le rsultat */
      res_r = (res_r > 255.0)? 255.0 : ((res_r < 0.0)? 0.0:res_r);
      res_g = (res_g > 255.0)? 255.0 : ((res_g < 0.0)? 0.0:res_g);
      res_b = (res_b > 255.0)? 255.0 : ((res_b < 0.0)? 0.0:res_b);

      /* on place le rsultat dans pdest */      
      put_pix_alpha_replace(pdest, vx, vy, COL(
        (uint8)res_r,(uint8)res_g,(uint8)res_b));
    }
  }
}
/********************************************************************/

/************************************************************/
/* Fait une dtection de contours. Le rsultat est une      */
/* image noir ou blanc o les points blancs correspondent   */
/* aux points appartenant  un contour. Pour avoir de       */
/* bon rsultat, il est conseill de passer l'image         */
/* d'origine dans un filtre de flou (par exemple gaussien). */
/* psrc est l'image d'origine et pdest est l'image          */
/* destination. psrc et pdest doivent avoir la mme taille. */
void image_edge_max(image *pdest, image *psrc)
{ int32 vx,vy;
  int i,j;
  float grad_x;
  float grad_y;
  float intensity;
  pix vpix;
  /* filtre de calcul de la drive horizontale */
  float filter_sobel_x[3][3] = {{-1.0,0.0,1.0},
                                {-2.0,0.0,2.0},
		                {-1.0,0.0,1.0}};
  /* filtre de calcul de la drive verticale */
  float filter_sobel_y[3][3] = {{-1.0,-2.0,-1.0},
                                { 0.0, 0.0, 0.0},
		                { 1.0, 2.0, 1.0}};

  /* seuil de dcision qui permet de dire si la pente */
  /* est assez importante pour tre conserve.        */
  float seuil = 8.0;
  float *grad_x_tab;
  float *grad_y_tab;
  float *norme_tab;
  boolean keep;

  /* si les prconditions ne sont pas remplies, on quitte */
  if((pdest == NULL) || (psrc == NULL) ||
     (psrc->width != pdest->width) || (psrc->height != pdest->height))
    return;

  /* alloue un tableau pour conserver la valeur */
  /* de la drive horizontale en chaque point. */
  if((grad_x_tab = (float *)malloc(psrc->width*psrc->height*sizeof(float))) == NULL)
  { perror("malloc failed "); exit(1); }

  /* alloue un tableau pour conserver la valeur */
  /* de la drive veticale en chaque point.    */
  if((grad_y_tab = (float *)malloc(psrc->width*psrc->height*sizeof(float))) == NULL)
  { perror("malloc failed "); exit(1); }

  /* alloue un tableau pour conserver la valeur */
  /* de la norme du gradient en chaque point.   */
  if((norme_tab = (float *)malloc(psrc->width*psrc->height*sizeof(float))) == NULL)
  { perror("malloc failed "); exit(1); }

  for(vy = 0;vy < psrc->height; vy++)
  {
    for(vx = 0;vx < psrc->width; vx++)
    {
      grad_x = 0.0;
      grad_y = 0.0;
      for(j=0;j<3;j++)
      {
        for(i=0;i<3;i++)
	{
  	  vpix = get_pix(psrc, vx-(3>>1)+i, vy-(3>>1)+j);
          /* calcul l'intensit en ce point */
          intensity = 0.299*(float)COL_RED  (vpix) +
                      0.587*(float)COL_GREEN(vpix) +
                      0.114*(float)COL_BLUE (vpix);
          grad_x += intensity * filter_sobel_x[j][i];
          grad_y += intensity * filter_sobel_y[j][i];
	}
      }
      grad_x /= 4.0;
      grad_y /= 4.0;

      /* on stocke les valeurs */
      grad_x_tab[psrc->width*vy+vx] = grad_x;
      grad_y_tab[psrc->width*vy+vx] = grad_y;
      norme_tab [psrc->width*vy+vx] = sqrt(grad_x*grad_x+grad_y*grad_y);
    }
  }

  /* on recherche les extremas locaux */
  for(vy = 0;vy < psrc->height; vy++)
  {
    for(vx = 0;vx < psrc->width; vx++)
    { int32 delta_x, delta_y;
      float norme,norme1,norme2;
    
      /* on recupre les valeurs du gradient */
      grad_x = grad_x_tab[psrc->width*vy+vx];
      grad_y = grad_y_tab[psrc->width*vy+vx];
      norme  = norme_tab [psrc->width*vy+vx];
      /* on part comme si on ne gardera pas ce point */
      keep = FALSE;
      /* si la norme du gradient est infrieur  notre */
      /* seuil, on ne garde pas le point.              */
      if(norme > seuil)
      {
        /* on normalise le gradient */
        grad_x /= norme;
        grad_y /= norme;
        /* on dtermine le dplacement  faire pour */
        /* trouver le prochain point arondi sur la  */
        /* droite du gradient.                      */
	delta_x = (grad_x<-0.5)?-1:((grad_x>0.5)?1:0);
	delta_y = (grad_y<-0.5)?-1:((grad_y>0.5)?1:0);
        /* si le dplacement nous fait sortir de l'image  */
        /* alors on ne conserve pas le point (arbitraire) */
	if((delta_x+vx >= 0) && (delta_x+vx < psrc->width) &&
           (delta_y+vy >= 0) && (delta_y+vy < psrc->height) &&
           (-delta_x+vx >= 0) && (-delta_x+vx < psrc->width) &&
           (-delta_y+vy >= 0) && (-delta_y+vy < psrc->height))
	{
          /* rcupre le point prcdent et suivant sur la */
          /* droite du gradient.                           */
          norme1 = norme_tab[psrc->width*(vy+delta_y)+vx+delta_x];
          norme2 = norme_tab[psrc->width*(vy-delta_y)+vx-delta_x];
          /* si notre norme est suprieur au point prcdent */
          /* et au point suivant alors c'est que nous somme  */
          /* sur le sommet du contour donc on garde ce point */
          if((norme >= norme2) && (norme >= norme1))
	    keep = TRUE;
        }
      }
      /* en fonction de la dcision que l'on a prise */
      /* on affiche on point blanc ou on point noir  */
      if(keep)
	put_pix_alpha_replace(pdest, vx, vy, WHITE);
      else
	put_pix_alpha_replace(pdest, vx, vy, BLACK);
    }
  }
}
/************************************************************/


/**********************************************/
/* On calcule l'histogramme de l'image pimage */
/* L'histogramme est un histogramme 2D avec   */
/* rouge/intensit normalise en x            */
/*  vert/intensit normalise en y            */
/* en (u,v) on trouve la probabilit (0  1)  */
/* d'apparition de la couleur. Le tableau     */
/* retourn doit tre dallou avec free.     */
float *histo_real(image *pimage)
{ int x, y;
  pix   vpix;
  int   red, green;
  float *histo;
  float lum;

  /* l'histogramme est un histogramme en 2 dimensions */
  /* en x : r/intensit                                */
  /* en y : v/intensit                                */
  if((histo = (float *)malloc(256*256*sizeof(float))) == NULL)
  { perror("malloc failed "); exit(1); }

  /* on initialise l'histogramme a 0.0 */
  for(x = 0; x < 256*256; x++)
  {
    histo[x] = 0.0;
  }

  /* on parcours tous les points de l'image */
  for(y = 0; y < pimage->height; y++)
  {
    for(x = 0; x < pimage->width; x++)
    {
      /* on rcupre le pixel courant */
      vpix = pimage->buf[(y*pimage->width)+x];

      /* 0.0 <= valeur de l'intensite <= 255.0 */
      lum = (
        (((float)COL_RED  (vpix))*0.299)+
	(((float)COL_GREEN(vpix))*0.587)+
	(((float)COL_BLUE (vpix))*0.114));

      /* si l'intensit est  0, tous est  0                       */
      /* et on ne fait pas le calcul pour viter une division par 0 */
      if(lum == 0.0)
      {
        red = green = 0;
      } else {
        /* red/lum est compris entre 0.0 et 1.70, on normalise */
	red   = ((float)COL_RED(vpix)/(lum*1.71))*255.0;
        /* green/lum est compris entre 0.0 et 3.34, on normalise */
	green = ((float)COL_GREEN(vpix)/(lum*3.35))*255.0;
      }
      /* on incrmente la nombre d'apparition de la couleur dans l'histogramme */
      histo[(256*green)+red] += 1.0;
    }
  }
  /* on normalise les apparitions pour avoir des probas d'apparitions */
  for(x = 0; x < 256*256; x++)
    histo[x] /= pimage->width*pimage->height;
  
  /* on retourne l'histogramme */
  return(histo);
}
/**********************************************/

/**********************************************************************/
/* Retourne une image qui correspond aux probas que chaque            */
/* point de l'image pimage appartienne  l'histogramme hist_pattern   */
/* En d'autre terme, si hist_pattern correspond  l'histogramme       */
/* d'une texture, la proba indique si un point dans pimage correspond */
/* ou pas  ce type de texture. Le barycentre des probas est          */
/* reprsent par une croix rouge. Un rectangle autour du barycentre  */
/* indique la zone comprise dans l'cart type du barycentre (zone     */
/* dtect). L'image retourne doit tre libre avec image_free      */
image *find_histo(image *pimage, float *hist_pattern)
{ int x, y;
  pix   vpix;
  int   red, green;
  float lum;
  image *img_res;
  float bar_x = 0.0;
  float bar_y = 0.0;
  float sum = 0.0;
  float proba;
  float ecx = 0.0, ecy = 0.0;
  float *hist_img;

  /* calcul l'histogramme de l'image  analyser */
  hist_img = histo_real(pimage);

  /* alloue une nouvelle image pour le rsultat */
  img_res = image_new(pimage->width,pimage->height);

  /* parcours toute l'image */
  for(y = 0; y < img_res->height; y++)
    for(x = 0; x < img_res->width; x++)
    {
      /* on rcupre le pixel courant */
      vpix = pimage->buf[(y*pimage->width)+x];

      /* calcul l'intensit du pixel */
      lum = (
        (((float)COL_RED  (vpix))*0.299)+
	(((float)COL_GREEN(vpix))*0.587)+
	(((float)COL_BLUE (vpix))*0.114));

      /* calcul les coordonnes sur l'histogramme 2D */
      if(lum == 0.0)
      {
        red = green = 0;
      } else {
	red   = ((float)COL_RED(vpix)/(lum*1.71))*255.0;
	green = ((float)COL_GREEN(vpix)/(lum*3.35))*255.0;
      }

      /* calcul la probabilit que notre point appartienne */
      /*  l'histogramme pattern.                          */
      /* proba =  proba_pattern/proba_image                */
      if(hist_img[256*green+red] == 0.0)
        proba = 0.0;
      else
        proba = hist_pattern[256*green+red]/hist_img[256*green+red];
      
      img_res->buf[(y*pimage->width)+x] = COL(
        (uint8)(proba*255.0),
        (uint8)(proba*255.0),
        (uint8)(proba*255.0));

      /* on fait la somme des probas */	
      sum   += proba;
      bar_x += proba*(float)x;
      bar_y += proba*(float)y;
    }

  /* calcul le point barycentre */
  bar_x /= sum;
  bar_y /= sum;

  /* reparcourt toute l'image pour calculer les carts types */
  for(y = 0; y < img_res->height; y++)
    for(x = 0; x < img_res->width; x++)
    {
     /* on rcupre le pixel courant */
      vpix = pimage->buf[(y*pimage->width)+x];

      /* calcul l'intensit du pixel */
      lum = (
        (((float)COL_RED  (vpix))*0.299)+
	(((float)COL_GREEN(vpix))*0.587)+
	(((float)COL_BLUE (vpix))*0.114));

      /* calcul les coordonnes sur l'histogramme 2D */
      if(lum == 0.0)
      {
        red = green = 0;
      } else {
	red   = ((float)COL_RED(vpix)/(lum*1.71))*255.0;
	green = ((float)COL_GREEN(vpix)/(lum*3.35))*255.0;
      }

      /* calcul la probabilit que notre point appartienne */
      /*  l'histogramme pattern.                          */
      /* proba =  proba_pattern/proba_image                */      
      if(hist_img[256*green+red] == 0.0)
        proba = 0.0;
      else
        proba = hist_pattern[256*green+red]/hist_img[256*green+red];

      ecx += ((float)x-bar_x)*((float)x-bar_x)*proba;
      ecy += ((float)y-bar_y)*((float)y-bar_y)*proba;
	
      sum += proba;
    }

  ecx /= sum;
  ecx = sqrt(ecx);
  ecy /= sum;
  ecy = sqrt(ecy);

  /* on dessine une croix au niveau du barycentre */
  line_alpha_replace(img_res,bar_x-5,bar_y,bar_x+5,bar_y,RED);
  line_alpha_replace(img_res,bar_x,bar_y-5,bar_x,bar_y+5,RED);

  /* on dessine un rectangle autour du barycentre */
  /* et de la taille de l'cart type.             */
  line_alpha_replace(img_res,bar_x-ecx,bar_y-ecy,bar_x+ecx,bar_y-ecy,WHITE);
  line_alpha_replace(img_res,bar_x-ecx,bar_y+ecy,bar_x+ecx,bar_y+ecy,WHITE);
  line_alpha_replace(img_res,bar_x-ecx,bar_y-ecy,bar_x-ecx,bar_y+ecy,WHITE);
  line_alpha_replace(img_res,bar_x+ecx,bar_y-ecy,bar_x+ecx,bar_y+ecy,WHITE);

  /* libre l'histogramme de l'image */
  free(hist_img);

  /* on retourne l'image */
  return(img_res);
}
/**********************************************************************/

/*********************************************************************/
/* Egalize l'histogramme de l'intensit de la lumire pour           */
/* l'image psrc. Cela veut dire que l'image pdest comporte           */
/* autant de point pour chaque intensit de lumire aprs traitement */
void image_equalize(image *pdest, image *psrc)
{ int32 vx,vy;
  int i,j;
  float res_r, res_g, res_b, y, u, v;
  pix vpix;
  float histo_lum[256];
  float equalize[256];
  float s1 = 0.0, s2 = 0.0;
 
  /* si les prconditions ne sont pas remplies, on quitte */
  if((pdest == NULL) || (psrc == NULL) ||
     (psrc->width != pdest->width) || (psrc->height != pdest->height))
    return;

  /* initialise l'histogramme */
  for(i = 0; i < 256; i++)
    histo_lum[i] = 0.0;

  /* calcule l'histogramme de la lumire */
  for(vy = 0;vy < psrc->height; vy++)
  {
    for(vx = 0;vx < psrc->width; vx++)
    {
      vpix = get_pix(psrc, vx, vy);
      y = (float)COL_RED  (vpix)*0.299 +
          (float)COL_GREEN(vpix)*0.587 +
          (float)COL_BLUE (vpix)*0.114;
      histo_lum[(int)y]++;
    }
  }
  /* normalize l'histogramme */
  for(i = 0; i < 256; i++)
    histo_lum[i] /= psrc->height*psrc->height;

  /* calcule la fonction d'galisation */
  for(j = 0; j < 256; j++)
    s2 += histo_lum[j];

  for(i = 0; i < 256; i++)
  {
    s1 = 0.0;
    for(j = 0; j <= i; j++) 
      s1 += histo_lum[j];
    equalize[i] = 255.0*s1/s2;
  }

  for(vy = 0;vy < psrc->height; vy++)
  {
    for(vx = 0;vx < psrc->width; vx++)
    {
      vpix = get_pix(psrc, vx, vy);
      res_r = (float)COL_RED  (vpix);
      res_g = (float)COL_GREEN(vpix);
      res_b = (float)COL_BLUE (vpix);

      /* converti en luminence, chrominance */
      y = 0.299*res_r + 0.587*res_g + 0.114*res_b;
      u = res_r - y;
      v = res_b - y;
      
      /* applique l'galisation sur l'intensit */
      y = equalize[(int)y];

      /* converti en RGB */
      res_r = u + y;
      res_b = v + y;
      res_g = y*1.7 - res_r*0.509 - res_b*0.194;

      /* on sature le rsultat */
      res_r = (res_r > 255.0)? 255.0 : ((res_r < 0.0)? 0.0:res_r);
      res_g = (res_g > 255.0)? 255.0 : ((res_g < 0.0)? 0.0:res_g);
      res_b = (res_b > 255.0)? 255.0 : ((res_b < 0.0)? 0.0:res_b);

      /* on place le rsultat dans l'image destination */      
      put_pix_alpha_replace(pdest, vx, vy, COL(
        (uint8)res_r,(uint8)res_g,(uint8)res_b));
    }
  }
}
/*********************************************************************/

/********************************************************/
/* Calcul le laplacien en chaque points. Le laplacien   */
/* est calcul pour chaque couleur. psrc est l'image    */
/* d'origine et pdest est l'image destination. psrc et  */
/* pdest doivent avoir la mme taille.                  */
void image_laplacien(image *pdest, image *psrc)
{ int32 vx,vy;
  int i,j;
  float res_r, res_g, res_b;
  pix vpix;
  /* filtre de calcul du laplacien */
  float filter_laplacien[3][3] = {{1.0, 1.0,1.0},
                                  {1.0,-8.0,1.0},
		                  {1.0, 1.0,1.0}};

  /* si les prconditions ne sont pas remplies, on quitte */
  if((pdest == NULL) || (psrc == NULL) ||
     (psrc->width != pdest->width) || (psrc->height != pdest->height))
    return;

  for(vy = 0;vy < psrc->height; vy++)
  {
    for(vx = 0;vx < psrc->width; vx++)
    {
      res_r = 0.0;
      res_g = 0.0;
      res_b = 0.0;
      for(j=0;j<3;j++)
        for(i=0;i<3;i++)
	{
  	  vpix = get_pix(psrc, vx-(3>>1)+i, vy-(3>>1)+j);
	  res_r += (float)COL_RED  (vpix) * filter_laplacien[j][i];
	  res_g += (float)COL_GREEN(vpix) * filter_laplacien[j][i];
	  res_b += (float)COL_BLUE (vpix) * filter_laplacien[j][i];
	}

      res_r += 128.0;
      res_g += 128.0;
      res_b += 128.0;

      /* on sature les valeurs */
      res_r = (res_r > 255.0)? 255.0 : ((res_r < 0.0)? 0.0:res_r);
      res_g = (res_g > 255.0)? 255.0 : ((res_g < 0.0)? 0.0:res_g);
      res_b = (res_b > 255.0)? 255.0 : ((res_b < 0.0)? 0.0:res_b);

      /* on place le rsultat dans pdest */      
      put_pix_alpha_replace(pdest, vx, vy, COL(
        (uint8)res_r,(uint8)res_g,(uint8)res_b));
    }
  }
}
/********************************************************/

/********************************************************/
/* Calcul le laplacien en chaque points. Le laplacien   */
/* est calcul pour chaque couleur. psrc est l'image    */
/* d'origine et pdest est l'image destination. psrc et  */
/* pdest doivent avoir la mme taille.                  */
void image_laplacien_edge(image *pdest, image *psrc)
{ int32 vx,vy;
  int i,j;
  float res_r, res_g, res_b;
  pix vpix;
  /* filtre de calcul du laplacien */
  float filter_laplacien[3][3] = {{1.0, 1.0,1.0},
                                  {1.0,-8.0,1.0},
		                  {1.0, 1.0,1.0}};
  float *red, *green, *blue;
  boolean border;

  /* si les prconditions ne sont pas remplies, on quitte */
  if((pdest == NULL) || (psrc == NULL) ||
     (psrc->width != pdest->width) || (psrc->height != pdest->height))
    return;

  if((red = (float *)malloc(psrc->width*psrc->height*sizeof(float))) == NULL)
  { perror("malloc failed "); exit(1); }

  if((green = (float *)malloc(psrc->width*psrc->height*sizeof(float))) == NULL)
  { perror("malloc failed "); exit(1); }
  
  if((blue = (float *)malloc(psrc->width*psrc->height*sizeof(float))) == NULL)
  { perror("malloc failed "); exit(1); }

  for(vy = 0;vy < psrc->height; vy++)
  {
    for(vx = 0;vx < psrc->width; vx++)
    {
      res_r = 0.0;
      res_g = 0.0;
      res_b = 0.0;
      for(j=0;j<3;j++)
        for(i=0;i<3;i++)
	{
  	  vpix = get_pix(psrc, vx-(3>>1)+i, vy-(3>>1)+j);
	  res_r += (float)COL_RED  (vpix) * filter_laplacien[j][i];
	  res_g += (float)COL_GREEN(vpix) * filter_laplacien[j][i];
	  res_b += (float)COL_BLUE (vpix) * filter_laplacien[j][i];
	}
      red  [vy*psrc->width+vx] = res_r;
      green[vy*psrc->width+vx] = res_g;
      blue [vy*psrc->width+vx] = res_b;
    }
  }

  for(vy = 1;vy < psrc->height-1; vy++)
  {
    for(vx = 1;vx < psrc->width-1; vx++)
    {
      border = FALSE;

      for(j=0;j<3;j++)
        for(i=0;i<3;i++)
	{
          if((red[vy*psrc->width+vx] >= 0.0) &&
             (red[(vy-(3>>1)+j)*psrc->width+(vx-(3>>1)+i)] <= 0.0))
          {
            border = TRUE;
          }
          
          if((green[vy*psrc->width+vx] >= 0.0) &&
             (green[(vy-(3>>1)+j)*psrc->width+(vx-(3>>1)+i)] <= 0.0))
          {
            border = TRUE;
          }
          
          if((blue[vy*psrc->width+vx] >= 0.0) &&
             (blue[(vy-(3>>1)+j)*psrc->width+(vx-(3>>1)+i)] <= 0.0))
          {
            border = TRUE;
          }
          /* conserve en fonction d'un seuil arbitraire (120) */
          if(red[vy*psrc->width+vx]+green[vy*psrc->width+vx]+blue[vy*psrc->width+vx] < 120.0)
            border = FALSE;
	}
      if(border)
        put_pix_alpha_replace(pdest, vx, vy, WHITE);
      else
        put_pix_alpha_replace(pdest, vx, vy, BLACK);
    }
  }
}
/********************************************************/
