/****************************************************/
/* Pour cree une image a partir d'un fichier GIF    */
/* image_from_gif.c                                 */
/*                                                  */
/* Ecrit par : Daniel Lacroix (all rights reserved) */
/*                                                  */
/****************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>
#include <image_from_gif.h>

/********************************************************/
/* Ne supporte aucune extension                         */
/* Ne supporte pas non plus les fichiers gif avec       */
/* plusieurs images seul la premiere sera charge        */
/* Ne supporte pas les gifs en mode entrelaces          */
/* Ne supporte pas la transparence                      */
/*                                                      */
/* ATTENTION, il faut une license de la part de         */
/* Unisys car la compression LZW utilise dans les       */
/* gifs est brevete. Cette license n'est necessaire     */
/* que pour les logiciels payant (shareware y compris). */
/********************************************************/
 
#define	MAX_LZW_BITS	12	          /* taille maximum du code LZW  */
#define LZW_TABLE_SIZE	(1<<MAX_LZW_BITS) /* nombre de code LZW possible */

typedef struct {
  char  signature[6];     /* GIF87a, GIF89a*/
  uint8 screen_width[2];  /* LSB d'abord */
  uint8 screen_height[2]; /* LSB d'abord */
  uint8 info;
  uint8 background_color;
  uint8 zero;
} EnTete;

typedef struct {
  uint8 red,green,blue;
} Palette;

typedef struct {
  uint8 image_left[2]; /* LSB d'abord */
  uint8 image_top[2];
  uint8 image_width[2];
  uint8 image_height[2];
  uint8 info;
} ImageDescriptor;

/* structure pour la decompression LZW */
typedef struct {
  uint16 *code_prefix;	/* table des prefixes des codes */
  uint8  *code_suffix;	/* table des suffixes des codes */
  uint8  *code_stack;   /* tas pour decoder les codes   */  
  uint8  *sp;           /* pointeur de pile             */
  
  uint8   LZW_code[256+4]; /* bloc de code LZW sur lequel on travaille */
  uint16  nb_bytes;        /* nombre d'octets dans LZW_code            */
  uint16  nb_bits;         /* nombre de bits dans LZW_code             */
  uint16  current_bit;	   /* prochain bit a lire dans LZW_code        */
  boolean data_end;        /* TRUE si il n'y a plus de bloc de donnees */
                           /* a lire dans le fichier GIF               */

  int input_code_size;      /* codesize given in GIF file     */
  int clear_code,end_code;	/* values for Clear and End codes */

  int code_size;		/* current actual code size */
  int limit_code;		/* 2^code_size */
  int max_code;			/* first unused code value */
  boolean first_time;		/* flags first call to NextIndex */

  /* Private state for NextIndex */
  int oldcode;			/* previous LZW symbol */
  int firstcode;		/* first byte of oldcode's expansion */

  char *buf_pos;   /* pointeur qui ce deplace sur l'image */
  int  size_left;  /* nombre d'octets restant dans l'image a partir de buf_pos */
} LZW_Info;


void ReInitLZW(LZW_Info *pLZW_Info)
/* (Re)initialize LZW state; shared code for startup and Clear processing */
{
  pLZW_Info->code_size  = pLZW_Info->input_code_size + 1;
  pLZW_Info->limit_code = pLZW_Info->clear_code << 1;	/* 2^code_size */
  pLZW_Info->max_code   = pLZW_Info->clear_code + 2;	/* first unused code value */
  pLZW_Info->sp         = pLZW_Info->code_stack;	/* init stack to empty */
}

void InitLZWCode(LZW_Info *pLZW_Info)
/* Initialize for a series of NextIndex (and hence NextCode) calls */
{
  /* NextCode initialization */
  pLZW_Info->nb_bytes    = 2;		/* make safe to "recopy last two bytes" */
  pLZW_Info->nb_bits     = 0;   /* rien dans LZW_code */
  pLZW_Info->current_bit = 0;		/* force buffer load on first call */
  pLZW_Info->data_end = FALSE;

  /* NextIndex initialization: */
  /* compute special code values (note that these do not change later) */
  pLZW_Info->clear_code = 1 << pLZW_Info->input_code_size;
  pLZW_Info->end_code   = pLZW_Info->clear_code + 1;
  pLZW_Info->first_time = TRUE;
  ReInitLZW(pLZW_Info);
}

/************************************************/
/* Renvoi le nombre d'octets d'un bloc ou -1 si */
/* une erreur est apparue. buf est alors rempli */
/* avec les donnees. Si le nombre d'octets      */
/* retourne est 0 alors il n'y a plus de blocs. */
int16 NextBlock(LZW_Info *pLZW_Info, char *buf)
{ int16 taille;
  /* verifie que l'on a assez de donnees pour lire la taille */
  pLZW_Info->size_left -= 1;
  if(pLZW_Info->size_left < 0) return(-1);
  /* si la taille du bloc est 0 alors il n'y a plus de blocs */
  if((taille = *pLZW_Info->buf_pos++ & 0xFF) == 0) return(0);
  /* verifie que l'on a assez de donnees en memoire */
  pLZW_Info->size_left -= taille;
  if(pLZW_Info->size_left < 0) return(-1);
  /* recopie dans buf les donnees d'un bloc */
  memcpy(buf,pLZW_Info->buf_pos,taille);
  pLZW_Info->buf_pos += taille;
  /* renvoi la taille du bloc de donnees */
  return(taille);
}
/************************************************/

/*************************************************************/
/* Renvoi le prochain code LZW ou -1 si il y a une erreur.   */
/* Pour que cette procedure fonctionne il faut que les codes */
/* fassent moins de 16 bits.                                 */
int16 NextCode(LZW_Info *pLZW_Info)
{
  uint32 buf;
  int16  count, pos, valeur;

  while((pLZW_Info->current_bit + pLZW_Info->code_size) > pLZW_Info->nb_bits) {
    /* recharge LZW_code */
    if(pLZW_Info->data_end) {
      printf("GIF : pas assez de donnnees.\n");
      /* simule quelque chose d'utile */
      return(pLZW_Info->end_code);
    }
    /* conserve les 2 derniers octets. On doit avoir code_size <= 16 */
    pLZW_Info->LZW_code[0] = pLZW_Info->LZW_code[pLZW_Info->nb_bytes-2];
    pLZW_Info->LZW_code[1] = pLZW_Info->LZW_code[pLZW_Info->nb_bytes-1];
    /* Charge un bloc d'octets */
    if((count = NextBlock(pLZW_Info, &pLZW_Info->LZW_code[2])) == -1) return(-1);
    /* si on est a la fin des blocs */
    if(count == 0) {
      pLZW_Info->data_end = TRUE;
      printf("GIF : pas assez de donnnees.\n");
      /* simule quelque chose d'utile */
      return(pLZW_Info->end_code);
    }
    /* met a jour les compteurs */
    pLZW_Info->current_bit = (pLZW_Info->current_bit - pLZW_Info->nb_bits) + 16;
    pLZW_Info->nb_bytes    = 2 + count;
    pLZW_Info->nb_bits     = pLZW_Info->nb_bytes << 3; /* mul par 8 */
  }

  /* prepare les prochains 24 bits */
  pos = pLZW_Info->current_bit >> 3; /* octet contenant current_bit */

  buf =  ((uint8)pLZW_Info->LZW_code[pos+2]<<16) + 
         ((uint8)pLZW_Info->LZW_code[pos+1]<< 8) +
	 ((uint8)pLZW_Info->LZW_code[pos]);

  /* aligne les bits utiles a droite, et ne conserve que cela */
  buf >>= (pLZW_Info->current_bit & 0x7);
  valeur =  buf & ((1 << pLZW_Info->code_size) - 1);
  
  pLZW_Info->current_bit += pLZW_Info->code_size;
  /* renvoi le code LZW */
  return(valeur);
}
/*************************************************************/

/****************************************************/
/* Renvoi le prochain index (pixel) dans la palette */
/* ou -1 si il y a eu une erreur                    */
int16 NextIndex(LZW_Info *pLZW_Info)
{
  int16 code;     /* code sur lequel on travaille */
  int16 tmp_code;

  if(pLZW_Info->first_time) {
    /* cela ne sera plus jamais la premiere fois */
    pLZW_Info->first_time = FALSE;
    /* recherche le premier code valid */
    do {
      /* si erreur, on quitte */
      if((code = NextCode(pLZW_Info)) == -1) return(-1);
    } while (code == pLZW_Info->clear_code);
    /* verifie qu'il s'agit bien d'une donnee valide */
    if(code > pLZW_Info->clear_code) {
      printf("GIF : donnees invalides.\n");
      /* tant pis, on remplace par quelque chose de possible */
      code = 0;
    }
    /* on conserve le code */
    pLZW_Info->firstcode = pLZW_Info->oldcode = code;
    return code;
  }
  
  /* si il y a des codes precedemment empiles, on les retournes */
  if(pLZW_Info->sp > pLZW_Info->code_stack)
    return((int16) *(--pLZW_Info->sp));

  /* on lit un nouveau code */
  if((code = NextCode(pLZW_Info)) == -1) return(-1);

  /* si c'est un code de reinitialisation, on reinitialise */
  /* et on renvoi le prochain code valide.                  */
  if(code == pLZW_Info->clear_code) {
    ReInitLZW(pLZW_Info);
    do {
      if((code = NextCode(pLZW_Info)) == -1) return(-1);
    } while (code == pLZW_Info->clear_code);
    /* verifie qu'il s'agit bien d'une donnee valide */
    if(code > pLZW_Info->clear_code) {
      printf("GIF : donnees invalides.\n");
      /* tant pis, on remplace par quelque chose de possible */
      code = 0;
    }
    /* on conserve le code */
    pLZW_Info->firstcode = pLZW_Info->oldcode = code;
    return code;
  }

  /* si on tombe sur le code de fin */
  if(code == pLZW_Info->end_code) {
    /* pour etre sur que l'on n'ira pas lire des blocs au dela */
    pLZW_Info->data_end = TRUE;
    printf("GIF : pas assez de donnnees.\n");
    /* renvoi une valeur possible (bidon) */
    return(0);
  }

  tmp_code = code;

  /* cas ou le code n'est pas encore definie */
  if(code >= pLZW_Info->max_code) {
    /* si code = max_code c'est bon, sinon c'est mauvais */
    if(code > pLZW_Info->max_code) {
      printf("GIF : donnees invalides.\n");
      tmp_code = 0; /* pour eviter les boucles dans la table des codes */
    }
    /* ce code sera definie comme oldcode/firstcode */
    *(pLZW_Info->sp++) = (uint8)pLZW_Info->firstcode;
    code = pLZW_Info->oldcode;
  }

  /* si c'est un code, on le decompresse dans la pile */
  while(code >= pLZW_Info->clear_code) {
    *(pLZW_Info->sp++) = pLZW_Info->code_suffix[code]; /* le suffix est un octets */
    code = pLZW_Info->code_prefix[code];      /* le preffix est un autre code LZW */
  }
  /* maintenant code est une donnee */
  pLZW_Info->firstcode = code;	/* on garde de cote */

  /* si il y a de la place dans la table */
  if((code = pLZW_Info->max_code) < LZW_TABLE_SIZE) {
    /* definie un nouveau code = ancien code + preffix du nouveau code etandu */
    pLZW_Info->code_prefix[code] = pLZW_Info->oldcode;
    pLZW_Info->code_suffix[code] = (uint8)pLZW_Info->firstcode;
    pLZW_Info->max_code++;
    /* faut-il augmenter la taille des codes ? */
    if((pLZW_Info->max_code >= pLZW_Info->limit_code) &&
       (pLZW_Info->code_size < MAX_LZW_BITS)) {
      pLZW_Info->code_size++;
      pLZW_Info->limit_code = 1 << pLZW_Info->code_size;
    }
  }
  
  pLZW_Info->oldcode = tmp_code; /* sauve le dernier code */
  return pLZW_Info->firstcode;	 /* renvoi le premier octets du code decompresse */
}
/****************************************************/


/****************************************/
/* Renvoi l'image ou NULL si impossible */
image *image_new_from_gif(char *file_name)
{ int    handle;
  int    file_size, size_left;
  uint16 block_size;
  char   *buf,*buf_pos;
  image  *vimage;
  int    i,x,y;
  int32  width,height;
  EnTete *vEnTete; /* l'entete des fichiers GIF */
  ImageDescriptor *vImageDescriptor;
  Palette *global_palette = NULL;
  uint16  global_palette_nb_color = 0;
  Palette *local_palette = NULL;
  uint16  local_palette_nb_color = 0;
  Palette *current_palette = NULL;
  int16   index;
  LZW_Info vLZW_Info;

  /* charge toute l'image en memoire d'un coup              */
  /* buf contient alors le fichier gif, file_size sa taille */
  if((handle = open(file_name,O_RDONLY)) == -1) return(NULL);
  if((file_size = lseek(handle, 0, SEEK_END)) == -1)
  { close(handle); return(NULL); }
  if(lseek(handle, 0, SEEK_SET) == -1)
  { close(handle); return(NULL); }
  if((file_size > MAX_GIF_FILE_SIZE) || 
     (file_size < sizeof(EnTete)))
  { close(handle); return(NULL); }
  if((buf = (char *)malloc(file_size)) == NULL)
  { perror("malloc failed "); exit(1); }
  if(read(handle, buf, file_size) == -1)
  { free(buf); close(handle); return(NULL); }
  close(handle);

  /* size_left permet de savoir combien d'octets non pas ete traites */
  /* pour ne pas sortir de la memoire alloue pour l'image            */
  size_left = file_size;

  /* test si il y a assez pour lire l'entete avant de la lire */
  size_left -= sizeof(EnTete);
  if(size_left < 0) { free(buf); return(NULL);}
    
  vEnTete = (EnTete *)buf;
  buf_pos = (char *)(buf + sizeof(EnTete));

  /* ce n'est pas un fichier gif */
  if((vEnTete->signature[0] != 'G') || (vEnTete->signature[1] != 'I') ||
     (vEnTete->signature[2] != 'F'))
  { free(buf); return(NULL); }

  /* il y a une palette global */
  if(vEnTete->info >> 7)
  { 
    global_palette = (Palette *)(buf_pos); 
    global_palette_nb_color = 1 << ((vEnTete->info & 0x7) + 1);
    
    size_left -= (global_palette_nb_color<<1) + global_palette_nb_color;
    if(size_left < 0) { free(buf); return(NULL);}
    
    buf_pos += (global_palette_nb_color<<1) + global_palette_nb_color;
    current_palette = global_palette;
  }

  size_left -= 1;
  if(size_left < 0) { free(buf); return(NULL); }
  
  while(*buf_pos != ',')
  {
    if(*buf_pos == ';')
    { printf("GIF : aucune image trouve\n"); free(buf); return(NULL); }
    /* les extensions sont ignorees */
    if(*buf_pos == '!')
    { printf("GIF : une extension ignore\n");
    
      size_left -= 3;
      if(size_left < 0) { free(buf); return(NULL); }

      buf_pos += 2;
      block_size = *buf_pos & 0xFF;

      while(block_size != 0)
      {
        size_left -= block_size;
        if(size_left < 0) { free(buf); return(NULL); }
            
        buf_pos += block_size + 1;
        block_size = *buf_pos & 0xFF;
      }
    }
    size_left -= 1;
    if(size_left < 0) { free(buf); return(NULL); }

    buf_pos++;
  }
    
  size_left -= 1;
  if(size_left < 0) { free(buf); return(NULL); }
  buf_pos++;

  size_left -= sizeof(ImageDescriptor);
  if(size_left < 0) { free(buf); return(NULL); }

  /* on a trouve une image */  
  vImageDescriptor = (ImageDescriptor *)buf_pos;
  width  = (vImageDescriptor->image_width[1] <<8) + vImageDescriptor->image_width[0];
  height = (vImageDescriptor->image_height[1]<<8) + vImageDescriptor->image_height[0];

  buf_pos += sizeof(ImageDescriptor);

  /* Il y a une palette local */
  if(vImageDescriptor->info >> 7)
  {
    local_palette = (Palette *)buf_pos;
    local_palette_nb_color = 1<<((vImageDescriptor->info & 0x7) + 1);

    size_left -= (local_palette_nb_color<<1) + local_palette_nb_color;
    if(size_left < 0) { free(buf); return(NULL); }

    buf_pos += (local_palette_nb_color<<1) + local_palette_nb_color;
    current_palette = local_palette;
  }

  /* pas de palette pour l'image, on quitte */
  if(current_palette == NULL) { free(buf); return(NULL); }

  size_left -= 1;
  if(size_left < 0) { free(buf); return(NULL); }

  vLZW_Info.input_code_size = *buf_pos++;
  
  if((vLZW_Info.input_code_size < 2) || (vLZW_Info.input_code_size >= MAX_LZW_BITS)) { free(buf); return(NULL); }

  if((vLZW_Info.code_prefix  = (uint16 *)malloc(LZW_TABLE_SIZE*sizeof(uint16))) == NULL)
  { perror("malloc failed "); exit(1); }
  if((vLZW_Info.code_suffix  = (uint8  *)malloc(LZW_TABLE_SIZE*sizeof(uint8))) == NULL)
  { perror("malloc failed "); exit(1); }
  if((vLZW_Info.code_stack = (uint8  *)malloc(LZW_TABLE_SIZE*sizeof(uint8))) == NULL)
  { perror("malloc failed "); exit(1); }

  InitLZWCode(&vLZW_Info);
  
  vLZW_Info.buf_pos   = buf_pos;
  vLZW_Info.size_left = size_left;
    
  /* image en mode entrelace */
  if(vImageDescriptor->info & 0x40)
  {
    /* cree l'image */
    vimage = image_new(width,height);
        
    /* premiere  passe */
    for(y = 0; y < height; y += 8)
    {
      for(x = 0; x < width; x++)
      {
        if((index = NextIndex(&vLZW_Info)) == -1)
        { image_free(vimage); free(vLZW_Info.code_prefix);
          free(vLZW_Info.code_suffix); free(vLZW_Info.code_stack);
          free(buf); return(NULL);
        }
	
        vimage->buf[y*width + x] = COL(
          current_palette[index].red,
          current_palette[index].green,
          current_palette[index].blue);
      }
    }
    /* deuxieme  passe */
    for(y = 4; y < height; y += 8)
      for(x = 0; x < width; x++)
      {
        if((index = NextIndex(&vLZW_Info)) == -1)
        { image_free(vimage); free(vLZW_Info.code_prefix);
          free(vLZW_Info.code_suffix); free(vLZW_Info.code_stack);
          free(buf); return(NULL);
        }
	
        vimage->buf[y*width + x] = COL(
          current_palette[index].red,
          current_palette[index].green,
          current_palette[index].blue);
      }
    /* troisieme passe */
    for(y = 2; y < height; y += 4)
      for(x = 0; x < width; x++)
      {
        if((index = NextIndex(&vLZW_Info)) == -1)
        { image_free(vimage); free(vLZW_Info.code_prefix);
          free(vLZW_Info.code_suffix); free(vLZW_Info.code_stack);
          free(buf); return(NULL);
        }
	
        vimage->buf[y*width + x] = COL(
          current_palette[index].red,
          current_palette[index].green,
          current_palette[index].blue);
      }
    /* quatrieme passe */
    for(y = 1; y < height; y += 2)
      for(x = 0; x < width; x++)
      {
        if((index = NextIndex(&vLZW_Info)) == -1)
        { image_free(vimage); free(vLZW_Info.code_prefix);
          free(vLZW_Info.code_suffix); free(vLZW_Info.code_stack);
          free(buf); return(NULL);
        }
	
        vimage->buf[y*width + x] = COL(
          current_palette[index].red,
          current_palette[index].green,
          current_palette[index].blue);
      }

  } else {
    /* cree l'image */
    vimage = image_new(width,height);
    /* rempli l'image */
    for(i = 0; i < width*height; i++)
    {
      if((index = NextIndex(&vLZW_Info)) == -1)
      { image_free(vimage); free(vLZW_Info.code_prefix);
        free(vLZW_Info.code_suffix); free(vLZW_Info.code_stack);
        free(buf); return(NULL);
      }

      vimage->buf[i] = COL(
        current_palette[index].red,
        current_palette[index].green,
        current_palette[index].blue);
    }
  }
  
  /* libere la memoire alloue */
  free(vLZW_Info.code_prefix);  free(vLZW_Info.code_suffix);
  free(vLZW_Info.code_stack); free(buf);
  /* renvoi l'image */
  return(vimage);
}
/****************************************/

/****************************************/
/* Renvoi l'image ou NULL si impossible */
image *image_new_from_gif_stream(char *file_name)
{ int    handle;
  int    file_size, size_left;
  uint16 block_size;
  char   *buf,*buf_pos;
  image  *vimage;
  int    i,x,y;
  int32  width,height;
  EnTete *vEnTete; /* l'entete des fichiers GIF */
  ImageDescriptor *vImageDescriptor;
  Palette *global_palette = NULL;
  uint16  global_palette_nb_color = 0;
  Palette *local_palette = NULL;
  uint16  local_palette_nb_color = 0;
  Palette *current_palette = NULL;
  int16   index;
  LZW_Info vLZW_Info;

  /* charge toute l'image en memoire d'un coup              */
  /* buf contient alors le fichier gif, file_size sa taille */
  if((handle = open(file_name,O_RDONLY)) == -1) return(NULL);
  if((file_size = lseek(handle, 0, SEEK_END)) == -1)
  { close(handle); return(NULL); }
  if(lseek(handle, 0, SEEK_SET) == -1)
  { close(handle); return(NULL); }
  if((file_size > MAX_GIF_FILE_SIZE) || 
     (file_size < sizeof(EnTete)))
  { close(handle); return(NULL); }
  if((buf = (char *)malloc(file_size)) == NULL)
  { perror("malloc failed "); exit(1); }
  if(read(handle, buf, file_size) == -1)
  { free(buf); close(handle); return(NULL); }
  close(handle);

  /* size_left permet de savoir combien d'octets non pas ete traites */
  /* pour ne pas sortir de la memoire alloue pour l'image            */
  size_left = file_size;

  /* test si il y a assez pour lire l'entete avant de la lire */
  size_left -= sizeof(EnTete);
  if(size_left < 0) { free(buf); return(NULL);}
    
  vEnTete = (EnTete *)buf;
  buf_pos = (char *)(buf + sizeof(EnTete));

  /* ce n'est pas un fichier gif */
  if((vEnTete->signature[0] != 'G') || (vEnTete->signature[1] != 'I') ||
     (vEnTete->signature[2] != 'F'))
  { free(buf); return(NULL); }

  /* il y a une palette global */
  if(vEnTete->info >> 7)
  { 
    global_palette = (Palette *)(buf_pos); 
    global_palette_nb_color = 1 << ((vEnTete->info & 0x7) + 1);
    
    size_left -= (global_palette_nb_color<<1) + global_palette_nb_color;
    if(size_left < 0) { free(buf); return(NULL);}
    
    buf_pos += (global_palette_nb_color<<1) + global_palette_nb_color;
    current_palette = global_palette;
  }

  size_left -= 1;
  if(size_left < 0) { free(buf); return(NULL); }
  
  while(*buf_pos != ',')
  {
    if(*buf_pos == ';')
    { printf("GIF : aucune image trouve\n"); free(buf); return(NULL); }
    /* les extensions sont ignorees */
    if(*buf_pos == '!')
    { printf("GIF : une extension ignore\n");
    
      size_left -= 3;
      if(size_left < 0) { free(buf); return(NULL); }

      buf_pos += 2;
      block_size = *buf_pos & 0xFF;

      while(block_size != 0)
      {
        size_left -= block_size;
        if(size_left < 0) { free(buf); return(NULL); }
            
        buf_pos += block_size + 1;
        block_size = *buf_pos & 0xFF;
      }
    }
    size_left -= 1;
    if(size_left < 0) { free(buf); return(NULL); }

    buf_pos++;
  }
    
  size_left -= 1;
  if(size_left < 0) { free(buf); return(NULL); }
  buf_pos++;

  size_left -= sizeof(ImageDescriptor);
  if(size_left < 0) { free(buf); return(NULL); }

  /* on a trouve une image */  
  vImageDescriptor = (ImageDescriptor *)buf_pos;
  width  = (vImageDescriptor->image_width[1] <<8) + vImageDescriptor->image_width[0];
  height = (vImageDescriptor->image_height[1]<<8) + vImageDescriptor->image_height[0];

  buf_pos += sizeof(ImageDescriptor);

  /* Il y a une palette local */
  if(vImageDescriptor->info >> 7)
  {
    local_palette = (Palette *)buf_pos;
    local_palette_nb_color = 1<<((vImageDescriptor->info & 0x7) + 1);

    size_left -= (local_palette_nb_color<<1) + local_palette_nb_color;
    if(size_left < 0) { free(buf); return(NULL); }

    buf_pos += (local_palette_nb_color<<1) + local_palette_nb_color;
    current_palette = local_palette;
  }

  /* pas de palette pour l'image, on quitte */
  if(current_palette == NULL) { free(buf); return(NULL); }

  size_left -= 1;
  if(size_left < 0) { free(buf); return(NULL); }

  vLZW_Info.input_code_size = *buf_pos++;
  
  if((vLZW_Info.input_code_size < 2) || (vLZW_Info.input_code_size >= MAX_LZW_BITS)) { free(buf); return(NULL); }

  if((vLZW_Info.code_prefix  = (uint16 *)malloc(LZW_TABLE_SIZE*sizeof(uint16))) == NULL)
  { perror("malloc failed "); exit(1); }
  if((vLZW_Info.code_suffix  = (uint8  *)malloc(LZW_TABLE_SIZE*sizeof(uint8))) == NULL)
  { perror("malloc failed "); exit(1); }
  if((vLZW_Info.code_stack = (uint8  *)malloc(LZW_TABLE_SIZE*sizeof(uint8))) == NULL)
  { perror("malloc failed "); exit(1); }

  InitLZWCode(&vLZW_Info);
  
  vLZW_Info.buf_pos   = buf_pos;
  vLZW_Info.size_left = size_left;
    
  /* image en mode entrelace */
  if(vImageDescriptor->info & 0x40)
  {
    /* cree l'image */
    vimage = image_new(width,height);
        
    /* premiere  passe */
    for(y = 0; y < height; y += 8)
    {
      for(x = 0; x < width; x++)
      {
        if((index = NextIndex(&vLZW_Info)) == -1)
        { image_free(vimage); free(vLZW_Info.code_prefix);
          free(vLZW_Info.code_suffix); free(vLZW_Info.code_stack);
          free(buf); return(NULL);
        }
	
        vimage->buf[y*width + x] = COL(
          current_palette[index].red,
          current_palette[index].green,
          current_palette[index].blue);
      }
    }
    /* deuxieme  passe */
    for(y = 4; y < height; y += 8)
      for(x = 0; x < width; x++)
      {
        if((index = NextIndex(&vLZW_Info)) == -1)
        { image_free(vimage); free(vLZW_Info.code_prefix);
          free(vLZW_Info.code_suffix); free(vLZW_Info.code_stack);
          free(buf); return(NULL);
        }
	
        vimage->buf[y*width + x] = COL(
          current_palette[index].red,
          current_palette[index].green,
          current_palette[index].blue);
      }
    /* troisieme passe */
    for(y = 2; y < height; y += 4)
      for(x = 0; x < width; x++)
      {
        if((index = NextIndex(&vLZW_Info)) == -1)
        { image_free(vimage); free(vLZW_Info.code_prefix);
          free(vLZW_Info.code_suffix); free(vLZW_Info.code_stack);
          free(buf); return(NULL);
        }
	
        vimage->buf[y*width + x] = COL(
          current_palette[index].red,
          current_palette[index].green,
          current_palette[index].blue);
      }
    /* quatrieme passe */
    for(y = 1; y < height; y += 2)
      for(x = 0; x < width; x++)
      {
        if((index = NextIndex(&vLZW_Info)) == -1)
        { image_free(vimage); free(vLZW_Info.code_prefix);
          free(vLZW_Info.code_suffix); free(vLZW_Info.code_stack);
          free(buf); return(NULL);
        }
	
        vimage->buf[y*width + x] = COL(
          current_palette[index].red,
          current_palette[index].green,
          current_palette[index].blue);
      }

  } else {
    /* cree l'image */
    vimage = image_new(width,height);
    /* rempli l'image */
    for(i = 0; i < width*height; i++)
    {
      if((index = NextIndex(&vLZW_Info)) == -1)
      { image_free(vimage); free(vLZW_Info.code_prefix);
        free(vLZW_Info.code_suffix); free(vLZW_Info.code_stack);
        free(buf); return(NULL);
      }

      vimage->buf[i] = COL(
        current_palette[index].red,
        current_palette[index].green,
        current_palette[index].blue);
    }
  }
  
  /* libere la memoire alloue */
  free(vLZW_Info.code_prefix);  free(vLZW_Info.code_suffix);
  free(vLZW_Info.code_stack); free(buf);
  /* renvoi l'image */
  return(vimage);
}
/****************************************/
