/****h* ROBODoc/HeaderTypes
 * FUNCTION
 *    Headers come in different types.  This module defines what kind
 *    of headertypes ROBODoc recognizes, and contains functions to add
 *    new headertypes and to compare headertypes.  All the headertypes
 *    are stored in an array, header_type_lookup_table.
 ******
 */

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include "headertypes.h"
#include "util.h"

/****v* HeaderTypes/header_type_lookup_table
 * FUNCTION
 *   A lookup table for all the header types that ROBODoc recognizes.
 *   At the moment is has about 127 entries.  About as many as there
 *   are characters in the standard ASCII set.  The first 32 entries
 *   can be used for special purposes.
 *
 *   Two of them are use:
 *     HT_MASTERINDEXTYPE
 *   and
 *     HT_SOURCEHEADERTYPE
 *
 *   HT_MASTERINDEXTYPE is a wildcard type. All headertypes match this
 *   type.  This is used to collect all the headers for the
 *   masterindex.
 *
 *   HT_SOURCEHEADERTYPE is used to pretend that the name of
 *   a sourcefile is a kind of header.  This makes it possible to
 *   include the names of the source files in the master index.
 * SOURCE
 */

struct RB_HeaderType header_type_lookup_table[ MAX_HEADER_TYPE + 1 ] =
{
    { '\0', NULL, NULL },
    {  HT_SOURCEHEADERTYPE, "Sourcefiles" , "robo_sourcefiles" },
    {  HT_MASTERINDEXTYPE,  "Index" , "masterindex" },   /* no robo_ prefix for backwards compatibility */
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { '\0', NULL, NULL },
    { ' ', NULL, NULL },
    { '!', NULL, NULL },
    { '"', NULL, NULL },
    { '#', NULL, NULL },
    { '$', NULL, NULL },
    { '%', NULL, NULL },
    { '&', NULL, NULL },
    { '\0', NULL, NULL },
    { '(', NULL, NULL },
    { ')', NULL, NULL },
    { '*', "Generics" ,  "robo_generics" },
    { '+', NULL, NULL },
    { ',', NULL, NULL },
    { '-', NULL, NULL },
    { '.', NULL, NULL },
    { '/', NULL, NULL },
    { '0', NULL, NULL },
    { '1', NULL, NULL },
    { '2', NULL, NULL },
    { '3', NULL, NULL },
    { '4', NULL, NULL },
    { '5', NULL, NULL },
    { '6', NULL, NULL },
    { '7', NULL, NULL },
    { '8', NULL, NULL },
    { '9', NULL, NULL },
    { ':', NULL, NULL },
    { ';', NULL, NULL },
    { '<', NULL, NULL },
    { '=', NULL, NULL },
    { '>', NULL, NULL },
    { '?', NULL, NULL },
    { '@', NULL, NULL },
    { 'A', NULL, NULL },
    { 'B', "Businessrules" ,    "robo_businessrules" },
    { 'C', "Contracts" ,        "robo_contracts" },
    { 'D', "Datasources" ,      "robo_datasources" },
    { 'E', "Ensure contracts",  "robo_ensure_contracts" },
    { 'F', NULL, NULL },
    { 'G', NULL, NULL },
    { 'H', NULL, NULL },
    { 'I', "Invariants",        "robo_invariants" },
    { 'J', NULL, NULL },
    { 'K', NULL, NULL },
    { 'L', NULL, NULL },
    { 'M', "Metadata", "robo_metadata" },
    { 'N', NULL, NULL },
    { 'O', NULL, NULL },
    { 'P', "Process",  "robo_processes" },
    { 'Q', NULL , NULL },
    { 'R', "Require contracts",  "robo_require_contracts" },
    { 'S', "Subjects" ,          "robo_subjects" },
    { 'T', NULL, NULL },
    { 'U', NULL, NULL },
    { 'V', NULL, NULL },
    { 'W', NULL, NULL },
    { 'X', NULL, NULL },
    { 'Y', NULL, NULL },
    { 'Z', NULL, NULL },
    { '[', NULL, NULL },
    { '\0', NULL, NULL },  /* Seperator /  */
    { ']', NULL, NULL },
    { '^', NULL, NULL },
    { '_', NULL, NULL },
    { '`', NULL, NULL },
    { 'a', NULL, NULL },
    { 'b', NULL, NULL },
    { 'c', "Classes",      "robo_classes" },
    { 'd', "Definitions",  "robo_definitions" },
    { 'e', "Exceptions",   "robo_exceptions" },
    { 'f', "Functions",    "robo_functions" },
    { 'g', NULL, NULL },
    { 'h', "Modules",      "robo_modules" },
    { '\0', NULL, NULL },    /* Internal header flag */
    { 'j', NULL, NULL },
    { 'k', NULL, NULL },
    { 'l', NULL, NULL },
    { 'm', "Methods",      "robo_methods" },
    { 'n', NULL, NULL },
    { 'o', NULL, NULL },
    { 'p', "Procedures",   "robo_procedures" },
    { 'q', NULL, NULL },
    { 'r', NULL, NULL },
    { 's', "Structures", "robo_strutures" },
    { 't', "Types",      "robo_types" },
    { 'u', "Unittest",   "robo_unittests" },
    { 'v', "Variables",  "robo_variables" },
    { 'w', "Warehouses", "robo_warehouses" },
    { 'x', NULL, NULL },
    { 'y', NULL, NULL },
    { 'z', NULL, NULL },
    { '{', NULL, NULL },
    { '|', NULL, NULL },
    { '}', NULL, NULL },
    { '~', NULL, NULL },
    { '\0', NULL, NULL },
};

/*****/


/****f* HeaderTypes/RB_AddHeaderType
 * FUNCTION
 *   Add a new headertype to the list of header type
 *   that robodoc recognizes.
 * RESULT
 *   * FALSE -- it is a new header type.
 *   * TRUE  -- header type already existed.
 * SOURCE
 */

int RB_AddHeaderType( unsigned int typeCharacter, char*  indexName, char* fileName )
{
    if ( header_type_lookup_table[ typeCharacter ].typeCharacter )
    {
        header_type_lookup_table[ typeCharacter ].indexName = indexName;
        header_type_lookup_table[ typeCharacter ].fileName = fileName;
    }
    else
    {
        RB_Panic( "The character %c is not allowed as a headertype character\n", 
                  typeCharacter );
    }
    /* Unused */
    return 0;
}

/*****/



/****f* HeaderTypes/RB_CompareHeaderTypes
 * FUNCTION
 *   Compare two header types and check if they are equal.  If one of
 *   the header types is a HT_MASTERINDEXTYPE the comparison is always
 *   TRUE.  (This to make sure that all headers appear in the Index.)
 * SYNOPSIS
 *    int RB_CompareHeaderTypes( 
 *    struct RB_HeaderType* ht1, struct RB_HeaderType* ht2 )
 * INPUTS
 *    o ht1 and ht2 -- the header types to compare.
 * RESULT
 *    o 0     -- header types are not equal
 *    o != 0  -- header type are equal
 * SOURCE
 */

int RB_CompareHeaderTypes( struct RB_HeaderType* ht1, struct RB_HeaderType* ht2 )
{
    assert( ht1 );
    assert( ht2 );
    return ( ht1->typeCharacter == HT_MASTERINDEXTYPE ) ||
           ( ht2->typeCharacter == HT_MASTERINDEXTYPE ) ||
           ( ht1->typeCharacter == ht2->typeCharacter );
}

/******/



/****f* HeaderTypes/RB_FindHeaderType
 * FUNCTION
 *   Return the header type that corresponds to the type character.
 * RESULT
 *   * 0  -- there is no such header type
 *   * pointer to the header type otherwise.
 * SOURCE
 */

struct RB_HeaderType* RB_FindHeaderType( unsigned char typeCharacter )
{
    struct RB_HeaderType*  headertype = 0;

    if ( typeCharacter < MAX_HEADER_TYPE )
    {
        headertype = &( header_type_lookup_table[ typeCharacter ] );
        if ( ( headertype->typeCharacter == typeCharacter ) &&
             ( headertype->indexName ) )
        {
            return headertype;
        }
    }
    return 0;
}

/*****/


/****f* HeaderTypes/RB_IsInternalHeader
 * FUNCTION
 *   Given the typeCharacter is this an internal header?
 * RESULT
 *   * TRUE  -- yes it is
 *   * FALSE -- no it is not
 * SOURCE
 */

int RB_IsInternalHeader( unsigned char type_character )
{
    return ( type_character == 'i' );
}

/*****/


#if 0
char* RB_GetIndexName( struct RB_HeaderType* ht1 )
{
    /* TODO  should be used to access indexName */
    return 0;
}
#endif
