/****h* ROBODoc/Items
 * FUNCTION
 *   This module contains functions that deal with items.  The
 *   documentation consists of headers, and headers contains one of
 *   more items.  Each item has a name and a body.  All possible items
 *   are listed in configuration.items.  A uses can specify that
 *   certain items are not to be added to the documentation.  These
 *   items are listed in configuration.ignore_items.
 * AUTHOR
 *   Frans Slothouber
 *******
 */

#include <stddef.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <assert.h>

#include "globals.h"
#include "robodoc.h"
#include "items.h"
#include "roboconfig.h"
#include "util.h"

/****v* Items/item_name_buffer
 * FUNCTION
 *   Stores the name of the last item that was found.
 * SOURCE
 */

#define MAX_ITEM_NAME_LENGTH 1024
char                item_name_buffer[MAX_ITEM_NAME_LENGTH];

/*****/

/* TODO Documentation */
char* RB_Get_Item_Name( void )
{
    return item_name_buffer;
}

/****f* Items/RB_Create_Item
 *
 * SOURCE
 */

struct RB_Item* RB_Create_Item( enum ItemType arg_item_type )
{
    struct RB_Item* item = malloc( sizeof( struct RB_Item ) );
    assert( item );

    item->next = 0;
    item->type = arg_item_type;
    item->begin_index = 0;
    item->end_index = 0;

    return item;
}

/*****/

/****f* Items/RB_Get_Item_Type [3.0b]
 * FUNCTION
 *   return the item_type represented by the given string.
 * SYNOPSIS
 *   int RB_Get_Item_Type( char *cmp_name )
 * INPUTS
 *   char *cmp_name -- item_name to evaluate
 * RESULT
 *   int            -- the right item_type or NO_ITEM
 * SOURCE
 */

int
RB_Get_Item_Type( char *cmp_name )
{
    unsigned int        item_type;

    assert( configuration.items.number );
    for ( item_type = 0; 
          item_type < configuration.items.number; 
          ++item_type )
    {
        if ( !strcmp( configuration.items.names[item_type], cmp_name ) )
        {
            return ( item_type );
        }
    }
    return ( NO_ITEM );
}

/*** RB_Get_Item_Type ***/



/****f* Items/RB_Is_ItemName
 * FUNCTION
 *   Is there an itemname in the line.  Ignores leading spaces and
 *   remark markers.
 * INPUTS
 *   line -- line to be searched.
 * RESULT
 *   The kind of item that was found or NO_ITEM if no item could be found.
 *   The name of the item will be stored in item_name_buffer.
 * NOTES
 *   We used to check for misspelled items names by testing if
 *   the item name buffer consists of only upper case characters.
 *   However checking for a misspelled item name this way results in
 *   many false positives. For instance many warnings are given for
 *   FORTRAN code as all the keywords are in uppercase.  We need to
 *   find a better method for this.
 * SOURCE
 */

enum ItemType
RB_Is_ItemName( char *line )
{
    char               *cur_char = line;
    int                 i = 0;

    cur_char = RB_Skip_Whitespace( cur_char );
    if ( RB_Has_Remark_Marker( cur_char ) ) 
    {
        cur_char = RB_Skip_Remark_Marker( cur_char );
        cur_char = RB_Skip_Whitespace( cur_char );
        /* It there anything left? */
        if ( strlen( cur_char ) )
        {
            enum ItemType item_type = NO_ITEM;

            /* Copy the name */
            strcpy( item_name_buffer, cur_char );
            /* remove any trailing spaces */
            for ( i = strlen( item_name_buffer ) - 1;
                    i >= 0 && isspace( item_name_buffer[i] ); --i )
            {
                item_name_buffer[i] = '\0';
            }
            /* No check and see if this is an item name */
            if ( strlen( item_name_buffer ) )
            {
                item_type = RB_Get_Item_Type( item_name_buffer );
#if 0   /* Until we find a better method */
                if ( item_type == NO_ITEM ) 
                {
                    /* Check if it is a misspelled item name */
                    item_type = POSSIBLE_ITEM;
                    for ( i = 0; i < strlen( item_name_buffer ); ++i )
                    {
                        if ( !( isupper( item_name_buffer[ i ] ) ||
                                    isspace( item_name_buffer[ i ] ) ) )
                        {
                            /* No it is not */
                            item_type = NO_ITEM;
                            break;
                        }
                    }
                }
#endif
            }
            return item_type;
        }
        else
        {
            return NO_ITEM;
        }
    }
    else
    {
        return NO_ITEM;
    }
}

/******/

int Is_Ignore_Item( char* name )
{
    unsigned int       i;

    for ( i = 0; i < configuration.ignore_items.number; ++i )
    {
        if ( !strcmp( configuration.ignore_items.names[i], name ) )
        {
            return TRUE;
        }
    }
    return FALSE;
}

