/****h* ROBODoc/Utilities
 * FUNCTION
 *   Set of general purpose utility functions that are used
 *   more than one module.
 *****
 * $Id: util.c,v 1.30 2004/05/18 20:45:13 gumpu Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>             /* for RB_Say() */
#include <time.h>
#include <assert.h>

#include "robodoc.h"
#include "globals.h"
#include "links.h"
#include "headers.h"
#include "util.h"

#ifdef DMALLOC
#include <dmalloc.h>
#endif


static void RB_Swap( void** array, int left, int right );


/*===============================================================================*/


/****f* Util/ExpandTab
 * FUNCTION
 *   Expand the tabs in a line of text.
 * SYNOPSIS
 *   char* RB_ExpandTab(char* line)
 * INPUTS
 *   line -- the line to be expanded
 *   line_buffer -- global. (No point in allocating memory for each
 *                  line).
 *   tab_size    -- global.
 * RETURN
 *   pointer to the expanded line.
 * NOTE
 *   This function is not reentrant.
 * SOURCE
 */

char               *ExpandTab( char *line )
{
    char               *cur_char = line;
    int                 n = 0;
    int                 jump = 0;

    for ( ; *cur_char; ++cur_char )
    {
        if ( *cur_char == '\t' )
        {
            int                 i;

            jump = tab_size - ( n % tab_size );
            for ( i = 0; i < jump; i++ )
            {
                line_buffer[n] = ' ';
                ++n;
            }
        }
        else
        {
            line_buffer[n] = *cur_char;
            ++n;
        }
    }
    line_buffer[n] = '\0';
    return line_buffer;
}

/******/

/****f* Utilities/RB_Alloc_Header [2.01]
 * FUNCTION
 *   allocate the struct RB_header
 * SYNOPSIS
 *   struct RB_header *RB_Alloc_Header( void )
 * RESULT
 *   struct RB_header *      -- all attributes/pointers set to zero
 * AUTHOR
 *   Koessi
 * SEE ALSO
 *   RB_Free_Header()
 * SOURCE
 */

struct RB_header   *
RB_Alloc_Header( void )
{
    struct RB_header   *new_header;

    if ( ( new_header = malloc( sizeof( struct RB_header ) ) ) != NULL )
    {
        memset( new_header, 0, sizeof( struct RB_header ) );
    }
    else
    {
        RB_Panic( "out of memory! [Alloc Header]\n" );
    }
    return ( new_header );
}

/********/


/****f* Utilities/RB_Free_Header [2.01]
 * NAME
 *   RB_Free_Header             -- oop
 * SYNOPSIS
 *   void RB_Free_Header( struct RB_header *header )
 * FUNCTION
 *   free struct RB_header and associated strings
 * INPUTS
 *   struct RB_header *header -- this one
 * AUTHOR
 *   Koessi
 * SEE ALSO
 *   RB_Alloc_Header(), RB_Close_The_Shop()
 * SOURCE
 */

void
RB_Free_Header( struct RB_header *header )
{
    if ( header )
    {
        if ( header->function_name )
        {
            free( header->function_name );
        }
        if ( header->version )
        {
            free( header->version );
        }
        if ( header->name )
        {
            free( header->name );
        }
        if ( header->unique_name )
        {
            free( header->unique_name );
        }
        if ( header->lines )
        {
            int                 i;

            for ( i = 0; i < header->no_lines; ++i )
            {
                free( header->lines[i] );
            }
            free( header->lines );
        }
        free( header );
    }
}

/************/


/****i* Utilities/RB_WordLen [2.01]
 * NAME
 *   RB_WordLen -- like strlen
 * SYNOPSIS
 *   int RB_WordLen( char *str )
 * FUNCTION
 *   get the amount of bytes until next space
 * INPUTS
 *   char *str -- the word
 * RESULT
 *   int -- length of the next word or 0
 * AUTHOR
 *   Koessi
 * SEE ALSO
 *   RB_Find_Header_Name()
 * SOURCE
 */

int
RB_WordLen( char *str )
{
    int                 len;
    char                c;

    for ( len = 0; ( ( c = *str ) != '\0' ) && !isspace( c ) && ( c != '\n' );
          ++str, ++len )
    {
        /* empty */
    }
    return ( len );
}

/*** RB_WordLen ***/


/****if* Utilities/RB_StrDup [2.01]
 * NAME
 *   RB_StrDup
 * SYNOPSIS
 *   char *RB_StrDup( char *str )
 * FUNCTION
 *   duplicate the given string
 * INPUTS
 *   char *str               -- source
 * RESULT
 *   char *                  -- destination
 * AUTHOR
 *   Koessi
 * SOURCE
 */

char               *
RB_StrDup( char *str )
{
    char               *dupstr;
    if ( ( dupstr =
           malloc( ( strlen( str ) + 1 ) * sizeof( char ) ) ) != NULL )
    {
        strcpy( dupstr, str );
    }
    else
    {
        RB_Panic( "out of memory! [StrDup]\n" );
    }
    return ( dupstr );
}

/*** RB_StrDup ***/

/****f* Utilities/RB_Say [2.01]
 * NAME
 *   RB_Say                     -- varargs
 * SYNOPSIS
 *   void RB_Say( char *what, char *why, ... )
 * FUNCTION
 *   Say what's going on.  Goes to stdout.
 * INPUTS
 *   char *format            -- formatstring
 *    ...                    -- parameters
 * AUTHOR
 *   Koessi
 * SOURCE
 */

void
RB_Say( char *format, ... )
{
    va_list             ap;

    if ( course_of_action & DO_TELL )
    {
        va_start( ap, format );
        printf( "%s: ", whoami );
        vprintf( format, ap );
        va_end( ap );
    }
}

/*** RB_Say ***/


/* TODO Documentation */
void
RB_SetCurrentFile( char *filename )
{
    current_file = filename;
}

/* TODO Documentation */
char               *
RB_GetCurrentFile(  )
{
    return current_file;
}


/****f* Utilities/RB_Panic [2.01]
 * NAME
 *   RB_Panic -- free resources and shut down
 * SYNOPSIS
 *   void RB_Panic( char *format, char *why, ... )
 * FUNCTION
 *   Print error message.  Frees all resources used by robodoc.
 *   Terminates program.  Output goes to stderr
 * INPUTS
 *   char *format            -- formatstring
 *   ...                     -- parameters
 * AUTHOR
 *   Koessi
 * SOURCE
 */

void
RB_Panic( char *format, ... )
{
    va_list             ap;
    char               *name;

    va_start( ap, format );

    name = RB_GetCurrentFile(  );

    if ( name )
    {
        char* buffer_copy = RB_StrDup( line_buffer );
        RB_StripCR( buffer_copy );
        fprintf( stderr, "%s: FATAL ERROR - %s:%d\n", whoami, name, line_number );
        fprintf( stderr, "   %s\n%s: ", whoami, buffer_copy, whoami );
        free( buffer_copy );
    }
    else
    {
        fprintf( stderr, "%s: ", whoami );
    }
    vfprintf( stderr, format, ap );
    fprintf( stderr, "%s: closing down...\n", whoami );
    va_end( ap );
    RB_Close_The_Shop(  );
    exit( EXIT_FAILURE );
}

/*** RB_Panic ***/


/* TODO Documentation */

void RB_Warning_Full( char* arg_filename, int arg_line_number, char* arg_format, ... )
{
    va_list             ap;
    ++number_of_warnings;
    va_start( ap, arg_format );
    fprintf( stdout, "%s: Warning - %s:%d\n", whoami, arg_filename, arg_line_number );
    fprintf( stdout, "   " );
    vfprintf( stdout, arg_format, ap );
    va_end( ap );
}


void RB_Warning( char* format, ... )
{
    va_list             ap;
    char               *name;
    ++number_of_warnings;
    va_start( ap, format );

    name = RB_GetCurrentFile(  );

    if ( name )
    {
//        char* buffer_copy = RB_StrDup( line_buffer );
//        RB_StripCR( buffer_copy );
        fprintf( stdout, "%s: Warning - %s:%d\n", whoami, name, line_number );
//        fprintf( stderr, "%s: %s\n%s: ", whoami, buffer_copy, whoami );
//        fprintf( stdout, "   ", whoami );
//        free( buffer_copy );
    }
    else
    {
//        fprintf( stderr, "%s: ", whoami );
    }
    fprintf( stdout, "   " );
    vfprintf( stdout, format, ap );
    va_end( ap );
}


/****f* Utilities/RB_Str_Case_Cmp
 * FUNCTION
 *   Compare two strings, regardless of the case of the characters.
 * SYNOPSIS
 *   int      RB_Str_Case_Cmp(char *, char *)
 *   result = RB_Str_Case_Cmp(s, t)
 * RESULT
 *    0  s == t
 *   -1  s < t
 *    1  s > t
 * SOURCE
 */

int
RB_Str_Case_Cmp( char *s, char *t )
{
    assert( s );
    assert( t );
    for ( ; tolower( *s ) == tolower( *t ); s++, t++ )
    {
        if ( *s == '\0' )
        {
            return 0;
        }
    }
    return ( int ) ( tolower( *s ) - tolower( *t ) );
}

/*********/


/****f* Utilities/RB_TimeStamp
 * NAME
 *   RB_TimeStamp -- print a time stamp
 * SOURCE
 */

void
RB_TimeStamp( FILE * f )
{
    time_t              ttp;
    char                timeBuffer[255];

    time( &ttp );
    strftime( timeBuffer, 255, "%a %b %d %H:%M:%S %Y\n", localtime( &ttp ) );
    fprintf( f, "%s", timeBuffer );
}

/******/

/****f* Utilities/RB_Skip_Whitespace
 * SYNOPSIS
 *   char * RB_Skip_Whitespace(char *buf)
 * FUNCTION
 *   Skip space and tab chars from the start *buf. This is needed when
 *   searching for indented headers and items. 
 * NOTES
 *   We should extract some info about indentation level and save it to
 *   global variable in order to write out source items (that originate from
 *   indented headers) neatly. 
 * SEE ALSO
 *   RB_Find_Marker, RB_Find_End_Marker, RB_Find_Item, RB_Generate_Item_Body
 * SOURCE
 */

char               *
RB_Skip_Whitespace( char *buf )
{
    char               *c;

    for ( c = buf; *c; c++ )
    {
        if ( isspace( *c ) )
        {

        }
        else
        {
            return c;
        }
    }
    return c;                 /* buf was null */
}

/*** RB_Skip_Whitespace ***/


/****f* Utilities/RB_FputcLatin1ToUtf8
 * NAME
 *   RB_FputcLatin1ToUtf8
 * SYNOPSIS
 *   void RB_FputcLatin1ToUtf8(FILE *fp, int c)
 * BUGS
 *   This wrongly assumes that input is always Latin-1.
 * SOURCE
 */

void
RB_FputcLatin1ToUtf8( FILE * fp, int c )
{
    if ( c < 0x80 )
    {
        if ( fputc( c, fp ) == EOF )
            RB_Panic( "RB_FputcLatin1ToUtf8: write error" );
    }
    else
    {
        if ( fputc( ( 0xC0 | ( c >> 6 ) ), fp ) == EOF )
            RB_Panic( "RB_FputcLatin1ToUtf8: write error" );
        if ( fputc( ( 0x80 | ( c & 0x3F ) ), fp ) == EOF )
            RB_Panic( "RB_FputcLatin1ToUtf8: write error" );
    }
}

/*** RB_FputcLatin1ToUtf8 ***/


/****f* Utilities/RB_CopyFile
 * NAME
 *   RB_CopyFile -- copy a file to another file
 * SYNOPSIS
 *   void RB_CopyFile( char* sourceFileName, char* destinationFileName )
 * RESULT
 *   Program Exit if one of the specified files did not open.
 * SOURCE
 */

void RB_CopyFile( char* sourceFileName, char* destinationFileName )
{
    FILE* source;

    source = fopen( sourceFileName, "r" );
    if ( source ) 
    {
        FILE* dest;
        dest = fopen( destinationFileName, "w" );
        if ( dest ) 
        {
            for (; fgets( line_buffer, MAX_LINE_LEN, source ); )
            {
                fputs( line_buffer, dest );
            }
        }
        else
        {
            fclose( source );
            RB_Panic( "can't open file %s for writing",
                    destinationFileName );
        }
    }
    else
    {
        RB_Panic( "can't open file %s for reading", sourceFileName );
    }
}

/*****/



/****f* Utilities/RB_Match
 * FUNCTION
 *   See if a wildcard expression matches a target string.  The wildcard
 *   expression can consists of any literal character and the two
 *   wildcards characters '*' and '?'.  '*' matches the longest string
 *   of zero or more characters that fit.  '?' matches any single
 *   character.
 *
 *   Examples:
 *      "*aap"   matches "aapaapaapaap"
 *      "?inux"  matches "linux"
 *      "lin*ux" matches "linux"
 *      "linux*" matches "linux"
 * NOTES
 *   This is a recursive function.
 * SYNOPSIS
 *   int RB_Match( char* target, char* wildcard_expression )
 * INPUTS
 *   o target -- the string to be matched agains the
 *               wildcard_expression.
 *   o wildcard_expression -- the wildcard expression
 * RETURN VALUE
 *   TRUE  -- the target matches the wildcard expression
 *   FALSE -- it does not match.
 * SOURCE
 */

int RB_Match( char* target, char* wildcard_expression )
{
    if ( ( *wildcard_expression == '\0' ) &&
         ( *target == '\0' ) ) 
    {
        /* a match, since both strings are now "" */
        return TRUE;
    }
    else if ( *wildcard_expression == '\0' ) 
    {
        /* we reached the end of the wildcard_expression,
         * but not the end of the target, this is not
         * a match.
         */
        return FALSE;
    }
    else if ( *target == '\0' )
    {
        /* we reached the end of the target but not the end of the
         * wildcard_expression.  Only if the whole wildcard_expression
         * consists of * we have a match.
         */
        unsigned int i;
        for ( i = 0; i < strlen( wildcard_expression ); ++i )
        {
            if ( wildcard_expression[ i ] != '*' )
            {
                return FALSE;
            }
        }
        return TRUE;
    }
    else
    {
        /* There are wildcard_expression characters left
         * and target characters left.
         */
        char wildcard = wildcard_expression[ 0 ];

        if ( wildcard == '?' )
        {
            /* Match a single character and see if the
             * rest of the target matches.
             */
            return RB_Match( target + 1, wildcard_expression + 1 );
        }
        else if ( wildcard == '*' )
        {
            int match = FALSE;
            int l = strlen( target );
            int i;
            /* First try to match all of the target string, and
             * then work back to the begin of the target string.
             * ( including the "" string. )
             */
            for ( i = l; i >= 0; --i )
            {
                if ( RB_Match( target + i, wildcard_expression + 1 ) )
                {
                    match = TRUE;
                    break;
                }
            }
            return match;
        }
        else 
        {
            int l_w = strlen( wildcard_expression );
            int l_t = strlen( target );
            /* The minimum of the length of the wildcard_expression
             * and target expression 
             */
            int l = ( l_w <= l_t ) ? l_w : l_t ;
            int i;
            for ( i = 0; i < l; ++i )
            {
                if( ( wildcard_expression[ i ] != '*' ) &&
                    ( wildcard_expression[ i ] != '?' ) )
                {
                    /* Some OS-es are not case-sensitive when it comes
                     * to file names, and consider Readme to be equal
                     * to README.  On these OS-es it can be handy if
                     * robodoc is also not case-sensitive.
                     */
#ifdef IGNORE_CASE_FILENAMES
                    if ( tolower( wildcard_expression[ i ] ) != tolower( target[ i ] ) )
#else
                    if ( wildcard_expression[ i ] != target[ i ] )
#endif
                    {
                        return FALSE;
                    }
                }
                else
                {
                    return RB_Match( target + i, wildcard_expression + i );
                }
            }
            /* The first l characters of the target and
             * wildcard_expression matched, now see if the rest
             * matches too.
             */
            return RB_Match( target + l, wildcard_expression + l );
        }
    }
}

/******/

/****f* Utilities/RB_Swap
 * FUNCTION
 *   Swap two elements in a array of pointers.  This function is used
 *   by RB_QuickSort().
 * SOURCE
 */

static void RB_Swap( void** array, int left, int right )
{
    void* p = array[ left ];
    array[ left ] = array[ right ];
    array[ right ] = p;
}

/*****/

/****f* Utilities/RB_QuickSort
 * FUNCTION
 *   Sort an array of pointers according to the lexical order
 *   of the elements the pointers point to.
 *   This is based on the quicksort routine in 
 *   "The C programming language" by B Kerninghan en D Ritchie.
 * INPUTS
 *   array -- the array of pointers.
 *   left  -- the most left element in the array.
 *   right -- the most right element in the array.
 *   f     -- pointer to a function that can compare
 *            the objects two elements of the array
 *            point to.
 * RESULT
 *   array -- A sorted array of pointers.
 *
 * EXAMPLE
 *   The following is an example program that shows
 *   the use
 *    #define TEST_SIZE 10
 *
 *    char* test[ TEST_SIZE ] = { "ape", "zebra", 
 *       "duck", "goofbal", "dodo", "rabit", 
 *       "crow", "cow", "pig", "goat" };
 *
 *    int string_compare( void* p1, void* p2 )
 *    {
 *       char *cp1 = p1;
 *       char *cp2 = p2;
 *       return strcmp( cp1, cp2 );
 *    }
 *
 *    RB_QuickSort( test, 0, TEST_SIZE - 1, string_compare ); 
 *
 * SOURCE
 */

void RB_QuickSort( void** array, int left, int right, TCompare f )
{
    int i;
    int last;

    if ( left >= right ) { return; }

    RB_Swap( array, left, ( left + right ) / 2 ) ;
    last = left;
    for ( i = left + 1; i <= right; ++i )
    {
        if ( (*f)( array[ i ], array[ left ] ) < 0 )
        {
            RB_Swap( array, ++last, i ) ;
        }
    }
    RB_Swap( array, left, last );
    RB_QuickSort( array, left, last - 1, f );
    RB_QuickSort( array, last + 1, right, f );
}

/*******/


/* TODO Documentation */

void RB_StripCR( char *line )
{
    char               *c;

    for ( c = line; *c; ++c )
    {
        if ( *c == '\n' )
        {
            *c = '\0';
        }
    }
}

