/*
 * tnmJob.c --
 *
 *	The simple job scheduler used to implement monitoring scripts.
 *	This version is derived from the original job scheduler
 *	written in Tcl by Stefan Schoek (schoek@ibr.cs.tu-bs.de).
 *
 * Copyright (c) 1994-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmInt.h"
#include "tnmPort.h"

/*
 * Structure used to describe a job.
 */

typedef struct Job {
    char *cmd;			/* The command to evaluate. */
    char *newCmd;		/* The new command to replaces the current. */
    int interval;		/* The time interval value in ms. */
    int remtime;		/* The remaining time in ms. */
    int iterations;		/* The number of iterations of this job. */
    int status;			/* The status of this job (see below). */
    Tcl_HashTable attributes;	/* The has table of job attributes. */
    Tcl_Command token;		/* The command token used by Tcl. */
    Tcl_Interp *interp;		/* The interpreter which owns this job. */
    struct Job *nextPtr;	/* Next job in our list of jobs. */
} Job;

/*
 * Every Tcl interpreter has an associated JobControl record. It
 * keeps track of the timers and the list of jobs known by the
 * interpreter. The name tnmJobControl is used to get/set the 
 * JobControl record.
 */

static char tnmJobControl[] = "tnmJobControl";

typedef struct JobControl {
    Job *jobList;		/* The list of jobs for this interpreter. */
    Tcl_TimerToken timer;	/* The token for the Tcl timer. */
    Tcl_Time lastTime;		/* The last time stamp. */
} JobControl;

/* 
 * These are all possible status codes of an existing job.
 */

#define TNM_JOB_SUSPEND 1
#define TNM_JOB_WAITING 2
#define TNM_JOB_RUNNING 3
#define TNM_JOB_EXPIRED 4

static TnmTable statusTable[] =
{
    { TNM_JOB_SUSPEND,	"suspended" },
    { TNM_JOB_WAITING,	"waiting" },
    { TNM_JOB_RUNNING,	"running" },
    { TNM_JOB_EXPIRED,	"expired" },
    { 0, NULL }
};

/*
 * The pointer to the currently running job.
 */

static Job *currentJob = NULL;

/*
 * Forward declarations for procedures defined later in this file:
 */

static void
AssocDeleteProc	_ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp));

static void
CmdDeleteProc	_ANSI_ARGS_((ClientData clientData));

static void
DestroyProc	_ANSI_ARGS_((char *memPtr));

static void
ScheduleProc	_ANSI_ARGS_((ClientData clientData));

static void 
NextSchedule	_ANSI_ARGS_((Tcl_Interp *interp, JobControl *control));

static void 
AdjustTime	_ANSI_ARGS_((JobControl *control));

static void
Schedule	_ANSI_ARGS_((Tcl_Interp *interp, JobControl *control));

static int
CreateJob	_ANSI_ARGS_((Tcl_Interp *interp, int argc, char **argv));

static int
Attributes	_ANSI_ARGS_((Job *jobPtr, Tcl_Interp *interp, 
			     int argc, char **argv));
static char *
GetOption	_ANSI_ARGS_((Tcl_Interp *interp, ClientData object, 
			     int option));
static int
SetOption	_ANSI_ARGS_((Tcl_Interp *interp, ClientData object, 
			     int option, char *value));
static int
JobCmd		_ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp, 
			     int argc, char **argv));

/*
 * The options used to configure job objects.
 */

#define TNM_JOB_OPT_COMMAND    1
#define TNM_JOB_OPT_INTERVAL   2
#define TNM_JOB_OPT_ITERATIONS 3
#define TNM_JOB_OPT_STATUS     4
#define TNM_JOB_OPT_TIME       5

static TnmTable optionTable[] = {
    { TNM_JOB_OPT_COMMAND,    "-command" },
    { TNM_JOB_OPT_INTERVAL,   "-interval" },
    { TNM_JOB_OPT_ITERATIONS, "-iterations" },
    { TNM_JOB_OPT_STATUS,     "-status" },
    { TNM_JOB_OPT_TIME,       "-time" },
    { 0, NULL }
};

static TnmConfig config = {
    optionTable,
    SetOption,
    GetOption
};

/*
 *----------------------------------------------------------------------
 *
 * AssocDeleteProc --
 *
 *	This procedure is called when a Tcl interpreter gets destroyed
 *	so that we can clean up the data associated with this interpreter.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
AssocDeleteProc(clientData, interp)
    ClientData clientData;
    Tcl_Interp *interp;
{
    JobControl *control = (JobControl *) clientData;

    /*
     * Note, we do not care about job objects since Tcl first
     * removes all commands before calling this delete procedure.
     * However, we have to delete the timer to make sure that
     * no further events are processed for this interpreter.
     */

    if (control) {
	if (control->timer) {
	    Tcl_DeleteTimerHandler(control->timer);
	}
	ckfree((char *) control);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * CmdDeleteProc --
 *
 *	This procedure removes the job for the list of active jobs and 
 *	releases all memory associated with a job object.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A job is destroyed.
 *
 *----------------------------------------------------------------------
 */

static void
CmdDeleteProc(clientData)
    ClientData clientData;
{
    Job **jobPtrPtr;
    Job *jobPtr = (Job *) clientData;
    JobControl *control = (JobControl *)
	Tcl_GetAssocData(jobPtr->interp, tnmJobControl, NULL);

    /*
     * First, update the list of all known jobs.
     */

    jobPtrPtr = &control->jobList;
    while (*jobPtrPtr && (*jobPtrPtr) != jobPtr) {
	jobPtrPtr = &(*jobPtrPtr)->nextPtr;
    }

    if (*jobPtrPtr) {
	(*jobPtrPtr) = jobPtr->nextPtr;
    }

    Tcl_EventuallyFree((ClientData) jobPtr, DestroyProc);
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyProc --
 *
 *	This procedure is invoked by Tcl_EventuallyFree or Tcl_Release
 *	to clean up the internal structure of a job at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the job is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
DestroyProc(memPtr)
    char *memPtr;
{
    Job *jobPtr = (Job *) memPtr;

    Tcl_DeleteHashTable(&(jobPtr->attributes));
    ckfree(jobPtr->cmd);
    if (jobPtr->newCmd) ckfree(jobPtr->newCmd);
    ckfree((char *) jobPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ScheduleProc --
 *
 *	This procedure is the callback of the Tcl event loop that
 *	invokes the scheduler.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The scheduler gets called.
 *
 *----------------------------------------------------------------------
 */

static void
ScheduleProc(clientData)
    ClientData clientData;
{
    Tcl_Interp *interp = (Tcl_Interp *) clientData;
    JobControl *control = (JobControl *) 
	Tcl_GetAssocData(interp, tnmJobControl, NULL);

    if (control) {
	Schedule(interp, control);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * NextSchedule --
 *
 *	This procedure registers a timer handler that will call us when
 *	the next job should be executed. An internal token is used to 
 *	make sure that we do not have more than one timer event pending
 *	for the job scheduler.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A new timer handler might be registered.
 *
 *----------------------------------------------------------------------
 */

static void
NextSchedule(interp, control)
    Tcl_Interp *interp;
    JobControl *control;
{
    Job *jobPtr;
    int ms;

    if (control->timer) {
	Tcl_DeleteTimerHandler(control->timer);
	control->timer = NULL;
    }

    /* 
     * Calculate the minimum of the remaining times of all jobs
     * waiting. Tell the event manager to call us again if we found
     * a waiting job with remaining time >= 0. We also wait for 
     * expired jobs so that they will get removed from the job list.
     */

    ms = -1;
    for (jobPtr = control->jobList; jobPtr != NULL; jobPtr = jobPtr->nextPtr) {
        if (jobPtr->status == TNM_JOB_WAITING
	    || jobPtr->status == TNM_JOB_EXPIRED) {
	    if (ms < 0 || jobPtr->remtime < ms) {
		ms = (jobPtr->remtime < 0) ? 0 : jobPtr->remtime;
	    }
	}
    }
    
    if (ms < 0) {
	control->lastTime.sec = 0;
	control->lastTime.usec = 0;
    } else {
        control->timer = Tcl_CreateTimerHandler(ms, ScheduleProc, 
						(ClientData) interp);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * AdjustTime --
 *
 *	This procedure updates the remaining time of all jobs in the 
 *	queue that are not suspended and sets lastTime to the current 
 *	time.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The remaining times of all active jobs is updated.
 *
 *----------------------------------------------------------------------
 */

static void
AdjustTime(control)
    JobControl *control;
{
    int delta;
    Job *jobPtr;
    Tcl_Time currentTime;

    if (control->lastTime.sec == 0 && control->lastTime.usec == 0) {
	TnmGetTime(&control->lastTime);
	return;
    }

    TnmGetTime(&currentTime);

    delta = (currentTime.sec - control->lastTime.sec) * 1000 
	    + (currentTime.usec - control->lastTime.usec) / 1000;

    control->lastTime = currentTime;

    /*
     * Do not increase remaining times of any jobs if the clock
     * has moved backwards.
     */

    if (delta <= 0) {
	return;
    }

    for (jobPtr = control->jobList; jobPtr; jobPtr = jobPtr->nextPtr) {
        if (jobPtr->status != TNM_JOB_SUSPEND) {
	    jobPtr->remtime -= delta;
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Schedule --
 *
 *	This procedure is called to schedule the next job. It first 
 *	checks for jobs that must be processed. It finally tells the 
 *	event mechanism to repeat itself when the next job needs
 *	attention. This function also cleans up the job queue by 
 *	removing all jobs that are waiting in the state expired.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Jobs might be activated and expired jobs are removed.
 *
 *----------------------------------------------------------------------
 */

static void
Schedule(interp, control)
    Tcl_Interp *interp;
    JobControl *control;
{
    Job *jobPtr;

    /*
     * Refresh the remaining time of the active jobs.
     */

    AdjustTime(control);

    /*
     * Execute waiting jobs with remaining time less or equal 0. 
     * Set the job status to expired if the number of iterations
     * reaches zero.
     */

  restart:
    for (jobPtr = control->jobList; jobPtr != NULL; jobPtr = jobPtr->nextPtr) {

	if (jobPtr->newCmd) {
	    ckfree(jobPtr->cmd);
	    jobPtr->cmd = jobPtr->newCmd;
	    jobPtr->newCmd = NULL;
	}

	if ((jobPtr->status == TNM_JOB_WAITING) && (jobPtr->remtime <= 0)) {

	    int code;

	    Tcl_Preserve((ClientData) jobPtr);
	    currentJob = jobPtr;
	    jobPtr->status = TNM_JOB_RUNNING;

	    Tcl_AllowExceptions(interp);
	    code = Tcl_GlobalEval(interp, jobPtr->cmd);
	    if (code == TCL_ERROR) {
		Tcl_AddErrorInfo(interp, 
				 "\n    (script bound to job - job deleted)");
		Tcl_BackgroundError(interp);
		jobPtr->status = TNM_JOB_EXPIRED;
	    }
    
	    Tcl_ResetResult(interp);
	    if (jobPtr->status == TNM_JOB_RUNNING) {
		jobPtr->status = TNM_JOB_WAITING;
	    }
	    currentJob = NULL;
	    
	    jobPtr->remtime = jobPtr->interval;
	    if (jobPtr->iterations > 0) {
		jobPtr->iterations--; 
		if (jobPtr->iterations == 0) {
		    jobPtr->status = TNM_JOB_EXPIRED;
		}
	    }
	    Tcl_Release((ClientData) jobPtr);
	    goto restart;
	}
    }

    /*
     * Delete all jobs which have reached the status expired.
     * We must restart the loop for every deleted job as the
     * job list is modified by calling Tcl_DeleteCommand().
     */

  repeat:
    for (jobPtr = control->jobList; jobPtr != NULL; jobPtr = jobPtr->nextPtr) {
        if (jobPtr->status == TNM_JOB_EXPIRED) {
	    char *name = Tcl_GetCommandName(interp, jobPtr->token);
#if 0
	    char cmd[40];
	    sprintf(cmd, "event raise %s", name);
	    Tcl_GlobalEval(interp, cmd);
	    Tcl_ResetResult(interp);
#endif
	    Tcl_DeleteCommand(interp, name);
	    goto repeat;
        }
    }
    
    /*
     * Compute and subtract the time needed to execute the jobs
     * and schedule the next pass through the scheduler.
     */

    AdjustTime(control);
    NextSchedule(interp, control);
}

/*
 *----------------------------------------------------------------------
 *
 * CreateJob --
 *
 *	This procedure creates a new job and appends it to our job
 *	list. The scheduler is called to updates its timer.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	The job list is updated and the scheduler is started.
 *
 *----------------------------------------------------------------------
 */

static int
CreateJob(interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;
    Job *jobPtr, *p;
    char name[20];
    int code;
    JobControl *control = (JobControl *) 
	Tcl_GetAssocData(interp, tnmJobControl, NULL);

    if (! control) {
	control = (JobControl *) ckalloc(sizeof(JobControl));
	memset((char *) control, 0, sizeof(JobControl));
	Tcl_SetAssocData(interp, tnmJobControl, AssocDeleteProc, 
			 (ClientData) control);
    }

    jobPtr = (Job *) ckalloc(sizeof(Job));
    memset((char *) jobPtr, 0, sizeof(Job));
    jobPtr->cmd = ckstrdup("");
    jobPtr->interval = 1000;
    jobPtr->status = TNM_JOB_WAITING;
    jobPtr->interp = interp;
    Tcl_InitHashTable(&(jobPtr->attributes), TCL_STRING_KEYS);

    code = TnmSetConfig(interp, &config, (ClientData) jobPtr, argc, argv);
    if (code != TCL_OK) {
        ckfree((char *) jobPtr);
        return TCL_ERROR;
    }

    /*
     * Put the new job into the job list. We add it at the end
     * to preserve the order in which the jobs were created.
     */

    if (control->jobList == NULL) {
        control->jobList = jobPtr;
    } else {
        for (p = control->jobList; p->nextPtr != NULL; p = p->nextPtr) ;
	p->nextPtr = jobPtr;
    }

    /*
     * Create a new scheduling point for this new job.
     */

    NextSchedule(interp, control);

    /*
     * Create a new Tcl command for this job object.
     */

    sprintf(name, "job%d", lastid++);
    jobPtr->token = Tcl_CreateCommand(interp, name, JobCmd,
				      (ClientData) jobPtr, CmdDeleteProc);
    Tcl_SetResult(interp, name, TCL_VOLATILE);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Attributes --
 *
 *	This procedure retrieves or sets job attributes.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Job attributes are created, modified or deleted.
 *
 *----------------------------------------------------------------------
 */

static int
Attributes(jobPtr, interp, argc, argv)
    Job *jobPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Tcl_HashTable *tablePtr = &(jobPtr->attributes);
    Tcl_HashEntry *entryPtr;
    Tcl_HashSearch search;

    if (argc < 2 || argc > 4) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " attribute ?name ?value??\"", (char *) NULL);
        return TCL_ERROR;
    }

    if (argc == 4) {
	int isnew;
	entryPtr = Tcl_CreateHashEntry(tablePtr, argv[2], &isnew);
	if (! isnew) {
	    ckfree((char *) Tcl_GetHashValue(entryPtr));
	}
	Tcl_SetHashValue(entryPtr, ckstrdup(argv[3]));
    } else if (argc == 3) {
	entryPtr = Tcl_FindHashEntry(tablePtr, argv[2]);
	if (entryPtr) {
	    Tcl_SetResult(interp, (char *) Tcl_GetHashValue(entryPtr), 
			  TCL_STATIC);
	}
    } else {
	entryPtr = Tcl_FirstHashEntry(tablePtr, &search);
	while (entryPtr) {
	    Tcl_AppendElement(interp, Tcl_GetHashKey(tablePtr, entryPtr));
	    entryPtr = Tcl_NextHashEntry(&search);
	}
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * GetOption --
 *
 *	This procedure retrieves the value of a job option.
 *
 * Results:
 *	A pointer to the value formatted as a string.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char *
GetOption(interp, object, option)
    Tcl_Interp *interp;
    ClientData object;
    int option;
{
    static char buffer[20], *status;
    Job *jobPtr = (Job *) object;
    JobControl *control = (JobControl *)
	Tcl_GetAssocData(jobPtr->interp, tnmJobControl, NULL);

    switch (option) {
    case TNM_JOB_OPT_COMMAND:
	return jobPtr->newCmd ? jobPtr->newCmd : jobPtr->cmd;
    case TNM_JOB_OPT_INTERVAL:
	sprintf(buffer, "%d", jobPtr->interval);
	return buffer;
    case TNM_JOB_OPT_ITERATIONS:
	sprintf(buffer, "%d", jobPtr->iterations);
	return buffer;
    case TNM_JOB_OPT_STATUS:
	status = TnmGetTableValue(statusTable, jobPtr->status);
	return status ? status : "unknown";
    case TNM_JOB_OPT_TIME:
	if (control) {
	    AdjustTime(control);
	}
	sprintf(buffer, "%d", jobPtr->remtime);
	return buffer;
    }
    return "";
}

/*
 *----------------------------------------------------------------------
 *
 * SetOption --
 *
 *	This procedure modifies a single option of a job object.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
SetOption(interp, object, option, value)
    Tcl_Interp *interp;
    ClientData object;
    int option;
    char *value;
{
    Job *jobPtr = (Job *) object;
    int num, status;
    JobControl *control = (JobControl *)
	Tcl_GetAssocData(jobPtr->interp, tnmJobControl, NULL);

    switch (option) {
    case TNM_JOB_OPT_COMMAND:
	if (jobPtr->newCmd) ckfree(jobPtr->newCmd);
	jobPtr->newCmd = ckstrdup(value);
	break;
    case TNM_JOB_OPT_INTERVAL:
	if (TnmGetUnsigned(interp, value, &num) != TCL_OK) {
	    return TCL_ERROR;
	}
	jobPtr->interval = num;
	break;
    case TNM_JOB_OPT_ITERATIONS:
	if (TnmGetUnsigned(interp, value, &num) != TCL_OK) {
	    return TCL_ERROR;
	}
	jobPtr->iterations = num;
	break;
    case TNM_JOB_OPT_STATUS:
	status = TnmGetTableKey(statusTable, value);
	if (status < 0) {
	    Tcl_AppendResult(interp, "unknown status \"", value, 
			     "\": should be ", TnmGetTableValues(statusTable), 
			     (char *) NULL);
	    return TCL_ERROR;
	}
	jobPtr->status = (status == TNM_JOB_RUNNING) 
	    ? TNM_JOB_WAITING : status;

	/*
	 * Compute the current time offsets and create a new 
	 * scheduling point. A suspended job may have resumed 
	 * and we must make sure that our scheduler is running.
	 */
	
	if (control) {
	    AdjustTime(control);
	    NextSchedule(interp, control);
	}
	break;
    case TNM_JOB_OPT_TIME:
	break;
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * JobCmd --
 *
 *	This procedure implements the job object command.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

static int
JobCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int result = TCL_OK;
    Job *job = (Job *) clientData;
    JobControl *control = (JobControl *) 
	Tcl_GetAssocData(interp, tnmJobControl, NULL);

    if (argc < 2) {
        Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " option ?args?\"", (char *) NULL);
        return TCL_ERROR;
    }

    Tcl_Preserve((ClientData) job);

    if (strcmp(argv[1], "attribute") == 0) {

	result = Attributes(job, interp, argc, argv);
	
    } else if (strcmp(argv[1], "configure") == 0) {

	result = TnmSetConfig(interp, &config, (ClientData) job, argc, argv);

    } else if (strcmp(argv[1], "cget") == 0) {

	result = TnmGetConfig(interp, &config, (ClientData) job, argc, argv);

    } else if (strcmp(argv[1], "destroy") == 0) {

        if (argc > 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " destroy\"", (char *) NULL);
	    goto error;
	}

	job->status = TNM_JOB_EXPIRED;

    } else if (strcmp(argv[1], "wait") == 0) {

	if (argc > 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " wait\"", (char *) NULL);
	    goto error;
	}

	if (control) {
	repeat:
	    for (job = control->jobList; job; job = job->nextPtr) {
		char *name = Tcl_GetCommandName(interp, job->token);
		if (job->status == TNM_JOB_WAITING 
		    && (strcmp(name, argv[0]) == 0)) {
		    Tcl_DoOneEvent(0);
		    goto repeat;
		}
	    }
	}

    } else {

	Tcl_AppendResult(interp, "bad option \"", argv[1],
			 "\": should be attribute, cget, configure, destroy, ",
			 "or wait", (char *) NULL);
	goto error;
    }

    Tcl_Release((ClientData) job);
    return result;

 error:
    Tcl_Release((ClientData) job);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_JobCmd --
 *
 *	This procedure is invoked to process the "job" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_JobCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Job *job;
    JobControl *control = (JobControl *) 
	Tcl_GetAssocData(interp, tnmJobControl, NULL);

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (strcmp(argv[1], "create") == 0) {
	return CreateJob(interp, argc, argv);

    } else if (strcmp(argv[1], "info") == 0) {
        if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " info\"", (char *) NULL);        
	    return TCL_ERROR;
	}
	if (control) {
	    for (job = control->jobList; job; job = job->nextPtr) {
		Tcl_AppendElement(interp, 
				  Tcl_GetCommandName(interp, job->token));
	    }
	}
	return TCL_OK;
    } else if (strcmp(argv[1], "current") == 0) {
        if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " current\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (currentJob && currentJob->interp == interp) {
	    Tcl_SetResult(interp, 
			  Tcl_GetCommandName(interp, currentJob->token),
			  TCL_VOLATILE);
	}
	return TCL_OK;
    } else if (strcmp(argv[1], "wait") == 0) {
        if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " wait\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (control) {
	repeat:
	    for (job = control->jobList; job; job = job->nextPtr) {
		if (job->status == TNM_JOB_WAITING) {
		    Tcl_DoOneEvent(0);
		    goto repeat;
		}
	    }
	}
	return TCL_OK;
    } else if (strcmp(argv[1], "schedule") == 0) {
        if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " schedule\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (control) {
	    Schedule(interp, control);
	}
	return TCL_OK;
    }

    Tcl_AppendResult(interp, "bad option \"", argv[1], 
		     "\": should be create, current, info, wait, or schedule",
		     (char *) NULL);
    return TCL_ERROR;
}
