/*
 * tnmSnmpAgent.c --
 *
 *	This is the SNMP agent interface of scotty.
 *
 * Copyright (c) 1994-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmSnmp.h"
#include "tnmMib.h"

/*
 * The following structure is used to implement a cache that
 * is used to remember queries so that we can respond to
 * retries quickly. This is needed because side effects can
 * break the agent down if we do them for each retry.
 */

typedef struct CacheElement {
    SNMP_Session *session;
    SNMP_PDU request;
    SNMP_PDU response;
    time_t timestamp;
} CacheElement;

#define CACHE_SIZE 64
static CacheElement cache[CACHE_SIZE];

/*
 * Flags used by the SNMP set processing code to keep state information
 * about individual variables in the varbind list.
 */

#define NODE_CREATED 0x01

/*
 * Forward declarations for procedures defined later in this file:
 */

static void
CacheInit		_ANSI_ARGS_((void));

static SNMP_PDU*
CacheGet		_ANSI_ARGS_((SNMP_Session *session, SNMP_PDU *pdu));

static SNMP_PDU*
CacheHit		_ANSI_ARGS_((SNMP_Session *session, SNMP_PDU *pdu));

static char*
TraceSysUpTime		_ANSI_ARGS_((ClientData clientData,
				     Tcl_Interp *interp,
				     char *name1, char *name2, int flags));
#ifdef TNM_SNMPv2U
static char*
TraceAgentTime		_ANSI_ARGS_((ClientData clientData,
				     Tcl_Interp *interp,
				     char *name1, char *name2, int flags));
#endif

static char*
TraceUnsignedInt	_ANSI_ARGS_((ClientData clientData,
				     Tcl_Interp *interp,
				     char *name1, char *name2, int flags));
static TnmSnmpNode*
FindInstance		_ANSI_ARGS_((SNMP_Session *session,
				     Tnm_Oid *oid, int oidlen));
static TnmSnmpNode*
FindNextInstance	_ANSI_ARGS_((SNMP_Session *session,
				     Tnm_Oid *oid, int oidlen));
static int
GetRequest		_ANSI_ARGS_((Tcl_Interp *interp, SNMP_Session *session,
				     SNMP_PDU *request, SNMP_PDU *response));
static int
SetRequest		_ANSI_ARGS_((Tcl_Interp *interp, SNMP_Session *session,
				     SNMP_PDU *request, SNMP_PDU *response));


/*
 * The global variable to keep the snmp statistics is defined here.
 */

SNMP_Statistics snmpStats;

/*
 * The following table is used to register build-in instances
 * bound to counter inside of the protocol stack.
 */

struct StatReg {
    char *name;
    unsigned int *value;
};

static struct StatReg statTable[] = {
    { "snmpInPkts.0",		      &snmpStats.snmpInPkts },
    { "snmpOutPkts.0",		      &snmpStats.snmpOutPkts },
    { "snmpInBadVersions.0",	      &snmpStats.snmpInBadVersions },
    { "snmpInBadCommunityNames.0",    &snmpStats.snmpInBadCommunityNames },
    { "snmpInBadCommunityUses.0",     &snmpStats.snmpInBadCommunityUses },
    { "snmpInASNParseErrs.0",	      &snmpStats.snmpInASNParseErrs },
    { "snmpInTooBigs.0",	      &snmpStats.snmpInTooBigs },
    { "snmpInNoSuchNames.0",	      &snmpStats.snmpInNoSuchNames },
    { "snmpInBadValues.0",	      &snmpStats.snmpInBadValues },
    { "snmpInReadOnlys.0",	      &snmpStats.snmpInReadOnlys },
    { "snmpInGenErrs.0",	      &snmpStats.snmpInGenErrs },
    { "snmpInTotalReqVars.0",	      &snmpStats.snmpInTotalReqVars },
    { "snmpInTotalSetVars.0",	      &snmpStats.snmpInTotalSetVars },
    { "snmpInGetRequests.0",	      &snmpStats.snmpInGetRequests },
    { "snmpInGetNexts.0",	      &snmpStats.snmpInGetNexts },
    { "snmpInSetRequests.0",	      &snmpStats.snmpInSetRequests },
    { "snmpInGetResponses.0",	      &snmpStats.snmpInGetResponses },
    { "snmpInTraps.0",		      &snmpStats.snmpInTraps },
    { "snmpOutTooBigs.0",	      &snmpStats.snmpOutTooBigs },
    { "snmpOutNoSuchNames.0",	      &snmpStats.snmpOutNoSuchNames },
    { "snmpOutBadValues.0",	      &snmpStats.snmpOutBadValues },
    { "snmpOutGenErrs.0",	      &snmpStats.snmpOutGenErrs },
    { "snmpOutGetRequests.0",	      &snmpStats.snmpOutGetRequests },
    { "snmpOutGetNexts.0",	      &snmpStats.snmpOutGetNexts },
    { "snmpOutSetRequests.0",	      &snmpStats.snmpOutSetRequests },
    { "snmpOutGetResponses.0",	      &snmpStats.snmpOutGetResponses },
    { "snmpOutTraps.0",		      &snmpStats.snmpOutTraps },
    { "snmpStatsPackets.0",	      &snmpStats.snmpStatsPackets },
    { "snmpStats30Something.0",	      &snmpStats.snmpStats30Something },
    { "snmpStatsUnknownDstParties.0", &snmpStats.snmpStatsUnknownDstParties },
    { "snmpStatsDstPartyMismatches.0",&snmpStats.snmpStatsDstPartyMismatches },
    { "snmpStatsUnknownSrcParties.0", &snmpStats.snmpStatsUnknownSrcParties},
    { "snmpStatsBadAuths.0",	      &snmpStats.snmpStatsBadAuths },
    { "snmpStatsNotInLifetimes.0",    &snmpStats.snmpStatsNotInLifetimes },
    { "snmpStatsWrongDigestValues.0", &snmpStats.snmpStatsWrongDigestValues },
    { "snmpStatsUnknownContexts.0",   &snmpStats.snmpStatsUnknownContexts },
    { "snmpStatsBadOperations.0",     &snmpStats.snmpStatsBadOperations },
    { "snmpStatsSilentDrops.0",	      &snmpStats.snmpStatsSilentDrops },
    { "snmpV1BadCommunityNames.0",    &snmpStats.snmpInBadCommunityNames },
    { "snmpV1BadCommunityUses.0",     &snmpStats.snmpInBadCommunityUses },
#ifdef TNM_SNMPv2U
    { "usecStatsUnsupportedQoS.0",    &snmpStats.usecStatsUnsupportedQoS },
    { "usecStatsNotInWindows.0",      &snmpStats.usecStatsNotInWindows },
    { "usecStatsUnknownUserNames.0",  &snmpStats.usecStatsUnknownUserNames },
    { "usecStatsWrongDigestValues.0", &snmpStats.usecStatsWrongDigestValues },
    { "usecStatsUnknownContexts.0",   &snmpStats.usecStatsUnknownContexts },
    { "usecStatsUnknownBadParameters.0", 
	&snmpStats.usecStatsBadParameters },
    { "usecStatsUnauthorizedOperations.0",   
	&snmpStats.usecStatsUnauthorizedOperations },
#endif
    { 0, 0 }
};


/*
 *----------------------------------------------------------------------
 *
 * CacheInit --
 *
 *	This procedure initializes the cache of answered requests.
 *	This procedure is only called once when we become an agent.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
CacheInit()
{
    int i;
    memset((char *) cache, 0, sizeof(cache));
    for (i = 0; i < CACHE_SIZE; i++) {
	Tcl_DStringInit(&cache[i].request.varbind);
	Tcl_DStringInit(&cache[i].response.varbind);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * CacheGet --
 *
 *	This procedure gets a free cache element. The cache is a
 *	simple FIFO ring buffer, so we just return the next element.
 *
 * Results:
 *	A pointer to a free cache element.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static SNMP_PDU*
CacheGet(session, pdu)
    SNMP_Session *session;
    SNMP_PDU *pdu;
{
    static int last = 0;
    last = (last + 1 ) % CACHE_SIZE;
    Tcl_DStringFree(&cache[last].request.varbind);
    Tcl_DStringFree(&cache[last].response.varbind);
    cache[last].session = session;
    cache[last].response.request_id = 0;
    cache[last].response.error_status = TNM_SNMP_NOERROR;
    cache[last].response.error_index = 0;
    cache[last].response.addr = pdu->addr;
    Tcl_DStringAppend(&cache[last].request.varbind, 
		      Tcl_DStringValue(&pdu->varbind), 
		      Tcl_DStringLength(&pdu->varbind));
    cache[last].timestamp = time((time_t *) NULL);
    return &(cache[last].response);
}

/*
 *----------------------------------------------------------------------
 *
 * CacheHit --
 *
 *	This procedure checks if the request identified by session and
 *	request id is in the cache so we can send the answer without
 *	further processing.
 *
 * Results:
 *      A pointer to the PDU or NULL if the lookup failed.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static SNMP_PDU*
CacheHit(session, pdu)
    SNMP_Session *session;
    SNMP_PDU *pdu;
{
    int i;
    time_t now = time((time_t *) NULL);

    for (i = 0; i < CACHE_SIZE; i++) {
	if (cache[i].response.request_id != pdu->request_id
	    || cache[i].session != session) {
	    continue;
	}
	if (!cache[i].timestamp || now - cache[i].timestamp > 5) {
	    continue;
	}
	if (Tcl_DStringLength(&pdu->varbind) 
	    == Tcl_DStringLength(&cache[i].request.varbind)
	    && strcmp(Tcl_DStringValue(&pdu->varbind), 
		      Tcl_DStringValue(&cache[i].request.varbind)) == 0
	    ) {
	    cache[i].response.addr = pdu->addr;
	    return &cache[i].response;
	}
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * CacheClear --
 *
 *     This procedure clears the cache for a given session.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *     None.
 *
 *----------------------------------------------------------------------
 */

static void
CacheClear(session)
    SNMP_Session *session;
{
    int i;

    for (i = 0; i < CACHE_SIZE; i++) {
	if (cache[i].session == session) {
	    cache[i].timestamp = 0;
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TraceSysUpTime --
 *
 *	This procedure is a trace callback which is called by the
 *	Tcl interpreter whenever the sysUpTime variable is read.
 *
 * Results:
 *      Always NULL.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char*
TraceSysUpTime(clientData, interp, name1, name2, flags)
    ClientData clientData;
    Tcl_Interp *interp;
    char *name1;
    char *name2;
    int flags;
{
    char buf[20];
    sprintf(buf, "%u", Tnm_SnmpSysUpTime());
    Tcl_SetVar2(interp, name1, name2, buf, TCL_GLOBAL_ONLY);
    return NULL;
}

#ifdef TNM_SNMPv2U
/*
 *----------------------------------------------------------------------
 *
 * TraceAgentTime --
 *
 *	This procedure is a trace callback which is called by the
 *	Tcl interpreter whenever the usecAgentTime variable is read.
 *
 * Results:
 *      Always NULL.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char*
TraceAgentTime(clientData, interp, name1, name2, flags)
    ClientData clientData;
    Tcl_Interp *interp;
    char *name1;
    char *name2;
    int flags;
{
    char buf[20];
    sprintf(buf, "%u", Tnm_SnmpSysUpTime() / 100);
    Tcl_SetVar2(interp, name1, name2, buf, TCL_GLOBAL_ONLY);
    return NULL;
}
#endif

/*
 *----------------------------------------------------------------------
 *
 * TraceUnsignedInt --
 *
 *	This procedure writes the unsigned value pointed to by
 *	clientData into the Tcl variable under trace. Used to
 *	implement snmp statistics.
 *
 * Results:
 *      Always NULL.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char*
TraceUnsignedInt(clientData, interp, name1, name2, flags)
    ClientData clientData;
    Tcl_Interp *interp;
    char *name1;
    char *name2;
    int flags;
{
    char buf[20];
    sprintf(buf, "%u", *(unsigned *) clientData);
    Tcl_SetVar2(interp, name1, name2, buf, TCL_GLOBAL_ONLY);
    return NULL;    
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpAgentInit --
 *
 *	This procedure initializes the agent by registering some
 *	default MIB variables.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_SnmpAgentInit(interp, session)
    Tcl_Interp *interp;
    SNMP_Session *session;
{
    static int done = 0;
    char tclvar[80], buffer[255], *value;
    struct StatReg *p;

    if (Tnm_SnmpAgentOpen(session->agentInterp, session) != TCL_OK) {
	if (interp != session->agentInterp) {
	    Tcl_SetResult(interp, session->agentInterp->result, TCL_STATIC);
	}
	return TCL_ERROR;
    }

    /*
     * Make sure we are only called once - at least until we support
     * multiple agent entities in one scotty process.
     */

    if (done) {
	return TCL_OK;
    }

    done = 1;
    CacheInit();

#ifdef TNM_SNMPv2U
    /*
     * This is a hack. We are required to store these values
     * in NVRAM - I need to think about how to implement this.
     * For now, we assume that the agent rebooted every second
     * since the beginning of the USEC epoch (1/1/1996 0:0:0).
     */
    {
	u_char *p = session->agentID;
	int id = 1575;
	*p++ = (id >> 24) & 0xff;
	*p++ = (id >> 16) & 0xff;
	*p++ = (id >> 8) & 0xff;
	*p++ = id & 0xff;
	id = session->maddr.sin_addr.s_addr;
	*p++ = (id >> 24) & 0xff;
	*p++ = (id >> 16) & 0xff;
	*p++ = (id >> 8) & 0xff;
	*p++ = id & 0xff;
	memcpy(p, "tubs", 4);
    }
    session->agentTime = time((time_t *) NULL);
    session->agentBoots = session->agentTime - 820454400;
    Tnm_SnmpUsecSetAgentID(session);
#endif

    strcpy(buffer, "scotty agent");
    value = Tcl_GetVar2(interp, "tnm", "version", TCL_GLOBAL_ONLY);
    if (value) {
	strcat(buffer, " version ");
        strcat(buffer, value);
    }
    value = Tcl_GetVar2(interp, "tnm", "arch", TCL_GLOBAL_ONLY);
    if (value) {
	strcat(buffer, " (");
        strcat(buffer, value);
	strcat(buffer, ")");
    }

    /*
     * Copy the variable in a writable memory location because Tcl_SetVar
     * tries to modify the name which can cause core dumps depending on
     * how clever the compiler is.
     */

    Tnm_SnmpCreateNode(interp, "sysDescr.0", 
		       "tnm_system(sysDescr)", buffer);
    Tnm_SnmpCreateNode(interp, "sysObjectID.0", 
		       "tnm_system(sysObjectID)", "1.3.6.1.4.1.1575.1.1");
    Tnm_SnmpCreateNode(interp, "sysUpTime.0", 
		       "tnm_system(sysUpTime)", "0");
    Tcl_TraceVar2(interp, "tnm_system", "sysUpTime", 
		  TCL_TRACE_READS | TCL_GLOBAL_ONLY, 
		  TraceSysUpTime, (ClientData) NULL);
    Tnm_SnmpCreateNode(interp, "sysContact.0", 
		       "tnm_system(sysContact)", "");
    Tnm_SnmpCreateNode(interp, "sysName.0", 
		       "tnm_system(sysName)", "");
    Tnm_SnmpCreateNode(interp, "sysLocation.0", 
		       "tnm_system(sysLocation)", "");
    Tnm_SnmpCreateNode(interp, "sysServices.0", 
		       "tnm_system(sysServices)", "72");

    for (p = statTable; p->name; p++) {
	strcpy(tclvar, "tnm_snmp(");
	strcat(tclvar, p->name);
	strcat(tclvar, ")");
	Tnm_SnmpCreateNode(interp, p->name, tclvar, "0");
	Tcl_TraceVar2(interp, "tnm_snmp", p->name, 
		      TCL_TRACE_READS | TCL_GLOBAL_ONLY,
		      TraceUnsignedInt, (ClientData) p->value);
    }

    /* XXX snmpEnableAuthenTraps.0 should be implemented */

#ifdef TNM_SNMPv2U
    Tnm_SnmpBinToHex((char *) session->agentID, USEC_MAX_AGENTID, buffer);
    Tnm_SnmpCreateNode(interp, "agentID.0", "tnm_usec(agentID)", buffer);
    sprintf(buffer, "%u", session->agentBoots);
    Tnm_SnmpCreateNode(interp, "agentBoots.0", "tnm_usec(agentBoots)", buffer);
    Tnm_SnmpCreateNode(interp, "agentTime.0", "tnm_usec(agentTime)", "0");
    Tcl_TraceVar2(interp, "tnm_usec", "agentTime",
		  TCL_TRACE_READS | TCL_GLOBAL_ONLY, 
		  TraceAgentTime, (ClientData) NULL);
    sprintf(buffer, "%d", session->maxSize);
    Tnm_SnmpCreateNode(interp, "agentSize.0", "tnm_usec(agentSize)", buffer);
#endif

    Tcl_ResetResult(interp);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * FindInstance --
 *
 *	This procedure locates the instance given by the oid in the
 *	instance tree. Ignores all tree nodes without a valid syntax
 *	that are only used internally as non leaf nodes.
 *
 * Results:
 *      A pointer to the instance or NULL is not found.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static TnmSnmpNode*
FindInstance(session, oid, oidlen)
    SNMP_Session *session;
    Tnm_Oid *oid;
    int oidlen;
{
    TnmSnmpNode *inst = Tnm_SnmpFindNode(session, oid, oidlen);
    return (inst && inst->syntax) ? inst : NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * FindNextInstance --
 *
 *	This procedure locates the next instance given by the oid in
 *	the instance tree. Ignores all tree nodes without a valid 
 *	syntax that are only used internally as non leaf nodes.
 *
 * Results:
 *      A pointer to the instance or NULL is not found.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static TnmSnmpNode*
FindNextInstance(session, oid, oidlen)
    SNMP_Session *session;
    Tnm_Oid *oid;
    int oidlen;
{
    TnmSnmpNode *inst = Tnm_SnmpFindNextNode(session, oid, oidlen);
    return (inst && inst->syntax) ? inst : NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * GetRequest --
 *
 *	This procedure is called to process get, getnext and getbulk
 *	requests.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
GetRequest(interp, session, request, response)
    Tcl_Interp *interp;
    SNMP_Session *session;
    SNMP_PDU *request;
    SNMP_PDU *response;
{
    int i, code, oidlen;
    Tnm_Oid *oid;
    SNMP_VarBind *inVarBindPtr;
    int inVarBindSize;
    TnmSnmpNode *inst;

    code = Tnm_SnmpSplitVBList(interp, Tcl_DStringValue(&request->varbind),
			       &inVarBindSize, &inVarBindPtr);
    if (code != TCL_OK) {
	return TCL_ERROR;
    }

    for (i = 0; i < inVarBindSize; i++) {

	char *value, *syntax;

	oid = Tnm_StrToOid(inVarBindPtr[i].soid, &oidlen);
	if (request->type == TNM_SNMP_GETNEXT 
	    || request->type == TNM_SNMP_GETBULK) {
	    inst = FindNextInstance(session, oid, oidlen);
	} else {
	    inst = FindInstance(session, oid, oidlen);
	}

	if (! inst) {

	    /*
	     * SNMPv1 handles this case by sending back an error PDU
	     * while SNMPv2 uses exceptions. We create an exception
	     * varbind to reflect the error types defined in RFC 1905.
	     */

	    if (session->version == TNM_SNMPv1) {
		response->error_status = TNM_SNMP_NOSUCHNAME;
		snmpStats.snmpOutNoSuchNames++;
		goto varBindError;
	    }

	    Tcl_DStringStartSublist(&response->varbind);
	    Tcl_DStringAppendElement(&response->varbind, inVarBindPtr[i].soid);
	    if (request->type == TNM_SNMP_GET) {
		Tnm_MibNode *nodePtr = Tnm_MibFindNode(inVarBindPtr[i].soid, 
						       NULL, 0);
		if (!nodePtr || nodePtr->childPtr) {
		    Tcl_DStringAppendElement(&response->varbind,
					     "noSuchObject");
		} else {
		    Tcl_DStringAppendElement(&response->varbind, 
					     "noSuchInstance");
		}
	    } else {
		Tcl_DStringAppendElement(&response->varbind, "endOfMibView");
	    }
	    Tcl_DStringAppendElement(&response->varbind, "");
	    Tcl_DStringEndSublist(&response->varbind);
	    continue;
	}

	Tcl_DStringStartSublist(&response->varbind);
	Tcl_DStringAppendElement(&response->varbind, inst->label);
	syntax = TnmGetTableValue(tnmSnmpTypeTable, inst->syntax);
	Tcl_DStringAppendElement(&response->varbind, syntax ? syntax : "");
	code = Tnm_SnmpEvalNodeBinding(session, request, inst, 
			       TNM_SNMP_GET_EVENT, inVarBindPtr[i].value, 
			       (char *) NULL);
	if (code == TCL_ERROR) {
	    goto varBindTclError;
	}
	value = Tcl_GetVar(interp, inst->tclVarName, 
			   TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG);
	if (!value) {
	    response->error_status = TNM_SNMP_GENERR;
	    goto varBindError;
	}
	Tcl_DStringAppendElement(&response->varbind, value);
	Tcl_ResetResult(interp);

	snmpStats.snmpInTotalReqVars++;

	Tcl_DStringEndSublist(&response->varbind);
	
	continue;

      varBindTclError:
	response->error_status = TnmGetTableKey(tnmSnmpErrorTable, 
						interp->result);
	if (response->error_status < 0) {
	    response->error_status = TNM_SNMP_GENERR;
	}
	snmpStats.snmpOutGenErrs += 
	    (response->error_status == TNM_SNMP_GENERR);
       
      varBindError:
	response->error_index = i+1;
	break;
    }

    /*
     * We check here if the varbind length in string representation
     * exceeds our buffer used to build the packet. This is not always
     * correct, but we should be on the safe side in most cases.
     */

    if (Tcl_DStringLength(&response->varbind) >= TNM_SNMP_MAXSIZE) {
	response->error_status = TNM_SNMP_TOOBIG;
	response->error_index = 0;
    }

    Tnm_SnmpFreeVBList(inVarBindSize, inVarBindPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * SetRequest --
 *
 *	This procedure is called to process set requests which is
 *	much more difficult to do than get* requests.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
SetRequest(interp, session, request, response)
    Tcl_Interp *interp;
    SNMP_Session *session;
    SNMP_PDU *request;
    SNMP_PDU *response;
{
    int i, code, oidlen;
    Tnm_Oid *oid;
    SNMP_VarBind *inVarBindPtr;
    int inVarBindSize;
    TnmSnmpNode *inst;
    int varsToRollback = 0;

    code = Tnm_SnmpSplitVBList(interp, Tcl_DStringValue(&request->varbind),
			       &inVarBindSize, &inVarBindPtr);
    if (code != TCL_OK) {
	return TCL_ERROR;
    }

    for (i = 0; i < inVarBindSize; i++) {

	char *value, *syntax;
	int setAlreadyDone = 0;
	varsToRollback = i;
	
	oid = Tnm_StrToOid(inVarBindPtr[i].soid, &oidlen);
	inst = FindInstance(session, oid, oidlen);

	if (! inst) {
	    char *access = Tnm_MibGetAccess(inVarBindPtr[i].soid, 0);
	    if (access && (strcmp(access, "read-create") == 0)) {
		char *name = Tnm_MibGetName(inVarBindPtr[i].soid, 0);
		char *tmp = ckalloc(strlen(name) + 2);
		char *c = tmp;
		strcpy(tmp, name);
		for (c = tmp; *c && *c != '.'; c++);
		if (*c) *c = '(';
		while (*c) c++;
		*c++ = ')';
		*c = '\0';
		Tnm_SnmpCreateNode(interp, inVarBindPtr[i].soid, tmp, "");
		inVarBindPtr[i].flags |= NODE_CREATED;
		ckfree(tmp);
		inst = FindInstance(session, oid, oidlen);
		if (! inst) {
		    response->error_status = TNM_SNMP_NOCREATION;
		    varsToRollback--;
		    goto varBindError;
		}
		code = Tnm_SnmpEvalNodeBinding(session, request, inst, 
				    TNM_SNMP_CREATE_EVENT, inVarBindPtr[i].value,
				    (char *) NULL);
		if (code == TCL_ERROR) {
		    goto varBindTclError;
		}
		if (code != TCL_BREAK) {
		    Tcl_SetVar(interp, inst->tclVarName, 
			       inVarBindPtr[i].value, TCL_GLOBAL_ONLY);
		}
		setAlreadyDone = 1;
	    } else {
		response->error_status = TNM_SNMP_NOCREATION;
		varsToRollback--;
		goto varBindError;
	    }
	}

	if (!setAlreadyDone) {

	    /*
	     * Check if the instance is writable.
	     */

	    if (inst->access == TNM_MIB_READONLY) {
		response->error_status = TNM_SNMP_NOTWRITABLE;
		varsToRollback--;
		goto varBindError;
	    }

	    /*
	     * Check if the received value is of approriate type.
	     */
	    
	    if (TnmGetTableKey(tnmSnmpTypeTable, 
			       inVarBindPtr[i].syntax) != inst->syntax) {
		response->error_status = TNM_SNMP_WRONGTYPE;
		varsToRollback--;
		goto varBindError;
	    }

	    value = Tcl_GetVar(interp, inst->tclVarName, 
			       TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG);
	    inVarBindPtr[i].clientData = (ClientData) ckstrdup(value);
	    code = Tnm_SnmpEvalNodeBinding(session, request, inst, 
					TNM_SNMP_SET_EVENT, inVarBindPtr[i].value,
					(char *) inVarBindPtr[i].clientData);
	    if (code == TCL_ERROR) {
		goto varBindTclError;
	    }
	    if (code != TCL_BREAK) {
	        Tcl_SetVar(interp, inst->tclVarName, 
			   inVarBindPtr[i].value, TCL_GLOBAL_ONLY);
	    }
	    snmpStats.snmpInTotalSetVars++;
	}
	
	Tcl_DStringStartSublist(&response->varbind);
	Tcl_DStringAppendElement(&response->varbind, inst->label);
	syntax = TnmGetTableValue(tnmSnmpTypeTable, inst->syntax);
	Tcl_DStringAppendElement(&response->varbind, syntax ? syntax : "");
	value = Tcl_GetVar(interp, inst->tclVarName, 
			   TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG);
	if (!value) {
	    response->error_status = TNM_SNMP_GENERR;
	    goto varBindError;
	}
	Tcl_DStringAppendElement(&response->varbind, value);
	Tcl_ResetResult(interp);

	Tcl_DStringEndSublist(&response->varbind);
	
	continue;

      varBindTclError:
	response->error_status = TnmGetTableKey(tnmSnmpErrorTable, 
						interp->result);
	if (response->error_status < 0) {
	    response->error_status = TNM_SNMP_GENERR;
	}
	snmpStats.snmpOutGenErrs += 
	    (response->error_status == TNM_SNMP_GENERR);
       
      varBindError:
	response->error_index = i+1;
	break;
    }

    /*
     * We check here if the varbind length in string representation
     * exceeds our buffer used to build the packet. This is not always
     * correct, but we should be on the safe side in most cases.
     */
    
    if (Tcl_DStringLength(&response->varbind) >= TNM_SNMP_MAXSIZE) {
	response->error_status = TNM_SNMP_TOOBIG;
	response->error_index = 0;
    }

    /*
     * Another check for consistency errors before we start 
     * the commit/rollback phase. This additional check was 
     * suggested by Peter.Polkinghorne@gec-hrc.co.uk.
     */
    
    if (response->error_status == TNM_SNMP_NOERROR) {
	for (i = 0; i < inVarBindSize; i++) {
	    oid = Tnm_StrToOid(inVarBindPtr[i].soid, &oidlen);
	    inst = FindInstance(session, oid, oidlen);
	    if (inst) {
		code = Tnm_SnmpEvalNodeBinding(session, request, inst, 
				    TNM_SNMP_CHECK_EVENT, inVarBindPtr[i].value,
				    (char *) inVarBindPtr[i].clientData);
	    } else {
	        Tcl_ResetResult(interp);
	        code = TCL_ERROR;
	    }

	    if (code != TCL_OK) {
		response->error_status = TnmGetTableKey(tnmSnmpErrorTable, 
							interp->result);
		if (response->error_status < 0) {
		    response->error_status = TNM_SNMP_GENERR;
		}
		snmpStats.snmpOutGenErrs +=
		    (response->error_status == TNM_SNMP_GENERR);
		response->error_index = i+1;
		break;
	    }
	}
    }

    /*
     * We now start the commit/rollback phase. Note, we must be
     * careful to do rollbacks in the correct order and only
     * on those instances that were actually processed.
     */
	
    if (response->error_status == TNM_SNMP_NOERROR) {

        /*
	 * Evaluate commit bindings if we have no error yet.
	 * Ignore all errors now since we have already decided 
	 * that this PDU has been processed successfully.
	 */
      
        for (i = 0; i < inVarBindSize; i++) {
	    oid = Tnm_StrToOid(inVarBindPtr[i].soid, &oidlen);
	    inst = FindInstance(session, oid, oidlen);
	    if (inst) {
	        Tnm_SnmpEvalNodeBinding(session, request, inst,
			     TNM_SNMP_COMMIT_EVENT, inVarBindPtr[i].value,
			     (char *) inVarBindPtr[i].clientData);
	    }
	    if (inVarBindPtr[i].clientData) {
	        char *oldValue = (char *) inVarBindPtr[i].clientData;
	        ckfree(oldValue);
	        inVarBindPtr[i].clientData = NULL;
	    }
	}

    } else {

        /*
	 * Evaluate the rollback bindings for all the instances
	 * that have been processed so far. Ignore all errors now
	 * as they would overwrite the error that caused us to
	 * rollback the set request processing.
	 */

        for (i = varsToRollback; i >= 0; i--) {
	    oid = Tnm_StrToOid(inVarBindPtr[i].soid, &oidlen);
	    inst = FindInstance(session, oid, oidlen);
	    if (inst) {
	        Tnm_SnmpEvalNodeBinding(session, request, inst, 
			     TNM_SNMP_ROLLBACK_EVENT, inVarBindPtr[i].value,
			     (char *) inVarBindPtr[i].clientData);
		if (inVarBindPtr[i].flags & NODE_CREATED) {
		    Tcl_UnsetVar(interp, inst->tclVarName, TCL_GLOBAL_ONLY);
		} else if (inVarBindPtr[i].clientData) {
		    Tcl_SetVar(interp, inst->tclVarName,
			       (char *) inVarBindPtr[i].clientData,
			       TCL_GLOBAL_ONLY);
		}
		if (inVarBindPtr[i].clientData) {
		    ckfree((char *) inVarBindPtr[i].clientData);
		    inVarBindPtr[i].clientData = NULL;
		}
	    }
	}
    }

    Tnm_SnmpFreeVBList(inVarBindSize, inVarBindPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpAgentRequest --
 *
 *	This procedure is called when the agent receives a get, getnext
 *	getbulk or set request. It splits the varbind, looks up the 
 *	variables and assembles a response.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_SnmpAgentRequest(interp, session, pdu)
    Tcl_Interp *interp;
    SNMP_Session *session;
    SNMP_PDU *pdu;
{
    int rc;
    SNMP_PDU *reply;

    switch (pdu->type) {
      case TNM_SNMP_GET: 
	  snmpStats.snmpInGetRequests++;
	  break;
      case TNM_SNMP_GETNEXT:
	  snmpStats.snmpInGetNexts++;
	  break;
      case TNM_SNMP_GETBULK:
	  break;
      case TNM_SNMP_SET:
	  snmpStats.snmpInSetRequests++;
	  break;
      case TNM_SNMP_INFORM:
	  break;
    }

    if (pdu->type == TNM_SNMP_SET) {
	CacheClear(session);
    }

    reply = CacheHit(session, pdu);
    if (reply != NULL) {
	rc = Tnm_SnmpEncode(interp, session, reply, NULL, NULL);
	return rc;
    }

    Tnm_SnmpEvalBinding(interp, session, pdu, TNM_SNMP_BEGIN_EVENT);

    reply = CacheGet(session, pdu);

    if (pdu->type == TNM_SNMP_SET) {
	rc = SetRequest(interp, session, pdu, reply);
    } else {
	rc = GetRequest(interp, session, pdu, reply);
    }
    if (rc != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * Throw the reply varbind away and re-use the one in the original
     * request if we had an error.
     */

    if (reply->error_status != TNM_SNMP_NOERROR) {
	Tcl_DStringFree(&reply->varbind);
	Tcl_DStringAppend(&reply->varbind,
			  Tcl_DStringValue(&pdu->varbind),
			  Tcl_DStringLength(&pdu->varbind));
    }
 
    reply->type = TNM_SNMP_RESPONSE;
    reply->request_id = pdu->request_id;

    Tnm_SnmpEvalBinding(interp, session, reply, TNM_SNMP_END_EVENT);

    if (Tnm_SnmpEncode(interp, session, reply, NULL, NULL) != TCL_OK) {
	Tcl_AddErrorInfo(interp, "\n    (snmp send reply)");
	Tcl_BackgroundError(interp);
	Tcl_ResetResult(interp);
	reply->error_status = TNM_SNMP_GENERR;
	Tcl_DStringFree(&reply->varbind);
        Tcl_DStringAppend(&reply->varbind,
			  Tcl_DStringValue(&pdu->varbind),
			  Tcl_DStringLength(&pdu->varbind));
	return Tnm_SnmpEncode(interp, session, reply, NULL, NULL);
    } else {
	return TCL_OK;
    }
}
