/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 

#ifdef UNIX
#include <sys/types.h>
#include <netinet/in.h>
#endif // UNIX

#include <string.h>
#include <math.h>

#include "mdpMessage.h"
#include "debug.h"

#include "sysdefs.h"

#ifdef NS2
double MDP_BACKOFF_WINDOW = 2.0; // Number of GRTT periods for NACK backoff
unsigned int MDP_GROUP_SIZE = 1000;
#endif // NS2

// These are very conservative Pack() & Unpack() message routines ...
// If we pay more attention to Mdp protocol field alignment and to
// the ability of various machines/compilers to follow alignment
// rules as we would expect, these routines could be optimized somewhat
// Note that protocol field alignment may result in some wasted protocol
// header space ... we'll go with the implementation below until some
// committee forces us into something else (good or bad)

// (TBD) Get rid of the use of "temp_short" & "temp_long" even though
//       it will possibly invalidate messages after packing ???

int MdpMessage::Pack(char *buffer)
{
	register unsigned int len = 0;
    register unsigned long temp_long;
    register unsigned short temp_short;
    
    // If these assertions fail, the code base needs some reworking
    // for compiler with different size settings (or set compiler
    // settings so these values work)
    ASSERT(sizeof(char) == 1);
    ASSERT(sizeof(short) == 2);
    ASSERT(sizeof(long) == 4);
    
    // Pack common fields first
    buffer[len++] = type;
    buffer[len++] = version;
    temp_long = htonl(sender);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);

    // Now pack fields specific to message type    
    switch(type)
    {	
	    case MDP_REPORT:
            buffer[len++] = report.status;
            buffer[len++] = report.flavor;
            switch(report.flavor)
            {
                case MDP_REPORT_HELLO:
                    strncpy(&buffer[len], report.node_name, MDP_NODE_NAME_MAX);
                    len += MDP_NODE_NAME_MAX;
                    if (report.status & MDP_CLIENT)
                        len += report.client_stats.Pack(&buffer[len]);
                    break;
                    
                default:
                    DMSG(0, "MdpMessage::Pack() Invalid report type!\n");
                    return 0;
            }
	        break;

        case MDP_INFO:
            temp_short = htons(object.sequence);    // sequence
            memcpy(&buffer[len], &temp_short, sizeof(short));
            len += sizeof(short);
            
            temp_long = htonl(object.object_id);      // object_id
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);
            
            temp_long = htonl(object.object_size);    // object_size
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);
            
            buffer[len++] = object.ndata;             // ndata
            buffer[len++] = object.nparity;           // nparity
            buffer[len++] = object.flags;             // flags
			buffer[len++] = object.grtt;              // grtt
            
            temp_short = htons(object.info.segment_size);  // segment_size
            memcpy(&buffer[len], &temp_short, sizeof(short));
            len += sizeof(short);
            
            // Copy the info data into the packet
            memcpy(&buffer[len], object.info.data, object.info.len); // info content
            
            if (object.flags & MDP_DATA_FLAG_FILE)
            {
                // Properly convert file name DIR_DELIMITERs
                char* ptr = buffer + len;
                char* end = ptr + object.info.len;
			    while((ptr < end) && (*ptr != '\0'))
			    {
                    if (DIR_DELIMITER == *ptr) *ptr = '/';
                    ptr++;
			    }
            }
            len += object.info.len;       
            break;

        case MDP_DATA:
            temp_short = htons(object.sequence);      // sequence
            memcpy(&buffer[len], &temp_short, sizeof(short));
            len += sizeof(short);
            
            temp_long = htonl(object.object_id);      // object_id
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);
            
            temp_long = htonl(object.object_size);    // object_size
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);
            
            buffer[len++] = object.ndata;             // ndata
            buffer[len++] = object.nparity;           // nparity
            buffer[len++] = object.flags;             // flags
            buffer[len++] = object.grtt;              // grtt
            
            temp_long = htonl(object.data.offset);         // offset
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);
            
            
            // Pack optional "seg_size" field for "RUNT" data msg
            if (object.flags & MDP_DATA_FLAG_RUNT)
            {
                temp_short = htons(object.data.segment_size);  // opt. segment_size
                memcpy(&buffer[len], &temp_short, sizeof(short));
                len += sizeof(short);  
            }
           
            memcpy(&buffer[len], object.data.data, object.data.len);  // data
            len += object.data.len;
            break;

        case MDP_PARITY:
            temp_short = htons(object.sequence);      // sequence
            memcpy(&buffer[len], &temp_short, sizeof(short));
            len += sizeof(short);
            
            temp_long = htonl(object.object_id);      // object_id
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);
            
            temp_long = htonl(object.object_size);    // object_size
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);
            
            buffer[len++] = object.ndata;             // ndata
            buffer[len++] = object.nparity;           // nparity
            buffer[len++] = object.flags;             // flags
            buffer[len++] = object.grtt;              // grtt
            
            temp_long = htonl(object.parity.offset);         // offset
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);
            
            buffer[len++] = object.parity.id;                // id 
            
            memcpy(&buffer[len], object.parity.data, object.parity.len);      // data
            len += object.parity.len;
            break;
            
        case MDP_CMD:
            temp_short = htons(cmd.sequence);                   // sequence
            memcpy(&buffer[len], &temp_short, sizeof(short));
            len += sizeof(short);
            buffer[len++] = cmd.grtt;                           // grtt
            
            buffer[len++] = cmd.flavor;                         // flavor
            switch(cmd.flavor)
            {
                case MDP_CMD_FLUSH:
                    buffer[len++] = cmd.flush.flags;
                    temp_long = htonl(cmd.flush.object_id);     // object_id
                    memcpy(&buffer[len], &temp_long, sizeof(long));
                    len += sizeof(long);
                    break;
                    
                case MDP_CMD_SQUELCH:
                    temp_long = htonl(cmd.squelch.sync_id);   // object_id
                    memcpy(&buffer[len], &temp_long, sizeof(long));
                    len += sizeof(long);
                    memcpy(&buffer[len], cmd.squelch.data, cmd.squelch.len);
                    len += cmd.squelch.len;
                    break;
                    
                case MDP_CMD_ACK_REQ:
                    temp_long = htonl(cmd.ack_req.object_id);   // object_id
                    memcpy(&buffer[len], &temp_long, sizeof(long));
                    len += sizeof(long);
                    
                    memcpy(&buffer[len], cmd.ack_req.data, 
                           cmd.ack_req.len);                    // data
                    len += cmd.ack_req.len;
                    break;
                    
                case MDP_CMD_GRTT_REQ:
                    buffer[len++] = cmd.grtt_req.flags;         // flags
                    
                    buffer[len++] = cmd.grtt_req.sequence;       // sequence
                    
                    temp_long = htonl(cmd.grtt_req.send_time.tv_sec);
                    memcpy(&buffer[len], &temp_long, sizeof(long));
                    len += sizeof(long);                        // send_time
                    temp_long = htonl(cmd.grtt_req.send_time.tv_usec);
                    memcpy(&buffer[len], &temp_long, sizeof(long));
                    len += sizeof(long);

                    temp_long = htonl(cmd.grtt_req.hold_time.tv_sec);
                    memcpy(&buffer[len], &temp_long, sizeof(long));
                    len += sizeof(long);                        // hold_time
                    temp_long = htonl(cmd.grtt_req.hold_time.tv_usec);
                    memcpy(&buffer[len], &temp_long, sizeof(long));
                    len += sizeof(long);
                    
                    temp_short = htons(cmd.grtt_req.segment_size); // segment_size
                    memcpy(&buffer[len], &temp_short, sizeof(short));
                    len += sizeof(short); 
                    
                    temp_long = htonl(cmd.grtt_req.rate);       // rate
                    memcpy(&buffer[len], &temp_long, sizeof(long));
                    len += sizeof(long);
                    
                    buffer[len++] = cmd.grtt_req.rtt;           // rtt
                    
                    temp_short = htons(cmd.grtt_req.loss);      // loss
                    memcpy(&buffer[len], &temp_short, sizeof(short));
                    len += sizeof(short);
                    
                    memcpy(&buffer[len], cmd.grtt_req.data, cmd.grtt_req.len);
                    len += cmd.grtt_req.len;                    // data
                    break;
                    
                case MDP_CMD_NACK_ADV:
                    memcpy(&buffer[len], cmd.nack_adv.data, cmd.nack_adv.len);
                    len += cmd.nack_adv.len;                    // data
                    break;
                
                default:
                    DMSG(0, "mdp: MdpMessage::Pack() : Bad server cmd!\n");
                    return 0;
            }
            break;  
            
        case MDP_NACK:
            temp_long = htonl(nack.server_id);          // server_id
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);
            
            temp_long = htonl(nack.grtt_response.tv_sec);
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);                        // grtt_response
            temp_long = htonl(nack.grtt_response.tv_usec);
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);
            
            temp_short = htons(nack.loss_estimate);
            memcpy(&buffer[len], &temp_short, sizeof(short));
            len += sizeof(short);                       // loss_estimate
            
            buffer[len++] = nack.grtt_req_sequence;     // grtt_req_sequence
            
            memcpy(&buffer[len], nack.data, nack.nack_len);  // data
            len += nack.nack_len;
            break;
            
        case MDP_ACK:
            temp_long = htonl(ack.server_id);           // server_id
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);
            
            temp_long = htonl(ack.grtt_response.tv_sec);
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);                        // grtt_response
            temp_long = htonl(ack.grtt_response.tv_usec);
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);
            
            temp_short = htons(ack.loss_estimate);
            memcpy(&buffer[len], &temp_short, sizeof(short));
            len += sizeof(short);                       // loss_estimate
            
            buffer[len++] = ack.grtt_req_sequence;      // grtt_req_sequence
            
            buffer[len++] = ack.type;                   // type
            
            temp_long = htonl(ack.object_id);           // object_id
            memcpy(&buffer[len], &temp_long, sizeof(long));
            len += sizeof(long);
            break;

	    default:
	        DMSG(0, "mdp: MdpMessage::Pack() : Bad message type!\n");
            return 0;
	        break;
    }
    return len;
}  // end MdpMessage::Pack()

bool MdpMessage::Unpack(char *buffer, int packet_length)
{
	register unsigned int len = 0;
    
    // Unpack common fields first.
    type = (unsigned char) buffer[len++];
    version = (unsigned char) buffer[len++];
    memcpy(&sender, &buffer[2], sizeof(long));
    sender = ntohl(sender);
    len += sizeof(long);
    
    // Now unpack fields unique to message type
    switch(type)
    {
	    case MDP_REPORT:
            report.status = (unsigned char) buffer[len++];
            report.flavor = (unsigned char) buffer[len++];
            switch(report.flavor)
            {
                case MDP_REPORT_HELLO:
                    strncpy(report.node_name, &buffer[len], MDP_NODE_NAME_MAX);
                    len += MDP_NODE_NAME_MAX;
                    if (report.status & MDP_CLIENT)
                        len += report.client_stats.Unpack(&buffer[len]); 
                    break;
                    
                default:
                    DMSG(0, "MdpMessage::Unpack(): Invalid report type!\n");
                    return false;
            }
            return true;
            
        case MDP_INFO:
            memcpy(&object.sequence, &buffer[len], sizeof(short));
            object.sequence = ntohs(object.sequence);       // sequence
            len += sizeof(short);
            
            memcpy(&object.object_id, &buffer[len], sizeof(long));
            object.object_id = ntohl(object.object_id);     // object_id
            len += sizeof(long);
            
            memcpy(&object.object_size, &buffer[len], sizeof(long));
            object.object_size = ntohl(object.object_size); // object_size
            len += sizeof(long);
            
            object.ndata = buffer[len++];                 // ndata
            object.nparity = buffer[len++];               // nparity
            object.flags = buffer[len++];                 // flags
			object.grtt = buffer[len++];                  // grtt
            
            memcpy(&object.info.segment_size, &buffer[len], sizeof(short));
            object.info.segment_size = ntohs(object.info.segment_size);  
            len += sizeof(short);                       // segment_size
            
            object.info.len = packet_length - len;             // info len
            object.info.data = &buffer[len];                   // info content
            
            // (TBD) This file hack stuff needs to go somewhere else
            if (object.flags & MDP_DATA_FLAG_FILE)
            {
                // Properly convert fifaille name DIR_DELIMITERs
                char* ptr = object.info.data;
			    char* end = ptr + object.info.len;
			    while((ptr < end) && (*ptr != '\0'))
			    {
                    if ('/' == *ptr) *ptr = DIR_DELIMITER;
                    ptr++;
			    }
            }
            return true;
        
        case MDP_DATA:
            memcpy(&object.sequence, &buffer[len], sizeof(short));
            object.sequence = ntohs(object.sequence);       // sequence
            len += sizeof(short);
            
            memcpy(&object.object_id, &buffer[len], sizeof(long));
            object.object_id = ntohl(object.object_id);     // object_id
            len += sizeof(long);
            
            memcpy(&object.object_size, &buffer[len], sizeof(long));
            object.object_size = ntohl(object.object_size); // object_size
            len += sizeof(long);
            
            object.ndata = (unsigned char) buffer[len++]; // ndata
            object.nparity = (unsigned char) buffer[len++]; // nparity
            object.flags = (unsigned char) buffer[len++]; // flags
            object.grtt = buffer[len++];                  // grtt
            
            memcpy(&object.data.offset, &buffer[len], sizeof(long));
            object.data.offset = ntohl(object.data.offset);           // offset
            len += sizeof(long);
            
            if (object.flags & MDP_DATA_FLAG_RUNT)    // opt. segment_size needed
            {
                memcpy(&object.data.segment_size, &buffer[len], sizeof(short));
                object.data.segment_size = ntohs(object.data.segment_size);
                len += sizeof(short);
            }
            object.data.len = packet_length - len;
            object.data.data = &buffer[len];
            return true;
        
        case MDP_PARITY:
            memcpy(&object.sequence, &buffer[len], sizeof(short));
            object.sequence = ntohs(object.sequence);       // sequence
            len += sizeof(short);
            
            memcpy(&object.object_id, &buffer[len], sizeof(long));
            object.object_id = ntohl(object.object_id);     // object_id
            len += sizeof(long);
            
            memcpy(&object.object_size, &buffer[len], sizeof(long));
            object.object_size = ntohl(object.object_size); // object_size
            len += sizeof(long);
            
            object.ndata = (unsigned char) buffer[len++]; // ndata
            object.nparity = (unsigned char) buffer[len++]; // nparity
            object.flags = (unsigned char) buffer[len++]; // flags
            object.grtt = buffer[len++];                  // grtt
            
            memcpy(&object.parity.offset, &buffer[len], sizeof(long));
            object.parity.offset = ntohl(object.parity.offset);           // offset
            len += sizeof(long);
            
            object.parity.id = (unsigned char) buffer[len++];      // id
                
            object.parity.len = packet_length - len;
            object.parity.data = &buffer[len];
            return true;
            
        case MDP_CMD:
            memcpy(&cmd.sequence, &buffer[len], sizeof(short));
            cmd.sequence = ntohs(cmd.sequence);       // sequence
            len += sizeof(short);
            cmd.grtt = buffer[len++];                 // grtt
                    
            cmd.flavor = buffer[len++];
            switch (cmd.flavor)
            {
                case MDP_CMD_FLUSH:
                    cmd.flush.flags = buffer[len++];
                    memcpy(&cmd.flush.object_id, &buffer[len], sizeof(long));
                    cmd.flush.object_id = 
                            ntohl(cmd.flush.object_id);    // object_id  
                    break;
                
                case MDP_CMD_SQUELCH:
                    memcpy(&cmd.squelch.sync_id, &buffer[len], sizeof(long));
                    cmd.squelch.sync_id = 
                            ntohl(cmd.squelch.sync_id);   // sync_id
                    len += sizeof(long);
                    cmd.squelch.len = packet_length - len;  // len 
                    cmd.squelch.data = &buffer[len];        // data
                    break;
                    
                case MDP_CMD_ACK_REQ:
                    memcpy(&cmd.ack_req.object_id, &buffer[len], sizeof(long));
                    cmd.ack_req.object_id = 
                            ntohl(cmd.ack_req.object_id);   // object_id
                    len += sizeof(long);
                    
                    cmd.ack_req.len = packet_length - len;  // len 
                    cmd.ack_req.data = &buffer[len];        // data
                    break;
                
                case MDP_CMD_GRTT_REQ:
                    cmd.grtt_req.flags = buffer[len++];     // flags
                    
                    cmd.grtt_req.sequence = buffer[len++];  // sequence
                    
                    memcpy(&cmd.grtt_req.send_time.tv_sec, &buffer[len], sizeof(long));
                    cmd.grtt_req.send_time.tv_sec = ntohl(cmd.grtt_req.send_time.tv_sec);
                    len += sizeof(long);                    // send_time
                    memcpy(&cmd.grtt_req.send_time.tv_usec, &buffer[len], sizeof(long));
                    cmd.grtt_req.send_time.tv_usec = ntohl(cmd.grtt_req.send_time.tv_usec);
                    len += sizeof(long);

                    memcpy(&cmd.grtt_req.hold_time.tv_sec, &buffer[len], sizeof(long));
                    cmd.grtt_req.hold_time.tv_sec = ntohl(cmd.grtt_req.hold_time.tv_sec);
                    len += sizeof(long);                    // hold_time
                    memcpy(&cmd.grtt_req.hold_time.tv_usec, &buffer[len], sizeof(long));
                    cmd.grtt_req.hold_time.tv_usec = ntohl(cmd.grtt_req.hold_time.tv_usec);
                    len += sizeof(long);
                    memcpy(&cmd.grtt_req.segment_size, &buffer[len], sizeof(short));
                    cmd.grtt_req.segment_size = ntohs(cmd.grtt_req.segment_size);
                    len += sizeof(short);                   // segment_size
                    memcpy(&cmd.grtt_req.rate, &buffer[len], sizeof(long));
                    cmd.grtt_req.rate = ntohl(cmd.grtt_req.rate);
                    len += sizeof(long);                    // rate
                    
                    cmd.grtt_req.rtt = buffer[len++];       // rtt
                    
                    memcpy(&cmd.grtt_req.loss, &buffer[len], sizeof(short));
                    cmd.grtt_req.loss = ntohs(cmd.grtt_req.loss);
                    len += sizeof(short);                   // loss
                    
                    cmd.grtt_req.len = packet_length - len; // len
                    cmd.grtt_req.data = &buffer[len];       // data
                    break;
                
                case MDP_CMD_NACK_ADV:
                    cmd.nack_adv.data = &buffer[len];       // data
                    cmd.nack_adv.len = packet_length - len; // len
                    break;
                                            
                default:
                    type = MDP_MSG_INVALID;
                    return false;
            }
            return true;
            
        case MDP_NACK:
            memcpy(&nack.server_id, &buffer[len], sizeof(long));
            nack.server_id = ntohl(nack.server_id); // server_id
            len += sizeof(long);
            
            memcpy(&nack.grtt_response.tv_sec, &buffer[len], sizeof(long));
            nack.grtt_response.tv_sec = ntohl(nack.grtt_response.tv_sec);
            len += sizeof(long);                    // grtt_response
            memcpy(&nack.grtt_response.tv_usec, &buffer[len], sizeof(long));
            nack.grtt_response.tv_usec = ntohl(nack.grtt_response.tv_usec);
            len += sizeof(long);
            
            memcpy(&nack.loss_estimate, &buffer[len], sizeof(short));
            nack.loss_estimate = ntohs(nack.loss_estimate);
            len += sizeof(short);                   // loss_estimate
            
            nack.grtt_req_sequence = buffer[len++]; // grtt_req_sequence
            
            nack.nack_len = packet_length - len;    // len
            nack.data = &buffer[len];               // data           
            return true;
            
        case MDP_ACK:
            memcpy(&ack.server_id, &buffer[len], sizeof(long));
            ack.server_id = ntohl(ack.server_id);   // server_id
            len += sizeof(long);
            
            memcpy(&ack.grtt_response.tv_sec, &buffer[len], sizeof(long));
            ack.grtt_response.tv_sec = ntohl(ack.grtt_response.tv_sec);
            len += sizeof(long);                    // grtt_response
            memcpy(&ack.grtt_response.tv_usec, &buffer[len], sizeof(long));
            ack.grtt_response.tv_usec = ntohl(ack.grtt_response.tv_usec);
            len += sizeof(long);
            
            memcpy(&ack.loss_estimate, &buffer[len], sizeof(short));
            ack.loss_estimate = ntohs(ack.loss_estimate);
            len += sizeof(short);                   // loss_estimate
            
            ack.grtt_req_sequence = buffer[len++];  // grtt_req_sequence
            
            ack.type = (MdpAckType) buffer[len++];  // type
            
            memcpy(&ack.object_id, &buffer[len], sizeof(long));
            ack.object_id = ntohl(ack.object_id);   // object_id
            
            return true;

	    default:
	        type = MDP_MSG_INVALID;
	        return false;
    }
}  // end MdpMessage::Unpack()

/***********************************************************************
 *  MdpMessageQueue implementation
 */

// These are the priorities used for queuing messages
// Bigger number is higher priority (FIFO otherwise)
static const int MSG_PRIORITY[] = 
{
    -1,     // MDP_INVALID_MSG
     0,     // MDP_REPORT
    10,     // MDP_INFO
     6,     // MDP_DATA
     8,     // MDP_PARITY
    12,     // MDP_CMD
    12,     // MDP_NACK
    12      // MDP_ACK
};

MdpMessageQueue::MdpMessageQueue()
    : head(NULL), tail(NULL)
{
}

/*void MdpMessageQueue::QueueMessage(MdpMessage *theMsg)
{
    if ((theMsg->prev = tail))
        theMsg->prev->next = theMsg;
    else
        head = theMsg;
    tail = theMsg;
    theMsg->next = NULL;
}  // end MdpMessageQueue::QueueMessage()*/


void MdpMessageQueue::QueueMessage(MdpMessage *theMsg)
{
    MdpMessage *prev = tail;
    while (prev)
    {
        if (MSG_PRIORITY[theMsg->type] > MSG_PRIORITY[prev->type])
        {
            prev = prev->prev;  // Go up the queue
        }
        else
        {
            theMsg->prev = prev;
            if ((theMsg->next = prev->next))
                theMsg->next->prev = theMsg;
            else
                tail = theMsg;
            prev->next = theMsg;
            return;
        }
    }
    
    // theMsg goes to top of the queue
    if ((theMsg->next = head))
        head->prev = theMsg;
    else
        tail = theMsg;
    theMsg->prev = NULL;
    head = theMsg;
}  // end MdpMessageQueue::QueueMessage() 

MdpMessage* MdpMessageQueue::GetNextMessage()
{
    MdpMessage *theMsg = head;
    if (theMsg)
    {
        if((head = head->next))
            head->prev = NULL;
        else
            tail = NULL;
    }
    return theMsg;
}  // end MdpMessageQueue::GetNextMessage()

void MdpMessageQueue::Remove(MdpMessage *theMsg)
{
    ASSERT(theMsg);    
    if (theMsg->prev)
        theMsg->prev->next = theMsg->next;
    else
        head = theMsg->next;
    if (theMsg->next)
        theMsg->next->prev = theMsg->prev;
    else
        tail = theMsg->prev;
}  // end MdpMessageQueue::Remove()

MdpMessage* MdpMessageQueue::FindNackAdv()
{
    MdpMessage *theMsg = head;
    while (theMsg)
    {
        if ((MDP_CMD == theMsg->type) &&
            (MDP_CMD_NACK_ADV == theMsg->cmd.flavor))
            return theMsg;
        theMsg = theMsg->next;
    }
    return (MdpMessage*)NULL;       
}  // end MdpMessageQueue::FindNackAdv()


/**********************************************************************
 * Routines for packing and parsing NACK data content
 * (TBD) Think of better ways to pack NACKs than raw masks ???
 */
    
const unsigned int MDP_NACK_OBJECT_HEADER_LEN = sizeof(long) + sizeof(short);

bool MdpObjectNack::Init(unsigned long id, char *buffer, unsigned int buflen)
{
    // Make sure there's room for at least one RepairNack
    if (buflen > MDP_NACK_OBJECT_HEADER_LEN)
    {
        object_id = id;
        data = &buffer[MDP_NACK_OBJECT_HEADER_LEN];
        nack_len = 0;
        max_len = buflen - MDP_NACK_OBJECT_HEADER_LEN;
        return true;
    }
    else
    {
        return false;
    }
}  // end MdpObjectNack::Init()


bool MdpObjectNack::AppendRepairNack(MdpRepairNack *nack)
{
    int len = nack->Pack(&data[nack_len], max_len);
    if (len)
    {
        nack_len += len;
        max_len -= len;
        return true;
    }
    else
    {
        return false;
    }
}  // end 

int MdpObjectNack::Pack()
{
    if (nack_len > 0)
    {
        // Pack object_id & nack_len
        unsigned long temp_long = htonl(object_id);
        char *header_ptr = data - MDP_NACK_OBJECT_HEADER_LEN;
        memcpy(header_ptr, &temp_long, sizeof(long));
        unsigned short temp_short = htons(nack_len);
        memcpy(&header_ptr[sizeof(long)], &temp_short, sizeof(short));
        return (nack_len + MDP_NACK_OBJECT_HEADER_LEN);
    }
    else
    {
        return 0;
    }
} // end MdpObjectNack::Pack()

int MdpObjectNack::Unpack(char *buffer)
{
    register int len = 0;
    memcpy(&object_id, buffer, sizeof(long));
    object_id = ntohl(object_id);
    len += sizeof(long);    
    memcpy(&nack_len, &buffer[len], sizeof(short));
    nack_len = ntohs(nack_len);
    len += sizeof(short);
    data = &buffer[len];
    len += nack_len;
    return len;
}  // end MdpObjectNack::Unpack()


int MdpRepairNack::Pack(char* buffer, unsigned int buflen)
{
    register int len = 0;
    if (buflen-- > 0)
        buffer[len++] = type;    
    else
        return 0;
    switch (type)
    {
        case MDP_REPAIR_OBJECT:
        case MDP_REPAIR_INFO:
            return 1;
        
        case MDP_REPAIR_SEGMENTS:
            // Note: MDP_REPAIR_SEGMENT masks should be sent intact
            if (buflen-- > 0)
                buffer[len++] = nerasure;  
            else
                return 0;
            if (buflen >= (sizeof(long) + sizeof(short) + mask_len))
            {
                unsigned long temp_long = htonl(offset);
                memcpy(&buffer[len], &temp_long, sizeof(long));
                len += sizeof(long);
                unsigned short temp_short = htons(mask_len);
                memcpy(&buffer[len], &temp_short, sizeof(short));
                len += sizeof(short);
                memcpy(&buffer[len], mask, mask_len);
                len += mask_len; 
                return len;
            }
            else
            {
                return 0;
            }      
            
        case MDP_REPAIR_BLOCKS:
            // Note: MDP_REPAIR_BLOCK masks may get shortened to fit
            if (buflen > (sizeof(long) + sizeof(short)))
            {
                unsigned long temp_long = htonl(offset);
                memcpy(&buffer[len], &temp_long, sizeof(long));
                len += sizeof(long);
                unsigned short temp_short = htons(mask_len);
                memcpy(&buffer[len], &temp_short, sizeof(short));
                len += sizeof(short);
                buflen -= (sizeof(long) + sizeof(short));
                if (buflen < mask_len)
                {
                    memcpy(&buffer[len], mask, buflen);
                    len += buflen;
                }
                else
                {
                    memcpy(&buffer[len], mask, mask_len);
                    len += mask_len;
                }
                return len;
            }
            else
            {
                return 0;
            }
        
        default:
            // this shouldn't happen
            DMSG(0, "MdpRepairNack::Pack() Invalid repair nack!\n");
            return 0;  
    }
}  // end MdpRepairNack::Pack()

int MdpRepairNack::Unpack(char* buffer)
{
    register int len = 0;
    type = (MdpRepairType) buffer[len++];
    switch (type)
    {
        case MDP_REPAIR_OBJECT:
        case MDP_REPAIR_INFO:
            break;
        
        case MDP_REPAIR_SEGMENTS:
            nerasure = buffer[len++];
        case MDP_REPAIR_BLOCKS:
            memcpy(&offset, &buffer[len], sizeof(long));
            offset = ntohl(offset);
            len += sizeof(long);
            memcpy(&mask_len, &buffer[len], sizeof(short));
            mask_len = ntohs(mask_len);
            len += sizeof(short);
            mask = &buffer[len];
            len += mask_len;
            break;
            
        default:
            DMSG(0, "MdpRepairNack::Unpack() Invalid repair nack!\n");
            type = MDP_REPAIR_INVALID;  // this shouldn't happen
            break;    
    }
    return len;
}  // end MdpRepairNack::Unpack()


bool MdpGrttReqCmd::AppendRepresentative(unsigned long repId, 
                                         double repRtt,
                                         unsigned short maxLen)
{
    if (!data) 
    {
        len = 0;
        return false;
    }
    if (maxLen < (len+sizeof(long)+1))
    {
        return false;
    }
    else
    {
        char* ptr = data + len;
        // Put into network order
        repId = ntohl(repId);
        memcpy(ptr, &repId, sizeof(long));
        ptr += sizeof(long);
        *ptr = QuantizeRtt(repRtt);
        len += (sizeof(long) + 1);
        return true;
    }   
}  // end MdpGrttReqCmd::PackRepresentative()

        
bool MdpGrttReqCmd::FindRepresentative(unsigned long repId, double* repRtt)
{
    repId = htonl(repId);  // data in message is in network order
    char* end = data + len;
    while (data < end)
    {
        if (0 == memcmp(data, &repId, sizeof(long))) 
        {
            data += sizeof(long);
            *repRtt = UnquantizeRtt((unsigned char)(*data));
            return true;   
        }
        data += (sizeof(long) + 1);
    }
    return false; 
}  // end MdpAckReqCmd::FindRepresentative()


bool MdpAckReqCmd::AppendAckingNode(unsigned long  nodeId, 
                                    unsigned short maxLen)
{
    if (!data) 
    {
        len = 0;
        return false;
    }
    if (maxLen < (len+sizeof(long)))
    {
        return false;
    }
    else
    {
        // Put into network order
        nodeId = ntohl(nodeId);
        memcpy((data+len), &nodeId, sizeof(long));
        len += sizeof(long);
        return true;
    }   
}  // end MdpAckReqCmd::AppendAckingNode()


bool MdpAckReqCmd::FindAckingNode(unsigned long nodeId)
{
    nodeId = htonl(nodeId);  // data in message is in network order
    const char* ptr = data;
    const char* end = ptr + len;
    while (ptr < end)
    {
        if (0 == memcmp(ptr, &nodeId, sizeof(long))) return true; 
        ptr += sizeof(long);
    }
    return false; 
}  // end MdpAckReqCmd::FindAckingNode()


/****************************************************************
 *  RTT quantization routines:
 *  These routines are valid for 1.0e-06 <= RTT <= 1000.0 seconds
 *  They allow us to pack our RTT estimates into a 1 byte fields
 */

// valid for rtt = 1.0e-06 to 1.0e+03
unsigned char QuantizeRtt(double rtt)
{
    if (rtt > RTT_MAX)
        rtt = RTT_MAX;
    else if (rtt < RTT_MIN)
        rtt = RTT_MIN;
    if (rtt < 3.3e-05) 
        return ((unsigned char)ceil((rtt*RTT_MIN)) - 1);
    else
        return ((unsigned char)(ceil(255.0 - (13.0*log(RTT_MAX/rtt)))));
}  // end QuantizeRtt()




/***********************************************************************
 *  MdpMessagePool implementation
 */

MdpMessagePool::MdpMessagePool()
    : message_count(0), message_total(0), 
      message_list(NULL)
{
}

MdpMessagePool::~MdpMessagePool()
{
    Destroy();
}

MdpError MdpMessagePool::Init(unsigned long count)
{
    ASSERT(!message_list);
    unsigned long i;
    for(i = 0; i < count; i++)
    {
        MdpMessage* theMsg;
        if((theMsg = new MdpMessage))
        {
            theMsg->next = message_list;
            message_list = theMsg;
        }
        else
        {
            break;
        }
    }
    message_count = message_total = i;
    if (i != count)
    {
        Destroy();
        return MDP_ERROR_MEMORY;
    }
    else
    {
        return MDP_ERROR_NONE;
    }
}  // end MdpMessagePool::Init()

void MdpMessagePool::Destroy()
{
    ASSERT(message_count == message_total);
    MdpMessage* theMsg = message_list;
    while(theMsg)
    {
        MdpMessage* nextMsg = theMsg->next;
        delete theMsg;
        theMsg = nextMsg;
    }
    message_list = NULL;
    message_count = message_total = 0;
}  // end MdpMessagePool::Destroy()

MdpMessage *MdpMessagePool::Get()
{
    MdpMessage *theMsg = message_list;
    if(theMsg)
    {
        message_count--;
        message_list = theMsg->next;
    }
    else
    {
        DMSG(0, "MdpMessagePool has been exhausted!\n");
    }
    return theMsg;
}  // end MdpMessagePool::Get()

void MdpMessagePool::Put(MdpMessage *theMsg)
{
    ASSERT(theMsg);
    theMsg->next = message_list;
    message_list = theMsg;
    message_count++;
}  // end MdpMessagePool::Put()

/***************************************************************
 * Client statistics class implementations
 */

int MdpBlockStats::Pack(char *buffer)
{
    unsigned long temp_long = htonl(count);
    memcpy(buffer, &temp_long, sizeof(long));
    int len = sizeof(long);
    temp_long = htonl(lost_00);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    temp_long = htonl(lost_05);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    temp_long = htonl(lost_10);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    temp_long = htonl(lost_20);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    temp_long = htonl(lost_40);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    temp_long = htonl(lost_50);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    return len;
}  // end MdpBlockStats::Pack()

int MdpBlockStats::Unpack(char *buffer)
{
      unsigned long temp_long;
      memcpy(&temp_long, buffer, sizeof(long));
      count = ntohl(temp_long);
      int len = sizeof(long); 
      memcpy(&temp_long, &buffer[len], sizeof(long));
      lost_00 = ntohl(temp_long);
      len += sizeof(long);  
      memcpy(&temp_long, &buffer[len], sizeof(long));
      lost_05 = ntohl(temp_long);
      len += sizeof(long);  
      memcpy(&temp_long, &buffer[len], sizeof(long));
      lost_10 = ntohl(temp_long);
      len += sizeof(long);  
      memcpy(&temp_long, &buffer[len], sizeof(long));
      lost_20 = ntohl(temp_long);
      len += sizeof(long);  
      memcpy(&temp_long, &buffer[len], sizeof(long));
      lost_40 = ntohl(temp_long);
      len += sizeof(long);  
      memcpy(&temp_long, &buffer[len], sizeof(long));
      lost_50 = ntohl(temp_long);
      len += sizeof(long); 
      return len;
}  // end MdpBlockStats::Unpack()
    

int MdpBufferStats::Pack(char *buffer)
{
    unsigned long temp_long = htonl(buf_total);
    memcpy(buffer, &temp_long, sizeof(long));
    int len = sizeof(long);
    temp_long = htonl(peak);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    temp_long = htonl(overflow);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    return len;
}  // end MdpBufferStats::Pack()
 
int MdpBufferStats::Unpack(char *buffer)
{
    unsigned long temp_long;
    memcpy(&temp_long, buffer, sizeof(long));
    buf_total = ntohl(temp_long);
    int len = sizeof(long); 
    memcpy(&temp_long, &buffer[len], sizeof(long));
    peak = ntohl(temp_long);
    len += sizeof(long);  
    memcpy(&temp_long, &buffer[len], sizeof(long));
    overflow = ntohl(temp_long);
    len += sizeof(long); 
    return len;
}  // end MdpBufferStats::Unpack()

int MdpClientStats::Pack(char *buffer)
{
    unsigned long temp_long = htonl(duration);
    memcpy(buffer, &temp_long, sizeof(long));
    int len = sizeof(long);
    temp_long = htonl(success);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    temp_long = htonl(active);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    temp_long = htonl(fail);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    temp_long = htonl(resync);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    len += blk_stat.Pack(&buffer[len]);
    temp_long = htonl(tx_rate);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    temp_long = htonl(nack_cnt);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    temp_long = htonl(supp_cnt);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    len += buf_stat.Pack(&buffer[len]);
    temp_long = htonl(goodput);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    temp_long = htonl(rx_rate);
    memcpy(&buffer[len], &temp_long, sizeof(long));
    len += sizeof(long);
    return len;
}  // end MdpClientStats::Pack()
    

int MdpClientStats::Unpack(char *buffer)
{
    unsigned long temp_long;
    memcpy(&temp_long, buffer, sizeof(long));
    duration = ntohl(temp_long);
    int len = sizeof(long);
    memcpy(&temp_long, &buffer[len], sizeof(long));
    success = ntohl(temp_long);
    len += sizeof(long);   
    memcpy(&temp_long, &buffer[len], sizeof(long));
    active = ntohl(temp_long);
    len += sizeof(long);   
    memcpy(&temp_long, &buffer[len], sizeof(long));
    fail = ntohl(temp_long);
    len += sizeof(long);
    memcpy(&temp_long, &buffer[len], sizeof(long));
    resync = ntohl(temp_long);
    len += sizeof(long);
    len += blk_stat.Unpack(&buffer[len]);   
    memcpy(&temp_long, &buffer[len], sizeof(long));
    tx_rate = ntohl(temp_long);
    len += sizeof(long);  
    memcpy(&temp_long, &buffer[len], sizeof(long));
    nack_cnt = ntohl(temp_long);
    len += sizeof(long);
    memcpy(&temp_long, &buffer[len], sizeof(long));
    supp_cnt = ntohl(temp_long);
    len += sizeof(long);
    len += buf_stat.Unpack(&buffer[len]);
    memcpy(&temp_long, &buffer[len], sizeof(long));
    goodput = ntohl(temp_long);
    len += sizeof(long);
    memcpy(&temp_long, &buffer[len], sizeof(long));
    rx_rate = ntohl(temp_long);
    len += sizeof(long);
    return len;
}  // end MdpClientStats::Unpack()


    

