/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
// This file contains the MdpSession & MdpObject methods 
// for handling recv'd MDPv2 protocol messages


#include "debug.h"  
#include "mdpSession.h"
#include <time.h>  // for gmtime()

bool MdpSession::RxSocketRecvHandler(UdpSocket* /*theSocket*/)
{
    char buffer[MSG_SIZE_MAX];
    unsigned int len = MSG_SIZE_MAX;
    NetworkAddress src;
    if (UDP_SOCKET_ERROR_NONE != rx_socket.RecvFrom(buffer, &len, &src))
    {
        DMSG(0, "mdp: node:%lu UdpSocket::RecvFrom() error.\n");
        return false;
    }        
    return HandleRecvPacket(buffer, len, &src, false);
}

bool MdpSession::TxSocketRecvHandler(UdpSocket* /*theSocket*/)
{
    char buffer[MSG_SIZE_MAX];
    unsigned int len = MSG_SIZE_MAX;
    NetworkAddress src;
    if (UDP_SOCKET_ERROR_NONE != tx_socket.RecvFrom(buffer, &len, &src))
    {
        DMSG(0, "mdp: node:%lu UdpSocket::RecvFrom() error.\n");
        return false;
    }        
    return HandleRecvPacket(buffer, len, &src, true);
}

    
bool MdpSession::HandleRecvPacket(char* buffer, unsigned int len, NetworkAddress* src, 
                                  bool isUnicast)
{
    MdpMessage msg; // This doesn't have to be static
    if(!msg.Unpack(buffer, len))
    {
        DMSG(0, "mdp: node:%lu Recv'd bad message from \"%s\"\n", 
				LocalNodeId(), src->HostAddressString());
        return true;
    }
#ifdef PROTO_DEBUG
    if (recv_drop_rate > 0.0)
    {
        if (UniformRand(100.0) < recv_drop_rate)
        {
            DMSG(6, "mdp: node:%lu Recv packet dropped.\n", LocalNodeId());
            return true;
        }
    }
#endif  // PROTO_DEBUG
       

    // Some machines or OS's (Win32) don't ever disable mcast loopback
    // Only receive your own packets if "loopback" is enabled
    if (!loopback && (LocalNodeId() == msg.sender))  return true;

    
    if (MDP_REPORT != msg.type) client_stats.rx_rate += len;  // for client rx_rate reporting
    
#ifdef PROTO_DEBUG    
    if (mdp_trace) MessageTrace(false, LocalNodeId(), &msg, len, src);
#endif // PROTO_DEBUG
      
    if (msg.version != MDP_PROTOCOL_VERSION)
    {
        DMSG(0, "mdp: node:%lu Received packet type:%d with wrong protocol "
				"version from \"%s\"\n", LocalNodeId(), msg.type, 
				src->HostAddressString());
        return true;
    }
    switch(msg.type)
    {
        case MDP_REPORT:
            HandleMdpReport(&msg, src);
            break;
            
        case MDP_INFO:
        case MDP_DATA:
        case MDP_PARITY:
            // Only clients care about MDP_PARITY
            if (IsClient()) HandleObjectMessage(&msg, src);
            break;
            
        case MDP_CMD:
            // Only clients care about MDP_CMD
            if (IsClient()) HandleMdpServerCmd(&msg, src);
            break;
            
        case MDP_NACK:
            // Both clients and servers care about MDP_NACKs
            if (IsServer()) ServerHandleMdpNack(&msg, isUnicast);
            if (IsClient()) ClientHandleMdpNack(&msg);
            break;
            
        case MDP_ACK:
            // Servers only for now
            if (IsServer()) ServerHandleMdpAck(&msg);
            break;
             
        default:
            // should never get here
            DMSG(0, "mdp: node:%lu Recv'd unsupported message type %d from \"%s\"\n",
                    LocalNodeId(), msg.type, src->HostAddressString());
            break;
    }
    // If the application has deleted/aborted the session in 
    // a notify callback during this event, we must do housekeeping
    if (notify_abort) mgr->DeleteSession(this);   
    return true;
}  // end MdpSession::HandleRecvPacket()

void MdpSession::HandleMdpReport(MdpMessage *theMsg, NetworkAddress *src)
{
    // Update Positive Acknowledgement List as needed
    // (TBD) Manage acking node list better ... remove nodes that have timed out
    if (theMsg->report.status & MDP_ACKING)
        AddAckingNode(theMsg->sender);
    else
        RemoveAckingNode(theMsg->sender);

    if (theMsg->report.status & MDP_CLIENT)
    {
        // (TBD) In the future we might use "NotifyUser" to post
        // report info to application.  Meanwhile we'll just output
        // a debug message
#ifdef PROTO_DEBUG
        MdpClientStats *stat = &theMsg->report.client_stats;
        DMSG(2, "*******************************************************\n");
        DMSG(2, "MDP_REPORT from client: \"%.64s\" (%s)\n", 
                    theMsg->report.node_name,
                    src->HostAddressString());
        struct timeval  rxTime;
        ::GetSystemTime(&rxTime);
        struct tm *rx_time = gmtime((time_t *)&rxTime.tv_sec);
        DMSG(2, "   Date & time (GMT): %02d/%02d %02d:%02d:%02d.%06lu\n",
                        rx_time->tm_mon,
                        rx_time->tm_mday, 
                        rx_time->tm_hour, 
                        rx_time->tm_min,
                        rx_time->tm_sec,
                        rxTime.tv_usec);
        DMSG(2, "   Session duration : %lu sec.\n", stat->duration);
        DMSG(2, "   Objects completed: %lu\n", stat->success);
        DMSG(2, "   Objects pending  : %lu\n", stat->active);
        DMSG(2, "   Objects failed   : %lu\n", stat->fail);
        DMSG(2, "   Server resyncs   : %lu\n", stat->resync);
		DMSG(2, "   Current goodput  : %.3f kbps\n", 
                        ((double)stat->goodput * 8.0)/1000.0);
        DMSG(2, "   Current rx rate  : %.3f kbps\n", 
                        ((double)stat->rx_rate * 8.0)/1000.0);
        DMSG(2, "   Current tx rate  : %.3f kbps\n", 
                        ((double)stat->tx_rate * 8.0)/1000.0);
        DMSG(2, "   NACKs transmitted: %lu\n", stat->nack_cnt);
        DMSG(2, "   NACKs suppressed : %lu\n", stat->supp_cnt);
        DMSG(2, "   Block loss stats : %lu, %lu, %lu, %lu, %lu, %lu\n",
                        stat->blk_stat.lost_00, stat->blk_stat.lost_05,
                        stat->blk_stat.lost_10, stat->blk_stat.lost_20,
                        stat->blk_stat.lost_40, stat->blk_stat.lost_50);
        DMSG(2, "   Buffer usage     : peak>%luK overruns>%lu\n",
                        (stat->buf_stat.peak+500)/1000, 
                        stat->buf_stat.overflow);
        DMSG(2, "*******************************************************\n"); 
#endif // PROTO_DEBUG       
    }    
    return;
}  // end MdpSession::HandleMdpReport()


void MdpSession::HandleObjectMessage(MdpMessage* theMsg, NetworkAddress* src)
{
    // Find state for the server
    MdpServerNode* theServer = (MdpServerNode*) server_list.FindNodeById(theMsg->sender);
    if (!theServer)
    {
        if(!(theServer = NewRemoteServer(theMsg->sender)))
        {
            DMSG(0, "mdp: node:%lu Error allocating new Mdp Server state!\n", LocalNodeId());
            return;
        }
    }    
    // Update general server state 
    theServer->UpdateGrtt(theMsg->object.grtt);   
    theServer->ResetActivityTimer();  
    theServer->SetAddress(src);
    theServer->UpdateLossEstimate(theMsg->object.sequence, ecn_status);
       
    unsigned short segmentSize;
    unsigned char numData = theMsg->object.ndata;
    unsigned char numParity = theMsg->object.nparity;
    unsigned long blockId;
    char* infoPtr = NULL;
    unsigned short infoLen = 0;
    
    MdpMessageType msgType = (MdpMessageType) theMsg->type;
    switch(msgType)
    {
        case MDP_INFO:
            segmentSize = theMsg->object.info.segment_size;
            blockId = 0;
            infoPtr = theMsg->object.info.data;
            infoLen = theMsg->object.info.len;
            break;
            
        case MDP_DATA:
            if (theMsg->object.flags & MDP_DATA_FLAG_RUNT)
                segmentSize = theMsg->object.data.segment_size;
            else
                segmentSize = theMsg->object.data.len;   
            blockId = theMsg->object.data.offset / (numData * segmentSize);      
            break;
            
        case MDP_PARITY:
            segmentSize = theMsg->object.parity.len;
            blockId = theMsg->object.parity.offset / (numData * segmentSize); 
            break;
        
        default:
            // This should _never_ occur
            ASSERT(0);
            return;
    }
    
    // Check to make sure server hasn't restarted with new coding parameters
    if (!theServer->VerifyCoding(segmentSize, numData, numParity))
    {
        // Reset server for new activation
        if (theServer->IsActive()) theServer->Deactivate();
        theServer->SetCoding(segmentSize, numData, numParity);
    }  
    
    // Find/create the appropriate object state
    unsigned long objectId = theMsg->object.object_id;
    MdpObject* theObject = theServer->FindRecvObjectById(objectId);
    if (!theObject)
    {
        DMSG(10, "mdp: node: %lu Receiving new object:%lu from server:%lu ...\n",
					LocalNodeId(), objectId, theServer->Id());
        // Pick up new objects only if already synchronized or
        // if it's _not_ a repair transmission _and_ it's in the first
        // coding block
		if (theServer->ServerSynchronized() || 
            (!(theMsg->object.flags & MDP_DATA_FLAG_REPAIR) && (0 == blockId)))
        {       
            // Is it in range
            MdpRecvObjectStatus objectStatus = 
                theServer->ObjectSequenceCheck(objectId);
            switch (objectStatus)
            {
                case  MDP_RECV_OBJECT_INVALID:
                    DMSG(10, "mdp: node: %lu New object status: MDP_RECV_OBJECT_INVALID\n",
					            LocalNodeId());
                    theServer->Sync(objectId);
                    ASSERT(MDP_RECV_OBJECT_NEW == theServer->ObjectSequenceCheck(objectId));
                    client_stats.resync++;
                    objectStatus = MDP_RECV_OBJECT_NEW;
                    break;

                case MDP_RECV_OBJECT_NEW:
					DMSG(10, "mdp: node: %lu New object status: MDP_RECV_OBJECT_NEW\n",
					     LocalNodeId());
                    // New object in valid range
                    break;

                case MDP_RECV_OBJECT_PENDING:
					DMSG(10, "mdp: node: %lu New object status: MDP_RECV_OBJECT_PENDING\n",
					     LocalNodeId());
                    break;

                case MDP_RECV_OBJECT_COMPLETE :
					DMSG(10, "mdp: node: %lu New object status: MDP_RECV_OBJECT_COMPLETE\n",
					     LocalNodeId());
                    // We have already received this object
                    break;
            }

            if (MDP_RECV_OBJECT_COMPLETE != objectStatus)
            {
                // Allocate buffers if not already allocated
                if (!theServer->IsActive())
                {
                    if (!theServer->Activate(client_window_size)) 
                    {
                        DMSG(0, "MdpSession::UpdateServer() Error activating server!\n");
                        return;
                    }
                }                          
#if defined(NS2) || defined(OPNET) 
                MdpObjectType objectType = MDP_OBJECT_SIM;
#else
                MdpObjectType objectType = ((theMsg->object.flags & MDP_DATA_FLAG_FILE) ?
                                        MDP_OBJECT_FILE : MDP_OBJECT_DATA);
#endif
				theObject = theServer->NewRecvObject(objectId, 
                                                     objectType, 
                                                     theMsg->object.object_size,
                                                     infoPtr, 
                                                     infoLen);               
                // If no info is available, don't ask for it              
                if (theObject && !(theMsg->object.flags & MDP_DATA_FLAG_INFO))
                    theObject->SetHaveInfo(true);
            }    
        }  // end if ("it's a good object to pick up on");
    }  // end if (!theObject)
   
    if (theObject)
    {
        ASSERT(theServer->ServerSynchronized());
        bool result = true;
        switch (msgType)
        {
            case MDP_INFO:
                if (!theObject->HaveInfo())
                {

                    ASSERT(theObject->IsOpen());
                    theObject->SetHaveInfo(true); 
                    theObject->SetInfo(theMsg->object.info.data, 
                                       theMsg->object.info.len);
#ifdef PROTO_DEBUG
                    char content[64];
                    unsigned short contentLen = theMsg->object.info.len;
                    contentLen = MIN(contentLen, 63);
                    memcpy(content, theMsg->object.info.data, contentLen);
                    content[contentLen] = '\0';
                    DMSG(6, "mdp: node:%lu Client got info for object:%lu (%s)\n",
					        LocalNodeId(), objectId, content);
                    if (MDP_OBJECT_FILE == theObject->Type())
                    {
                        DMSG(6, "mdp: node:%lu Client object:%lu equals file:%32s\n", 
                                LocalNodeId(), objectId, ((MdpFileObject*)theObject)->Info());
                    }
#endif // PROTO_DEBUG 
                    theObject->Notify(MDP_NOTIFY_RX_OBJECT_INFO, MDP_ERROR_NONE);
                }
                else
                {
                    // received redundant info pkt   
                    // (TBD) we may want to verify that the info hasn't changed ??
                }
                if (!theServer->WasDeleted()) 
                    theServer->ObjectRepairCheck(objectId, 0);
                break;
                
            case MDP_DATA:       
                // Now update the object with received MDP_DATA content
                result = theObject->HandleRecvData(theMsg->object.data.offset, 0, 
                                              (0 != (theMsg->object.flags & MDP_DATA_FLAG_BLOCK_END)),
                                              theMsg->object.data.data, 
                                              theMsg->object.data.len);
                break;
                         
            case MDP_PARITY:       
                // Now update the object with received MDP_PARITY content
                result = theObject->HandleRecvData(theMsg->object.parity.offset, 
                                              theMsg->object.parity.id, 
                                              (0 != (theMsg->object.flags & MDP_DATA_FLAG_BLOCK_END)),
                                              theMsg->object.parity.data, 
                                              theMsg->object.parity.len);
                break;               
            
            default:
                // This should _never_ occur
                ASSERT(0);
                break;
        }  // end switch(msgType)
        
        // Check for object/remote server termination by app during callback
        if (theServer->WasDeleted())
        { 
            theServer->Delete();
            return;
        }
        else if (theObject->WasAborted()) 
        {
            theObject->RxAbort();
            theServer->ObjectRepairCheck(objectId, 0);
            return;
        }
        
        if (result)
        {
            // Is the object finished?
            if (!theObject->RxPending())
            {
	            ASSERT(!theObject->HaveBuffers());
                client_stats.success++;
                theServer->DeactivateRecvObject(theObject->TransportId(), theObject);
                DMSG(6, "mdp: node:%lu Client recv object:%lu complete.\n", 
					    LocalNodeId(), objectId);
                theObject->Notify(MDP_NOTIFY_RX_OBJECT_COMPLETE, MDP_ERROR_NONE);
                delete theObject;      
            }   
        }
        else  // Fatal DATA/PARITY handling error for this object
        {
            client_stats.fail++;
            DMSG(0, "mdp: node:%lu Client error handling reception of object:%lu\n",
                            LocalNodeId(), objectId);
            theObject->RxAbort();
        }
    }
    else  // if (!theObject)
    {
        if (theServer->ServerSynchronized()) 
            theServer->ObjectRepairCheck(objectId, 0);
    }  // end if/else(theObject)
}  // end MdpSession::HandleObjectMessage()


// This handles received data or parity for an object
// (TBD) Rewrite this to be a member of MdpServerNode ???
bool MdpObject::HandleRecvData(unsigned long    dataOffset, 
                               int              parityId, 
                               bool             blockEnd, 
                               char*            theData, 
                               unsigned short   dataLen)
{
	ASSERT(IsOpen());    
    unsigned long blockId = dataOffset/block_size;
    MdpBlock* theBlock = block_buffer.GetBlock(blockId);
    if (!theBlock)
    {
        // Make sure we haven't already received this block
        if(!transmit_mask.Test(blockId)) 
        {
            DMSG(6, "mdp: node:%lu Client receiving redundant %s", LocalNodeId(),
					parityId ? "parity " : "data ");
            DMSG(6, "for object:%lu block:%03lu\n", transport_id, blockId);
            sender->ObjectRepairCheck(transport_id, 
                            (blockEnd ? (blockId+1) : blockId));
            return true;
        }        
        // Create new block
        if(!(theBlock = sender->BlockGetFromPool()))
        {
            if (sender->ReclaimResources(this, blockId))
            {
                theBlock = sender->BlockGetFromPool();
                ASSERT(theBlock);
            }
            else
            {              
                DMSG(6, "mdp: node:%lu No buffers for recv data block of object:%lu block:%lu\n", 
                        LocalNodeId(), transport_id, blockId);
                sender->ObjectRepairCheck(transport_id, 
                                (blockEnd ? (blockId+1) : blockId));
                return true;
            }
        }       
        // Properly rx init the block (Note that last block may be short)
        if (blockId != last_block_id) 
            theBlock->RxInit(blockId, ndata, nparity);
        else
            theBlock->RxInit(blockId, last_block_len, nparity);        
        // Put it in our object's block_buffer 
        block_buffer.Prepend(theBlock);       
    }  // end if (!theBlock)
           
    // Which vector is this??
    unsigned int vectorId;
    if (parityId)
    {
        vectorId = parityId;
        DMSG(6, "mdp: node:%lu Client recv'd MDP_PARITY object:%lu block:%02lu vector:%02lu\n", 
                LocalNodeId(), transport_id, blockId, vectorId);
    }
    else
    {
        vectorId = (dataOffset / segment_size) - (blockId * ndata);
        DMSG(6, "mdp: node:%lu Client recv'd MDP_DATA  object:%lu block:%02lu vector:%02lu\n", 
                LocalNodeId(), transport_id, blockId, vectorId);
    }   
    // Check to see if we already have it
    if(theBlock->Vector(vectorId))
    {
        DMSG(6, "mdp: node:%lu Client recv'd redundant %s.\n", 
                LocalNodeId(), (parityId ? "MDP_PARITY" : "MDP_DATA"));
        sender->ObjectRepairCheck(transport_id, 
                        (blockEnd ? (blockId+1) : blockId));
        return true;
    }    
    
    // Copy data to a vector from the server's pool and attach to block
    char* vector = sender->VectorGetFromPool();
    if (!vector)
    {
        if (sender->ReclaimResources(this, blockId))
        {
            vector = sender->VectorGetFromPool();
            ASSERT(vector);
        }
        else
        {
            DMSG(6, "mdp: node:%lu No buffers for recv data vector of object:%lu block:%lu\n", 
                LocalNodeId(), transport_id, blockId);
            sender->ObjectRepairCheck(transport_id, 
                            (blockEnd ? (blockId+1) : blockId));
            return true;
        }
    }    
    memcpy(vector, theData, dataLen);
    if (dataLen < segment_size) 
		memset(&vector[dataLen], 0, segment_size - dataLen);	
        
    // It's non-redundant data or parity, call it "goodput" for client reporting
    session->client_stats.goodput += dataLen; 
    
    if (parityId)
        theBlock->IncrementParityCount();
    else
        theBlock->DecrementErasureCount();           
	
    theBlock->AttachVector(vector, vectorId);
    theBlock->UnsetMask(vectorId);  // "unmark" erasure    
    if (!parityId)
    {
        if (!WriteSegment(dataOffset, theData, &dataLen))
        {
            // fatal error for this object
            DMSG(0, "mdp: node:%lu Error writing object data vector\n",
					LocalNodeId());
            sender->ObjectRepairCheck(transport_id, 
                            ((blockEnd) ? (blockId+1) : blockId));
            return false; // error writing to object            
        }
    }  
    data_recvd += dataLen;  // track recv progress
	
    Notify(MDP_NOTIFY_RX_OBJECT_UPDATE, MDP_ERROR_NONE);
    if (sender->WasDeleted() || WasAborted()) return true;
    
    if(theBlock->ErasureCount() <= theBlock->ParityCount())
    {
        // Block fully received
        transmit_mask.Unset(blockId);  // block transmission complete       
        unsigned int numData;
        if (blockId != last_block_id) 
            numData = ndata;
        else
            numData = last_block_len;      
        // Decode if necessary
        if (theBlock->FirstSet() < numData)
        {
            DMSG(6, "mdp: node:%lu Client decoding object:%lu block:%lu with %d erasure(s) ...\n", 
                    LocalNodeId(), transport_id, blockId, theBlock->ErasureCount());
            // Fill missing (erased) vectors with zero filled vectors
            if(!theBlock->FillZero(sender->VectorPool(), segment_size, numData+nparity))
            {
                // This shouldn't happen since receiver allocate some extra vectors 
                // for this job among others
                DMSG(0, "mdp: node:%lu Error zero filling coding block!\n", LocalNodeId());
                sender->ObjectRepairCheck(transport_id, 
                                ((blockEnd) ? (blockId+1) : blockId));
                return false; // error writing to object     
            }
            sender->Decode(theBlock->VectorList(), numData, theBlock->Mask());   
        }        
        // Write filled erasures to object (erasures marked in block vector_mask)
        vectorId = theBlock->FirstSet();
        unsigned long blockOffset = blockId * block_size;
        dataLen = segment_size;
        while (vectorId < numData)
        {
            dataOffset = blockOffset + (vectorId * dataLen);
            if (last_offset == dataOffset) 
				dataLen = (unsigned short)(object_size - last_offset);
            if (!WriteSegment(dataOffset, theBlock->Vector(vectorId), &dataLen))
            {
                // fatal error for this object
                DMSG(0, "mdp: node:%lu Error writing object data vector\n",
						LocalNodeId());
                sender->ObjectRepairCheck(transport_id, 
                                ((blockEnd) ? (blockId+1) : blockId));
                return false; // error writing to object            
            }
            vectorId = theBlock->NextSet(++vectorId);   
        }
#ifdef PROTO_DEBUG       
        // Record first pass block loss statistics for full-size blocks
        // Note: Out of order packet reception at block boundaries
        //       can cause these estimated stat's to be skewed a little
        if (!theBlock->InRepair() && (numData == ndata)) 
        {
            if (0 == theBlock->ErasureCount())
            {
                session->client_stats.blk_stat.lost_00++;
            }
            else
            {
                double loss = (double)theBlock->ErasureCount() /
                             (double)(numData);
                if (loss <= 0.05) 
                    session->client_stats.blk_stat.lost_05++;
                else if (loss <= 0.10)
                    session->client_stats.blk_stat.lost_10++;
                else if (loss <= 0.20)
                    session->client_stats.blk_stat.lost_20++;
                else if (loss <= 0.40)
                    session->client_stats.blk_stat.lost_40++;
                else
                    session->client_stats.blk_stat.lost_50++;
            }
        }
#endif // PROTO_DEBUG
        // Since we're finished with the block, return block resources to pools
        block_buffer.Remove(theBlock);
		sender->BlockReturnToPool(theBlock);       
    }    
    sender->ObjectRepairCheck(transport_id, 
                        ((blockEnd) ? (blockId+1) : blockId));
    return true;  // Everything OK    
}  // end MdpObject::HandleRecvData()


void MdpSession::HandleMdpServerCmd(MdpMessage *theMsg, NetworkAddress *src)
{
    MdpServerNode* theServer = (MdpServerNode*) server_list.FindNodeById(theMsg->sender);
    if (!theServer)
    {
        if(!(theServer = NewRemoteServer(theMsg->sender)))
        {
            DMSG(0, "mdp: node:%lu Error allocating new MdpServer state!\n", LocalNodeId());
            return;
        }
    }   
    // Update general server state 
    theServer->UpdateGrtt(theMsg->cmd.grtt);
    theServer->ResetActivityTimer();  
    theServer->SetAddress(src);
    theServer->UpdateLossEstimate(theMsg->cmd.sequence, ecn_status);
    
    switch(theMsg->cmd.flavor)
    {
        case MDP_CMD_FLUSH:
            if (theMsg->cmd.flush.flags & MDP_CMD_FLAG_EOT)
            {
                theServer->Deactivate();  
                theServer->Close();
                DeleteRemoteServer(theServer);
            }
            else
            {
                if (theServer->ServerSynchronized())
                    theServer->FlushServer(theMsg->cmd.flush.object_id);
            }
            break;

        case MDP_CMD_SQUELCH:
            theServer->HandleSquelchCmd(theMsg->cmd.squelch.sync_id,
                                        theMsg->cmd.squelch.data,
                                        theMsg->cmd.squelch.len);
            break;

        case MDP_CMD_ACK_REQ:
        {
            // Are we in the list ?
            unsigned long objectId = theMsg->cmd.ack_req.object_id;
            if (theMsg->cmd.ack_req.FindAckingNode(LocalNodeId()))
            {
                // If object is complete, acknowledge it
                // Note this does not mean we actually received it,
                // but rather that the MdpClient is no longer interested
                // in it for whatever reason.
                if (MDP_RECV_OBJECT_COMPLETE == theServer->ObjectSequenceCheck(objectId))                  
                {
                    theServer->AcknowledgeRecvObject(objectId);
                }
                else
                {
                    // Being asked to ACK will wake us up
                    theServer->FlushServer(objectId);
                }
            }
            else
            {
                // "FlushServer()" will activate new objects as needed
                if (theServer->ServerSynchronized())
                        theServer->FlushServer(objectId);
            }
        }
        break;

        case MDP_CMD_GRTT_REQ:
            theServer->HandleGrttRequest(theMsg);
            break;
            
        case MDP_CMD_NACK_ADV:
            theServer->HandleRepairNack(theMsg->cmd.nack_adv.data,
                                        theMsg->cmd.nack_adv.len);
            break;

        default:
            DMSG(0, "mdp: node:%lu Client recv'd invalid server command from \"%.64s\"\n",
                    LocalNodeId(), theServer->Name());
            break;
    }
}  // end MdpSession::HandleMdpServerCmd()


void MdpSession::ServerHandleMdpNack(MdpMessage *theMsg, bool isUnicast)
{
    //if (isUnicast) DMSG(0, "mdp: Server received unicast NACK.\n");
    // Is the message for me ??
    if (LocalNodeId() != theMsg->nack.server_id) return; 
    
#if defined(PROTO_DEBUG) && !defined (OPNET)  
    if (theMsg->nack.grtt_response.tv_sec || theMsg->nack.grtt_response.tv_usec)
    {
        struct in_addr theClient;
        theClient.s_addr = htonl(theMsg->sender);
        DMSG(6, "mdp: node:%lu Server recv'd MDP_NACK with GRTT response from \"%s\"\n", 
				LocalNodeId(), inet_ntoa(theClient));
    }
#endif // PROTO_DEBUG
    // Process embedded GRTT_REQ response 
    ServerProcessClientResponse(theMsg->sender, 
                                &theMsg->nack.grtt_response, 
                                theMsg->nack.loss_estimate, 
                                theMsg->nack.grtt_req_sequence);
    
    // Variables to deal with potential need to SQUELCH
    MdpMessage* squelchMsg = NULL;
    unsigned short squelchLen = 0;
    char* squelchPtr = NULL;
    unsigned long pendingLow = 0, pendingRange = 0;
    
    
    // For unicast nack suppression: If the "increaseRepairState"
    // gets set to "true" while processing this NACK, the server
    // will advertise its current repair state in the form of
    // a pseudo-nack sent to the group to suppress receivers with
    // equal or less repair needs
    bool increasedRepairState = false;
    
    // Unpack the concatenated Object/Repair Nacks and process them    
    char* optr = theMsg->nack.data;
    char* nack_end = optr + theMsg->nack.nack_len;
    while (optr < nack_end)
    {
        MdpObjectNack onack;
        optr += onack.Unpack(optr);
        unsigned long objectId = onack.object_id;
        MdpObject* theObject = tx_repair_queue.FindFromHead(objectId);
        if (!theObject)
        {
            if((theObject = tx_hold_queue.FindFromTail(objectId)))
            {
                // Put object onto session's (active) tx_repair_queue 
                ReactivateTxObject(theObject);
            }
            else
            {
                DMSG(6, "mdp: node:%lu Server couldn't find state for repair object: %lu "
                        "(squelching it)\n", LocalNodeId(), objectId);
                if (!squelchMsg) 
                {
                    if ((squelchMsg = msg_pool.Get()))
                    {
                        squelchMsg->type = MDP_CMD;
                        squelchMsg->version = MDP_PROTOCOL_VERSION;
                        squelchMsg->sender = LocalNodeId();
                        squelchMsg->cmd.flavor = MDP_CMD_SQUELCH;
                        if(!(squelchMsg->cmd.squelch.data = squelchPtr = server_vector_pool.Get()))
                        {
                            DMSG(0, "mdp: Server error getting vector for SquelchCmd!");
                            msg_pool.Put(squelchMsg);
                            squelchMsg = NULL;
                            return;
                        } 
                        // Init range of objects we're currently servicing
                        MdpObject* theHead = tx_hold_queue.Head();
                        if (theHead)
                        {
                            pendingLow = theHead->TransportId();
                            if ((theHead = tx_repair_queue.Head()))
                            {
                                unsigned long tempId = theHead->TransportId();
                                // if tempId < pendingLow
                                unsigned long diff = tempId - pendingLow;
                                if ((diff > 0x80000000) || 
                                    ((diff == 0x80000000) && (tempId > pendingLow)))
                                {
                                    pendingLow = tempId;       
                                }
                            }
                        }
                        else
                        {
                            if((theHead = tx_repair_queue.Head()))
                                pendingLow = theHead->TransportId();
                            else
                                pendingLow = current_tx_object_id;
                        }  // end if/else(theHead)
                        squelchMsg->cmd.squelch.sync_id = pendingLow;
                        pendingRange = current_tx_object_id - pendingLow;
                    }
                    else
                    {
                        DMSG(0, "mdp: node:%lu Server msg_pool empty, can't send MDP_SQUELCH!\n",
							LocalNodeId());
                    }    
                }
                if (squelchMsg)
                {
                    // Is the object in question in the valid range of objects
                    // we're currently servicing?
                    if ((objectId - pendingLow) <= pendingRange)
                    {
                        // Is there room in our "squelchVector"?
                        if (segment_size >= (squelchLen + sizeof(unsigned long)))
                        {
                            unsigned long tempId = htonl(objectId);
                            memcpy(squelchPtr, &tempId, sizeof(unsigned long));
                            squelchLen += sizeof(unsigned long);
                            squelchPtr += sizeof(unsigned long);
                        }
                    }
                }
                continue;
            }
        }  // end if (!theObject)
        char* rptr = onack.data;
        char* onack_end = rptr + onack.nack_len;
        while (rptr < onack_end)
        {
            MdpRepairNack rnack;
            rptr += rnack.Unpack(rptr);
            increasedRepairState |= theObject->ServerHandleRepairNack(&rnack);
        }	
        // If nack was obsolete or unable to activate repair cycle for
        // any reason, temporarily deactivate the TxObject (move to tx_hold_queue)
        if (!(theObject->TxPending() || theObject->RepairPending()))
            DeactivateTxObject(theObject);
    }  // end while (optr < nack_end)
    
    if (squelchMsg)
    {        
        squelchMsg->cmd.squelch.len = squelchLen;   
        // (TBD) fill in sync point
        QueueMessage(squelchMsg);
    }
    
    // Here's where we suppress non-multicast NACKs by advertising our
    // current repair state in the form of a NACK
    // Only do this when destination address is multicast
    if (isUnicast && Address()->IsMulticast())// && increasedRepairState)
    {
        //DMSG(1, "mdp: Server should re-advertise NACK ...\n");
        MdpMessage* nackAdv = NULL;
        nackAdv = tx_queue.FindNackAdv();
        bool enqueue;
        if (nackAdv)
        {
            enqueue = false;
            if (!increasedRepairState) nackAdv = NULL;
        }
        else
        {
            enqueue = true;
            nackAdv = msg_pool.Get();
            if (nackAdv)
            {
                char* vector = server_vector_pool.Get();
                if (vector)
                {
                    nackAdv->cmd.nack_adv.data = vector;
                }
                else
                {
                    msg_pool.Put(nackAdv);
                    nackAdv = NULL;
                    DMSG(0, "mdp: Server Warning! vector_pool empty. Couldn't"
                            "re-advertise unicast NACK ...\n");
                }
            }
        }
        if (nackAdv)
        {
            nackAdv->type = MDP_CMD;
            nackAdv->version = MDP_PROTOCOL_VERSION;
            nackAdv->sender = LocalNodeId();
            nackAdv->cmd.flavor = MDP_CMD_NACK_ADV;
            nackAdv->cmd.nack_adv.len = MIN(theMsg->nack.nack_len, segment_size);
            memcpy(nackAdv->cmd.nack_adv.data, theMsg->nack.data, nackAdv->cmd.nack_adv.len);
            if (enqueue) QueueMessage(nackAdv);
            DMSG(0, "mdp: Server Re-advertising unicast NACK ...\n");
            
        }
        else
        {
            if (increasedRepairState)
                DMSG(0, "mdp: Server Warning! msg_pool empty. Couldn't "
                        "re-advertise unicast NACK ...\n");
        }
    }
    else
    {
        //DMSG(1, "mdp: Server should dump NACK ...\n");
        if (isUnicast)
            DMSG(0, "mdp: Server non-increasing NACK adv ...\n");
    }
    
	// Wake up the server in case immediate reaction is required
	if (tx_queue.IsEmpty()) Serve();
}  // end MdpSession::ServerHandleMdpNack()

void MdpSession::ServerHandleMdpAck(MdpMessage* theMsg)
{
    // Is the message for me?
    if (LocalNodeId() != theMsg->ack.server_id) return;
    
#if defined(PROTO_DEBUG) && !defined (OPNET)    
    if (theMsg->ack.grtt_response.tv_sec || theMsg->ack.grtt_response.tv_usec)
    {
        struct in_addr theClient;
        theClient.s_addr = htonl(theMsg->sender);
        DMSG(6, "mdp: node:%lu Server recv'd MDP_ACK with GRTT response from \"%s\"\n", 
				LocalNodeId(), inet_ntoa(theClient));
    }
#endif // PROTO_DEBUG
    // Process embedded GRTT_REQ response 
    ServerProcessClientResponse(theMsg->sender, 
                                &theMsg->ack.grtt_response, 
                                theMsg->ack.loss_estimate,
                                theMsg->ack.grtt_req_sequence);
    switch (theMsg->ack.type)
    {
        case MDP_ACK_GRTT:
            // it was just a GRTT ack
            break;
        
        case MDP_ACK_OBJECT:
        {
            MdpAckingNode* theAcker = (MdpAckingNode*) pos_ack_list.FindNodeById(theMsg->sender);
            if (theAcker)
                theAcker->SetLastAckObject(theMsg->ack.object_id);
        }
        break;
        
        default:
        {
#if defined(PROTO_DEBUG) && !defined (OPNET) 
            struct in_addr theClient;
            theClient.s_addr = htonl(theMsg->sender);
            DMSG(0, "mdp: node:%lu Server recv'd invalid MDP_ACK from \"%s\"\n", LocalNodeId(),
					inet_ntoa(theClient));
#endif 
        }
        break;
    }
}  // end MdpSession::ServerHandleMdpAck()

void MdpSession::ClientHandleMdpNack(MdpMessage* theMsg)
{
    // Do we know this server ??
    MdpServerNode* theServer = (MdpServerNode*) server_list.FindNodeById(theMsg->nack.server_id);
    if (theServer)
        theServer->HandleRepairNack(theMsg->nack.data, theMsg->nack.nack_len);
    else
        DMSG(0, "mdp: node:%lu Heard NACK destined for unknown server ...\n", LocalNodeId());
}  // end MdpSession::ClientHandleMdpNack()    
    

/* Note  - I need to figure out how to get current object/block info
         - from another client's NACK so we can properly start up 
         - repair_timer ... I suggest clients provide their "current_object_id"
         - and "current_block_id" in their NACK content ... Then we can
         - do an "ObjectRepairCheck()" prior to parsing the NACK, the alternative
         - is to parse the NACK twice ... once to get the current obj/block info
         - and a second time to incorporate the NACK content into our
         - "repair_masks" ... Be wary of NACK race condition when GRTT estimate
         - is off!!! ... Think about this !!! 

        -> For now, we won't let other client's NACKs trigger a new repair cycle
        -> Only messages from the server will trigger client repair cycles ?!
        -> Client triggering of repair can lead to all sorts of problems ??        
*/
    
void MdpServerNode::HandleRepairNack(char* nackData, unsigned short nackLen)
{
    // Clients only care about NACKs for NACK suppression purposes
    if (repair_timer.IsActive() && repair_timer.RepeatCount())
    {
        char* optr = nackData;            
        char* nack_end = optr + nackLen;
        while (optr < nack_end)
        {
            MdpObjectNack onack;
            optr += onack.Unpack(optr);
            char* rptr = onack.data;
            char* onack_end = rptr + onack.nack_len;
            MdpRepairNack rnack;    
            MdpObject* theObject = FindRecvObjectById(onack.object_id);
            while (rptr < onack_end)
            {                
                rptr += rnack.Unpack(rptr);
                if (theObject)
                {
                    theObject->ClientHandleRepairNack(&rnack);
                    DMSG(6, "mdp: node:%lu Client heard MDP_NACK_REPAIR for object:%lu\n", 
                        session->LocalNodeId(), theObject->TransportId());
                }
                else
                {
                    if (MDP_RECV_OBJECT_PENDING == ObjectSequenceCheck(onack.object_id))
                    {
                        if (MDP_REPAIR_OBJECT == rnack.type)
                            object_repair_mask.Unset(onack.object_id);
                    }
                }
            }  // end while (rptr < onack_end)
        }  // end while(optr < nack_end)        
    }  // end if (repair_timer.IsActive() ...)
}  // end MdpServerNode::ClientHandleRepairNack()   


#ifdef PROTO_DEBUG

static const char cmd_flavor[6][13] = 
{
    "MDP_NULL_CMD",
    "MDP_FLUSH   ",
    "MDP_SQUELCH ",
    "MDP_ACK_REQ ",
    "MDP_GRTT_REQ",
    "MDP_NACK_ADV"
};
    
static const char dst[2][4] = 
{
    "src",
    "dst"
};
	
static void DataTrace(unsigned long  object_id,
				  unsigned short segment_size, 
		    	  unsigned char  ndata, 
				  unsigned long  offset,
		    	  unsigned char  parityId);
    
void MessageTrace(bool snd, unsigned long node_id, MdpMessage *msg, int len, const NetworkAddress *src)
{   
    struct timeval rxTime;	
    ::GetSystemTime(&rxTime);
    struct tm *rx_time = gmtime((time_t *)&rxTime.tv_sec);
    DMSG(0, "mdp_trace>%02d:%02d:%02d.%06lu ",
                    rx_time->tm_hour, 
                    rx_time->tm_min,
                    rx_time->tm_sec,
                    rxTime.tv_usec); 	
	DMSG(0, "node>%5lu ", node_id);
	
    if (snd) snd = 1;
    
    switch(msg->type)
    {
        case MDP_REPORT:
            DMSG(0, "MDP_REPORT       %s>%-16s len>%04d\n", 
                    dst[snd], src->HostAddressString(), len);
            break;
            
        case MDP_INFO:
            DMSG(0, "MDP_INFO         %s>%-16s len>%04d obj>%04lu\n", 
                dst[snd], src->HostAddressString(), len, msg->object.object_id);
            break;
            
        case MDP_DATA:
            DMSG(0, "MDP_DATA         %s>%-16s len>%04d ", dst[snd], 
				    src->HostAddressString(), len);
		    DataTrace(msg->object.object_id, (msg->object.flags & MDP_DATA_FLAG_RUNT ) ?
					   msg->object.data.segment_size : msg->object.data.len, 
					  msg->object.ndata, msg->object.data.offset, 0);
            break;
             
        case MDP_PARITY:
            DMSG(0, "MDP_PARITY       %s>%-16s len>%04d ", 
                dst[snd], src->HostAddressString(), len);
			DataTrace(msg->object.object_id, msg->object.parity.len, 
                      msg->object.ndata, 
					  msg->object.parity.offset, msg->object.parity.id);
            break;
            
        case MDP_CMD:
            DMSG(0, "%-16s %s>%-16s len>%04d ", 
                cmd_flavor[msg->cmd.flavor], dst[snd], src->HostAddressString(), 
                len);
            switch(msg->cmd.flavor)
            {
                case MDP_CMD_FLUSH:
                    DMSG(0, "%s\n", (msg->cmd.flush.flags & MDP_CMD_FLAG_EOT) ?
                                    "(End-of-Transmission)" : "");
                    break;
                    
                case MDP_CMD_SQUELCH:
                    DMSG(0, "obj>%04lu\n", msg->cmd.squelch.sync_id);
                    break;
                case MDP_CMD_ACK_REQ:
                    DMSG(0, "obj>%04lu\n", msg->cmd.ack_req.object_id);
                    break;
                default:
                    DMSG(0, "\n");
                    break;
            }
            break;
            
        case MDP_NACK:
            DMSG(0, "MDP_NACK         %s>%-16s len>%04d \n", 
                    dst[snd], src->HostAddressString(), len);
            break;
            
        case MDP_ACK:
            DMSG(0, "MDP_ACK          %s>%-16s len>%04d obj>%04lu \n", 
                    dst[snd], src->HostAddressString(), len, msg->ack.object_id);
            break;
            
        default:
            DMSG(0, "MDP_INVALID_MSG %s>%s len>%04d\n", 
                    dst[snd], src->HostAddressString(), len);
    }  // end switch(msg->type)   
}  // end MessageTrace()

static void DataTrace(unsigned long  object_id, 
		    		  unsigned short segment_size, 
		    		  unsigned char  ndata, 
					  unsigned long  offset,
		    		  unsigned char  parityId)
{
	unsigned long block_size = ndata * segment_size;
	unsigned long blockId = offset / block_size;
	unsigned int vector_id;
	if (parityId)
		vector_id = parityId;
	else
		vector_id = (offset - (blockId * block_size)) / segment_size;
	DMSG(0, "obj>%04lu blk>%04lu vec>%03d\n", object_id, blockId, vector_id);
}

#endif // PROTO_DEBUG
        
        
