/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 

#include "mdpSimAgent.h"

bool MdpSimAgent::TimerInstaller(ProtocolTimerInstallCmd cmd,
                                 double                  delay,
                                 ProtocolTimerMgr*       timerMgr, 
                                 const void*             installData)
{
    // For MDP, "installData" contains MdpSessionMgr pointer
    // (session mgr userdata has agent)
    MdpSessionMgr* sessionMgr = (MdpSessionMgr*)installData;
    MdpSimAgent* agent = (MdpSimAgent*)sessionMgr->GetUserData();
    switch(cmd)
	{
		case PROTOCOL_TIMER_INSTALL:
            agent->InstallTimer(delay);
			break;
			
		case PROTOCOL_TIMER_MODIFY:
            agent->ModifyTimer(delay);
			break;
			
		case PROTOCOL_TIMER_REMOVE:
            agent->RemoveTimer();
			break;
	}
	return true;
}  // end MdpSimAgent::TimerInstaller()

bool MdpSimAgent::SocketInstaller(UdpSocketCmd        cmd,
                                  class UdpSocket*    theSocket,
                                  const void*         installData)
{
    // MDP sets "installData" = to its SessionMgr.  For our
    // simulation udp socket we need to reset the socket install
    // data to point to the simulation agent instance.
    switch(cmd)
    {
        case UDP_SOCKET_INSTALL:
        {
            MdpSessionMgr* mgr = (MdpSessionMgr*)installData;
            MdpSimAgent* agent = (MdpSimAgent*)mgr->GetUserData();
            theSocket->Init(theSocket->Owner(), theSocket->RecvHandler(),
                            MdpSimAgent::SocketInstaller, (void*)agent);
            break;
        }
            
        case UDP_SOCKET_REMOVE:
        {
            // Restore socket to original state
            MdpSimAgent* agent = (MdpSimAgent*)installData;
            MdpSessionMgr* mgr = &(agent->session_mgr);
            theSocket->Init(theSocket->Owner(), theSocket->RecvHandler(),
                            NULL, (void*)mgr);
            break;   
        }
    }
    return true;
}  // end MdpSimAgent::SocketInstaller()

bool MdpSimAgent::Notifier(MdpNotifyCode      notifyCode,
                           MdpInstanceHandle  instanceHandle,
                           MdpSessionHandle   sessionHandle,
                           MdpNodeHandle      nodeHandle,
                           MdpObjectHandle    objectHandle,
                           MdpError           errorCode)
{	
    MdpSimAgent* agent = (MdpSimAgent*)
        ((MdpSessionMgr*)instanceHandle)->GetUserData();
    ASSERT(agent);
	agent->Notify(notifyCode, sessionHandle, nodeHandle,
                  objectHandle, errorCode);
	return true;
}  // end MdpSimAgent::Notifier()

MdpSimAgent::MdpSimAgent()
    : total_recvd_low(0), total_recvd_high(0),
      goodput_recvd_low(0), goodput_recvd_high(0),
      session(NULL), ttl(255), tx_rate(64000),
	  segment_size(1024), block_size(64), num_parity(32), auto_parity(0),
      tx_buffer_size(1024*512), rx_buffer_size(1024*512),
      base_object_id(0),
	  congestion_control(false),  fast_start(false), ecn_enable(false),
      tx_rate_min((double)0.0), tx_rate_max((double)0.0),
      unicast_nacks(false), emcon(false), tx_drop_rate(0.0), rx_drop_rate(0.0), 
      initial_grtt(0.5), current_tx_object_size(0), num_repeats(0), 
      logging(false)
      
{
    session_mgr.SetUserData((void*)this);
    session_mgr.SetTimerInstaller(MdpSimAgent::TimerInstaller);
    session_mgr.SetSocketInstaller(MdpSimAgent::SocketInstaller);
    session_mgr.SetNotifyCallback(MdpSimAgent::Notifier);
}

MdpSimAgent::~MdpSimAgent()
{
}

// BEFORE SERVER/CLIENT START ONLY
bool MdpSimAgent::SetSessionTTL(unsigned char theTTL)
{
    if (session) return false;
    ttl = theTTL;
    return true;
}  // end MdpSimAgent::SetSessionTTL()

bool MdpSimAgent::SetSessionSegmentSize(unsigned short theSize)
{
    if (session) return false;
    segment_size = theSize;
    return true;
}  // end MdpSimAgent::SetSessionSegmentSize()

bool MdpSimAgent::SetSessionBlockSize(unsigned char theSize)
{
    if (session) return false;
    block_size = theSize;
    return true;
}  // end MdpSimAgent::SetSessionBlockSize()


bool MdpSimAgent::SetSessionNumParity(unsigned char numParity)
{
    if (session) return false;
    num_parity = numParity;
    return true;
}  // end MdpSimAgent::SetSessionNumParity()


bool MdpSimAgent::SetSessionTxBufferSize(unsigned long theSize)
{
    if (session) return false;
    tx_buffer_size = theSize;
    return true;
}  // end MdpSimAgent::SetSessionTxBufferSize()

bool MdpSimAgent::SetSessionRxBufferSize(unsigned long theSize)
{
    if (session) return false;
    rx_buffer_size = theSize;
    return true;
}  // end MdpSimAgent::SetSessionRxBufferSize()

bool MdpSimAgent::SetSessionBaseObjectId(unsigned long theId)
{
    if (session) return false;
    base_object_id = theId;
    return true;
}  // end MdpSimAgent::SetSessionBaseObjectId()

bool MdpSimAgent::SetSessionEmcon(bool state)
{
    if (session) return false;
    emcon = state;
    return true;
}  // end MdpSimAgent::SetSessionEmcon()



// ANYTIME
void MdpSimAgent::SetSessionTxRate(unsigned long theRate)
{
    tx_rate = theRate;
    if (session) session->SetTxRate(theRate);
}  // end MdpSimAgent::SetSessionTTL()

void MdpSimAgent::SetSessionAutoParity(unsigned char autoParity)
{
    auto_parity = autoParity;
    if (session) session->SetAutoParity(autoParity);
}  // end MdpSimAgent::SetSessionAutoParity()

void MdpSimAgent::SetSessionCongestionControl(bool theState)
{
    congestion_control = theState;
    if (session) session->SetCongestionControl(theState);
}  // end MdpSimAgent::SetSessionCongestionControl()

void MdpSimAgent::SetSessionFastStart(bool theState)
{
    fast_start = theState;
    if (session) session->SetFastStart(theState);
}  // end MdpSimAgent::SetSessionFastStart()

void MdpSimAgent::SetSessionEcnEnable(bool theState)
{
    ecn_enable = theState;
    if (session) session->SetEcnEnable(theState);
}  // end MdpSimAgent::SetSessionEcnEnable()

void MdpSimAgent::SetSessionEcnStatus(bool theState)
{
    ASSERT(session);
    session->SetEcnStatus(theState);
}  // end MdpSimAgent::SetSessionEcnStatus()

void MdpSimAgent::SetSessionUnicastNacks(bool theState)
{
    unicast_nacks = theState;
    if (session) session->SetUnicastNacks(theState);
}  // end MdpSimAgent::SetUnicastNacks()

void MdpSimAgent::SetSendDropRate(double percent)
{
    tx_drop_rate = percent;
    if (session) session->SetSendDropRate(percent);
}  // end  MdpSimAgent::SetSendDropRate()

void MdpSimAgent::SetRecvDropRate(double percent)
{
    rx_drop_rate = percent;
    if (session) session->SetRecvDropRate(percent);
}  // end  MdpSimAgent::SetRecvDropRate()

void MdpSimAgent::SetGrttEstimate(double theTime)
{
    initial_grtt = theTime;
    if (session) session->SetGrttEstimate(theTime);
}  // end  MdpSimAgent::SetGrttEstimate()

void MdpSimAgent::SetTxRateBounds(double txRateMin, double txRateMax)
{
    tx_rate_min = txRateMin;
    tx_rate_max = txRateMax;
    if (session) session->SetTxRateBounds((unsigned long)txRateMin, 
                                          (unsigned long)txRateMax);
}  // end  MdpSimAgent::SetGrttEstimate()

bool MdpSimAgent::StartServer(SIMADDR theAddr, unsigned short thePort)
{
    if (session)
	{
		fprintf(stderr, "NsMdpAgent(): Already open as %s\n",
			    session->IsServer() ? "server" : "client");
		return false;
	}
    
	// For now, use agent's address as MDP node id
	session_mgr.SetLocalNodeId(GetAgentId());
	char text[MDP_NODE_NAME_MAX];
	sprintf(text, "Node %lu", GetAgentId());
    session_mgr.SetNodeName(text);
    	
	if(!(session = session_mgr.NewSession()))
	{
		fprintf(stderr, "MdpSimAgent::StartServer() Error " 
                        "creating new session!\n");
		return false;
	}
	
    // Set server parameters
	session->SetSimAddress(theAddr, thePort);	
	session->SetTTL(ttl);
    session->SetTxRate(tx_rate);
    session->SetCongestionControl(congestion_control);
    session->SetFastStart(fast_start);
	session->SetAutoParity(auto_parity);
    session->SetGrttEstimate(initial_grtt);
    session->SetTxRateBounds((unsigned long)tx_rate_min, (unsigned long)tx_rate_max);
    session->SetSendDropRate(tx_drop_rate);
    session->SetRecvDropRate(rx_drop_rate);
    session->SetBaseObjectTransportId(base_object_id);
	session->SetEmconServer(emcon);
    
    if (MDP_ERROR_NONE != session->OpenServer(segment_size, block_size,
			 								  num_parity, tx_buffer_size))
	{
		fprintf(stderr, "NsMdpAgent::StartServer() Error opening server "
				        "session!\n");
		session_mgr.DeleteSession(session);
		session = NULL;
		return false;
	}
    return true;  
}  // end MdpSimAgent::StartServer()

bool MdpSimAgent::QueueTxObject(unsigned long dataSize)
{
    if (!session || !session->IsServer())
	{
		fprintf(stderr, "MdpSimAgent::QueueTxObject() Node is not a server!");
		return false;
	}
    MdpError err;
    MdpSimObject* theObject = session->NewTxSimObject(0, dataSize, &err);
    if (!theObject)
    {
        fprintf(stderr, "MdpSimAgent::QueueTxObject() Tx object create error!\n");
	    session->Close(true);
		session_mgr.DeleteSession(session);
		session = NULL;
		return false;
    }
    current_tx_object_size = dataSize;
    return true;
}  // end MdpSimAgent::QueueTxObject()

bool MdpSimAgent::StartClient(SIMADDR theAddr, unsigned short thePort)
{
	if (session)
	{
		fprintf(stderr, "MdpSimAgent::StartClient() Already open as %s\n",
			    session->IsServer() ? "server" : "client");
		return false;
	}
	
    // For now, use agent's local address/id as MDP node id
	session_mgr.SetLocalNodeId(GetAgentId());
	char text[MDP_NODE_NAME_MAX];
	sprintf(text, "Node %lu", GetAgentId());
    session_mgr.SetNodeName(text);
    	
	if(!(session = session_mgr.NewSession()))
	{
		fprintf(stderr, "MdpSimAgent::StartClient() Error creating new session!\n");
		return false;
	}

	// Set client parameters
	session->SetSimAddress(theAddr, thePort);
	session->SetTTL(ttl);
	session->SetTxRate(tx_rate);
    session->SetUnicastNacks(unicast_nacks);
    session->SetEcnEnable(ecn_enable);
    session->SetSendDropRate(tx_drop_rate);
    session->SetRecvDropRate(rx_drop_rate);
	session->SetEmconClient(emcon);
    
    // Reset recv volume/goodput tracking
    total_recvd_low = 0;
    total_recvd_high = 0;
    goodput_recvd_low = 0;
    goodput_recvd_high =0;
    
    // Open client operation
	if (MDP_ERROR_NONE != session->OpenClient(rx_buffer_size))
	{
		fprintf(stderr, "MdpSimAgent::StartClient() Error opening client!\n");
		session_mgr.DeleteSession(session);
		session = NULL;
		return false;
	}
	return true;
}  // end MdpSimAgent::StartClient()

void MdpSimAgent::Stop()
{
	if (session)
	{
        session->Close(true);
		session_mgr.DeleteSession(session);
		session = NULL;
	}
}  // end MdpSimAgent::Stop()


void MdpSimAgent::Notify(MdpNotifyCode      notifyCode,
						 MdpSessionHandle   sessionHandle,
                         MdpNodeHandle      nodeHandle,
                	     MdpObjectHandle    /*objectHandle*/,
                	     MdpError           /*errorCode*/)
{	
	switch(notifyCode)
	{
		case MDP_NOTIFY_ERROR:
			break;
			
		case MDP_NOTIFY_TX_OBJECT_START:
			break;
			
		case MDP_NOTIFY_TX_OBJECT_FIRST_PASS:
			break;
			
		case MDP_NOTIFY_TX_OBJECT_ACK_COMPLETE:
			break;
			
		case MDP_NOTIFY_TX_OBJECT_FINISHED:
			break;
			
		case MDP_NOTIFY_TX_QUEUE_EMPTY:
			if (num_repeats)
			{
				if (num_repeats > 0) num_repeats--;
                QueueTxObject(current_tx_object_size);
			}
			break;
			
		case MDP_NOTIFY_RX_OBJECT_START:
			break;
           	
		case MDP_NOTIFY_RX_OBJECT_UPDATE:
        {
            // On new goodput 
            MdpServerNode* src = (MdpServerNode*)nodeHandle;
            ASSERT(src);
            unsigned short dataLen = src->SegmentSize();
            goodput_recvd_low += dataLen;
            if (goodput_recvd_low < dataLen)
                goodput_recvd_high++;
        }
		break;
            
	    case MDP_NOTIFY_RX_OBJECT_INFO:
            break;
            
		case MDP_NOTIFY_RX_OBJECT_COMPLETE:
			//SetReceiptStatus(true);
			break;
            
        case MDP_NOTIFY_OBJECT_DELETE:
            break;
            
        case MDP_NOTIFY_REPRESENTATIVE_ELECT:
            //SetReceiptStatus(true);
            break;
            
        case MDP_NOTIFY_REPRESENTATIVE_IMPEACH:
            //SetReceiptStatus(false);
            break;
			
		default:
			fprintf(stdout, "MdpSimAgent::Notify(code=%d) UNKNOWN?\n", notifyCode);
			break;
	}
}  // end MdpSimAgent()
