/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>  // for PATH_MAX
#include <string.h>

#define VERSION "2.3"

#define MIN(X,Y) ((X<Y)?X:Y)
#define MAX(X,Y) ((X>Y)?X:Y)

const int MAX_LINE = 256;

class FastReader
{
    public:
        FastReader();
        bool Read(FILE* filePtr, char* buffer, unsigned int* len);
        bool Readline(FILE* filePtr, char* buffer, unsigned int* len);
    
    private:
        char         savebuf[256];
        char*        saveptr;
        unsigned int savecount;
};  // end class FastReader

class Flow
{
    friend class FlowList;
    
    public:
        Flow();
        ~Flow();
        void PrintDescription(FILE* f);
        const char* Type() {return type;}
        bool SetType(const char* theType);
        int SrcAddr() {return src_addr;}
        void SetSrcAddr(int value) {src_addr = value;}
        int SrcPort() {return src_port;}
        void SetSrcPort(int value) {src_port = value;}
        int DstAddr() {return dst_addr;}
        void SetDstAddr(int value) {dst_addr = value;}
        int DstPort() {return dst_port;}
        void SetDstPort(int value) {dst_port = value;}
        
        bool IsComposite();
        
        bool TypeMatch(const char* theType)
        {
            if (theType && type)
            {
                return (0 == strcmp(theType, type));
            }
            else
            {
                return (theType == type);
            }
        }
        
        bool Match(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort);
        bool ExactMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort);
        
        double Bytes() {return (1.0e06*((double)megabytes) + ((double)bytes));}
        void AddBytes(unsigned short pktSize) 
        {
            bytes += pktSize;
            if (bytes > 999999)
            {
                bytes -= 999999;
                megabytes += 1;   
            }
        }
        
        void Reset()
        {
            megabytes = 0;
            bytes = 0;
        }
        Flow* Next() {return next;}
        
            
    private:
        char* type;
        int   type_len;
        
        int   src_addr;
        int   src_port;
        int   dst_addr;
        int   dst_port;
        
        // Two part count accumulator
        unsigned long   megabytes;
        unsigned long   bytes;
            
        Flow* prev;
        Flow* next;  
}; // end class Flow

class FlowList
{
    public:
        FlowList();
        ~FlowList();
        void Destroy();
        void Append(Flow* theFlow);
        void Remove(Flow* theFlow);
        Flow* FindFlowByMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort);
        Flow* Head() {return head;}
    
    private:
        Flow* head;
        Flow* tail;
    
};


Flow::Flow()
    : type(NULL), type_len(0), 
      src_addr(-1), src_port(-1), dst_addr(-1), dst_port(-1),
      megabytes(0), bytes(0),
      prev(NULL), next(NULL)
{
}

Flow::~Flow()
{
    if (type) delete []type;
}

bool Flow::SetType(const char* theType)
{
    if (type) delete []type;
    int len = strlen(theType) + 1;
    if(!(type = new char[len]))
    {
        perror("bwplot: Error allocating flow type storage");
        return false;
    }
    strcpy(type, theType);
    type_len = len - 1;
    return true;
}  // end Flow::SetName()

bool Flow::IsComposite()
{
    if ((NULL != type) || 
        (src_addr != -1) || (src_port != -1) ||
        (dst_addr >= 0) || (dst_port >= 0))
    {
        return false;   
    }
    else
    {
        return true;
    }
}  // end Flow::IsComposite()

bool Flow::Match(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort)
{
    if ((type && !TypeMatch(theType)) ||    
        ((dst_port >= 0) && (dst_port != dstPort)) ||
        ((dst_addr != -1) && (dst_addr != dstAddr)) ||
        ((src_port >= 0) && (src_port != srcPort)) ||
        ((src_addr != -1) && (src_addr != srcAddr)))
    {
        return false;
    }
    else
    {
        return true;
    }
}  // end Flow::Match()

bool Flow::ExactMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort)
{
    if ((!TypeMatch(theType)) ||     
        (dst_port != dstPort) ||
        (dst_addr != dstAddr) ||
        (src_port != srcPort) ||
        (src_addr != srcAddr))
    {
        return false;
    }
    else
    {
        return true;
    }
}  // end Flow::ExactMatch()

void Flow::PrintDescription(FILE* f)
{
    if (type)
        fprintf(f, "%s:", type);
    else
        fprintf(f, "*:");
    if (src_addr == -1)
        fprintf(f, "*.");
    else
        fprintf(f, "%d.", src_addr);
    if (src_port < 0)
        fprintf(f, "*->");
    else
        fprintf(f, "%d->", src_port);
    if (dst_addr == -1)
        fprintf(f, "*.");
    else
        fprintf(f, "%d.", dst_addr);
    if (dst_port < 0)
        fprintf(f, "*");
    else
        fprintf(f, "%d", dst_port);
}  // end Flow::PrintDescription()

FlowList::FlowList()
    : head(NULL), tail(NULL)
{
}

FlowList::~FlowList()
{
    Destroy();
}

void FlowList::Append(Flow* theFlow)
{
    if ((theFlow->prev = tail))
        theFlow->prev->next = theFlow;
    else
        head = theFlow;
    theFlow->next = NULL;
    tail = theFlow;
}  // end FlowList::Append()


void FlowList::Remove(Flow* theFlow)
{
    if (theFlow->prev)
        theFlow->prev->next = theFlow->next;
    else
        head = theFlow->next;
    
    if (theFlow->next)
        theFlow->next->prev = theFlow->prev;
    else
        tail = theFlow->prev;
}  // end FlowList::Remove()
        
Flow* FlowList::FindFlowByMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort)
{
    Flow* next_flow = head;
    while (next_flow)
    {
        if (next_flow->ExactMatch(theType, srcAddr, srcPort, dstAddr, dstPort))
            return next_flow;
        next_flow = next_flow->next;
    }
    return NULL;   
}  // end FlowList:FindFlowByName()

void FlowList::Destroy()
{
    Flow* next;
    while ((next = head))
    {
        Remove(next);
        delete next;
    }   
}  // end Destroy()


inline void usage()
{
    fprintf(stderr, "Usage: bwplot [version][testId <id>][len <protoNameLen>]\n"
                    "              [lag <lagTime>][flow <flowType>] link <srcNode,dstNode>\n"
                    "              trace <traceFile> [output <outputFile>]\n");
}
        
int main(int argc, char* argv[])
{ 
    double windowSize = 1.0;
    char* traceFile = NULL;
    char* outputFile = NULL;
    int link_src = 0;
    int link_dst = 0;
    bool wildcardFlowType = true;
    unsigned int protoNameLen = 31;
    const char* testId = "1";
    double lagTime = 0.0;
    
    FlowList validFlowList;  // list of flow types we're looking for
    
    if (argc < 5)
    {
        usage();
        exit(-1);
    }
    
    fprintf(stderr, "bwplot Version %s\n", VERSION);
            
    // Parse command line
    fprintf(stderr, "\n");
    int i = 1;
    while(i < argc)
    {
        if (!strcmp("link", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "bwplot: Insufficient \"link\" arguments!\n");
                usage();
                exit(-1);
            }
            if (2 != sscanf(argv[i++], "%d,%d", &link_src, &link_dst))
            {
                fprintf(stderr, "bwplot: Error parsing \"link\" description!\n");
                usage();
                exit(-1);
            }
        } 
        else if (!strcmp("testId", argv[i]))
        {
            i++; 
            if (i >= argc)
            {
                fprintf(stderr, "bwplot: Insufficient \"testId\" arguments!\n");
                usage();
                exit(-1);
            }
            testId = argv[i++];
        } 
        else if (!strcmp("lag", argv[i]))
        {
            i++; 
            if (i >= argc)
            {
                fprintf(stderr, "bwplot: Insufficient \"lag\" arguments!\n");
                usage();
                exit(-1);
            }
            float theTime;
            if (1 != sscanf(argv[i++], "%f", &theTime))
            {
                fprintf(stderr, "bwplot: Error parsing <lagTime>.\n");
                usage();
                exit(-1);
            }
            lagTime = theTime;
        }
        else if (!strcmp("len", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "bwplot: Insufficient \"len\" arguments!\n");
                usage();
                exit(-1);
            }
            int len = atoi(argv[i++]);
            if (len < 1)
            {
                fprintf(stderr, "bwplot: Error! Invalid \"len\" argument.\n");
                usage();
                exit(-1);
            }   
            protoNameLen = len;
        }
        else if (!strcmp("flow", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "bwplot: Insufficient \"flow\" arguments!\n");
                usage();
                exit(-1);
            }
            Flow* theFlow = new Flow;
            if (!theFlow)
            {
                perror("bwplot: Error creating flow");
                exit(-1);   
            }
            if (!theFlow->SetType(argv[i++])) exit(-1);
            validFlowList.Append(theFlow);
            wildcardFlowType = false;
        }
        else if (!strcmp("trace", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "bwplot: Insufficient \"trace\" arguments!\n");
                usage();
                exit(-1);
            }
            traceFile = argv[i++];
        }    
        else if (!strcmp("version", argv[i]))
        {
            exit(0);   
        }
        else if (!strcmp("output", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "bwplot: Insufficient \"output\" arguments!\n");
                usage();
                exit(-1);
            }
            outputFile = argv[i++];
        }
        else
        {
            fprintf(stderr, "bwplot: Error! Invalid command: %s\n", argv[i]);
            usage();
            exit(-1);
        }
    }      
    
    // Open ns-2 trace input file
    FILE* inFile;
    if (traceFile)
    {
        if(!(inFile = fopen(traceFile, "r")))
        {
            perror("bwplot: Error opening input trace file");
            usage();
            exit(-1);
        }
    }
    else
    {
        fprintf(stderr, "bwplot: No \"trace\" file provided!\n");
        usage();
        exit(-1);
    }
    
    
    // Read input file, identify unique flows, 
    // and cumulatively add the number of bytes
    char buffer[MAX_LINE];
    unsigned int len = MAX_LINE;
    unsigned int line = 0;
    FastReader reader;
    unsigned int flowCount = 0;
    Flow cumulativeFlow;
    FlowList foundFlowList;  // list of unique flows we find
    while (reader.Readline(inFile, buffer, &len))
    {
        line++;
        float theTime;
        char proto[32], junkText[16];
        unsigned int linkSrc, linkDst, pktSize, flowId;
        int srcAddr, srcPort, dstAddr, dstPort;
        if (len && ('-' == buffer[0]))
        {
            if (11 == sscanf(&buffer[1], "%f %u %u %32s %u %16s %d %d.%d %d.%d", &theTime, 
                                          &linkSrc, &linkDst, proto, &pktSize, junkText, 
                                          &flowId, &srcAddr, &srcPort, &dstAddr, &dstPort))
            {
                // Is this line on the our link?
                if ((theTime >= lagTime) && (linkSrc == link_src) && (linkDst == link_dst))
                {
                    proto[31] = '\0';  // to be safe, should be cool anyway
                    unsigned int protoLen = strlen(proto);
                    protoLen = MIN(protoLen, protoNameLen);
                    proto[protoLen] = '\0';
                                    
                    // Is this a flow we already know?
                    Flow* nextFlow = foundFlowList.Head();
                    while (nextFlow)
                    {
                        if (nextFlow->Match(proto, srcAddr, srcPort, dstAddr, dstPort))
                        {
                            break;
                        }
                        nextFlow = nextFlow->Next();
                    }
                    
                    if (!nextFlow)
                    {
                        // Is this of an existing or new flow type?
                        Flow* nextFlow = validFlowList.Head();
                        while (nextFlow)
                        {
                            if (nextFlow->TypeMatch(proto)) break;
                            nextFlow = nextFlow->Next(); 
                        }
                        // If we're wildcarding flow types (default)
                        // add unique flow types to "validFlowList" on the fly
                        if (!nextFlow && wildcardFlowType)
                        {
                            if ((nextFlow = new Flow))
                            {
                                if (!nextFlow->SetType(proto)) exit(-1);
                                validFlowList.Append(nextFlow);
                            }
                            else
                            {
                                perror("bwplot: Error creating flow");
                                exit(-1); 
                            }
                        }
                        if (nextFlow)
                        {
                            nextFlow->AddBytes(1);  // use "bytes" for flow counter
                            nextFlow = NULL;
                        }
                        else
                        {
                            break; // Skip over this non-valid flow  
                        }
                    }
                    
                    if (!nextFlow)
                    {
                        if (!(nextFlow = new Flow))
                        {
                            perror("bwplot: Error allocating memory for new Flow");
                            fclose(inFile);
                            exit(-1);
                        }
                        
                        
                        unsigned int protoLen = strlen(proto);
                        protoLen = MIN(protoNameLen, protoLen);
                        proto[protoLen] = '\0';
                        if (protoLen) 
                        {
                            if (!nextFlow->SetType(proto)) exit(-1);
                        }   
                        nextFlow->SetSrcAddr(srcAddr);
                        nextFlow->SetSrcPort(srcPort);
                        nextFlow->SetDstAddr(dstAddr);
                        nextFlow->SetDstPort(dstPort);
                        foundFlowList.Append(nextFlow);
                        flowCount++;  // Track count of valid, unique flows
                    }
                    
                    // If it's a valid flow, count it
                    if (nextFlow)
                    {
		                nextFlow->AddBytes(pktSize);
                        cumulativeFlow.AddBytes(pktSize);
                    }
                }  // end if ((linkSrc == link_src) && (linkDst == link_dst))
            }
            else
            {
                fprintf(stderr, "bwplot: Error parsing ns-2 trace file at line %u.\n", line);
            }           
            
        }
        len = MAX_LINE;
    }    
    fclose(inFile);
    
    if (!flowCount)
    {
        fprintf(stderr, "bwplot: No valid flows were found!\n");
        exit(0);   
    }
    
    // Open output file
    FILE* outFile;
    if (outputFile)
    {
        if(!(outFile = fopen(outputFile, "w+")))
        {
            perror("bwplot: Error opening output file");
            usage();
            exit(-1);
        } 
    }
    else
    {
        outputFile = "stdout";
        outFile = stdout; 
    }
    
    double expectedThroughput = cumulativeFlow.Bytes() / ((double)flowCount);
    
    fprintf(outFile, "test %s\n", testId);
    Flow* nextValidFlow = validFlowList.Head();
    while (nextValidFlow)
    {
        Flow* nextFoundFlow = foundFlowList.Head();
        double normalSum = 0.0;
        double normalCount = 0.0;
        while (nextFoundFlow)
        {
            if (!strcmp(nextFoundFlow->Type(), nextValidFlow->Type()))
            {
                double normalizedThroughput = 
                    nextFoundFlow->Bytes() / expectedThroughput;
                fprintf(outFile, "flow ");
                nextFoundFlow->PrintDescription(outFile);
                fprintf(outFile, " %f\n", normalizedThroughput);
                normalSum += normalizedThroughput;
                normalCount += 1.0;
            }
            nextFoundFlow = nextFoundFlow->Next();
        }
        if (normalCount > 0.0)
        {
            double normalMean = normalSum / normalCount;
            fprintf(outFile, "mean ");
            nextValidFlow->PrintDescription(outFile);
            fprintf(outFile, " %f\n", normalMean);
            normalSum = 0.0;
            normalCount = 0.0;
        }
        nextValidFlow = nextValidFlow->Next();
    }  // end while(nextValidFlow)
    fprintf(outFile, "\n");
    fclose(outFile);
    fprintf(stderr, "bwplot: Done.\n");
    exit(0);
}  // end main()



FastReader::FastReader()
    : savecount(0)
{
    
}

bool FastReader::Read(FILE* filePtr, char* buffer, unsigned int* len)
{
    unsigned int want = *len;   
    if (savecount)
    {
        unsigned int ncopy = MIN(want, savecount);
        memcpy(buffer, saveptr, ncopy);
        savecount -= ncopy;
        saveptr += ncopy;
        buffer += ncopy;
        want -= ncopy;
    }
    while (want)
    {
        unsigned int result = fread(savebuf, sizeof(char), 256, filePtr);
        if (result)
        {
            unsigned int ncopy= MIN(want, result);
            memcpy(buffer, savebuf, ncopy);
            savecount = result - ncopy;
            saveptr = savebuf + ncopy;
            buffer += ncopy;
            want -= ncopy;
        }
        else  // end-of-file
        {
            *len -= want;
            if (*len)
                return true;  // we read something
            else
                return false; // we read nothing
        }
    }
    return true;
}  // end FastReader::Read()

// An OK text readline() routine (reads what will fit into buffer incl. NULL termination)
// if *len is unchanged on return, it means the line is bigger than the buffer and 
// requires multiple reads

bool FastReader::Readline(FILE* filePtr, char* buffer, unsigned int* len)
{   
    unsigned int count = 0;
    unsigned int length = *len;
    char* ptr = buffer;
    unsigned int one = 1;
    while ((count < length) && Read(filePtr, ptr, &one))
    {
        if (('\n' == *ptr) || ('\r' == *ptr))
        {
            *ptr = '\0';
            *len = count;
            return true;
        }
        count++;
        ptr++;
    }
    // Either we've filled the buffer or hit end-of-file
    if (count < length) *len = 0; // Set *len = 0 on EOF
    return false;
}  // end FastReader::Readline()
