/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>  // for PATH_MAX
#include <string.h>

#define VERSION "2.3"

#define MIN(X,Y) ((X<Y)?X:Y)
#define MAX(X,Y) ((X>Y)?X:Y)

const int MAX_LINE = 256;

class FastReader
{
    public:
        FastReader();
        bool Read(FILE* filePtr, char* buffer, unsigned int* len);
        bool Readline(FILE* filePtr, char* buffer, unsigned int* len);
    
    private:
        char         savebuf[256];
        char*        saveptr;
        unsigned int savecount;
};  // end class FastReader

class Flow
{
    friend class FlowList;
    
    public:
        Flow();
        ~Flow();
        void PrintDescription(FILE* f);
        const char* Type() {return type;}
        bool SetType(const char* theType);
        int SrcAddr() {return src_addr;}
        void SetSrcAddr(int value) {src_addr = value;}
        int SrcPort() {return src_port;}
        void SetSrcPort(int value) {src_port = value;}
        int DstAddr() {return dst_addr;}
        void SetDstAddr(int value) {dst_addr = value;}
        int DstPort() {return dst_port;}
        void SetDstPort(int value) {dst_port = value;}
        
        bool IsComposite();
        
        bool TypeMatch(const char* theType)
        {
            if (theType && type)
                return (0 == strcmp(theType, type));
            else
                return (theType == type);
        }
        
        bool Match(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort);
        bool ExactMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort);
        
        unsigned long Bytes() {return byte_count;}
        void AddBytes(unsigned long pktSize) {byte_count += pktSize;}
        
        void Reset()
        {
            byte_count = 0;
        }
        Flow* Next() {return next;}
        
            
    private:
        char* type;
        int   type_len;
        
        int   src_addr;
        int   src_port;
        int   dst_addr;
        int   dst_port;
        
        // Byte count accumulator
        unsigned long   byte_count;
            
        Flow* prev;
        Flow* next;  
}; // end class Flow

class FlowList
{
    public:
        FlowList();
        ~FlowList();
        void Destroy();
        void Append(Flow* theFlow);
        void Remove(Flow* theFlow);
        Flow* FindFlowByMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort);
        Flow* Head() {return head;}
    
    private:
        Flow* head;
        Flow* tail;
    
};


Flow::Flow()
    : type(NULL), type_len(0), 
      src_addr(-1), src_port(-1), dst_addr(-1), dst_port(-1),
      byte_count(0), 
      prev(NULL), next(NULL)
{
}

Flow::~Flow()
{
    if (type) delete []type;
}

bool Flow::SetType(const char* theType)
{
    if (type) delete []type;
    int len = strlen(theType) + 1;
    if(!(type = new char[len]))
    {
        perror("rateplot: Error allocating flow type storage");
        return false;
    }
    strcpy(type, theType);
    type_len = len - 1;
    return true;
}  // end Flow::SetName()

bool Flow::IsComposite()
{
    if ((NULL != type) || 
        (src_addr != -1) || (src_port != -1) ||
        (dst_addr >= 0) || (dst_port >= 0))
    {
        return false;   
    }
    else
    {
        return true;
    }
}  // end Flow::IsComposite()

bool Flow::Match(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort)
{
    if ((type && !TypeMatch(theType)) ||    
        ((dst_port >= 0) && (dst_port != dstPort)) ||
        ((dst_addr != -1) && (dst_addr != dstAddr)) ||
        ((src_port >= 0) && (src_port != srcPort)) ||
        ((src_addr != -1) && (src_addr != srcAddr)))
    {
        return false;
    }
    else
    {
        return true;
    }
}  // end Flow::Match()

bool Flow::ExactMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort)
{
    if ((!TypeMatch(theType)) ||     
        (dst_port != dstPort) ||
        (dst_addr != dstAddr) ||
        (src_port != srcPort) ||
        (src_addr != srcAddr))
    {
        return false;
    }
    else
    {
        return true;
    }
}  // end Flow::ExactMatch()

void Flow::PrintDescription(FILE* f)
{
    if (type)
        fprintf(f, "%s:", type);
    else
        fprintf(f, "*:");
    if (src_addr == -1)
        fprintf(f, "*.");
    else
        fprintf(f, "%d.", src_addr);
    if (src_port < 0)
        fprintf(f, "*->");
    else
        fprintf(f, "%d->", src_port);
    if (dst_addr == -1)
        fprintf(f, "*.");
    else
        fprintf(f, "%d.", dst_addr);
    if (dst_port < 0)
        fprintf(f, "*");
    else
        fprintf(f, "%d", dst_port);
}  // end Flow::PrintDescription()

FlowList::FlowList()
    : head(NULL), tail(NULL)
{
}

FlowList::~FlowList()
{
    Destroy();
}

void FlowList::Append(Flow* theFlow)
{
    if ((theFlow->prev = tail))
        theFlow->prev->next = theFlow;
    else
        head = theFlow;
    theFlow->next = NULL;
    tail = theFlow;
}  // end FlowList::Append()


void FlowList::Remove(Flow* theFlow)
{
    if (theFlow->prev)
        theFlow->prev->next = theFlow->next;
    else
        head = theFlow->next;
    
    if (theFlow->next)
        theFlow->next->prev = theFlow->prev;
    else
        tail = theFlow->prev;
}  // end FlowList::Remove()
        
Flow* FlowList::FindFlowByMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort)
{
    Flow* next_flow = head;
    while (next_flow)
    {
        if (next_flow->ExactMatch(theType, srcAddr, srcPort, dstAddr, dstPort))
            return next_flow;
        next_flow = next_flow->next;
    }
    return NULL;   
}  // end FlowList:FindFlowByName()

void FlowList::Destroy()
{
    Flow* next;
    while ((next = head))
    {
        Remove(next);
        delete next;
    }   
}  // end Destroy()


const char WILDCARD = 'X';

inline void usage()
{
    fprintf(stderr, "Usage: rateplot [version][gif][post][raw][window <sec>][auto][len <protoNameLen>]\n"
                    "                [flow <type,srcAddr.port,dstAddr.port>]\n"
                    "                link <srcNode,dstNode> trace <traceFile> [<outputFile>]\n");
    fprintf(stderr, "          (Wildcard type/address/port parameters with 'X')\n");
}
        
int main(int argc, char* argv[])
{ 
    FlowList flow_list;
    double windowSize = 1.0;
    char* trace_file;
    char* output_file;
    bool use_gnuplot = true;
    bool make_gif = false;
    bool make_post = false;
    int link_src = 0;
    int link_dst = 0;
    
    bool auto_detect = false;
    unsigned int detect_proto_len = 31;
    
    if (argc < 2)
    {
        usage();
        exit(-1);
    }
    
    fprintf(stderr, "Rateplot Version %s\n", VERSION);
            
    // Parse command line
    fprintf(stderr, "\n");
    int i = 1;
    while(i < argc)
    {
        if (!strcmp("window", argv[i]))
        {
            i++;
            float w;
            if (1 != sscanf(argv[i], "%f", &w))
            {
               fprintf(stderr, "rateplot: Error parsing \"window\" size!\n");
               usage();
               exit(-1);
            }
            windowSize = w;
            i++;
        }
        else if (!strcmp("flow", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "rateplot: Insufficient \"flow\" arguments!\n");
                usage();
                exit(-1);
            }
            // Pull out flow type, checking for wildcard
            Flow* theFlow = new Flow;
            if (!theFlow)
            {
                perror("rateplot: Error allocating memory for flow");
                exit(-1);
            }
            char* flow_info = strtok(argv[i++], ",");
            if (flow_info) 
            {
                if(WILDCARD != flow_info[0])
                    theFlow->SetType(flow_info);
            }
            else
            {
                fprintf(stderr, "rateplot: Error parsing \"flow\" description!\n");
                usage();
                exit(1);
            }
            // Pull out source addr/port, checking for wildcards
            flow_info = strtok(NULL, ",");
            if (flow_info)
            {
                // Parse source address/port
                char* ptr = strchr(flow_info, '.');
                if (ptr) 
                {
                    *ptr++ = '\0';
                    if (WILDCARD != ptr[0])
                        theFlow->SetSrcPort(atoi(ptr));
                }
                if (WILDCARD != flow_info[0])
                    theFlow->SetSrcAddr(atoi(flow_info));
            
                // Pull out destination addr/port, checking for wildcards
                flow_info = strtok(NULL, ",");
                if (flow_info)
                {
                    // Parse source address/port
                    char* ptr = strchr(flow_info, '.');
                    if (ptr) 
                    {
                        *ptr++ = '\0';
                        if (WILDCARD != ptr[0])
                            theFlow->SetDstPort(atoi(ptr));
                    }
                    if (WILDCARD != flow_info[0])
                       theFlow->SetDstAddr(atoi(flow_info));
                }
            }
            flow_list.Append(theFlow);
            fprintf(stderr, "rateplot: Adding flow: ");
            theFlow->PrintDescription(stderr);
            fprintf(stderr, "\n");
        }
        else if (!strcmp("link", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "rateplot: Insufficient \"link\" arguments!\n");
                usage();
                exit(-1);
            }
            if (2 != sscanf(argv[i++], "%d,%d", &link_src, &link_dst))
            {
                fprintf(stderr, "rateplot: Error parsing \"link\" description!\n");
                usage();
                exit(-1);
            }
        } 
        else if (!strcmp("raw", argv[i]))
        {
            i++;
            use_gnuplot = false;
        } 
        else if (!strcmp("gif", argv[i]))
        {
            i++;
            make_gif = true;
        }  
        else if (!strcmp("post", argv[i]))
        {
            i++;
            make_post = true;
        } 
        else if (!strcmp("trace", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "rateplot: Insufficient \"trace\" arguments!\n");
                usage();
                exit(-1);
            }
            trace_file = argv[i++];
        }    
        else if (!strcmp("len", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "rateplot: Insufficient \"len\" arguments!\n");
                usage();
                exit(-1);
            }   
	        detect_proto_len = atoi(argv[i++]);
        }
        else if (!strcmp("auto", argv[i]))
        {
            i++;
            auto_detect = true;
        }    
        else if (!strcmp("version", argv[i]))
        {
            exit(0);   
        }
        else
        {
            output_file = argv[i++];
        }
    }   
    
    // Open ns-2 trace input file
    FILE* infile;
    if (trace_file)
    {
        if(!(infile = fopen(trace_file, "r")))
        {
            perror("rateplot: Error opening input trace file");
            usage();
            exit(-1);
        }
    }
    else
    {
        fprintf(stderr, "rateplot: No \"trace\" file provided!\n");
        usage();
        exit(-1);
    }
    
    // Open output file
    FILE* outfile;
    char temp_file[PATH_MAX];
    if (output_file)
    {
	strcpy(temp_file, output_file);
	if (use_gnuplot) strcat(temp_file, ".tmp");
        if(!(outfile = fopen(temp_file, "w+")))
        {
            perror("rateplot: Error opening output file");
            usage();
            exit(-1);
        } 
    }
    else
    {
        output_file = "stdout";
        outfile = stdout;  
        use_gnuplot = false;
    }
    
    
    
    
    char buffer[MAX_LINE];
    unsigned int len = MAX_LINE;
    unsigned int line = 0;
    
    
    double windowStart = 0.0;
    double windowEnd = windowStart + windowSize;
    bool validWindow = false;
    
    FastReader reader;
    while (reader.Readline(infile, buffer, &len))
    {
        line++;
        float theTime;
        char proto[32], junkText[16];
        unsigned int linkSrc, linkDst, pktSize, flowId;
        int srcAddr, srcPort, dstAddr, dstPort;
        
        if (len && ('-' == buffer[0]))
        {
            if (11 == sscanf(&buffer[1], "%f %u %u %32s %u %16s %d %d.%d %d.%d", &theTime, 
                                          &linkSrc, &linkDst, proto, &pktSize, junkText, 
                                          &flowId, &srcAddr, &srcPort, &dstAddr, &dstPort))
            {
                proto[31] = '\0';
                unsigned protoLen = MIN(strlen(proto), 32);
                protoLen = MIN(detect_proto_len, protoLen);
                proto[protoLen] = '\0';
                            
                if (theTime < windowStart)
                {
                    windowStart = theTime;
                    windowEnd = windowStart + windowSize;
                    validWindow = false;
                }
                
                if (validWindow && (theTime > windowEnd))
                {
                    // Plot points
                    // Window start point
                    fprintf(outfile, "%7.3f", windowStart);
                    Flow* next_flow = flow_list.Head();
                    while (next_flow)
                    {
                        fprintf(outfile, ", %6.3f", (8.0/1000.0) * ((double)next_flow->Bytes())/windowSize);
                        next_flow = next_flow->Next();
                    }
                    fprintf(outfile, "\n");
                    // Window end point
                    fprintf(outfile, "%7.3f", windowEnd);
                    next_flow = flow_list.Head();
                    while (next_flow)
                    {
                        fprintf(outfile, ", %6.3f", (8.0/1000.0) * ((double)next_flow->Bytes())/windowSize);
                        next_flow->Reset();
                        next_flow = next_flow->Next();
                    }
                    fprintf(outfile, "\n");
                    // Update window start/end
                    windowStart = windowEnd;
                    windowEnd += windowSize;
                }                  
                // Is this line on the our link?
                if ((linkSrc == link_src) && (linkDst == link_dst))
                {
                    bool match = false;
                    Flow* next_flow = flow_list.Head();
                    while (next_flow)
                    {
                        //fprintf(stderr, "Matching flow: ");
                        //next_flow->PrintDescription(stderr);
                        //fprintf(stderr, "to proto:%3s src:%u.%u dst:%u:%u\n",
                        //        proto, srcAddr, srcPort, dstAddr, dstPort);
                        if (next_flow->Match(proto, srcAddr, srcPort, dstAddr, dstPort))
                        {
                            //fprintf(stderr, "MATCHED!\n");
                            if (!next_flow->IsComposite()) match = true;
                            if (!validWindow)
                            {
                                windowStart = theTime;
                                windowEnd = windowStart + windowSize;
                                validWindow = true;
                            }
                            next_flow->AddBytes(pktSize);
                        }
                        next_flow = next_flow->Next();
                    }
		    
                    if (auto_detect && !match)
                    {
		    	        Flow* theFlow = new Flow;
                        if (theFlow)
                        {
                            theFlow->SetType(proto);
                            theFlow->SetSrcAddr(srcAddr);
                            theFlow->SetSrcPort(srcPort);
                            theFlow->SetDstAddr(dstAddr);
                            theFlow->SetDstPort(dstPort);
                            flow_list.Append(theFlow);
			                fprintf(stderr, "rateplot: At time %f - Adding flow: ", theTime);;
        	                        theFlow->PrintDescription(stderr);
        	                        fprintf(stderr, "\n");
                            if (!validWindow)
                            {
                                windowStart = theTime;
                                windowEnd = windowStart + windowSize;
                                validWindow = true;
                            }
                            theFlow->AddBytes(pktSize);
                        }  
                        else
                        {
                            perror("rateplot: Error allocating memory for new Flow");
                            fclose(infile);
                            fclose(outfile);
                            exit(0);   
                        }                      
                    }
                }  // end if ((linkSrc == link_src) && (linkDst == link_dst))
            }
            else
            {
                fprintf(stderr, "rateplot: Error parsing ns-2 trace file at line %u.\n", line);
            }           
            
        }
        len = MAX_LINE;
    }
    
    fclose(infile);
    fclose(outfile);
    
    // Create final output file with gnuplot header if applicable
    if (use_gnuplot && flow_list.Head())
    {
    	if (!output_file)
        {
            fprintf(stderr, "rateplot: Must specify output file name for gnuplot file generation!\n");
            exit(-1);
        }
    	if(!(outfile = fopen(output_file, "w+")))
	    {
	        perror("rateplot: Error opening output file");
	        exit(-1);
	    }	
        if (make_post)
        {
            char postName[256];
            strcpy(postName, output_file);
            strcat(postName, ".ps");
            fprintf(outfile, "set term post color solid\n");
            fprintf(outfile, "set output '%s'\n", postName); 
        }
        else if (make_gif)
        {           
            char gifName[256];
            strcpy(gifName, output_file);
            strcat(gifName, ".gif");
            fprintf(outfile, "set term gif\n");
            fprintf(outfile, "set output '%s'\n", gifName);   
        }
        fprintf(outfile, "set title '%s'\n", output_file);
        fprintf(outfile, "set xlabel 'Time (sec)'\n");
        fprintf(outfile, "set ylabel 'Rate (kbps)'\n");
        fprintf(outfile, "set data style lines\n");
        fprintf(outfile, "set key bottom right\n");
        fprintf(outfile, "plot ");
        Flow* next_flow = flow_list.Head();
        unsigned int x = 2;
        while (next_flow)
        {
            if (x > 2) 
                fprintf(outfile, ",\\\n'%s' index 1 using 1:%u t '",
                              output_file, x);
            else
                fprintf(outfile, "\\\n'-' using 1:%u t '", x);
            next_flow->PrintDescription(outfile);
            fprintf(outfile, "'");
            next_flow = next_flow->Next();
            x++;
        }
        fprintf(outfile, "\n\n\n");
        fflush(outfile);
	
	    // Append data from temp file to output file
	    if(!(infile = fopen(temp_file, "r")))
	    {
	        perror("rateplot: Error opening our temp file");
	        exit(-1);
	    }
	    int result;
	    char buffer[64];
	    while ((result = fread(buffer, sizeof(char), 64, infile)))
	    {
	         fwrite(buffer, sizeof(char), result, outfile);
	    }
	    fclose(infile);
        unlink(temp_file);
	    fclose(outfile);
    }
    else if (use_gnuplot)
    {
        fprintf(stderr, "rateplot: No data to plot!\n");    
    }
    
    fprintf(stderr, "rateplot: Done.\n");
    exit(0);
}  // end main()



FastReader::FastReader()
    : savecount(0)
{
    
}

bool FastReader::Read(FILE* filePtr, char* buffer, unsigned int* len)
{
    unsigned int want = *len;   
    if (savecount)
    {
        unsigned int ncopy = MIN(want, savecount);
        memcpy(buffer, saveptr, ncopy);
        savecount -= ncopy;
        saveptr += ncopy;
        buffer += ncopy;
        want -= ncopy;
    }
    while (want)
    {
        unsigned int result = fread(savebuf, sizeof(char), 256, filePtr);
        if (result)
        {
            unsigned int ncopy= MIN(want, result);
            memcpy(buffer, savebuf, ncopy);
            savecount = result - ncopy;
            saveptr = savebuf + ncopy;
            buffer += ncopy;
            want -= ncopy;
        }
        else  // end-of-file
        {
            *len -= want;
            if (*len)
                return true;  // we read something
            else
                return false; // we read nothing
        }
    }
    return true;
}  // end FastReader::Read()

// An OK text readline() routine (reads what will fit into buffer incl. NULL termination)
// if *len is unchanged on return, it means the line is bigger than the buffer and 
// requires multiple reads

bool FastReader::Readline(FILE* filePtr, char* buffer, unsigned int* len)
{   
    unsigned int count = 0;
    unsigned int length = *len;
    char* ptr = buffer;
    unsigned int one = 1;
    while ((count < length) && Read(filePtr, ptr, &one))
    {
        if (('\n' == *ptr) || ('\r' == *ptr))
        {
            *ptr = '\0';
            *len = count;
            return true;
        }
        count++;
        ptr++;
    }
    // Either we've filled the buffer or hit end-of-file
    if (count < length) *len = 0; // Set *len = 0 on EOF
    return false;
}  // end FastReader::Readline()
