/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>  // for PATH_MAX
#include <string.h>

#define VERSION "2.0"

#define MIN(X,Y) ((X<Y)?X:Y)
#define MAX(X,Y) ((X>Y)?X:Y)

const int MAX_LINE = 256;

class FastReader
{
    public:
        FastReader();
        bool Read(FILE* filePtr, char* buffer, unsigned int* len);
        bool Readline(FILE* filePtr, char* buffer, unsigned int* len);
    
    private:
        char         savebuf[256];
        char*        saveptr;
        unsigned int savecount;
};  // end class FastReader

enum TransType {OFF, ON};
        
class Transition
{
    friend class TransitionList;
    
    public:
        Transition(TransType theType, double theTime);
        Transition* Next() {return next;}
        TransType Type() {return type;}
        double Time() {return time;}
    private:
        TransType   type;
        double      time;
            
        Transition* prev;
        Transition* next;  
};


class TransitionList
{
    public:
        TransitionList();
        Transition* Head() {return head;}
        void Append(Transition* t);
        void Remove(Transition* t);
    
    private:
        Transition* head;
        Transition* tail;  
};


TransitionList::TransitionList()
    : head(NULL), tail(NULL)
{
}

Transition::Transition(TransType theType, double theTime)
    : type(theType), time(theTime) 
      
{
}

class RepHist
{
    friend class RepHistList;
    
    public:
        RepHist(int theId);
        int Id() {return node_id;}
        void AddTransition(Transition* t)
            {list.Append(t);}
        Transition* FirstTransition() {return list.Head();}
        RepHist* Next() {return next;}
        
    private:    
        int             node_id;
        TransitionList  list;
        RepHist*        prev;
        RepHist*        next;
};

RepHist::RepHist(int theId)
    : node_id(theId)
{
}

class RepHistList
{
    public:
        RepHistList();
        RepHist* FindRepById(int nodeId);
        RepHist* Head() {return head;}
        void Append(RepHist* theRep);
        
    private:
        RepHist* head;
        RepHist* tail;  
};

  
RepHistList::RepHistList()
    : head(NULL), tail(NULL)
{
}

RepHist* RepHistList::FindRepById(int nodeId)
{
    RepHist* nextRep = head;
    while(nextRep)
    {
        if (nodeId == nextRep->Id())
            return nextRep;
        nextRep = nextRep->next;   
    }
    return NULL;
}  // end RepHistList::FindRepById()


void RepHistList::Append(RepHist* theRep)
{
    if ((theRep->prev = tail))
        theRep->prev->next = theRep;
    else
        head = theRep;
    theRep->next = NULL;
    tail = theRep;
}  // end RepHistList::Append()


void TransitionList::Append(Transition* t)
{
    if ((t->prev = tail))
        t->prev->next = t;
    else
        head = t;
    t->next = NULL;
    tail = t;
}  // end TransitionList::Append()


void TransitionList::Remove(Transition* t)
{
    if (t->prev)
        t->prev->next = t->next;
    else
        head = t->next;
    
    if (t->next)
        t->next->prev = t->prev;
    else
        tail = t->prev;
}  // end TransitionList::Remove()


void usage()
{
    fprintf(stderr, "Usage: repplot [gif] <mdpLogFile> <outFile>\n");
}  //end usage()


int main(int argc, char* argv[])
{
    bool makeGif = false;
    if (argc < 3)
    {
        usage();
        exit(-1);
    }
    
    fprintf(stderr, "Repplot Version %s\n", VERSION);
            
    // Parse command line
    fprintf(stderr, "\n");
    int i = 1;
    while(i < (argc - 2))
    {
        if (!strcmp("gif", argv[i]))
        {
            i++;
            makeGif = true;
        }
    }
    
    // Infile/Outfile are last 2 arguments
    char* inFileName = argv[i];
    char* outFileName = argv[i+1];
    
    // Open input file
    FILE* infile;
    if(!(infile = fopen(inFileName, "r")))
    {
        perror("repplot: Error opening input file");
        usage();
        exit(-1);
    }
    
    // Open output file
    FILE* outfile;
    if(!(outfile = fopen(outFileName, "w+")))
    {
        perror("repplot: Error opening output file");
        usage();
        exit(-1);
    }
    
    char buffer[MAX_LINE];
    unsigned int len = MAX_LINE;
    
    
    RepHistList repList;
    // This loop reads tracefile and builds a list of reps & their transitions
    float currentTime = 0.0;
    unsigned long maxId = 0, minId = 0xffffffff;
    FastReader reader;
    while (reader.Readline(infile, buffer, &len))
    {
        unsigned long serverId, theId;
        char theStatus[32];
        theStatus[31] = '\0';
        if (4 == sscanf(buffer, "RepresentativeStatus> time>%f server>%lu node>%lu %31s",
                &currentTime, &serverId, &theId, &theStatus))
        { 
            RepHist* theRep = repList.FindRepById(theId);
            if (!theRep)
            {
                if(!(theRep = new RepHist(theId)))
                {
                    perror("repplot: Error allocating memory for RepHist");
                    exit(-1);   
                } 
                repList.Append(theRep);
                printf("repplot: New rep(%d.%d) at time:%f\n",
                        theId>>8, theId&0xff, currentTime);
                maxId = (MAX(maxId, theId));
                minId = (MIN(minId, theId));
            }

            // now we have theRep
            TransType theType;
            if (!strcmp(theStatus, "elected."))
            {
                theType = ON;
            }
            else if (!strcmp(theStatus, "impeached."))
            {
                theType = OFF;
            }
            else
            {
                fprintf(stderr, "repplot: Invalid line!\n");
                exit(-1);
            }
            
            Transition* trans = new Transition(theType, (double)currentTime);
            if (!trans)
            {
                perror("repplot: Error allocating memory for Transition");
                exit(-1);   
            }
            theRep->AddTransition(trans);           
        }
        len = MAX_LINE;
    }  // end while(reader.Readline)
    fclose(infile);
    
    // Write gnuplot header ...
    float max = (maxId >> 8) + 1.0;
    float min = (minId >> 8) - 1.0;
    if (makeGif)
    {           
        char gifName[PATH_MAX];
        strcpy(gifName, outFileName);
        strcat(gifName, ".gif");
        fprintf(outfile, "set term gif\n");
        fprintf(outfile, "set output '%s'\n", gifName);   
    }
    fprintf(outfile, "set yrange [%f:%f]\n", min, max);
    fprintf(outfile, "set title 'Representative Node Migration'\n");
    fprintf(outfile, "set xlabel 'Time(sec)'\n");
    fprintf(outfile, "set ylabel 'Node Number'\n");
    fprintf(outfile, "set data style lines\n");
    fprintf(outfile, "set nokey \n");
    fprintf(outfile, "set ytics 1\n");
    fprintf(outfile, "set grid ytics\n");
    fprintf(outfile, "plot '-'");
    
    // This loop writes out data set for gnuplot
    RepHist* nextRep = repList.Head();
    while (nextRep)
    {
        // Make 2 blank lines for data set
        fprintf(outfile, "\n\n");
        Transition* nextTrans = nextRep->FirstTransition();
        while (nextTrans)
        {
            fprintf(outfile, "%f    %d.%d\n", 
                    nextTrans->Time(), nextRep->Id()>>8, nextRep->Id()&0xff);
            if (OFF == nextTrans->Type())
                fprintf(outfile, "\n");
            else if (!nextTrans->Next())
                fprintf(outfile, "%f    %d.%d\n", 
                    currentTime*1.1, nextRep->Id()>>8, nextRep->Id()&0xff);
            nextTrans = nextTrans->Next();
        }   // end while(nextTrans)
        nextRep = nextRep->Next();
    }   // end while(nextRep)
    
    fprintf(stderr, "repplot: Done.\n");
    fclose(outfile);
    exit(0);
}  // end main()



FastReader::FastReader()
    : savecount(0)
{
    
}

bool FastReader::Read(FILE* filePtr, char* buffer, unsigned int* len)
{
    unsigned int want = *len;   
    if (savecount)
    {
        unsigned int ncopy = MIN(want, savecount);
        memcpy(buffer, saveptr, ncopy);
        savecount -= ncopy;
        saveptr += ncopy;
        buffer += ncopy;
        want -= ncopy;
    }
    while (want)
    {
        unsigned int result = fread(savebuf, sizeof(char), 256, filePtr);
        if (result)
        {
            unsigned int ncopy= MIN(want, result);
            memcpy(buffer, savebuf, ncopy);
            savecount = result - ncopy;
            saveptr = savebuf + ncopy;
            buffer += ncopy;
            want -= ncopy;
        }
        else  // end-of-file
        {
            *len -= want;
            if (*len)
                return true;  // we read something
            else
                return false; // we read nothing
        }
    }
    return true;
}  // end FastReader::Read()

// An OK text readline() routine (reads what will fit into buffer incl. NULL termination)
// if *len is unchanged on return, it means the line is bigger than the buffer and 
// requires multiple reads

bool FastReader::Readline(FILE* filePtr, char* buffer, unsigned int* len)
{   
    unsigned int count = 0;
    unsigned int length = *len;
    char* ptr = buffer;
    unsigned int one = 1;
    while ((count < length) && Read(filePtr, ptr, &one))
    {
        if (('\n' == *ptr) || ('\r' == *ptr))
        {
            *ptr = '\0';
            *len = count;
            return true;
        }
        count++;
        ptr++;
    }
    // Either we've filled the buffer or hit end-of-file
    if (count < length) *len = 0; // Set *len = 0 on EOF
    return false;
}  // end FastReader::Readline()
