/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
#include <unistd.h>
#include <string.h>  // for strerror()
#include <errno.h>

#include "mdpFile.h"

#ifdef NS2
#define HAVE_DIRFD  // Assume we're building NS on Linux
#define HAVE_LOCKF
#endif

// Most don't have the dirfd() function
#ifndef HAVE_DIRFD
static inline int dirfd(DIR *dir) {return (dir->dd_fd);}
#endif // HAVE_DIRFD    

MdpFile::MdpFile()
    : fd(-1)
{    
}

MdpFile::~MdpFile()
{
    if (IsOpen()) Close();
}  // end MdpFile::~MdpFile()

// This should be called with a full path only!
bool MdpFile::Open(char *path, int theFlags)
{
    ASSERT(!IsOpen());
	
    // See if we need to build a directory
    if (theFlags & O_CREAT)
    {
        char *ptr = path;
        if (DIR_DELIMITER == *ptr) ptr += 1;
        // Descend path, making and cd to dirs as needed 
        while((ptr = strchr(ptr, DIR_DELIMITER)))
        {
            *ptr = '\0';
            if (chdir(path))
            {
                if (mkdir(path, 0755))
                {
                    DMSG(0, "mdp: Error opening file \"%s\": %s", path,
                          strerror(errno));
                    return false;     
                }
            }
            *ptr++ = DIR_DELIMITER;
        }
    }
    
    if((fd = open(path, theFlags, 0640)) >= 0)
    {
        offset = 0;
        return true;  // no error
    }
    else
    {    
        DMSG(0, "mdp: Error opening file: %s\n", strerror(errno));   
        return false;
    }
}  // end MdpFile::Open()


// Routines to try to get an exclusive lock on
// a file
bool MdpFile::Lock()
{
    fchmod(fd, 0640 | S_ISGID);
#ifdef HAVE_FLOCK
    if (flock(fd, LOCK_EX | LOCK_NB))
#else
#ifdef HAVE_LOCKF
    if (lockf(fd, F_LOCK, 0))
#endif // HAVE_LOCKF
#endif // HAVE_FLOCK
        return false;
    else
        return true;
}  // end MdpFile::Lock()

void MdpFile::Unlock()
{
#ifdef HAVE_FLOCK
    flock(fd, LOCK_UN);
#else
#ifdef HAVE_LOCKF
    lockf(fd, F_ULOCK, 0);
#endif // HAVE_LOCKF
#endif // HAVE_FLOCK
    fchmod(fd, 0640);
}  // end MdpFile::UnLock()

bool MdpFileIsLocked(char *path)
{
    // First make sure file exists
    if (!MdpFileExists(path)) return false;  
    
    MdpFile testFile;
    if(!testFile.Open(path, O_WRONLY | O_CREAT))
    {
        return true;
    }
    else if (testFile.Lock())
    {
        // We were able to lock the file successfully
        testFile.Unlock();
        testFile.Close();
        return false;
    }
    else
    {
        testFile.Close();
        return true;
    }
}  // end MdpFileIsLocked()

bool MdpFile::Rename(char *old_name, char *new_name)
{
    // Make sure the new file name isn't an existing "busy" file
    // (This also builds sub-directories as needed)
    if (MdpFileIsLocked(new_name))
        return false;
    else if (rename(old_name, new_name))
        return false;
    else
        return true;
}  // end MdpFile::Rename()

bool MdpFile::Unlink(char *path)
{
    // Don't unlink a file that is open (locked)
    if (MdpFileIsLocked(path))
    {
        return false;
    }
    else if (unlink(path))
    {
        //DMSG(0, "MdpFile::Unlink() error: %s\n", strerror(errno));
        return false;
    }
    else
    {
        return true;
    }
}  // end

void MdpFile::Close()
{
    ASSERT(IsOpen());   
    close(fd);
    fd = -1;
}  // end MdpFile::Close()


bool MdpFile::Seek(unsigned long theOffset)
{
    ASSERT(IsOpen());
    off_t result = lseek(fd, theOffset, SEEK_SET);
    if (result < 0)
    {
        DMSG(0, "mdp: Error positioning file pointer: %s", strerror(errno));
        return false;
    }
    else
    {
        offset = result;
        return true; // no error
    }
}  // end MdpFile::Seek()

unsigned long MdpFile::Size()
{
    ASSERT(IsOpen());
    struct stat info;
    int result = fstat(fd, &info);
    if (result)
    {
        DMSG(0, "Error getting file size: %s\n", strerror(errno));
        return 0;   
    }
    else
    {
        return info.st_size;
    }
}  // end MdpFile::Size()

/******************************************
 * The MdpDirectory and MdpDirectoryIterator classes
 * are used to walk directory trees for file transmission
 */

MdpDirectory::MdpDirectory(const char *thePath, MdpDirectory *theParent)
    : dptr(NULL), parent(theParent)
{
    strncpy(path, thePath, PATH_MAX);
    int len  = MIN(PATH_MAX, strlen(path));
    if ((len < PATH_MAX) && (DIR_DELIMITER != path[len-1]))
    {
        path[len++] = DIR_DELIMITER;
        if (len < PATH_MAX) path[len] = '\0';
    }
}

bool MdpDirectory::Open()
{
    if (dptr) Close();
    char full_name[PATH_MAX];
    GetFullName(full_name);
    // Get rid of trailing DIR_DELIMITER
    int len = MIN(PATH_MAX, strlen(full_name));
    if (DIR_DELIMITER == full_name[len-1]) full_name[len-1] = '\0';
    if((dptr = opendir(full_name)))
        return true;
    else    
        return false;
} // end MdpDirectory::Open()

void MdpDirectory::Close()
{
    closedir(dptr);
    dptr = NULL;
}  // end MdpDirectory::Close()


void MdpDirectory::GetFullName(char *ptr)
{
    ptr[0] = '\0';
    RecursiveCatName(ptr);
}  // end GetFullName()

void MdpDirectory::RecursiveCatName(char *ptr)
{
    if (parent) parent->RecursiveCatName(ptr);
    int len = MIN(PATH_MAX, strlen(ptr));
    strncat(ptr, path, PATH_MAX-len);
}  // end GetFullName()



MdpDirectoryIterator::MdpDirectoryIterator()
    : current(NULL)
{

}

MdpDirectoryIterator::~MdpDirectoryIterator()
{
    Destroy();
}


bool MdpDirectoryIterator::Init(const char *thePath)
{
    if (current) Destroy();
    if (thePath && access(thePath, X_OK))
    {
        DMSG(0, "MdpDirectoryIterator: can't access directory: %s\n", thePath);
        return false;
    }
    current = new MdpDirectory(thePath);
    if (current && current->Open())
    {
        path_len = MIN(PATH_MAX, strlen(current->Path()));
        return true;
    }
    else
    {
        DMSG(0, "MdpDirectoryIterator: can't open directory: %s\n", thePath);
        if (current) delete current;
        current = NULL;
        return false;
    }
}  // end MdpDirectoryIterator::Init()

void MdpDirectoryIterator::Destroy()
{
    MdpDirectory *ptr;
    while ((ptr = current))
    {
        current = ptr->parent;
        ptr->Close();
        delete ptr;
    }
}  // end MdpDirectoryIterator::Destroy()

bool MdpDirectoryIterator::GetNextFile(char *file_name)
{   
    if (!current) return false;
    struct dirent *dp;
    while((dp = readdir(current->dptr)))
    {
        // Make sure it's not "." or ".."
        if (dp->d_name[0] == '.')
        {
            if ((1 == strlen(dp->d_name)) ||
                ((dp->d_name[1] == '.' ) && (2 == strlen(dp->d_name))))
            {
                continue;  // skip "." and ".." directory names
            }
        }
        current->GetFullName(file_name);
        strcat(file_name, dp->d_name);
        int type = MdpFileGetType(file_name);        
        if (MDP_FILE_NORMAL == type)
        {
            int name_len = MIN(PATH_MAX, strlen(file_name)) - path_len;
            memmove(file_name, file_name+path_len, name_len);
            if (name_len < PATH_MAX) file_name[name_len] = '\0';
            return true;
        } 
        else if (MDP_FILE_DIRECTORY == type)
        {
            MdpDirectory *dir = new MdpDirectory(dp->d_name, current);
            if (dir && dir->Open())
            {
                // Push sub-directory onto stack and search it
                current = dir;
                return GetNextFile(file_name);
            }
            else
            {
                // Couldn't open this one, try next one
                if (dir) delete dir;
            }
        }
        else
        {
            // MDP_FILE_INVALID, try next
        }
    }  // end while(readdir())
   
    // Popup a level and recursively continue or finish if done
    if (current->parent)
    {
        char path[PATH_MAX];
        current->parent->GetFullName(path);
        current->Close();
        MdpDirectory *dir = current;
        current = current->parent;
        delete dir;
        return GetNextFile(file_name);
    }
    else
    {
        current->Close();
        delete current;
        current = NULL;
        return false;  // no more files remain
    }      
}  // end MdpDirectoryIterator::NextFileItem()


// Is the named item a valid directory or file (or neither)??
MdpFileType MdpFileGetType(const char *path)
{
    struct stat file_info;  
    if (stat(path, &file_info)) 
        return MDP_FILE_INVALID;  // stat() error
    else if ((S_ISDIR(file_info.st_mode)))
        return MDP_FILE_DIRECTORY;
    else 
        return MDP_FILE_NORMAL;
}  // end MdpFileGetType

unsigned long MdpFileGetSize(const char* path)
{
    struct stat info;
    int result = stat(path, &info);
    if (result)
    {
        //DMSG(0, "Error getting file size: %s\n", strerror(errno));
        return 0;   
    }
    else
    {
        return info.st_size;
    }
}  // end MdpFileGetSize()

time_t MdpFileGetUpdateTime(const char *path)
{
    struct stat file_info;  
    if (stat(path, &file_info)) 
        return (time_t) 0;  // stat() error
    else 
        return file_info.st_ctime;  
}  // end GetMdpFileUpdateTime()


