/* This file is part of statist
**
** It is distributed under the GNU General Public License.
** See the file COPYING for details.
** 
** (c) 1997 Dirk Melcher
**  Doerper Damm 4
**  49134 Wallenhorst
**  GERMANY
**  Tel. 05407/7636
**  email: Dirk.Melcher@usf.Uni-Osnabrueck.DE
**
**  some changes by Bernhard Reiter  http://www.usf.Uni-Osnabrueck.DE/~breiter
**  $Id: statist.c,v 1.6 2001/11/15 20:28:55 breiter Exp breiter $
***************************************************************/

/* statist.c */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <signal.h>


#include "statist.h"
#include "menue.h"
#include "data.h"
#include "funcs.h"
#include "plot.h"
#include "gettext.h"

typedef struct {
  char *optstr;
  BOOLEAN *optvar;
} OPTREC;


BOOLEAN get_options(char **args, int options[], int nopt, OPTREC optrec[]);
void help_msg();

#ifndef MSDOS   
void handle_pipe_interrupt(int sig);
#endif
void handle_fpe_interrupt(int sig);
/* void handle_Ctrl_C_interrupt(int sig); */
void handle_term_interrupt(int sig);

PREAL   xx[MCOL], *yy, tempcol;
int     nn[MCOL], acol[MCOL], *ny;
BOOLEAN x_read[MCOL], empty, gnupl_open=FALSE;
int     ncol=0, status;
char    line[MLINE];
char    *alias[MCOL];
FILE    *logfile;

BOOLEAN silent=FALSE, noplot=FALSE, help=FALSE, log_set=FALSE,
        nofile=FALSE, nobell=FALSE, thist=FALSE, bernhard=FALSE,
        delrow=FALSE;

FILE    *pipef;

/* ================================================================== */


int main(int argc, char *argv[]) {
   int   i;
   char  *sourcename;
   BOOLEAN source_set=FALSE;
   int options[MOPT], nopt=0;

   OPTREC optrec[MOPT] = {
      {"-h", &help}, {"-help", &help}, {"-?", &help}, {"-silent", &silent},
      {"-noplot", &noplot}, {"-log", &log_set}, {"-nofile", &nofile},
      {"-nobell", &nobell}, {"-thist", &thist} , {"-delrow", &delrow},
      {"--bernhard",&bernhard}
   };

   sourcename = (char*) mycalloc(255, sizeof(char));
   
#ifndef MSDOS   
   signal(SIGPIPE, handle_pipe_interrupt);
#endif
   signal(SIGFPE, handle_fpe_interrupt);
   /*   signal(SIGINT, handle_Ctrl_C_interrupt);  */
   signal(SIGINT, handle_term_interrupt);
   signal(SIGTERM, handle_term_interrupt);


   for (i=1; i<argc; i++) {
     if (argv[i][0]=='-') {
       options[nopt] = i;
       nopt ++;
       if (nopt > MOPT) {
	 out_err(FAT, ERR_FILE, ERR_LINE, 
	 	_("Too many options",
		"Zu viele Optionen") );
       }
     }
     else {
       if (!source_set) {
	 strncpy(sourcename, argv[i], 254);
	 source_set = TRUE;
       }
       else {
	 out_err(FAT, ERR_FILE, ERR_LINE, 
	 	_(" More than one data file given!",
		" Mehrere Datenfiles angegeben!") );
       }
     }
   }

   if (!get_options(argv, options, nopt, optrec)) {
     exit(1);
   }
   out_d("\n%s\n\n", VERSION_INFO);
   if (help) {
     help_msg();
     exit(0);
   }
  
   if ( (!source_set) && (!nofile) ) {
     out_d(_("Name of data file: ", "Name der Datendatei: ") );
     GETNLINE
     if (empty) {
       out_d(_("No data file given. Load file using menu option\n\
Data management|Read another file\n",
	 "Es wurde keine Datei angegeben. Laden einer Datendatei\n\
im Menue Datenverwaltung|Neue Datei einlesen\n") );
       nofile = TRUE;
       /*       return 0;  */
     }
     else {
       strncpy(sourcename, line, 254);
     }
   }
   
   if (log_set) {
     FOPEN("statist.log", "wt", logfile);
     out_r(_("Source file = %s\n\n", "Quelldatei = %s\n\n"), sourcename);
   }

   for (i=0; i<MCOL; i++) {
     nn[i] = 0;
     x_read[i] = FALSE;
   }

   for (i=0; i<MCOL; i++) {  
 /*  alias[i] = (char*)mycalloc(3, sizeof(char));
     sprintf(alias[i], "%1i", (i+1));
  */
     alias[i] = (char*)mycalloc(2, sizeof(char));
     sprintf(alias[i], "%c", (char)(i+CHAR_OFFSET));
   }

   if (!nofile) {
     readsourcefile(sourcename);
     mywait();
   }
   myfree(sourcename);
   out_d("\n");
   main_menue();
   finish();
   return 0;
}

/* =================================================================== */

void finish() {
   erasetempfiles();
   if (log_set) {
     FCLOSE(logfile);
   }
#ifndef MSDOS
   if (gnupl_open) {
     fprintf(pipef, "quit\n");
     pclose(pipef);
   }
   if (myexist(GPL_DAT)) {
     remove(GPL_DAT);
   }
#endif
}

  

void help_msg() {
out_d(_("Usage: statist <Options> data file\n",
	  "Benutzung: statist <Optionen> Datenfile\n") );
out_d(_("Options:\n\
-h, -help, -? : print this help message and exit\n\
-silent       : don't print menu etc. (for batch/script usage)\n\
-log          : write results to log file `statist.log'\n\
-nofile       : don't read a data file when starting the program\n\
-nobell       : no beep at errors and warnings\n\
-thist        : histogram as text graphic instead of gnuplot-graphic\n\
-noplot       : no gnuplot-graphic\n\
-delrow	  : skip lines with missing values\n\
--bernhard    : special output changes from Bernhard, i.e.:\n\
		- table output at miscelaneous/standard deviation\n\
		- if -noplot defined no text histogram at\n\
			miscelaneous/standard deviation\n\
\n",
"Optionen:\n\
-h, -help, -? : dieser Hilfetext\n\
-silent       : Keine Menues u.a. anzeigen (fuer Batchbetrieb)\n\
-log          : Ergebnisse werden in Datei `statist.log' protokolliert\n\
-nofile       : Kein Datenfile bei Programmstart einlesen\n\
-nobell       : Kein Piepton bei Fehler und Warnungen\n\
-thist        : Histogramm als Textgrafik statt gnuplot-Graphik\n\
-noplot       : Keine gnuplot-Graphik\n\
-delrow	  : Zeilen mit fehlenden Werten berlesen\n\
--bernhard    : Bernhards Outputaenderungen\n\
	        also:   - Tabellenoutput bei Verschiedenes/Standardabweichung\n\
			- falls -noplot gegeben kein Texthistogramm bei\n\
				Verschiedenes/Standardabweichung\n\
\n") );
out_d(_("Missing values must be indecated by '%c'\n",
	  "Fehlende Daten: mssen mit \"%c\" gekennzeichnet sein\n"), NODATA);
}

/* =================================================================== */

BOOLEAN get_options(char **args, int options[], int nopt, OPTREC optrec[]) {
  int i, k;
  BOOLEAN found;

  for (i=0; i<nopt; i++) {
    found = FALSE;
    for (k=0; k<MOPT; k++) {
      if (strcmp(args[options[i]], optrec[k].optstr) == 0) {
	*(optrec[k].optvar) = TRUE;
	found = TRUE;
	break;
      }
    }
    if (!found) {
      out_err(ERR, ERR_FILE, ERR_LINE, 
		_("Illegal option: '%s'. Try -h for help.",
		"Ungueltige Option: '%s'. Hilfe anzeigen mit -h."), 
		args[options[i]]);
      return FALSE;
    }
  }
  return TRUE;
}

/* =================================================================== */

void mywait() {
  if (!silent) {
    out_d(_("  --- Please, continue with <RETURN> ... ---",
    		"   --- Bitte mit ENTER fortfahren ... ---") );
    GETRLINE
  }
}



BOOLEAN myexist(char *name) {
   if (fopen(name,"r") == NULL) {
     return FALSE;
   }
   else {
     return TRUE;
   }
}



void out_r(char *fmt, ...) {
  va_list argptr;

  va_start(argptr, fmt);
  vprintf(fmt, argptr);
  if (log_set) {
    vfprintf(logfile, fmt, argptr);
  }
  va_end(argptr);
/*  fflush(logfile);  */
}


void out_d(char *fmt, ...) {
  va_list argptr;

  if (!silent) {
    va_start(argptr, fmt);
    vprintf(fmt, argptr);
    va_end(argptr);
  }
}



void out_err(int errno, char *modulname, int lno, char *fmt, ...) {
  va_list argptr;
  int error;
  char where[128];

  va_start(argptr, fmt);
  switch (errno) {
    case WAR: strcpy(line, 
    		_("> statist-warning ",
		  "> statist-Warnung ") );
      break;
    case ERR: strcpy(line, 
    		_("> statist-error ",
		  "> statist-Fehler ") );
      break;
    case FAT: strcpy(line, 
    		_("\n> statist-fatal error ",
		  "\n> statist-fataler Fehler ") );
      break;
    case MAT: strcpy(line, 
    		_("> statist-numerical error ",
		  "> statist-Berechnungsfehler ") );
      break;
    case MWA: strcpy(line, 
    		_("> statist-warning ",
		  "> statist-Warnung ") );
      break;
  }

  error = errno;
  if (lno != 0) {
    sprintf(where, 
    		_("(Modul %15s, line %i):\n  ",
		"(Modul %15s Zeile %i):\n  "), modulname, lno);
    strncat(line, where, 254-strlen(line));
  }
  else {
    strncat(line, ":\n  ", 254-strlen(line));
  }
  strncat(line, fmt, 254-strlen(line));
  if (nobell) {
    strncat(line, "\n\n", 254-strlen(line));
  }
  else {
    strncat(line, "\a\n\n", 254-strlen(line));
  }
  vfprintf(stderr, line, argptr);
  if ( ((errno==MAT) || (errno==MWA)) && (log_set) ) {
    vfprintf(logfile, line, argptr);
  }
  va_end(argptr);

  if (error == FAT) {
    finish();
    exit(1);
  }
}



/* =================================================================== */

void print_histo(REAL x[], int y[], int n) {
  int maxclass;
  int   k, cols, i;
  char show[80];

  maxclass = get_maxint(y, n);
  for (i=0; i<n; i++) {
    cols = y[i]*60/maxclass;
    strcpy(show,"");
    for (k=0; k<cols; k++) {
      strcat(show, "*");
    }
    out_r("%9f %s %i\n", x[i], show, y[i]);
    if (((i+1) % SCREENSIZE) == 0) {
      mywait();
    }
  }
}


/* =================================================================== */

#ifndef MSDOS
void handle_pipe_interrupt(int sig) {
  gnupl_open = FALSE;
  out_err(WAR, ERR_FILE, ERR_LINE,
	_("Broken pipe to gnuplot. Re-connecting.\n",
	  "Pipe zu gnuplot gestoert. Baue neue Pipe auf.\n") );
  pclose(pipef);
}
#endif

void handle_fpe_interrupt(int sig) {
  out_err(FAT, ERR_FILE, ERR_LINE,
	_("Signal from operating system: floating point exception\n\
		Division by zero? Terminating!\n",
	  "Signal vom Betriebsystem: Fliesskommafehler gemeldet\n\
                      Division durch 0? Ich sage Adieu!\n") );
}
/*
void handle_Ctrl_C_interrupt(int sig) {
  out_d("\n  statist: Erhielt Signal SIGINT: Ctrl-C gedrueckt? Adieu!\n");
  finish();
  exit(1);
}
*/
void handle_term_interrupt(int sig) {
  out_d(
  	_("\n  statist: Received signal SIGTERM (Ctrl-C?). Terminating!\n",
	"\n  statist: Erhielt Signal SIGTERM: Ctrl-C gedrueckt? Adieu!\n") );
  finish();
  exit(1);
}  
