/*
 * hamming.c
 *
 * Code to encode and decode blocks of hamming encoded bytes.
 *
 * Copyright (c) 2003 Todd MacDermid <tmacd@synacklabs.net> 
 *
 * A word about what we're doing, here. This was first written for
 * stegtunnel, which needed a way of recovering if a packet needed
 * to be retransmitted. Since random IPIDs are not repeated for TCP
 * retransmits, the remote side would have an invalid packet being
 * passed to it, with no idea about whether it was the original
 * packet or a scrambled retransmit. By breaking the IPID field into
 * 16 parallel hamming encoded streams, we can ensure that if a single
 * retransmit occurs along the way, that the error is correctable.
 *
 * This is kind of an unusual implementation of a hamming code, due
 * to the fact that it is doing parallel encodes/decodes. I tried
 * to generalize it to a generic "width", rather than assuming a
 * 2 byte wide input stream, because I didn't know if I'd use
 * it again in the future over another header field. In stegtunnel,
 * as of this writing, though, width will always be 2. 
 */

#include <stdint.h>
#include <stdlib.h>

int data_bytes[] = {
  2, 4, 5, 6, 8, 9, 10, 11, 12, 13, 14, 16, 17, 18, 19, 20, 21, 22,
  23, 24, 25, 26, 27, 28, 29, 30, 32, 33, 34, 35, 36, 37, 38, 39,
  40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55,
  56, 57, 58, 59, 60, 61, 62, 64, 65, 66, 67, 68, 69, 70, 71, 72,
  73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88,
  89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103,
  104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116,
  117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 128, 129, 130,
  131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143,
  144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156,
  157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169,
  170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181, 182,
  183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195,
  196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208,
  209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221,
  222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234,
  235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 247,
  248, 249, 250, 251, 252, 253, 254, 255
};

int parity_bytes[] = { 0, 1, 3, 7, 15, 31, 63, 127 };

/* 
 * hamming_encode takes in a stream of bytes, a width that the output
 * should be broken up into, a pointer to memory to dump the encoded bytes
 * out to, and "k." k is the number of parity bits used. The higher
 * k is, the longer the input can be, and hamming code parity bits
 * will take up a smaller proportion of the data stream at higher
 * values of k, but remember, hamming codes can only correct one
 * error per block, so a value of k too high for the amount of noise
 * in the channel will be inefficient.
 *
 * Returns 0 on success, -1 on failure (due to an out-of-range parameter,
 * or other programmer error).
 */

int
hamming_encode(int width, int k, uint8_t *in_bytes, uint8_t *encoded_arr)
{
  int i;
  int j;
  int width_loc;
  int num_input;
  int num_output;
  int out_loc;
  int out_num;
  int data_num;

  if(width < 1)
    return(-1);

  if((k < 2) || (k > 8))
    return(-1);

  if((in_bytes == NULL) || (encoded_arr == NULL))
    return(-1);

  num_output = (1 << k) - 1;
  num_input = (1 << k) - k - 1;

  /* zero out the output array */

  for(i = 0; i < num_output * width; i++) {
    encoded_arr[i] = 0;
  }

  /* Place input into proper out locations */

  for(i = 0; i < num_input; i++) {
    out_loc = data_bytes[i];

    for(width_loc = 0; width_loc < width; width_loc++) {
      encoded_arr[(out_loc*width) + width_loc] = 
	in_bytes[(i*width) + width_loc];
    }
  }

  /* Calculate parity bytes */

  for( i = 0; i < k; i++) {
    out_loc = parity_bytes[i];
    out_num = out_loc + 1;
    for( j = 0; j < num_output; j++ ) {
      data_num = j+1;
      if((out_num & data_num) && (out_num != data_num)) {
        for(width_loc = 0; width_loc < width; width_loc++) {
          encoded_arr[out_loc * width + width_loc] ^= 
	    encoded_arr[j * width + width_loc];	  
        }
      }
    }
  }
 
  /*  printf("hamming encoded: ");
  for(i = 0; i < num_output * width; i++) {
    printf("%02x", encoded_arr[i]);
  }
  printf("\n"); */

  return(0);
}

/* 
 * hamming_decode takes a stream of bytes produced by hamming_encode,
 * and will reproduce the original data stream. The width and k values
 * provided must match those given to the original hamming_encode.
 * encoded_arr should point to the encoded bytes, and decoded_arr
 * must point to sufficient memory to contain the output.
 *
 * Will return -1 on programmer error, -2 on unrecoverable errors in the 
 * stream. Will produce no output on those errors.
 * Will return 0 on no errors in the stream, or 1 on a recoverable
 * error in the stream. In those cases, decoded_arr will hold the proper 
 * output bytes.
 */

int
hamming_decode(int width, int k, uint8_t *encoded_arr, uint8_t *decoded_arr)
{

  int have_err = 0;
  uint8_t err_loc;
  uint8_t err_mask;
  int encoded_sz;
  int decoded_sz;
  uint8_t tmp_err;
  int parity_num;
  int data_num;
  int width_loc;
  int i;
  int j;
  
  if(width < 1)
    return(-1);

  if((k < 2) || (k > 8))
    return(-1);
 
  decoded_sz = (1 << k) - k - 1;
  encoded_sz = (1 << k) - 1;

  /* zero out the output array */

  for(i = 0; i < decoded_sz * width; i++) {
    decoded_arr[i] = 0;
  }

  for(width_loc = 0; width_loc < width; width_loc ++) {
    err_mask = 0;
    err_loc = 0;

    for(i = 0; i < k; i++) {
      parity_num = parity_bytes[i] + 1;
      tmp_err = 0;
      for(j = 0; j < encoded_sz; j++) {
	data_num = j+1;
	if(parity_num & data_num) {
	  tmp_err ^= encoded_arr[width * j + width_loc];
	}
      }
      if(tmp_err != 0) {
	have_err = 1;
	err_loc |=  1 << i;
	if(err_mask == 0) {
	  err_mask = tmp_err;
	} else {
	  if(tmp_err == err_mask) {	    
	  } else {
	    return(-2);
	  }
	}
      }
    }

    if(err_loc) {
      encoded_arr[(err_loc - 1)*width + width_loc] ^= err_mask;
    }
    for(i = 0; i < decoded_sz; i++) {
      decoded_arr[width*i + width_loc] = 
	encoded_arr[data_bytes[i]*width + width_loc];
    }
  }

  /* printf("hamming encoded: ");
  for(i = 0; i < encoded_sz * width; i++) {
    printf("%02x", encoded_arr[i]);
  }
  printf("\n"); */

  if(have_err)
    return(-1);
  else
    return(0);
}

