/*
 * session.c
 *
 * Code to handle the finding, adding, and deleting of session state within
 * stegserver and stegclient
 *
 * Copyright (c) 2003 Todd MacDermid <tmacd@synacklabs.net> 
 *
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <netinet/in.h>

#include <dnet.h>
#include <fcntl.h>
#include <pcap.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <packetp.h>

#include "crypto.h"
#include "sha1.h"
#include "stegtunnel.h"

/*
 * stegt_session_add constructs the session object for a new
 * connection, and fills it in. It also adds the session object to the
 * st_ctx's hashtable. 
 *
 * If a server proxy IP is being used, It will create
 * two keys pointing to the entry. One key is simply the
 * proxy port that the proxy IP address is using to simulate local
 * connections, and the other is a hash of the remote IP address, and
 * the ports used by the remote IP address. This is so we can access
 * the same session information either inbound or outbound using a proxy.
 *
 * If not proxy is being used, or if it's not a server, it will
 * only create one key, the hash of the remote IP and ports
 *
 * Returns a pointer to the new session on success, NULL on failure.
 */


struct stegt_session *
stegt_session_add(struct packetp_ctx *pp_ctx, 
		  struct stegt_ctx *st_ctx, uint8_t *packet) 
{
  SHA1Context sha_ctx;
  struct ip_hdr *ip_header;
  struct tcp_hdr *tcp_header;
  struct stegt_session *new_session;
  uint8_t *hash;
  uint16_t *port;
  int i;
  
  if ((hash = (uint8_t *)calloc(1, SHA1HashSize)) == NULL) {
    fprintf(stderr, "server_session_add: calloc() failed\n");
    return(NULL);
  }

  ip_header = (struct ip_hdr *)packet;
  if (ip_header->ip_p != IP_PROTO_TCP) {
    free(hash);
    fprintf(stderr, "client_session_add: not a TCP packet\n");
    return(NULL);
  }
  tcp_header = (struct tcp_hdr *) (packet + (ip_header->ip_hl * 4));

  if((new_session = (struct stegt_session *)
      calloc(1, sizeof(struct stegt_session))) == NULL) {
    fprintf(stderr, "stegt_session_add: calloc() failed\n");
    free(hash);
    return(NULL);
  }

  SHA1Reset(&sha_ctx);
  if(st_ctx->server) {
    memcpy(&(new_session->remote_ip.addr_ip), 
	   &(ip_header->ip_src), IP_ADDR_LEN);
    SHA1Input(&sha_ctx, (uint8_t *)&(ip_header->ip_src), IP_ADDR_LEN);
    SHA1Input(&sha_ctx, (uint8_t *)&(tcp_header->th_sport), 2);
    SHA1Input(&sha_ctx, (uint8_t *)&(tcp_header->th_dport), 2);
  } else {
    memcpy(&(new_session->remote_ip.addr_ip), 
	   &(ip_header->ip_dst), IP_ADDR_LEN);
    SHA1Input(&sha_ctx, (uint8_t *)&(ip_header->ip_dst), IP_ADDR_LEN);
    SHA1Input(&sha_ctx, (uint8_t *)&(tcp_header->th_dport), 2);
    SHA1Input(&sha_ctx, (uint8_t *)&(tcp_header->th_sport), 2);
  }

  SHA1Result(&sha_ctx, hash);
  new_session->remote_ip.addr_bits = 32;
  new_session->remote_ip.addr_type = ADDR_TYPE_IP;

  if(hadd(st_ctx->sessions, hash, SHA1HashSize, new_session) == FALSE) {
    fprintf(stderr, "stegt_session_add: hadd() failed\n");
    free(hash);
    free(new_session);
    return(NULL);
  }
  
  if(st_ctx->server) {
    if (pp_ctx->wedge_type == PP_FAKEIP) {
      if ((port = (uint16_t *)calloc(1, sizeof(uint16_t))) == NULL) {
	fprintf(stderr, "stegt_session_add: calloc() failed\n");
	free(hash);
	free(new_session);
	return(NULL);
      }
      for(i = 1; i < TCP_PORT_MAX; i++) {
	if(!st_ctx->used_ports[i]) {
	  new_session->proxy_port = i;
	  *port = new_session->proxy_port;
	  if(hadd(st_ctx->sessions, 
		  (uint8_t *)port, sizeof(uint16_t), new_session) == FALSE) {
	    fprintf(stderr, "stegt_session_add: port hadd() failed\n");
	    goto fail;
	  }
	  st_ctx->used_ports[i] = 1;
	  break;
	}
      }
      if(i >= TCP_PORT_MAX) {
	fprintf(stderr, "server_session_add: proxy ports exhausted\n");
	goto fail;
      }
    }
    new_session->remote_port = tcp_header->th_sport;
    new_session->local_port = tcp_header->th_dport;
    new_session->msg_offset = 0;
  } else {
    new_session->keyed = 0;
  }

  new_session->remote_port = tcp_header->th_sport;
  new_session->local_port = tcp_header->th_dport;
  new_session->cur_file = NULL;
  
  return(new_session);
  
  fail:
  free(new_session);
  free(hash);
  if(pp_ctx->wedge_type == PP_FAKEIP) 
    free(port);

  return(NULL);
}

/*
 * stegt_session_del might have to delete two entries out of the session hash
 * table, one corresponding to the local proxy port, and one corresponding to 
 * the IP addr/remote port/local real port. It returns nothing, because
 * there's really no graceful way to half-exit from this.
 *
 * Note that stegt_session_find() has been presumed to have been called before
 * this, so the internal hash table positioning is already correct.
 */

void 
stegt_session_del(struct stegt_ctx *st_ctx, struct packetp_ctx *pp_ctx) 
{
  SHA1Context sha_ctx;
  uint8_t hash[SHA1HashSize];
  struct stegt_session *session;
  hitem  *h; 

  (h = st_ctx->sessions->ipos);
  session = (struct stegt_session *)h->stuff;
  if(st_ctx->server) {
  
    SHA1Reset(&sha_ctx);
    SHA1Input(&sha_ctx, (uint8_t *)&(session->remote_ip.addr_ip), IP_ADDR_LEN);
    SHA1Input(&sha_ctx, (uint8_t *)&(session->remote_port), 2);
    SHA1Input(&sha_ctx, (uint8_t *)&(session->local_port), 2);
    SHA1Result(&sha_ctx, hash);
    
    if(hfind(st_ctx->sessions, hash, SHA1HashSize) == TRUE) {
      h = st_ctx->sessions->ipos;
      free(h->key);
    }
    
    if(pp_ctx->wedge_type == PP_FAKEIP) {
      hdel(st_ctx->sessions);
      if(hfind(st_ctx->sessions, &(session->proxy_port), 
	       sizeof(uint16_t)) == TRUE) {
	h = st_ctx->sessions->ipos;
	st_ctx->used_ports[session->proxy_port] = 0;
	free(h->key);
      }
    }
  } else {
    free(h->key);
  }
  if(session->cur_file != NULL) {
    session->cur_file->active_session = NULL;
  }
  free(h->stuff);
  hdel(st_ctx->sessions);
}

/* 
 * stegt_session_find attempts to locate a session corresponding to the
 * provided packet. Returns a pointer to the session on success, NULL
 * on failure.
 */

struct stegt_session *
stegt_session_find(uint8_t *packet, struct stegt_ctx *st_ctx, int direction,
		    struct packetp_ctx *pp_ctx) 
{
  SHA1Context sha_ctx;
  struct ip_hdr *ip_header;
  struct tcp_hdr *tcp_header;
  uint8_t hash[SHA1HashSize];
  uint16_t proxy_port;
  int i;

  ip_header = (struct ip_hdr *)packet;
  if (ip_header->ip_p != IP_PROTO_TCP)
    return(NULL);
  tcp_header = (struct tcp_hdr *) (packet + (ip_header->ip_hl * 4));  

  SHA1Reset(&sha_ctx);
  
  if(st_ctx->server) {
    if(direction == OUT) {
      if(pp_ctx->wedge_type == PP_FAKEIP) {
	proxy_port = ntohs(tcp_header->th_dport);
	if(hfind(st_ctx->sessions, (uint8_t *)&proxy_port, 
		 sizeof(uint16_t)) == FALSE) {
	  return(NULL);
	} else {
	  return(hstuff(st_ctx->sessions));
	}
      } else {
	SHA1Input(&sha_ctx, (uint8_t *)&(ip_header->ip_dst), IP_ADDR_LEN);
	SHA1Input(&sha_ctx, (uint8_t *)&(tcp_header->th_dport), 2);
	SHA1Input(&sha_ctx, (uint8_t *)&(tcp_header->th_sport), 2);
      }
    } else {
      SHA1Input(&sha_ctx, (uint8_t *)&(ip_header->ip_src), IP_ADDR_LEN);
      SHA1Input(&sha_ctx, (uint8_t *)&(tcp_header->th_sport), 2);
      SHA1Input(&sha_ctx, (uint8_t *)&(tcp_header->th_dport), 2);
    }
  } else {
    if(direction == IN) {
      SHA1Input(&sha_ctx, (uint8_t *)&(ip_header->ip_src), IP_ADDR_LEN);
      SHA1Input(&sha_ctx, (uint8_t *)&(tcp_header->th_sport), 2);
      SHA1Input(&sha_ctx, (uint8_t *)&(tcp_header->th_dport), 2);
    } else {
      SHA1Input(&sha_ctx, (uint8_t *)&(ip_header->ip_dst), IP_ADDR_LEN);
      SHA1Input(&sha_ctx, (uint8_t *)&(tcp_header->th_dport), 2);
      SHA1Input(&sha_ctx, (uint8_t *)&(tcp_header->th_sport), 2);
    }
  }
  SHA1Result(&sha_ctx, hash);

  if(hfind(st_ctx->sessions, hash, SHA1HashSize) == FALSE) {
    return(NULL);
  } else {
    return(hstuff(st_ctx->sessions));
  }
}

struct stegt_file *
stegt_file_find(uint8_t *packet, struct stegt_ctx *st_ctx, int direction) 
{
  struct ip_hdr *ip_header;

  ip_header = (struct ip_hdr *)packet;
  
  if(direction == IN) {
    if(hfind(st_ctx->files, (uint8_t *)&(ip_header->ip_src), 
	     IP_ADDR_LEN) == FALSE) {
      return(NULL);
    } else {
      return(hstuff(st_ctx->files));
    }
  } else {
    if(hfind(st_ctx->files, (uint8_t *)&(ip_header->ip_dst), 
	     IP_ADDR_LEN) == FALSE) {
      return(NULL);
    } else {
      return(hstuff(st_ctx->files));
    }
  }
}

int
stegt_file_add(struct stegt_ctx *st_ctx, struct stegt_session *session,
	       struct stegt_file *file_ctx) 
{
  uint8_t *key;

  if(session->cur_file != NULL) 
    return(-1);
    
  if((key = calloc(1, IP_ADDR_LEN)) == NULL)
    return(-1);

  memcpy(key, &(session->remote_ip.addr_ip), IP_ADDR_LEN);

  if(hadd(st_ctx->files, key, IP_ADDR_LEN, file_ctx) == FALSE) {
    fprintf(stderr, "stegt_file_add: hadd() failed\n");
    return(-1);
  }
  session->cur_file = file_ctx;
  file_ctx->active_session = session;
  return(0);
}

