/*
 * stegtunnel.h 
 *
 * Structures used by both stegclient and stegserver.
 *
 * Copyright (c) 2003 Todd MacDermid <tmacd@synacklabs.net> 
 *
 */

#ifndef STEGTUNNEL_H
#define STEGTUNNEL_H

#include <dnet.h>
#include <time.h>

#include "hashtab.h"
#include "sha1.h"

/* 
 * tcp_state definitions. Note that these don't correspond identically
 * to the TCP FSM states in dnet/tcp.h, because we're really proxying
 * for another state machine. We have no concept, for example, of
 * TCP_LISTEN, we just forward on. Also, we can't assume we've ACKed
 * anything we've received, that takes place in two steps. Fun, ne?
 */

#define TCP_NO_STATE          0
#define TCP_SYN_SENT          1
#define TCP_SYN_RECEIVED      2
#define TCP_SYNACK_SENT       3
#define TCP_SYNACK_RECEIVED   4
#define TCP_ESTABLISHED       5
#define TCP_FIN_SENT          6
#define TCP_FIN_RECEIVED      7
#define TCP_FINACK_SENT       8
#define TCP_FINACK_RECEIVED   9
#define TCP_SENT_CLOSED       10
#define TCP_RECV_CLOSED       11

/*
 * stegt_session objects are allocated one per TCP connection, and are
 * deallocated when that session ends (either through FIN or RST).
 */

struct stegt_session {
  struct addr remote_ip;
  struct stegt_file *cur_file;
  int32_t seq_offset;     
  int msg_offset;
  int tcp_state;          /* Defined above */
  int keyed;              /* Whether the remote side knows our secret */
  uint16_t proxy_port;    /* Used in PP_FAKEIP mode */
  uint16_t remote_port;
  uint16_t local_port;
};

#define STEGT_MAXBLOCK 22
#define STEGT_HAMMING_BLOCK 30
#define STEGT_MAXNAME 80
#define STEGT_CHECKSUM_BYTES 4
#define SHA_INBUF_SZ 4096
#define STEGT_K 4
#define STEGT_SIZE_BYTES 4

#define STEGT_SENDING_NAME     0
#define STEGT_SENDING_SIZE     1
#define STEGT_SENDING_CHECKSUM 2
#define STEGT_SENDING_DATA     3
#define STEGT_CHECK_SUM        4
#define STEGT_FILE_DONE        5

struct stegt_file {
  struct stegt_session *active_session;
  struct timeval last_active;
  int phase;
  int size_loc;
  uint8_t size_buf[STEGT_SIZE_BYTES];
  uint32_t size;
  int check_loc;
  uint8_t checksum[STEGT_CHECKSUM_BYTES];
  int fd;
  uint32_t file_loc;
  uint32_t last_good_seq;
  uint32_t synch_seq;
  uint32_t synch_loc;
  int synch_set;
  uint8_t pot_resynch[2];
  uint8_t old_resynch[2];
  int need_resynch;
  int name_loc;
  char name_buf[STEGT_MAXNAME];
  int hamming_loc;
  uint8_t hamming_encoded[STEGT_HAMMING_BLOCK];
};

/*
 * stegt_ctx is the object-zilla which holds all the configuration settings
 * and other goodness which should be more or less globally available.
 */

/* 
 * Definitions for the mode used by stegtunnel. SEQ_AND_IPID is the only
 * supported one right now
 */

#define SEQ_AND_IPID 1

struct stegt_ctx {
  char message[65535];
  char served_name[80];
  uint8_t used_ports[TCP_PORT_MAX];
  struct addr target_ip;
  struct addr proxy_ip;
  uint8_t pw_hash[SHA1HashSize];
  htab *sessions;
  htab *hosts;
  htab *files;
  rand_t *rand_handle;
  int mode;
  int verbose;
  int repeat;
  int server;
  int file_server;
  int file_write_permit;
  int fd;
  int nat;
};
  
struct stegt_ctx *stegt_ctx_init(int);
void version(char *);
struct stegt_file *stegt_file_wcreate(void);
int stegt_file_wopen(struct stegt_file *);
struct stegt_file *stegt_file_ropen(char *);
int stegt_file_output(struct stegt_file *, uint8_t *, int);
int stegt_file_input(struct stegt_file *, struct stegt_ctx *, uint8_t *, int);
int stegt_file_read(struct stegt_file *);

#endif  /* STEGTUNNEL_H */
