package internal

import (
	"archive/zip"
	"io"
	"os"
	"path/filepath"

	"github.com/charmbracelet/bubbles/progress"
	"github.com/lithammer/shortuuid"
)

func zipSource(source, target string) error {
	id := shortuuid.New()
	prog := progress.New()
	prog.PercentageStyle = footerStyle

	totalFiles, err := countFiles(source)

	if err != nil {
		outPutLog("Zip file count files error: ", err)
	}

	p := process{
		name:     "zip files",
		progress: prog,
		state:    inOperation,
		total:    totalFiles,
		done:     0,
	}

	_, err = os.Stat(target)
	if os.IsExist(err) {
		p.name = "󰗄 File already exist"
		channel <- channelMessage{
			messageId:       id,
			processNewState: p,
		}
		return nil
	}

	f, err := os.Create(target)
	if err != nil {
		return err
	}
	defer f.Close()

	writer := zip.NewWriter(f)
	defer writer.Close()

	err = filepath.Walk(source, func(path string, info os.FileInfo, err error) error {
		p.name = "󰗄 " + filepath.Base(path)
		if len(channel) < 5 {
			channel <- channelMessage{
				messageId:       id,
				processNewState: p,
			}
		}

		if err != nil {
			return err
		}

		header, err := zip.FileInfoHeader(info)
		if err != nil {
			return err
		}

		header.Method = zip.Deflate

		header.Name, err = filepath.Rel(filepath.Dir(source), path)
		if err != nil {
			return err
		}
		if info.IsDir() {
			header.Name += "/"
		}

		headerWriter, err := writer.CreateHeader(header)
		if err != nil {
			return err
		}

		if info.IsDir() {
			return nil
		}

		f, err := os.Open(path)
		if err != nil {
			return err
		}
		defer f.Close()

		_, err = io.Copy(headerWriter, f)
		if err != nil {
			return err
		}
		p.done++
		if len(channel) < 5 {
			channel <- channelMessage{
				messageId:       id,
				processNewState: p,
			}
		}
		return nil
	})

	if err != nil {
		outPutLog("Error while zip file:", err)
		p.state = failure
		channel <- channelMessage{
			messageId:       id,
			processNewState: p,
		}
	}
	p.state = successful
	p.done = totalFiles
	channel <- channelMessage{
		messageId:       id,
		processNewState: p,
	}

	return nil
}