/* $Id$ */

/*
 * Copyright (c) 2015 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <getopt.h>
#include <signal.h>
#include <err.h>
#include <errno.h>
#include <fts.h>
#include <util.h>
#include <pwd.h>
#include <sysexits.h>
#include <libgen.h>
#include <syslog.h>
#include <paths.h>
#include <sys/cdefs.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/statvfs.h>
#include <sys/wait.h>
#include <ufs/ffs/fs.h>
#include <ufs/ufs/inode.h>

extern char **environ;

#include "syncffsd.h"

#define CHANGE_POLL 1
#define CHANGE_POLL_ONERROR 10

#define SYSLOG_DEBUG(...) if (debug) syslog(LOG_DEBUG, __VA_ARGS__);

#ifndef RSYNC_PREFIX
#define RSYNC_PREFIX "/usr/pkg"
#endif

#define DEFAULT_CMD_ARGC (ARG_MAX / MAXPATHLEN)

const char const default_cmd[] = RSYNC_PREFIX "/bin/rsync -rlptgzR --delete";
const char const superuser_cmd[] = RSYNC_PREFIX "/bin/rsync -azR --delete";
const char const dry_cmd[] = "/bin/echo ";

typedef union {
	struct fs fs;
	char _pad[SBLOCKSIZE];
} superblock_t;

struct options {
	int opt_foreground;
	const char *opt_pidfile;
	const char *opt_cmd;
	const char **opt_sources;
	const char **opt_targets;
	const char *opt_source_prefix;
	const char *opt_user;
	const char *opt_statefile;
	int opt_statedelay;
};

struct state {
	char st_dev[MAXPATHLEN];
	int st_children;
	int st_errors;
	time_t st_last_change;
	time_t st_last_start;
	time_t st_last_commit;
	int st_finished;
	time_t st_last_dump;
};

void  dump_state(struct state *, struct options *);
void  scheduled_dump_state(struct state *, struct options *);
void  check_signal(struct state *, struct options *);
void  signal_handler(int);
int   install_signal_handlers(void);
void  usage(void);
void  write_pid(const char *);
const char **append_argv(char *, const char **);
int   check_prefix(struct options *);
int   parse_options(int, char **, struct options *);
void  change_user(struct options *);
char *print_argv(const char **);
char *iso8601(time_t);
int   chg_dedup(char *, char *, int);
int   reset_argv(const char **, int, struct options *);
int   run_cmd(int, const char **, struct options *);
int   find_changes(struct state *, struct options *);
int   find_sb(const char **, int *, off_t *, char *, size_t);
void  collect_children(struct state *, struct options *);
int   main(int, char **);

sig_atomic_t signaled = 0;
int debug = 0;

const char *rsync_code[] = {
	/*  0 */	"Success",
	/*  1 */	"Syntax or usage error",
	/*  2 */	"Protocol incompatibility",
	/*  3 */	"Errors selecting input/output files, dirs",
	/*  4 */	"Requested  action  not supported",
	/*  5 */	"Error starting client-server protocol",
	/*  6 */	"Daemon unable to append to log-file",
	/*  7 */	"code 7",
	/*  8 */	"code 8",
	/*  9 */	"code 9",
	/* 10 */	"Error in socket I/O",
	/* 11 */	"Error in file I/O",
	/* 12 */	"Error in rsync protocol data stream",
	/* 13 */	"Errors with program diagnostics",
	/* 14 */	"Error in IPC code",
	/* 15 */	"code 15",
	/* 16 */	"code 16",
	/* 17 */	"code 17",
	/* 18 */	"code 18",
	/* 19 */	"code 19",
	/* 20 */	"Received SIGUSR1 or SIGINT",
	/* 21 */	"Some error returned by waitpid()",
	/* 22 */	"Error allocating core memory buffers",
	/* 23 */	"Partial transfer due to error",
	/* 24 */	"Partial transfer due to vanished source files",
	/* 25 */	"The --max-delete limit stopped deletions",
	/* 26 */	"code 26",
	/* 27 */	"code 27",
	/* 28 */	"code 28",
	/* 29 */	"code 29",
	/* 30 */	"Timeout in data send/receive",
	/* 31 */	"code 31",
	/* 32 */	"code 32",
	/* 33 */	"code 33",
	/* 34 */	"code 34",
	/* 35 */	"Timeout waiting for daemon connection",
};
#define RSYNC_CODE_MAX ((sizeof(rsync_code) / sizeof(*rsync_code)) - 1)

void
dump_state(st, opt)
	struct state *st;
	struct options *opt;
{
	FILE *s;
	time_t now;
	time_t sync_delay;
	time_t detect_delay;

	SYSLOG_DEBUG("state dump started to %s", opt->opt_statefile);

	if ((s = fopen(opt->opt_statefile, "w")) == NULL) {
		syslog(LOG_ERR, "Report request: cnannot open %s: %s",
		       opt->opt_statefile, strerror(errno));
		return;
	}

	now = time(NULL);
	detect_delay = now - st->st_last_change;
	if (st->st_last_start > st->st_last_commit)
		sync_delay = st->st_last_start - st->st_last_commit;
	else
		sync_delay = 0;

	fprintf(s, "timestamp:\t%lu\n", (unsigned long)now);
	fprintf(s, "sync_delay:\t%lu\n", (unsigned long)sync_delay);
	fprintf(s, "detect_delay:\t%lu\n", (unsigned long)detect_delay);
	fprintf(s, "sources:\t%s\n", print_argv(opt->opt_sources));
	fprintf(s, "source_device:\t%s\n", st->st_dev);
	fprintf(s, "source_prefix:\t%s\n", opt->opt_source_prefix);
	fprintf(s, "targets:\t%s\n", print_argv(opt->opt_targets));
	fprintf(s, "active_rsync:\t%d\n", st->st_children);
	fprintf(s, "error_rsync:\t%d\n", st->st_errors);
	fprintf(s, "last_change:\t%s\n", iso8601(st->st_last_change));
	fprintf(s, "last_start:\t%s\n", iso8601(st->st_last_start));
	fprintf(s, "last_commit:\t%s\n", iso8601(st->st_last_commit));

	(void)fclose(s);
	st->st_last_dump = time(NULL);

	SYSLOG_DEBUG("state dump to %s finished", opt->opt_statefile);

	return;
}

void
scheduled_dump_state(st, opt)
	struct state *st;
	struct options *opt;
{
	if (opt->opt_statedelay == 0)
		return;

	if (time(NULL) > st->st_last_dump + opt->opt_statedelay)
		dump_state(st, opt);

	return;
}

void
check_signal(st, opt)
	struct state *st;
	struct options *opt;
{
	switch (signaled) {
	case 0:
		break;
	case SIGINFO:
		syslog(LOG_INFO, "Got SIGINFO, dumping state");
		dump_state(st, opt);
		break;
	default:
		syslog(LOG_INFO, "Got signal %d, exitting", signaled);
		st->st_finished = 1;
		break;
	}

	signaled = 0;
	return;
}

void
signal_handler(signum)
	int signum;
{
	SYSLOG_DEBUG("got signal %d", signum);
	signaled = signum;
	return;
}

int
install_signal_handlers(void)
{
	int signums[] = { SIGTERM, SIGINT, SIGINFO, SIGHUP, 0 };
	int i;

	for (i = 0; signums[i]; i++) {
		if (signal(signums[i], *signal_handler) == SIG_ERR) {
			SYSLOG_DEBUG("signal(%d) failed: %s", 
				     signums[i], strerror(errno));
			return -1;
		}
	}

	return 0;
}

void
usage(void)
{
	fprintf(stderr, "Single source, single destination:\n");
	fprintf(stderr, "  syncffsd [common_options] "
	        "[-P src_prefix] src_dir [user@]dst_host:dst_dir\n");
	fprintf(stderr, "Mulitple sources and desintation "
	        "(multiple -s and -t flags allowed)\n");
	fprintf(stderr, "  syncffsd [common_options] "
	        "-P src_prefix -s src_dir -t [user@]dst_host:dst_dir\n");
	fprintf(stderr, "Common options:\n");
	fprintf(stderr, "  -u user     Switch to user during operation\n");
	fprintf(stderr, "  -p pidfile  Write PID to pidfile, default %s\n",
			DEFAULT_PIDFILE);
	fprintf(stderr, "  -f          Run in foreground\n");
	fprintf(stderr, "  -d          Enable debug messages\n");
	fprintf(stderr, "  -n          Dry mode, just print commands\n");
	fprintf(stderr, "  -S status   Replication state file, default %s\n",
			DEFAULT_STATEFILE);
	fprintf(stderr, "  -T delay    Delay between state dumps, default %d\n",
			DEFAULT_STATEDELAY);
	exit(EX_USAGE);
	/* NOTREACHED */
	return;
}

void
write_pid(pidfile)
	const char *pidfile;
{
	FILE *fp;

	if ((fp = fopen(pidfile, "w")) == NULL) {
		syslog(LOG_ERR, "cannot open PID file %s: %s",
		       pidfile, strerror(errno));
		return;
	}

	if (fprintf(fp, "%d\n", getpid()) == 0) {
		syslog(LOG_ERR, "cannot write PID to %s: %s",
		       pidfile, strerror(errno));
	}

	if (fclose(fp) != 0)
		syslog(LOG_WARNING, "close PID file %s error: %s",
		       pidfile, strerror(errno));

	return;
}

const char **
append_argv(item, argv)
	char *item;
	const char **argv;
{
	int argc = 0;
	size_t size;

	if (argv != NULL) {
		while (argv[argc] != NULL)
			argc++;
	}

	size = sizeof(*argv) * (argc + 2); /* +1 for item, +1 for NULL */

	if ((argv = realloc(argv, size)) == NULL) {
			syslog(LOG_ERR, "realloc failed: %s", strerror(errno));
			exit(EX_OSERR);
	}

	argv[argc] = item;
	argv[argc + 1] = NULL;

	return argv;
}

int
check_prefix(opt)
	struct options *opt;
{
	int i;
	char *p;
	int ret = 0;

	for(i = 0; opt->opt_sources[i] != NULL; i++) {
		p = strstr(opt->opt_sources[i], opt->opt_source_prefix);
		if (p != opt->opt_sources[i]) {
			warnx("source prefix \"%s\" does not match "
			      "source \"%s\"", opt->opt_source_prefix,
			      opt->opt_sources[i]);
			ret = -1;
		}
	}

	return ret;
}

int
parse_options(argc, argv, opt)
	int argc;
	char **argv;
	struct options *opt;
{
	int fl;
	int drymode = 0;

	(void)memset(opt, 0, sizeof(*opt));
	opt->opt_pidfile = DEFAULT_PIDFILE;
	opt->opt_statefile = DEFAULT_STATEFILE;
	opt->opt_statedelay = DEFAULT_STATEDELAY;

	while ((fl = getopt(argc, argv, "c:dfnp:P:s:S:t:T:u:")) != -1) {
		switch(fl) {
		case 'c':
			opt->opt_cmd = optarg;
			break;
		case 'd':
			debug = 1;
			break;
		case 'f':
			opt->opt_foreground = 1;
			break;
		case 'n':
			drymode = 1;
			break;
		case 'p':
			opt->opt_pidfile = optarg;
			break;
		case 'P':
			opt->opt_source_prefix = optarg;
			break;
		case 's':
			opt->opt_sources = append_argv(optarg,
						       opt->opt_sources);
			break;
		case 'S':
			opt->opt_statefile = optarg;
			break;
		case 't':
			opt->opt_targets = append_argv(optarg,	
						       opt->opt_targets);
			break;
		case 'T':
			opt->opt_statedelay = atoi(optarg);
			break;
		case 'u':
			opt->opt_user = optarg;
			break;
		default:
			usage();
			break;
		}
	}

	if (opt->opt_cmd == NULL) {
		if (opt->opt_user == NULL) {
			opt->opt_cmd = superuser_cmd;
		} else {
			opt->opt_cmd = default_cmd;
		}
	}

	if (drymode) {
		char *new_cmd;

		if (asprintf(&new_cmd, "%s %s", dry_cmd, opt->opt_cmd) == -1) {
			syslog(LOG_ERR, "asprintf failed: %s", strerror(errno));
			exit(EX_OSERR);
		}

		opt->opt_cmd = new_cmd;
	}

	return optind;
}

void
change_user(opt)
	struct options *opt;
{
	struct passwd *pw;
	uid_t uid;

	SYSLOG_DEBUG("change user to %s", opt->opt_user);

	errno = 0;
	uid = strtol(opt->opt_user, NULL, 10);
	if (errno == 0 && uid != 0)
		pw = getpwuid(uid);
	else
		pw = getpwnam(opt->opt_user);

	if (pw == NULL) {
		syslog(LOG_ERR, "inexistant user %s", opt->opt_user);
		exit(EX_NOUSER);
	}

	if (pw->pw_uid == geteuid()) {
		SYSLOG_DEBUG("already running uid %d", pw->pw_uid);
		return;
	}

	SYSLOG_DEBUG("switch to uid %d, gid %d", pw->pw_uid, pw->pw_gid);

	if (chown(opt->opt_pidfile, pw->pw_uid, pw->pw_gid) != 0)
		syslog(LOG_WARNING, "cannot chown \"%s\" to %s: %s",
		       opt->opt_pidfile, opt->opt_user,
	               strerror(errno));

	if (setgid(pw->pw_gid) != 0) {
		syslog(LOG_WARNING, "cannot setgid %d: %s",
		       pw->pw_gid, strerror(errno));
		if (getgid() != pw->pw_gid)
			exit(EX_NOUSER);
	}
		
	if (setegid(pw->pw_gid) != 0) {
		syslog(LOG_WARNING, "cannot setegid %d: %s",
		       pw->pw_gid, strerror(errno));
		if (getegid() != pw->pw_gid)
			exit(EX_NOUSER);
	}
		
	if (setuid(pw->pw_uid) != 0) {
		syslog(LOG_WARNING, "cannot setuid %d: %s",
		       pw->pw_uid, strerror(errno));
		if (getuid() != pw->pw_uid)
			exit(EX_NOUSER);
	}
		
	if (seteuid(pw->pw_uid) != 0) {
		syslog(LOG_WARNING, "cannot seteuid %d: %s",
		       pw->pw_uid, strerror(errno));
		if (geteuid() != pw->pw_uid)
			exit(EX_NOUSER);
	}
		
	return;
}

char *
print_argv(argv)
	const char **argv;
{
	static char buf[4096];
	char continued[] = " ... ";
	size_t maxlen = sizeof(buf) - sizeof(continued);
	const char **arg;

	buf[0] = '\0';

	for (arg = argv; *arg; arg++) {
		if (arg != argv) {
			if (strlcat(buf, ", ", maxlen) >= maxlen)
				break;
		}
		if (strlcat(buf, *arg, maxlen) >= maxlen)
			break;
	}

	if (*arg != NULL)
		(void)strlcat(buf, continued, sizeof(buf));
	
	return buf;
}

char *
iso8601(ts)
	time_t ts;
{
	static char buf[512];

	if (strftime(buf, sizeof(buf), "%F %H:%M:%S", localtime(&ts)) == 0)
		return NULL;
	
	return buf;
}

int
chg_dedup(path, chg, chgcnt)
	char *path;
	char *chg;
	int chgcnt;
{
	char *entry;
	int i;

	for (i = 0; i < chgcnt; i++) {
		entry = chg + (MAXPATHLEN * i);

		SYSLOG_DEBUG("%s: entry \"%s\", new \"%s\"", 
			     __func__, entry, path);

		/* already removed entry */
		if (*entry == '\0') {
			SYSLOG_DEBUG("%s:  => null entry", __func__);
			continue;
		}		

		/* we already have it: skip it */
		if (strcmp(entry, path) == 0) {
			SYSLOG_DEBUG("%s:  => dup", __func__);
			return -1;
		}

		/* existing entry is below new one: remove existing */
		if (strstr(entry, path) == entry) {
			SYSLOG_DEBUG("%s:  => rm entry", __func__);
			*entry = '\0';
		}

		/* new entry is below existing one: give up with new one */
		if (strstr(path, entry) == path) {
			SYSLOG_DEBUG("%s:  => rm new", __func__);
			return -1;
		}
	}

	SYSLOG_DEBUG("%s:  => add new", __func__);
	return 0;
}

int
reset_argv(argv, argc, opt)
	const char **argv;
	int argc;
	struct options *opt;
{
	char cmd[ARG_MAX];
	char *cp;
	char *st;
	int count = 0;
	int i = 0;
	
	for (i = 0; argv[i]; i++) {
		free(__UNCONST(argv[i]));
		argv[i] = NULL;
	}

	if (argc == 0)
		return 0;

	(void)strlcpy(cmd, opt->opt_cmd, sizeof(cmd));
	for (cp = strtok_r(cmd, " ", &st);
	     cp != NULL && count < argc;
	     cp = strtok_r(NULL, " ", &st)) {
		if ((argv[count++] = strdup(cp)) == NULL) {
			syslog(LOG_ERR, "strdup \"%s\" failed: %s",
			       cp, strerror(errno));
			exit(EX_OSERR);
		}
	}

	if (count >= argc) {
		syslog(LOG_ERR, "too many arguments: %d", count);
		exit(EX_DATAERR);
	}
	argv[count] = NULL;

	return count;
}

int
run_cmd(argc, argv, opt)
	int argc;
	const char **argv;
	struct options *opt;
{
	int i = 0;
	int children = 0;

	for (i = 0; opt->opt_targets[i] != NULL; i++) {
		switch(fork()) {
		case 0:
			argv[argc] = opt->opt_targets[i];
			argv[argc + 1] = NULL;

			SYSLOG_DEBUG("running argv = %s", print_argv(argv));

			execve(__UNCONST(argv[0]), __UNCONST(argv), environ);
			syslog(LOG_ERR, "exec \"%s\" failed: %s",
			       argv[0], strerror(errno));
			exit(EX_DATAERR);
			break;
		case -1:
			syslog(LOG_ERR, "cannot fork: %s", strerror(errno));
			return -1;
			break;
		default:
			children++;
			break;
		}
	}

	return children;
}

int
find_changes(st, opt)
	struct state *st;
	struct options *opt;
{
	uint32_t last_time = st->st_last_commit;
	FTS *fts;
	FTSENT *ftsent;
	const char **fts_argv = opt->opt_sources;
	char *chg = NULL;
	static int chglen = 256;	/* Some reaonsable default */
	int chgcnt = 0;
	const char *argv[DEFAULT_CMD_ARGC] = { NULL };
	unsigned int argc = 0;
	int dirty = 0;
	int ran = 0;
	int i;

	if (last_time == 0) {
		SYSLOG_DEBUG("initial sync");
		st->st_errors = 0;
		argc = reset_argv(argv, sizeof(argv), opt);

		if ((argv[argc++] = strdup("./")) == NULL) {
			syslog(LOG_ERR, "strdup \"./\" failed: %s",
			       strerror(errno));
			exit(EX_OSERR);
		}	

		ran += run_cmd(argc, argv, opt);

		(void)reset_argv(argv, 0, opt);
		return ran;
	}

	if ((chg = malloc(MAXPATHLEN * chglen)) == NULL) {
		syslog(LOG_ERR, "malloc(MAXPATHLEN * %d) failed: %s",
		       chglen, strerror(errno));
		exit(EX_OSERR);
	}

	if ((fts = fts_open(__UNCONST(fts_argv),
			    FTS_PHYSICAL|FTS_XDEV, NULL)) == NULL) {
		syslog(LOG_ERR, "fts_open(\"%s\") failed: %s",
		       print_argv(opt->opt_sources), strerror(errno));
		exit(EX_OSERR);
	}

	while ((ftsent = fts_read(fts)) != NULL) {
		switch(ftsent->fts_info) {
		case FTS_DNR:
		case FTS_ERR:
		case FTS_NS:
			syslog(LOG_WARNING, "fts_read(\"%s\") => %d: %s",
			       print_argv(opt->opt_sources), ftsent->fts_info, 
			       strerror(ftsent->fts_errno));
			continue;
		case FTS_NSOK:
			syslog(LOG_WARNING, "fts_read(\"%s\") => %d",
			       print_argv(opt->opt_sources), ftsent->fts_info);
			continue;
		default:
			break;
		}

		if ((ftsent->fts_statp->st_mtime == 0 ||
		     ftsent->fts_statp->st_mtime <= last_time) &&
		    (ftsent->fts_statp->st_ctime == 0 ||
		     ftsent->fts_statp->st_ctime <= last_time))
			continue;

		SYSLOG_DEBUG("changed on %s:\t%s",
			     iso8601(ftsent->fts_statp->st_mtime),
			     ftsent->fts_path);

		if (chg_dedup(ftsent->fts_path, chg, chgcnt) != 0)
			continue;

		SYSLOG_DEBUG("changed array count = %d -> %d, "
			     "len = %d adding \"%s\"",
			     chgcnt, chgcnt + 1, chglen, ftsent->fts_path);

		(void)strlcpy(chg + (MAXPATHLEN * chgcnt),
			      ftsent->fts_path, MAXPATHLEN);

		chgcnt++;

		if (chgcnt > chglen) {
			SYSLOG_DEBUG("changed array count = %d, len = %d -> %d",
				     chgcnt, chglen, chglen * 2);

			chglen *= 2;

			if ((chg = realloc(chg, MAXPATHLEN * chglen)) == NULL) {
				syslog(LOG_ERR,
				       "malloc(MAXPATHLEN * %d) failed: %s",
				       chglen, strerror(errno));
				exit(EX_OSERR);
			}
		}
	}

	if (fts_close(fts) != 0)
		syslog(LOG_WARNING, "fts_close(\"%s\") error: %s",
		       print_argv(opt->opt_sources), strerror(errno));

	SYSLOG_DEBUG("chgcnt = %d\n", chgcnt);

	st->st_errors = 0;
	argc = reset_argv(argv, sizeof(argv), opt);
	for (i = 0; i < chgcnt; i++) {
		char *entry = chg + (MAXPATHLEN * i);
		char *relative_entry = entry;

		if (*entry == '\0')
			continue;

		relative_entry = entry;
		if (strcmp(opt->opt_source_prefix, "/") != 0) {
			relative_entry += strlen(opt->opt_source_prefix) - 2;
			relative_entry[0] = '.';
			relative_entry[1] = '/';
		}

		if ((argv[argc++] = strdup(relative_entry)) == NULL) {
			syslog(LOG_ERR, "strdup(\"%s\") failed: %s",
			       entry, strerror(errno));
			exit(EX_OSERR);
		}

		dirty++;
		
		if (argc + 2 > sizeof(argv)) {
			ran += run_cmd(argc, argv, opt);

			argc = reset_argv(argv, sizeof(argv), opt);
			dirty = 0;
		}
	}	

	if (dirty)
		ran += run_cmd(argc, argv, opt);

	(void)reset_argv(argv, 0, opt);
	free(chg);

	return ran;
}

int
find_sb(sources, fdp, offp, dev, devlen)
	const char **sources;
	int *fdp;
	off_t *offp;
	char *dev;
	size_t devlen;
{
	int fd;
	struct statvfs vfs;
	char mntfromname[MAXPATHLEN + 1] = "";
	const off_t sblock_try[] = SBLOCKSEARCH;
	int i;
	superblock_t sb;

	for (i = 0; sources[i] != NULL; i++) {
		if (statvfs(sources[i], &vfs) != 0) {
			syslog(LOG_ERR, "statvfs(\"%s\") failed; %s",
			       sources[i], strerror(errno));
			return -1;
		}

		if (mntfromname[0] == '\0') {
			(void)strlcpy(mntfromname, 
				      vfs.f_mntfromname,
				      sizeof(mntfromname));
		} else {
			if (strcmp(vfs.f_mntfromname, mntfromname) != 0) {
				syslog(LOG_ERR, "sources must reside on the "
				       "same filesystem: \"%s\" was on \"%s\", "
				       "while \"%s\" is on \"%s\"",
				       sources[0], mntfromname,
				       sources[i], vfs.f_mntfromname);
				exit(EX_USAGE);
			}
		}
	}
			

	fd = opendisk(basename(mntfromname), O_RDONLY, dev, devlen, 0);
	if (fd == -1) {
		syslog(LOG_ERR, "open(\"%s\") failed; %s",
		       mntfromname, strerror(errno));
		return -1;
	}

	for (i = 0; sblock_try[i] != -1; i++) {
		if (pread(fd, &sb, sizeof(sb), sblock_try[i]) != sizeof(sb)) {
			syslog(LOG_ERR, "read(\"%s\") failed; %s",
			       mntfromname, strerror(errno));
			return -1;
		}

		switch(sb.fs.fs_magic) {
		case FS_UFS2_MAGIC:
			SYSLOG_DEBUG("Found UFS2 on %s", dev);
			goto out;
			break;
		case FS_UFS1_MAGIC:
			SYSLOG_DEBUG("Found UFS1 on %s", dev);
			goto out;
			break;
		case FS_UFS2_MAGIC_SWAPPED:
			SYSLOG_DEBUG("Found UFS2 swapped on %s", dev);
			goto out;
			break;
		case FS_UFS1_MAGIC_SWAPPED:
			SYSLOG_DEBUG("Found UFS1 swapped on %s", dev);
			goto out;
			break;
		default:
			break;
		}
	}
out:
	if (sblock_try[i] == -1) {
		if (close(fd) != 0)
			syslog(LOG_WARNING, "close(\"%s\") error: %s",
			       mntfromname, strerror(errno));
		return -1;
	}

	*fdp = fd;
	*offp = sblock_try[i];
	return 0;
}

void
collect_children(st, opt)
	struct state *st;
	struct options *opt;
{
	pid_t pid;
	int status;

	SYSLOG_DEBUG("wait %d children", st->st_children);

	while ((pid = waitpid(-1, &status, WNOHANG)) != 0) {
		check_signal(st, opt);
		scheduled_dump_state(st, opt);
		if (st->st_finished)
			break;

		if (pid == -1) {
			if (errno == ECHILD)
				break;

			syslog(LOG_ERR, "waitpid failed: %s",
			       strerror(errno));

		} else if (WIFSIGNALED(status)) {
			syslog(LOG_ERR, "child %d got signal %d",
			       pid, WTERMSIG(status));
			st->st_children--;
			st->st_errors++;

		} else if (WIFSTOPPED(status)) {
			syslog(LOG_ERR, "child %d stopped", pid);

		} else if (WIFEXITED(status)) {
			int code = WEXITSTATUS(status);

			if (code > 0 && code <= RSYNC_CODE_MAX) {
				syslog(LOG_ERR, "child %d exit: %s",
				       pid, rsync_code[code]);
				st->st_errors++;
			} else if (code != 0) {
				syslog(LOG_ERR, "child %d exit code %d",
				       pid, code);
				st->st_errors++;
			}
			st->st_children--;

		} else {
			syslog(LOG_ERR, "unexpected status %d for pid %d",
			       status, pid);
			exit(EX_SOFTWARE);
		}
	}

	return;
}

int
main(argc, argv)
	int argc;
	char **argv;
{
	char *progname = argv[0];
	int shift_args;
	struct options opt;
	off_t sb_off;
	int fd;
	superblock_t sb;
	struct state st;

	(void)memset(&st, 0, sizeof(st));

	shift_args = parse_options(argc, argv, &opt);
	argc -= shift_args;
	argv += shift_args;

	if ((opt.opt_sources == NULL && opt.opt_targets != NULL) ||
	    (opt.opt_sources != NULL && opt.opt_targets == NULL))
		usage();

	if (opt.opt_sources != NULL && opt.opt_targets != NULL && argc != 0)
		usage();

	if (opt.opt_sources == NULL && opt.opt_targets == NULL) {
		if (argc != 2)
			usage();
		opt.opt_sources = append_argv(argv[0], opt.opt_sources);
		opt.opt_targets = append_argv(argv[1], opt.opt_targets);
	}

	if (opt.opt_source_prefix == NULL) {
		if (opt.opt_sources[1] == NULL) /* single sources */
			opt.opt_source_prefix = opt.opt_sources[0];
		else
			errx(EX_USAGE, "Specifying a source prefix using -P "
				       "is mandatory with multiple sources");
	}

	if (check_prefix(&opt) != 0)
		errx(EX_USAGE, "source/prefix mismatch, giving up.");

	openlog(progname, (opt.opt_foreground ? LOG_PERROR : 0), LOG_DAEMON);

	if (!opt.opt_foreground)
		daemon(0, 0);

	SYSLOG_DEBUG("chdir to prefix dir \"%s\"", opt.opt_source_prefix);
	if (chdir(opt.opt_source_prefix) != 0) {
		syslog(LOG_ERR, "Cannot change directory to \"%s\": %s",
		       opt.opt_source_prefix, strerror(errno));
		exit(EX_DATAERR);
	}

	if (install_signal_handlers() != 0) {
		syslog(LOG_ERR, "Cannot install signal handler, exitting");
		exit(EX_OSERR);
	}

	write_pid(opt.opt_pidfile);

	if (find_sb(opt.opt_sources, &fd, &sb_off, 
		    st.st_dev, sizeof(st.st_dev)) == -1) {
		syslog(LOG_ERR, "cannot find superblock for %s",
		       opt.opt_sources[0]);
		exit(EX_DATAERR);
	}

	if (opt.opt_user)
		change_user(&opt);

	do {
		if (pread(fd, &sb, sizeof(sb), sb_off) != sizeof(sb)) {
			syslog(LOG_ERR, "read(\"%s\") failed: %s",
			       st.st_dev, strerror(errno));
			exit(EX_IOERR);
		}

		st.st_last_change = sb.fs.fs_time;

		check_signal(&st, &opt);
		scheduled_dump_state(&st, &opt);
		if (st.st_finished)
			break;

		if (st.st_last_change > st.st_last_commit) {
			SYSLOG_DEBUG("change on %s since %s",
				     opt.opt_sources[0],
				     iso8601(st.st_last_change));	

			if (st.st_children != 0) {
				SYSLOG_DEBUG("%d children running, skip", 
					     st.st_children);
			} else {
				st.st_children += find_changes(&st, &opt);
				st.st_last_start = st.st_last_change;
			}
		}

		if (st.st_children > 0) {
			collect_children(&st, &opt);
			SYSLOG_DEBUG("%d children running, %d errors",
				     st.st_children, st.st_errors);
		}

		if (st.st_children == 0 && st.st_errors == 0)
			st.st_last_commit = st.st_last_start;

		if (st.st_finished)
			break;

		if (st.st_errors == 0)
			sleep(CHANGE_POLL);
		else
			sleep(CHANGE_POLL_ONERROR);
	} while (1 /* CONSTCOND */);

	if (unlink(opt.opt_pidfile) != 0)
		syslog(LOG_WARNING, "cannot remove PID file %s: %s",
		       opt.opt_pidfile, strerror(errno));

	exit(EX_OK);
}
