#![allow(non_local_definitions)]
pub mod error;
pub mod params;
pub mod results;
pub mod util;

use std::fmt::Debug;

use async_trait::async_trait;
use lazy_static::lazy_static;
use serde::Deserialize;
use syncserver_db_common::{DbFuture, GetPoolState};

use error::DbErrorIntrospect;
use util::SyncTimestamp;

lazy_static! {
    /// For efficiency, it's possible to use fixed pre-determined IDs for
    /// common collection names.  This is the canonical list of such
    /// names.  Non-standard collections will be allocated IDs starting
    /// from the highest ID in this collection.
    pub static ref STD_COLLS: Vec<(i32, &'static str)> = {
        vec![
        (1, "clients"),
        (2, "crypto"),
        (3, "forms"),
        (4, "history"),
        (5, "keys"),
        (6, "meta"),
        (7, "bookmarks"),
        (8, "prefs"),
        (9, "tabs"),
        (10, "passwords"),
        (11, "addons"),
        (12, "addresses"),
        (13, "creditcards"),
        ]
    };
}

/// Rough guesstimate of the maximum reasonable life span of a batch
pub const BATCH_LIFETIME: i64 = 2 * 60 * 60 * 1000; // 2 hours, in milliseconds

/// The ttl to use for rows that are never supposed to expire (in seconds)
pub const DEFAULT_BSO_TTL: u32 = 2_100_000_000;

/// Non-standard collections will be allocated IDs beginning with this value
pub const FIRST_CUSTOM_COLLECTION_ID: i32 = 101;

#[async_trait]
pub trait DbPool: Sync + Send + Debug + GetPoolState {
    type Error;

    async fn get(&self) -> Result<Box<dyn Db<Error = Self::Error>>, Self::Error>;

    fn validate_batch_id(&self, params: params::ValidateBatchId) -> Result<(), Self::Error>;

    fn box_clone(&self) -> Box<dyn DbPool<Error = Self::Error>>;
}

impl<E> Clone for Box<dyn DbPool<Error = E>> {
    fn clone(&self) -> Box<dyn DbPool<Error = E>> {
        self.box_clone()
    }
}

pub trait Db: Debug {
    type Error: DbErrorIntrospect + 'static;

    fn lock_for_read(&mut self, params: params::LockCollection) -> DbFuture<'_, (), Self::Error>;

    fn lock_for_write(&mut self, params: params::LockCollection) -> DbFuture<'_, (), Self::Error>;

    fn begin(&mut self, for_write: bool) -> DbFuture<'_, (), Self::Error>;

    fn commit(&mut self) -> DbFuture<'_, (), Self::Error>;

    fn rollback(&mut self) -> DbFuture<'_, (), Self::Error>;

    fn get_collection_timestamps(
        &mut self,
        params: params::GetCollectionTimestamps,
    ) -> DbFuture<'_, results::GetCollectionTimestamps, Self::Error>;

    fn get_collection_timestamp(
        &mut self,
        params: params::GetCollectionTimestamp,
    ) -> DbFuture<'_, results::GetCollectionTimestamp, Self::Error>;

    fn get_collection_counts(
        &mut self,
        params: params::GetCollectionCounts,
    ) -> DbFuture<'_, results::GetCollectionCounts, Self::Error>;

    fn get_collection_usage(
        &mut self,
        params: params::GetCollectionUsage,
    ) -> DbFuture<'_, results::GetCollectionUsage, Self::Error>;

    fn get_storage_timestamp(
        &mut self,
        params: params::GetStorageTimestamp,
    ) -> DbFuture<'_, results::GetStorageTimestamp, Self::Error>;

    fn get_storage_usage(
        &mut self,
        params: params::GetStorageUsage,
    ) -> DbFuture<'_, results::GetStorageUsage, Self::Error>;

    fn get_quota_usage(
        &mut self,
        params: params::GetQuotaUsage,
    ) -> DbFuture<'_, results::GetQuotaUsage, Self::Error>;

    fn delete_storage(
        &mut self,
        params: params::DeleteStorage,
    ) -> DbFuture<'_, results::DeleteStorage, Self::Error>;

    fn delete_collection(
        &mut self,
        params: params::DeleteCollection,
    ) -> DbFuture<'_, results::DeleteCollection, Self::Error>;

    fn delete_bsos(
        &mut self,
        params: params::DeleteBsos,
    ) -> DbFuture<'_, results::DeleteBsos, Self::Error>;

    fn get_bsos(&mut self, params: params::GetBsos) -> DbFuture<'_, results::GetBsos, Self::Error>;

    fn get_bso_ids(
        &mut self,
        params: params::GetBsos,
    ) -> DbFuture<'_, results::GetBsoIds, Self::Error>;

    fn post_bsos(
        &mut self,
        params: params::PostBsos,
    ) -> DbFuture<'_, results::PostBsos, Self::Error>;

    fn delete_bso(
        &mut self,
        params: params::DeleteBso,
    ) -> DbFuture<'_, results::DeleteBso, Self::Error>;

    fn get_bso(
        &mut self,
        params: params::GetBso,
    ) -> DbFuture<'_, Option<results::GetBso>, Self::Error>;

    fn get_bso_timestamp(
        &mut self,
        params: params::GetBsoTimestamp,
    ) -> DbFuture<'_, results::GetBsoTimestamp, Self::Error>;

    fn put_bso(&mut self, params: params::PutBso) -> DbFuture<'_, results::PutBso, Self::Error>;

    fn create_batch(
        &mut self,
        params: params::CreateBatch,
    ) -> DbFuture<'_, results::CreateBatch, Self::Error>;

    fn validate_batch(
        &mut self,
        params: params::ValidateBatch,
    ) -> DbFuture<'_, results::ValidateBatch, Self::Error>;

    fn append_to_batch(
        &mut self,
        params: params::AppendToBatch,
    ) -> DbFuture<'_, results::AppendToBatch, Self::Error>;

    fn get_batch(
        &mut self,
        params: params::GetBatch,
    ) -> DbFuture<'_, Option<results::GetBatch>, Self::Error>;

    fn commit_batch(
        &mut self,
        params: params::CommitBatch,
    ) -> DbFuture<'_, results::CommitBatch, Self::Error>;

    fn check(&mut self) -> DbFuture<'_, results::Check, Self::Error>;

    fn get_connection_info(&self) -> results::ConnectionInfo;

    /// Retrieve the timestamp for an item/collection
    ///
    /// Modeled on the Python `get_resource_timestamp` function.
    fn extract_resource(
        &mut self,
        user_id: UserIdentifier,
        collection: Option<String>,
        bso: Option<String>,
    ) -> DbFuture<'_, SyncTimestamp, Self::Error> {
        Box::pin(async move {
            match collection {
                None => {
                    // No collection specified, return overall storage timestamp
                    self.get_storage_timestamp(user_id).await
                }
                Some(collection) => match bso {
                    None => self
                        .get_collection_timestamp(params::GetCollectionTimestamp {
                            user_id,
                            collection,
                        })
                        .await
                        .or_else(|e| {
                            if e.is_collection_not_found() {
                                Ok(SyncTimestamp::from_seconds(0f64))
                            } else {
                                Err(e)
                            }
                        }),
                    Some(bso) => self
                        .get_bso_timestamp(params::GetBsoTimestamp {
                            user_id,
                            collection,
                            id: bso,
                        })
                        .await
                        .or_else(|e| {
                            if e.is_collection_not_found() {
                                Ok(SyncTimestamp::from_seconds(0f64))
                            } else {
                                Err(e)
                            }
                        }),
                },
            }
        })
    }

    // Internal methods used by the db tests

    fn get_collection_id(&mut self, name: String) -> DbFuture<'_, i32, Self::Error>;

    fn create_collection(&mut self, name: String) -> DbFuture<'_, i32, Self::Error>;

    fn update_collection(
        &mut self,
        params: params::UpdateCollection,
    ) -> DbFuture<'_, SyncTimestamp, Self::Error>;

    fn timestamp(&self) -> SyncTimestamp;

    fn set_timestamp(&mut self, timestamp: SyncTimestamp);

    fn delete_batch(&mut self, params: params::DeleteBatch) -> DbFuture<'_, (), Self::Error>;

    fn clear_coll_cache(&mut self) -> DbFuture<'_, (), Self::Error>;

    fn set_quota(&mut self, enabled: bool, limit: usize, enforce: bool);
}

#[derive(Debug, Default, Deserialize, Clone, PartialEq, Eq, Copy)]
#[serde(rename_all = "lowercase")]
pub enum Sorting {
    #[default]
    None,
    Newest,
    Oldest,
    Index,
}

#[derive(Clone, Debug, Default, Eq, Hash, PartialEq)]
pub struct UserIdentifier {
    /// For MySQL database backends as the primary key
    pub legacy_id: u64,
    /// For NoSQL database backends that require randomly distributed primary keys
    pub fxa_uid: String,
    pub fxa_kid: String,
    pub hashed_fxa_uid: String,
    pub hashed_device_id: String,
}
