/*	$Id: stdio2pty.c,v 1.1.2.1 2011/01/15 12:48:44 steve Exp $	*/

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/wait.h>

#include <errno.h>
#include <fcntl.h>
#include <poll.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "buffer.h"
#include "stdio2pty.h"

#define	MAX_BUFF_LEN	32768

#ifndef WALLSIG
#define WALLSIG 0
#endif

static int all_done, exit_status, child_pid;

static struct buffer_ctx *stdout_buff;
static struct buffer_ctx *ptyout_buff;

static void
childsig(int dummy)
{
	int status;
	pid_t pid;

	if ((pid = wait3(&status, WNOHANG | WALLSIG, NULL)) == 0)
		return;

	if (WIFEXITED(status)) {
		exit_status = WEXITSTATUS(status);
		all_done = 1;
	} else
	if (WIFSIGNALED(status)) {
		exit_status = -1;
		all_done = 1;
	}
}

static int
copyfrom(int fd, struct buffer_ctx *bc)
{
	char buff[1024];
	ssize_t rv;

#ifndef MIN
#define	MIN(a,b)	(((a) < (b)) ? (a) : (b))
#endif

	for (;;) {
		if ((rv = read(fd, buff, sizeof(buff))) == 0)
			break;

		if (rv < 0) {
			if (errno == EINTR)
				continue;

			if (errno == EWOULDBLOCK)
				break;

			return (-1);
		}

		if ((rv = buffer_fill(bc, buff, (size_t)rv)) < 0)
			return ((int)rv);
	}

	return (0);
}

static int
mainloop(int fd)
{
	struct pollfd pfds[3];
	int nfds, rv = 0;

	while (all_done == 0 && rv >= 0) {
		pfds[0].fd = fd;
		pfds[0].events = POLLHUP;
		pfds[0].revents = 0;
		if (buffer_length(stdout_buff) < MAX_BUFF_LEN)
			pfds[0].events |= POLLRDNORM;
		if (buffer_length(ptyout_buff) > 0)
			pfds[0].events |= POLLWRNORM;

		pfds[1].fd = STDIN_FILENO;
		pfds[1].revents = 0;
		if (buffer_length(ptyout_buff) < MAX_BUFF_LEN)
			pfds[1].events = POLLRDNORM;
		else
			pfds[1].events = 0;

		pfds[2].fd = STDOUT_FILENO;
		pfds[2].revents = 0;
		if (buffer_length(stdout_buff) > 0)
			pfds[2].events = POLLWRNORM;
		else
			pfds[2].events = 0;

		if ((nfds = poll(pfds, 3, INFTIM)) < 0 && errno == EINTR)
			continue;

		if (nfds < 0) {
			perror("poll");
			return(-1);
		}

		if (pfds[0].revents & POLLHUP)
			return (3);	/* Child closed slave device */

		if (pfds[0].revents & POLLRDNORM)
			rv = copyfrom(pfds[0].fd, stdout_buff);

		if (rv >= 0 && pfds[2].revents & POLLWRNORM)
			rv = buffer_drain(stdout_buff);

		if (rv >= 0 && pfds[1].revents & POLLRDNORM)
			rv = copyfrom(pfds[1].fd, ptyout_buff);

		if (rv >= 0 && pfds[0].revents & POLLWRNORM)
			rv = buffer_drain(ptyout_buff);
	}

	return (rv);
}

int
stdio2pty_main(int argc, char **argv, char **envp)
{
	char *slavename, **cmdargs;
	int fd, rv;

	if ((fd = posix_openpt(O_RDWR | O_NONBLOCK)) < 0) {
		perror("posix_openpt");
		exit(1);
	}

	if (unlockpt(fd) < 0) {
		perror("unlockpt");
		(void) close(fd);
		exit(1);
	}

	if (grantpt(fd) < 0) {
		perror("grantpt");
		(void) close(fd);
		exit(1);
	}

	if ((slavename = ptsname(fd)) == NULL) {
		perror("ptsname");
		(void) close(fd);
		exit(1);
	}

	signal(SIGCHLD, childsig);

	if ((child_pid = fork()) < 0) {
		perror("fork");
		(void) close(fd);
		exit(1);
	} else
	if (child_pid == 0) {
		(void) close(fd);
		signal(SIGCHLD, SIG_DFL);

		if ((fd = open(slavename, O_RDWR)) < 0) {
			perror("ptsname");
			exit(1);
		}

		if (dup2(fd, STDIN_FILENO) < 0 || dup2(fd, STDOUT_FILENO) < 0) {
			perror("dup2");
			(void) close(fd);
			exit(1);
		}

		if ((cmdargs = calloc(argc, sizeof(*cmdargs))) == NULL) {
			perror("calloc");
			(void) close(fd);
			exit(1);
		}

		memcpy(cmdargs, &argv[1], (argc - 1) * sizeof(*cmdargs));
		cmdargs[argc - 1] = NULL;

		(void) execve(argv[1], cmdargs, envp);

		perror("execve");
		fprintf(stderr, "%s: Failed to exec '%s'\n", argv[0], argv[1]);
		(void) close(fd);
		exit(2);
	}

	if (buffer_init(&stdout_buff, STDOUT_FILENO) < 0 ||
	    buffer_init(&ptyout_buff, fd) < 0) {
		perror("buffer_init");
		sleep(1);
		kill(child_pid, SIGHUP);
		sleep(1);
		(void) close(fd);
		exit(1);
	}

	if (fcntl(STDIN_FILENO, F_SETFL, O_NONBLOCK) < 0) {
		perror("fcntl(O_NONBLOCK)");
		sleep(1);
		kill(child_pid, SIGHUP);
		sleep(1);
		(void) close(fd);
		exit(1);
	}

	if ((rv = mainloop(fd)) != 0) {
		sleep(1);
		kill(child_pid, SIGHUP);
		sleep(1);
		close(fd);
		exit(rv);
	}

	exit(exit_status);
}
