/* config.c -- routines to read and write to Windows style configuration files
 *
 * Copyright (c) 1998  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log: config.c,v $
 * Revision 1.1  1999/05/25 08:08:42  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

static const char rcsid[]="$Id: config.c,v 1.1 1999/05/25 08:08:42 jab Exp jab $";

#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<errno.h>
#include<limits.h>
#include<unistd.h>
#include<sys/stat.h>

#define MAXLENGTH   256   /* maximum length of a line */
#define NEWLINE     "\n"  /* change to "\r\n" if using DOS style text files */

static char *cache = NULL;
struct stat last;


/*
 * Case insensitive version of strstr
 */
char *FindString(const char *haystack, const char *needle)
{
	char *h,*n,*t;

	if (*needle=='\0')
		return NULL;

	for (h=(char *) haystack;*h!='\0';h++) {
		t = h;
		n = (char *) needle;
		
		for (t=h,n=(char *) needle;
			(((*t)|0x60)==((*n)|0x60)) && (*n!='\0') && (*t!='\0');
			t++,n++);

		if ((n>needle) && (*n=='\0'))
		      return h;   
	}

	return NULL;
}


/*
 * Read a configuration file into the global cache
 */
int ReadConfig(char *file)
{
	FILE *handle;
	struct stat info;

	if (stat(file, &info)!=0)
		return 0;

	if (S_ISREG(info.st_dev))
		return 0;

	/* is the config file the same as the last one read and unchanged? */

	if ((info.st_dev==last.st_dev) && (info.st_ino==last.st_dev) &&
		(info.st_mtime==last.st_mtime) && (cache!=NULL))
		return strlen(cache);

	/* free the cache ready  */

	if (cache!=NULL)
		free(cache);

	/* allocate the memory for the cache */

	if ((cache = (char *) malloc(info.st_size+1))==NULL) {
		return 0;
	}

	/* open the file and read it into the cache */

	if ((handle = fopen(file,"rb"))==NULL) {
		return 0;
	}
	fread(cache, info.st_size, 1, handle);
	cache[info.st_size] = '\0';
 
	fclose(handle);

	/* store the relevant fields of the stat structure */

	last.st_dev = info.st_dev;
	last.st_ino = info.st_ino;
	last.st_mtime = info.st_mtime;

	return strlen(cache)+1;
}


/*
 * Locate a given section and return a pointer to it, within the global cache
 */
char *FindSection(char *section)
{
	char find[MAXLENGTH];
	char *temp;

	strcpy(find, NEWLINE);
	strcat(find, "[");
	strcat(find, section);
	strcat(find, "]");

	temp = strstr(cache, find);

	if (temp==NULL) {
		temp = strstr(cache, find+1);
		if (temp==NULL)
			return NULL;
	}	

	return temp+strlen(NEWLINE);
}


/*
 * Return a pointer to the entry from the given section of configuration file
 */
char *GetEntry(char *section, char *entry)
{
	char find[MAXLENGTH];
	char *start,*toofar;

	strcpy(find, NEWLINE);
	strcat(find, entry);
	strcat(find, "=");

	if ((start = strstr(section, find))==NULL)
		return NULL;

	start += strlen(NEWLINE);

	if ((toofar = strstr(section+1, "["))==NULL)
		toofar = section+strlen(section);

	if (toofar<start)
		return NULL;
	else
		return start+strlen(find)-1;
}


/*
 * Locate and return an integer entry in a given section of the specified file
 */
int GetConfigInt(char *section, char *entry, int fault, char *file)
{
	char *start;
	long number;


	if (ReadConfig(file)==0)
		return fault;

	if ((start = FindSection(section))==NULL)
		return fault;

	if ((start = GetEntry(start, entry))==NULL)
		return fault;

	number = strtol(start, (char **) NULL, 0);

	if ((errno==ERANGE) && ((number==LONG_MAX) || (number==LONG_MIN)))
		return fault;

	return (int) number;
}


/*
 * Locate and return a string entry in a given section of the specified file
 */
int GetConfigString(char *section, char *entry, char *fault, char *buffer,
	int size, char *file)
{
	char *start,*end;


	if (ReadConfig(file)==0) {
		strcpy(buffer, fault);
		return strlen(fault)-1;
	}

	if ((start = FindSection(section))==NULL) {
		strcpy(buffer, fault);
		return strlen(fault)-1;
	}

	if ((start = GetEntry(start, entry))==NULL) {
		strcpy(buffer, fault);
		return strlen(fault)-1;
	}

	if (*start=='\"') {
		start++;
		if ((end = strstr(start, "\""))==NULL)
			end = start+strlen(start);
	}
	else if (*start=='\'') {
		start++;
		if ((end = strstr(start, "\'"))==NULL)
			end = start+strlen(start);
	}
	else {
		if ((end = strstr(start, NEWLINE))==NULL)
			end = start+strlen(start);
	}

	strncpy(buffer, start, end-start);
	buffer[end-start] = '\0';

	return end-start;
}


/*
 * Update or write a string to the configuration file creating the file
 * and/or section and/or entry as needed
 */
int WriteConfigString(char *section, char *entry, char *string, char *file)
{
	FILE *handle;
	int size;
	char find[8];
	char *start,*end,*line;
	struct stat info;

	/* check to see if the file exists, if not create it */

	if ((stat(file, &info)!=0) && (errno==ENOENT)) {
		if ((handle = fopen(file,"w"))==NULL)
			return 0;
		fprintf(handle, "[%s]%s", section, NEWLINE);
		fprintf(handle, "%s=%s%s", entry, string, NEWLINE);
		fclose(handle);
		free(cache);
		cache = NULL;
		return 1;
	}

	/* read in the configuration file as it may not be cached */

	size = ReadConfig(file);
	if (size==0) {
		return 0; /* even empty files have a cache size of one */
	}

	/* open the file in append mode */

	if ((handle = fopen(file,"w"))==NULL) {
		return 0;
	}

	/* if the section does not already exist, appened entry at the end */

	if ((start = FindSection(section))==NULL) {

		fwrite(cache, 1, size-1, handle);
		fprintf(handle, "%s[%s]%s", NEWLINE, section, NEWLINE);
		fprintf(handle, "%s=%s%s", entry, string, NEWLINE);
		fclose(handle);
		free(cache);
		cache = NULL;
		return 1;
	}

	/* if the entry does not exist in the section add entry at the end */

	if ((end = GetEntry(start, entry))==NULL) {

		/* try and find the end of the section */

		strcpy(find, NEWLINE);
		strcat(find, "[");

		/* if unable to find start line of the next section, must have
		   been the last section in the file */

		if ((line = strstr(start+1, find))==NULL) {
			fwrite(cache, 1, size-1, handle);
			fprintf(handle, "%s=%s%s", entry, string, NEWLINE);
		} else {
			fwrite(cache, 1, line-cache, handle);
			fprintf(handle, "%s=%s%s", entry, string, NEWLINE);
			fwrite(line, 1, size-(line-cache)-1, handle);
		}
	} else {
		/* write the configuration string over the existing one */

		strcpy(find, NEWLINE);
		line = strstr(end, find);
		fwrite(cache, 1, end-cache, handle);
		fprintf(handle, "%s", string);
		fwrite(line, 1, size-(line-cache)-1, handle);
	}
	
	fclose(handle);
	free(cache);
	cache = NULL;

	return 1;
}
