/****************************************************************************/
/*                                                                          */
/*                        TCPA TakeOwnerShip Routine                        */
/*                                                                          */
/* This file is copyright 2003 IBM. See "License" for details               */
/****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include <tcpa.h>
#include <buildbuff.h>
#include <oiaposap.h>
#include <hmac.h>
#include <keys.h>
#include <openssl/rsa.h>
#include <openssl/rand.h>
#include <openssl/bn.h>
#include <openssl/crypto.h>
#include <openssl/err.h>
#include <openssl/evp.h>

/****************************************************************************/
/*                                                                          */
/*  Take Ownership of the TPM                                               */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* ownpass   is the authorization data (password) for the new owner         */
/* dsrkpass  is the authorization data (password) for the new root key      */
/*           both authorization values MUST be 20 bytes long                */
/* key       a pointer to a keydata structure to receive the SRK public key */
/*           or NULL if this information is not required                    */
/*                                                                          */
/****************************************************************************/
uint32_t TPM_TakeOwnership(unsigned char *ownpass, unsigned char *srkpass,
                           keydata *key, FILE *log)
{
    unsigned char take_owner_fmt[] = "00 c2 T l s @ @ % L % 00 %";
    unsigned char tcpa_oaep_pad_str[] = { 'T', 'C', 'P', 'A' };
    unsigned char tcpadata[TCPA_MAX_BUFF_SIZE]; /* request/response buffer */
    pubkeydata tcpapubkey;      /* public endorsement key data */
    uint32_t srkparamsize;      /* SRK parameter buffer size */
    unsigned char nonceeven[TCPA_HASH_SIZE];
    RSA *pubkey;                /* PubEK converted to OpenSSL format */
    unsigned char padded[RSA_MODULUS_BYTE_SIZE];
    keydata srk;                /* key info for SRK */
    uint32_t ret;
    uint32_t command;           /* command ordinal */
    uint16_t protocol;          /* protocol ID */
    uint32_t oencdatasize;      /* owner auth data encrypted size */
    unsigned char ownerencr[RSA_MODULUS_BYTE_SIZE];
    uint32_t sencdatasize;      /* srk auth data encrypted size */
    unsigned char srkencr[RSA_MODULUS_BYTE_SIZE];
    unsigned char srk_param_buff[TCPA_SRK_PARAM_BUFF_SIZE];
    uint32_t authhandle;        /* auth handle (from OIAPopen) */
    unsigned char nonceodd[TCPA_HASH_SIZE];     /* odd nonce */
    unsigned char authdata[TCPA_HASH_SIZE];     /* auth data */

    /* check that parameters are valid */
    if (ownpass == NULL || srkpass == NULL)
        return 1;
    /* set up command and protocol values for TakeOwnership function */
    command = htonl(0x0d);
    protocol = htons(0x05);
    /* get the TCPA Endorsement Public Key */
    ret = TPM_ReadPubek(&tcpapubkey, log);
    if (ret)
        return 1;
    /* convert the public key to OpenSSL format */
    pubkey = convpubkey(&tcpapubkey);
    if (pubkey == NULL)
        return 1;
    memset(ownerencr, 0, sizeof ownerencr);
    memset(srkencr, 0, sizeof srkencr);
    /* Pad and then encrypt the owner data using the RSA public key */
    ret = RSA_padding_add_PKCS1_OAEP(padded, RSA_MODULUS_BYTE_SIZE,
                                     ownpass, TCPA_HASH_SIZE,
                                     tcpa_oaep_pad_str,
                                     sizeof tcpa_oaep_pad_str);
    if (ret == 0)
        return 1;
    ret = RSA_public_encrypt(RSA_MODULUS_BYTE_SIZE, padded, ownerencr,
                             pubkey, RSA_NO_PADDING);
    if (ret < 0)
        return 1;
    oencdatasize = htonl(ret);
    /* Pad and then encrypt the SRK data using the RSA public key */
    ret = RSA_padding_add_PKCS1_OAEP(padded, RSA_MODULUS_BYTE_SIZE,
                                     srkpass, TCPA_HASH_SIZE,
                                     tcpa_oaep_pad_str,
                                     sizeof tcpa_oaep_pad_str);
    if (ret == 0)
        return 1;
    ret = RSA_public_encrypt(RSA_MODULUS_BYTE_SIZE, padded, srkencr, pubkey,
                           RSA_NO_PADDING);
    if (ret < 0)
        return 1;
    sencdatasize = htonl(ret);
    RSA_free(pubkey);
    if (ntohl(oencdatasize) < 0)
        return 1;
    if (ntohl(sencdatasize) < 0)
        return 1;
    /* fill the SRK-params key structure */
    srk.version[0] = 1;         /* version 1.1.0.0 */
    srk.version[1] = 1;
    srk.version[2] = 0;
    srk.version[3] = 0;
    srk.keyusage = 0x0011;      /* Storage Key */
    srk.keyflags = 0;
    srk.authdatausage = 0x01;   /* Key usage must be authorized */
    srk.privkeylen = 0;         /* private key not specified here */
    srk.pub.algorithm = 0x00000001;     /* RSA */
    srk.pub.encscheme = 0x0003; /* RSA OAEP SHA1 MGF1 */
    srk.pub.sigscheme = 0x0001; /* NONE */
    srk.pub.keybitlen = RSA_MODULUS_BIT_SIZE;
    srk.pub.numprimes = 2;
    srk.pub.expsize = 0;        /* defaults to 0x010001 */
    srk.pub.keylength = 0;      /* not used here */
    srk.pub.pcrinfolen = 0;     /* not used here */
    /* convert to a memory buffer */
    srkparamsize = BuildKey(srk_param_buff, &srk);
    /* generate the odd nonce */
    ret = RAND_bytes(nonceodd, TCPA_NONCE_SIZE);
    if (ret != 1)
        return 1;
    /* initiate the OIAP protocol */
    ret = TPM_OIAP(&authhandle, nonceeven, log);
    if (ret)
        return ret;
    /* calculate the Authorization Data */
    ret = authhmac(authdata, ownpass, TCPA_HASH_SIZE, nonceeven,
                   nonceodd, 0, 4, &command, 2,
                   &protocol, 4, &oencdatasize,
                   ntohl(oencdatasize), ownerencr, 4,
                   &sencdatasize, ntohl(sencdatasize), srkencr,
                   srkparamsize, srk_param_buff, 0, 0);
    if (ret < 0) {
        TPM_Terminate_Handle(authhandle, log);
        return 1;
    }
    /* insert all the calculated fields into the request buffer */
    ret = buildbuff(take_owner_fmt, tcpadata,
                    command,
                    protocol,
                    ntohl(oencdatasize),
                    ownerencr,
                    ntohl(sencdatasize),
                    srkencr,
                    srkparamsize,
                    srk_param_buff,
                    authhandle,
                    TCPA_HASH_SIZE, nonceodd, TCPA_HASH_SIZE, authdata);
    if (ret <= 0) {
        TPM_Terminate_Handle(authhandle, log);
        return 1;
    }
    ret = TPM_Transmit(tcpadata, log, "Take Ownership");
    TPM_Terminate_Handle(authhandle, log);
    if (ret != 0)
        return ret;
    /* check the response HMAC */
    srkparamsize = KeySize(tcpadata + TCPA_DATA_OFFSET);
    ret = checkhmac1(tcpadata, command, nonceodd, ownpass,
                     TCPA_HASH_SIZE, srkparamsize, TCPA_DATA_OFFSET, 0, 0);
    if (ret != 0)
        return 1;
    if (key == NULL)
        return 0;
    KeyExtract(tcpadata + TCPA_DATA_OFFSET, key);
    return 0;
}
