/****************************************************************************/
/*                                                                          */
/*                           SEAL/UNSEAL routines                           */
/*                                                                          */
/* This file is copyright 2003 IBM. See "License" for details               */
/****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include <tcpa.h>
#include <buildbuff.h>
#include <oiaposap.h>
#include <hmac.h>
#include <pcrs.h>
#include <openssl/rand.h>

/****************************************************************************/
/*                                                                          */
/* Seal a data object with caller Specified PCR infro                       */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* keyhandle is the TCPA_KEY_HANDLE of the key used to seal the data        */
/*           0x40000000 for the SRK                                         */
/* pcrinfo   is a pointer to a TCPA_PCR_INFO structure containing           */
/*           a bit map of the PCR's to seal the data to, and a              */
/*           pair of TCPA_COMPOSITE_HASH values for the PCR's               */
/* pcrinfosize is the length of the pcrinfo structure                       */
/* keyauth   is the authorization data (password) for the key               */
/* dataauth  is the authorization data (password) for the data being sealed */
/*           both authorization values must be 20 bytes long ?              */
/* data      is a pointer to the data to be sealed                          */
/* datalen   is the length of the data to be sealed (max 256?)              */
/* blob      is a pointer to an area to received the sealed blob            */
/*           it should be long enough to receive the encrypted data         */
/*           which is 256 bytes, plus some overhead. 512 total recommended? */
/* bloblen   is a pointer to an integer which will receive the length       */
/*           of the sealed blob                                             */
/*                                                                          */
/****************************************************************************/
uint32_t TPM_Seal(uint32_t keyhandle,
                  unsigned char *pcrinfo, uint32_t pcrinfosize,
                  unsigned char *keyauth,
                  unsigned char *dataauth,
                  unsigned char *data, unsigned int datalen,
                  unsigned char *blob, unsigned int *bloblen, FILE * log)
{
    unsigned char seal_fmt[] = "00 C2 T l l % @ @ l % o %";
    unsigned char tcpadata[TCPA_MAX_BUFF_SIZE];
    unsigned char encauth[TCPA_HASH_SIZE];
    unsigned char pubauth[TCPA_HASH_SIZE];
    unsigned char xorwork[TCPA_HASH_SIZE * 2];
    unsigned char xorhash[TCPA_HASH_SIZE];
    unsigned char nonceodd[TCPA_NONCE_SIZE];
    osapsess sess;
    uint32_t ret;
    int i;
    unsigned char c;
    uint32_t ordinal;
    uint32_t pcrsize;
    uint32_t datsize;
    uint32_t keyhndl;
    uint16_t keytype;
    int sealinfosize;
    int encdatasize;
    int storedsize;

    /* check input arguments */
    if (keyauth == NULL || dataauth == NULL || data == NULL
        || blob == NULL)
        return 1;
    if (pcrinfosize != 0 && pcrinfo == NULL)
        return 1;
    if (keyhandle == 0x40000000)
        keytype = 0x0004;
    else
        keytype = 0x0001;
    /* Open OSAP Session */
    ret = TPM_OSAP(&sess, keyauth, keytype, keyhandle, log);
    if (ret)
        return ret;
    /* calculate encrypted authorization value */
    memcpy(xorwork, sess.ssecret, TCPA_HASH_SIZE);
    memcpy(xorwork + TCPA_HASH_SIZE, sess.enonce, TCPA_HASH_SIZE);
    sha1(xorwork, TCPA_HASH_SIZE * 2, xorhash);
    RAND_bytes(nonceodd, TCPA_NONCE_SIZE);
    /* move Network byte order data to variables for hmac calculation */
    ordinal = htonl(0x17);
    datsize = htonl(datalen);
    keyhndl = htonl(keyhandle);
    pcrsize = htonl(pcrinfosize);
    c = 0;
    /* encrypt data authorization key */
    for (i = 0; i < TCPA_HASH_SIZE; ++i)
        encauth[i] = xorhash[i] ^ dataauth[i];
    /* calculate authorization HMAC value */
    if (pcrinfosize == 0) {
        /* no pcr info specified */
        ret = authhmac(pubauth, sess.ssecret, TCPA_HASH_SIZE,
                       sess.enonce, nonceodd, c, 4,
                       &ordinal, TCPA_HASH_SIZE, encauth, 4,
                       &pcrsize, 4, &datsize, datalen, data, 0, 0);
    } else {
        /* pcr info specified */
        ret = authhmac(pubauth, sess.ssecret, TCPA_HASH_SIZE,
                       sess.enonce, nonceodd, c, 4,
                       &ordinal, TCPA_HASH_SIZE, encauth, 4,
                       &pcrsize, pcrinfosize, pcrinfo, 4,
                       &datsize, datalen, data, 0, 0);
    }
    if (ret < 0) {
        TPM_Terminate_Handle(sess.handle, log);
        return 1;
    }
    /* build the request buffer */
    ret = buildbuff(seal_fmt, tcpadata,
                    ordinal,
                    keyhndl,
                    TCPA_HASH_SIZE, encauth,
                    pcrinfosize, pcrinfo,
                    datalen, data,
                    sess.handle,
                    TCPA_NONCE_SIZE, nonceodd, c, TCPA_HASH_SIZE, pubauth);
    if (ret <= 0) {
        TPM_Terminate_Handle(sess.handle, log);
        return 1;
    }
    /* transmit the request buffer to the TPM device and read the reply */
    ret = TPM_Transmit(tcpadata, log, "Seal");
    if (ret != 0) {
        TPM_Terminate_Handle(sess.handle, log);
        return ret;
    }
    /* calculate the size of the returned Blob */
    sealinfosize = ntohl(*(uint32_t *) (tcpadata + TCPA_DATA_OFFSET + 4));
    encdatasize = ntohl(*(uint32_t *) (tcpadata +
                                       TCPA_DATA_OFFSET + 4 + 4 +
                                       sealinfosize));
    storedsize = 4 + 4 + sealinfosize + 4 + encdatasize;
    /* check the HMAC in the response */
    ret = checkhmac1(tcpadata, ordinal, nonceodd, sess.ssecret,
                     TCPA_HASH_SIZE, storedsize, TCPA_DATA_OFFSET, 0, 0);
    if (ret != 0) {
        TPM_Terminate_Handle(sess.handle, log);
        return 1;
    }
    /* copy the returned blob to caller */
    memcpy(blob, tcpadata + TCPA_DATA_OFFSET, storedsize);
    *bloblen = storedsize;
    TPM_Terminate_Handle(sess.handle, log);
    return 0;
}

/****************************************************************************/
/*                                                                          */
/* Seal a data object with current PCR information                          */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* keyhandle is the TCPA_KEY_HANDLE of the key used to seal the data        */
/*           0x40000000 for the SRK                                         */
/* pcrmap    is a 32 bit integer containing a bit map of the PCR register   */
/*           numbers to be used when sealing. e.g 0x0000001 specifies       */
/*           PCR 0. 0x00000003 specifies PCR's 0 and 1, etc.                */
/* keyauth   is the authorization data (password) for the key               */
/* dataauth  is the authorization data (password) for the data being sealed */
/*           both authorization values must be 20 bytes long ?              */
/* data      is a pointer to the data to be sealed                          */
/* datalen   is the length of the data to be sealed (max 256?)              */
/* blob      is a pointer to an area to received the sealed blob            */
/*           it should be long enough to receive the encrypted data         */
/*           which is 256 bytes, plus some overhead. 512 total recommended? */
/* bloblen   is a pointer to an integer which will receive the length       */
/*           of the sealed blob                                             */
/*                                                                          */
/****************************************************************************/
uint32_t TPM_Seal_CurrPCR(uint32_t keyhandle, uint32_t pcrmap,
                         unsigned char *keyauth,
                         unsigned char *dataauth,
                         unsigned char *data, unsigned int datalen,
                         unsigned char *blob, unsigned int *bloblen,
                         FILE * log)
{
    uint32_t ret;
    unsigned char pcrinfo[MAXPCRINFOLEN];
    uint32_t pcrlen;

    ret = GenPCRInfo(pcrmap, pcrinfo, &pcrlen, log);
    return TPM_Seal(keyhandle, pcrinfo, pcrlen,
                    keyauth, dataauth, data, datalen, blob, bloblen, log);
}

/****************************************************************************/
/*                                                                          */
/* Unseal a data object                                                     */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* keyhandle is the TCPA_KEY_HANDLE of the key used to seal the data        */
/*           0x40000000 for the SRK                                         */
/* keyauth   is the authorization data (password) for the key               */
/* dataauth  is the authorization data (password) for the data being sealed */
/*           both authorization values must be 20 bytes long ?              */
/* blob      is a pointer to an area to containing the sealed blob          */
/* bloblen   is the length of the sealed blob                               */
/* rawdata   is a pointer to an area to receive the unsealed data (max 256?)*/
/* datalen   is a pointer to a int to receive the length of the data        */
/*                                                                          */
/****************************************************************************/
uint32_t TPM_Unseal(uint32_t keyhandle,
                    unsigned char *keyauth,
                    unsigned char *dataauth,
                    unsigned char *blob, unsigned int bloblen,
                    unsigned char *rawdata, unsigned int *datalen,
                    FILE * log)
{
    unsigned char unseal_fmt[] = "00 C3 T l l % L % o % L % o %";
    unsigned char tcpadata[TCPA_MAX_BUFF_SIZE];
    unsigned char nonceodd[TCPA_NONCE_SIZE];
    unsigned char enonce1[TCPA_NONCE_SIZE];
    unsigned char enonce2[TCPA_NONCE_SIZE];
    unsigned char authdata1[TCPA_HASH_SIZE];
    unsigned char authdata2[TCPA_HASH_SIZE];
    uint32_t ret;
    unsigned char c;
    uint32_t ordinal;
    uint32_t keyhndl;
    uint32_t authhandle1;
    uint32_t authhandle2;

    /* check input arguments */
    if (keyauth == NULL || dataauth == NULL || rawdata == NULL
        || blob == NULL)
        return 1;
    /* open TWO OIAP sessions, one for the Key and one for the Data */
    ret = TPM_OIAP(&authhandle1, enonce1, log);
    if (ret)
        return ret;
    ret = TPM_OIAP(&authhandle2, enonce2, log);
    if (ret)
        return ret;
    /* move data to Network byte order variables for HMAC calculation */
    ordinal = htonl(0x18);
    keyhndl = htonl(keyhandle);
    /* generate odd nonce */
    RAND_bytes(nonceodd, TCPA_NONCE_SIZE);
    c = 0;
    /* calculate KEY authorization HMAC value */
    ret = authhmac(authdata1, keyauth, TCPA_HASH_SIZE, enonce1, nonceodd,
                   c, 4, &ordinal, bloblen, blob, 0, 0);
    if (ret < 0) {
        TPM_Terminate_Handle(authhandle1, log);
        TPM_Terminate_Handle(authhandle2, log);
        return 1;
    }
    /* calculate DATA authorization HMAC value */
    ret = authhmac(authdata2, dataauth, TCPA_HASH_SIZE, enonce2,
                   nonceodd, c, 4, &ordinal, bloblen, blob, 0, 0);
    if (ret < 0) {
        TPM_Terminate_Handle(authhandle1, log);
        TPM_Terminate_Handle(authhandle2, log);
        return -1;
    }
    /* build the request buffer */
    ret = buildbuff(unseal_fmt, tcpadata,
                    ordinal,
                    keyhndl,
                    bloblen, blob,
                    authhandle1,
                    TCPA_NONCE_SIZE, nonceodd,
                    c,
                    TCPA_HASH_SIZE, authdata1,
                    authhandle2,
                    TCPA_NONCE_SIZE, nonceodd,
                    c, TCPA_HASH_SIZE, authdata2);

    if (ret <= 0) {
        TPM_Terminate_Handle(authhandle1, log);
        TPM_Terminate_Handle(authhandle2, log);
        return 1;
    }
    /* transmit the request buffer to the TPM device and read the reply */
    ret = TPM_Transmit(tcpadata, log, "Unseal");
    if (ret != 0) {
        TPM_Terminate_Handle(authhandle1, log);
        TPM_Terminate_Handle(authhandle2, log);
        return ret;
    }
    /* TODO !!!!!!!!!!!!!!!!
       check HMAC in response
     */
    /* copy decrypted data back to caller */
    *datalen = ntohl(*(uint32_t *) (tcpadata + TCPA_DATA_OFFSET));
    memcpy(rawdata, tcpadata + TCPA_DATA_OFFSET + 4, *datalen);
    TPM_Terminate_Handle(authhandle1, log);
    TPM_Terminate_Handle(authhandle2, log);
    return 0;
}
