#!/usr/bin/ruby
#
# Author:: Oliver M. Bolzer <oliver@fakeroot.net>
# Copyright:: Copyright (c) 2002 Oliver M. Bolzer. All rights reserved.
# Licence:: Ruby licence.

require 'test/unit'
require 'mock'

require 'vapor/extent'


## test class for Extent 
class Test_Extent < Test::Unit::TestCase
  include Vapor

  # test set-up
  def setup
     @mock_pmgr = Mock.new
     def @mock_pmgr.get_object( oid ); end
  end # setup()


  # test type
  def test_type
    assert( Extent.included_modules.include?( Enumerable ) )
  end # test_type()

  # test constructor
  def test_constructor

    # test instanciation
    # three Arguments: Extent.new( PersistenceManager, Class, subclasses )
    assert_raises( ArgumentError ){ Extent.new }
    assert_raises( ArgumentError ){ Extent.new( nil ) }
    assert_raises( ArgumentError ){ Extent.new( nil, nil ) }
    assert_raises( ArgumentError ){ Extent.new( nil, Object ) }
    assert_raises( ArgumentError ){ Extent.new( @mock_pmgr , nil ) }
    assert_raises( ArgumentError ){ Extent.new( nil, nil, nil, nil ) } 
  end # test_constructor()
  
  # test addition of Persistable
  def test_add
    
    klass = Class.new
    
    extent = Extent.new( @mock_pmgr, klass, false )

    # check argument handling
    # only one argument, must be Integer
    assert_respond_to( extent, :add )
    assert_raises( ArgumentError ){ extent.add() }
    assert_raises( ArgumentError ){ extent.add( nil, nil ) }
    assert_raises( ArgumentError ){ extent.add( "fooo" ) }
    assert_nothing_raised{ extent.add( 1 ) }

    # must return itself
    extent2 = extent.add( 2 )
    assert_same( extent, extent2 )

    ## test alias method <<
    assert_nothing_raised{ extent << 3 }
    extent2 = extent << 4 << 5 << 6
    assert_same( extent, extent2 )
  end # test_add()

  # test if class of stored objects is correctly saved
  def test_candidate_class
    
    # prepare, use anonymous klass to make sure the answer is genuine
    klass = Class.new 
    extent = Extent.new( @mock_pmgr, klass, false )

    # test
    assert_respond_to( extent, :candidate_class )
    assert_same( klass, extent.candidate_class )

  end # test_candiate_class()

  # test associated PersistenceManager
  def test_persistence_manager
    extent = Extent.new( @mock_pmgr, Object, false )
    
    assert_respond_to( extent, :persistence_manager )
    assert_same( @mock_pmgr, extent.persistence_manager )
  end # test_persistence_manager

  # test each-Iterator over all contained Persistables
  def test_each
    
    # the Extent to test
    extent = Extent.new( @mock_pmgr, Foo, false )
    
    # prepare mock_pmgr
    # with Foo-objects, we want to put into the Extent
    def @mock_pmgr.foos
     @foos ||= [ Foo.new(0), Foo.new(1), Foo.new(2) ]
     return @foos              
    end

    def @mock_pmgr.get_object( oid )
      return @foos[ oid ]
    end

    # test method existence, correct parameter (block)
    assert_respond_to( extent, :each )

    # add the them to the extent
    @mock_pmgr.foos.each{|foo|
      extent.add( foo.oid )
    }
    
    # make sure iterator does not run without block
    assert_raises( LocalJumpError ){ extent.each }

    # prepare our PersistenceManager

    # now call our iterator, save results in a new Array
    returned_foos = Array.new
    assert_nothing_raised{
      extent.each{|object| returned_foos << object }
    }

    # the results should be equal to the Foos above
    assert_equal( @mock_pmgr.foos, returned_foos )

    @mock_pmgr.__verify

  end # test_each()

  ## test empty?() and size()
  def test_size_and_empty
 
    extent = Extent.new( @mock_pmgr, Object, false )

    assert_respond_to( extent, :empty? )
    assert_respond_to( extent, :size )

    assert_equal( true, extent.empty? )
    assert_equal( 0, extent.size )

    ## test size increase 10 times
    1.upto( 10 ){ |index|
      extent.add( index )

      assert_equal( false, extent.empty? )
      assert_equal( index, extent.size )
   }

  end # test_size_and_empty()

  # test listing of OIDs
  def test_oid_array
    extent = Extent.new( @mock_pmgr, Object, false )
    oids = [ 1, 2, 3, 4, 5 ]
    oids.each{|oid|  extent.add( oid ) }

    extent_oids = extent.oid_array

    #extent_oids.sort!

    assert_equal( oids, extent_oids )

  end # test_oid_array()

  def test_subclass
    # include subclasses
    extent = Extent.new( @mock_pmgr, Object, false )
    assert_respond_to( extent, :has_subclasses? )
    assert_equal( false, extent.has_subclasses? )

    extent = Extent.new( @mock_pmgr, Object, true )
    assert_equal( true, extent.has_subclasses? )
  end # test_subclass()
  
  # Mock-class for Testing purposes
  class Foo
    def initialize( oid )
      @oid = oid
    end
    attr_reader :oid
  end

end # class Test_Extent
