/* packet-bmc.c
 * Routines for Broadcast/Multicast Control dissection
 * Copyright 2011, Neil Piercy <Neil.Piercy@ipaccess.com>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

#include "config.h"

#include <epan/packet.h>
#include <wsutil/bitswap.h>
#include <epan/asn1.h> /* needed for packet-gsm_map.h */

#include "packet-cell_broadcast.h"
#include "packet-gsm_map.h"

void proto_register_bmc(void);

static int dissect_bmc_cbs_message     (tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree);
static int dissect_bmc_schedule_message(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree);
static int dissect_bmc_cbs41_message   (tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree);

static int proto_bmc;
static int hf_bmc_message_type;
static int hf_bmc_message_id;
static int hf_bmc_serial_number;
/* static int hf_bmc_data_coding_scheme; */
/* static int hf_bmc_cb_data; */
static int hf_bmc_offset_to_begin_ctch_bs_index;
static int hf_bmc_length_of_cbs_schedule_period;
static int hf_bmc_new_message_bitmap;
static int hf_bmc_message_description_type;
static int hf_bmc_offset_to_ctch_bs_index_of_first_transmission;
static int hf_bmc_broadcast_address;
static int hf_bmc_cb_data41;
static int hf_bmc_future_extension_bitmap;
static int hf_bmc_length_of_serial_number_list;
static int hf_bmc_ctch_bs_index;

#define MESSAGE_TYPE_CBS_MESSAGE        1
#define MESSAGE_TYPE_SCHEDULE_MESSAGE   2
#define MESSAGE_TYPE_CBS41_MESSAGE      3

static const value_string message_type_vals[] = {
    {MESSAGE_TYPE_CBS_MESSAGE, "CBS Message"},
    {MESSAGE_TYPE_SCHEDULE_MESSAGE, "Schedule Message"},
    {MESSAGE_TYPE_CBS41_MESSAGE, "CBS41 Message"},
    {0, NULL}
};

static const value_string message_description_type_vals[] = {
    {0, "Repetition of new BMC CBS message within schedule period"},
    {1, "New BMC CBS message (a BMC CBS message never previously sent)"},
    {2, "Reading advised"},
    {3, "Reading optional"},
    {4, "Repetition of old BMC CBS message within schedule period"},
    {5, "Old BMC CBS message (repetition of a BMC CBS message sent in a previous schedule period)"},
    {6, "Schedule message"},
    {7, "CBS41 message"},
    {8, "no message"},
    {0, NULL}
};

static int ett_bmc;
static int ett_bmc_message_description;

static int
dissect_bmc(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void *data _U_)
{
    uint8_t     message_type;
    uint8_t    *reversing_buffer;
    int         offset = 0;
    int         len;
    proto_item *ti;
    proto_tree *bmc_tree;
    tvbuff_t   *bit_reversed_tvb;

    col_set_str(pinfo->cinfo, COL_PROTOCOL, "BMC");
    col_clear(pinfo->cinfo, COL_INFO);

    ti = proto_tree_add_item(tree, proto_bmc, tvb, 0, -1, ENC_NA);
    bmc_tree = proto_item_add_subtree(ti, ett_bmc);

    /* Needs bit-reversing. Create a new buffer, copy the message to it and bit-reverse */
    len = tvb_reported_length(tvb);
    reversing_buffer = (uint8_t *)tvb_memdup(pinfo->pool, tvb, offset, len);
    bitswap_buf_inplace(reversing_buffer, len);

    /* Make this new buffer part of the display and provide a way to dispose of it */
    bit_reversed_tvb = tvb_new_child_real_data(tvb, reversing_buffer, len, len);
    add_new_data_source(pinfo, bit_reversed_tvb, "Bit-reversed Data");

    message_type = tvb_get_uint8(bit_reversed_tvb, offset);
    proto_tree_add_item(bmc_tree, hf_bmc_message_type, bit_reversed_tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;
    col_add_str(pinfo->cinfo, COL_INFO, val_to_str(pinfo->pool, message_type, message_type_vals,"Reserved 0x%02x"));

    switch (message_type) {
        case MESSAGE_TYPE_CBS_MESSAGE:
            offset = dissect_bmc_cbs_message(bit_reversed_tvb, pinfo, bmc_tree);
            break;

        case MESSAGE_TYPE_SCHEDULE_MESSAGE:
            offset = dissect_bmc_schedule_message(bit_reversed_tvb, pinfo, bmc_tree);
            break;

        case MESSAGE_TYPE_CBS41_MESSAGE:
            offset = dissect_bmc_cbs41_message(bit_reversed_tvb, pinfo, bmc_tree);
            break;

        default:
            break;
    }

    return offset;
}

static int
dissect_bmc_cbs_message(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree)
{
    tvbuff_t *cell_broadcast_tvb;
    int       offset = 1;

    dissect_cbs_message_identifier(tvb, tree, offset);
    offset += 2;

    dissect_cbs_serial_number(tvb, tree, offset);
    offset += 2;

    dissect_cbs_data_coding_scheme(tvb, pinfo, tree, offset);
    offset += 1;

    cell_broadcast_tvb = tvb_new_subset_remaining(tvb, offset);
    dissect_umts_cell_broadcast_message(cell_broadcast_tvb, pinfo, tree, NULL);
    offset = tvb_reported_length(cell_broadcast_tvb);

    return offset;
}

static int
dissect_bmc_schedule_message(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree)
{
    int         offset = 1, i, saved_offset;
    uint8_t     new_message_bitmap_len;
    uint8_t     length_of_cbs_schedule_period;
    uint8_t     message_description_type;
    uint8_t     future_extension_bitmap;
    uint8_t     length_of_serial_number_list;
    uint8_t     entry;
    uint8_t     bit;
    proto_tree *message_description_tree;
    proto_item *ti;

    proto_tree_add_item(tree, hf_bmc_offset_to_begin_ctch_bs_index, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    length_of_cbs_schedule_period = tvb_get_uint8(tvb,offset);
    proto_tree_add_item(tree, hf_bmc_length_of_cbs_schedule_period, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    new_message_bitmap_len = length_of_cbs_schedule_period>>3;
    if (length_of_cbs_schedule_period & 0x7)
        new_message_bitmap_len += 1;

    proto_tree_add_item(tree, hf_bmc_new_message_bitmap, tvb, offset, new_message_bitmap_len, ENC_NA);
    offset += new_message_bitmap_len;

    message_description_tree = proto_tree_add_subtree(tree, tvb, offset, 0,
                    ett_bmc_message_description, &ti, "Message Description" );
    saved_offset = offset;

    bit=1;
    for (i=0; i<new_message_bitmap_len; i++) {
        for(; bit<=length_of_cbs_schedule_period; bit++) {
            message_description_type = tvb_get_uint8(tvb,offset);
            proto_tree_add_uint_format(message_description_tree, hf_bmc_message_description_type,
                                       tvb, offset, 1, message_description_type,
                                       "Message %d Message Description Type: %s (%d)",
                                       bit,
                                       val_to_str_const(message_description_type, message_description_type_vals,"Unknown"),
                                       message_description_type);
            offset += 1;

            if ((message_description_type==1) || (message_description_type==5)) {
                proto_tree_add_item(message_description_tree, hf_bmc_message_id, tvb, offset, 2, ENC_BIG_ENDIAN);
                offset += 2;
            }
            else if ((message_description_type==0) || (message_description_type==4)) {
                proto_tree_add_item(message_description_tree, hf_bmc_offset_to_ctch_bs_index_of_first_transmission,
                                    tvb, offset, 1, ENC_BIG_ENDIAN);
                offset += 1;
            }
        }
    }
    proto_item_set_len(ti, offset-saved_offset);

    if (tvb_reported_length_remaining(tvb,offset)) {
        future_extension_bitmap = tvb_get_uint8(tvb,offset);
        proto_tree_add_item(tree, hf_bmc_future_extension_bitmap, tvb, offset, 1, ENC_BIG_ENDIAN);
        offset += 1;
        if (future_extension_bitmap & 0x01) {
            length_of_serial_number_list = tvb_get_uint8(tvb,offset);
            proto_tree_add_item(tree, hf_bmc_length_of_serial_number_list, tvb, offset, 1, ENC_BIG_ENDIAN);
            offset += 1;

            for (entry=0; entry<length_of_serial_number_list; entry++) {
                proto_tree_add_item(tree, hf_bmc_serial_number, tvb, offset, 2, ENC_BIG_ENDIAN);
                offset += 2;

                proto_tree_add_item(tree, hf_bmc_ctch_bs_index, tvb, offset, 1, ENC_BIG_ENDIAN);
                offset += 1;
            }
        }
    }

    return offset;
}

static int
dissect_bmc_cbs41_message(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree)
{
    int offset=1;

    proto_tree_add_item(tree, hf_bmc_broadcast_address, tvb, offset, 5, ENC_NA);
    offset += 5;

    proto_tree_add_item(tree, hf_bmc_cb_data41, tvb, offset, tvb_reported_length_remaining(tvb,offset), ENC_NA);
    offset = tvb_reported_length(tvb);

    return offset;
}

void
proto_register_bmc(void)
{
    static hf_register_info hf[] = {
        { &hf_bmc_message_type,
            { "Message Type", "bmc.message_type",
            FT_UINT8, BASE_DEC, VALS(message_type_vals), 0,
            NULL, HFILL }
        },
        { &hf_bmc_message_id,
            { "Message ID", "bmc.message_id",
            FT_UINT16, BASE_HEX, NULL, 0,
            NULL, HFILL }
        },
        { &hf_bmc_serial_number,
            { "Serial Number", "bmc.serial_number",
            FT_UINT16, BASE_HEX, NULL, 0,
            NULL, HFILL }
        },
#if 0
        { &hf_bmc_data_coding_scheme,
            { "Data Coding Scheme", "bmc.data_coding_scheme",
            FT_UINT8, BASE_HEX, NULL, 0,
            NULL, HFILL }
        },
        { &hf_bmc_cb_data,
            { "CB Data", "bmc.cb_data",
            FT_BYTES, BASE_NONE, NULL, 0,
            NULL, HFILL }
        },
#endif
        { &hf_bmc_offset_to_begin_ctch_bs_index,
            { "Offset to Begin CTCH Block Set Index", "bmc.offset_to_begin_ctch_bs_index",
            FT_UINT8, BASE_DEC, NULL, 0,
            NULL, HFILL }
        },
        { &hf_bmc_length_of_cbs_schedule_period,
            { "Length of CBS Schedule Period", "bmc.length_of_cbs_schedule_period",
            FT_UINT8, BASE_DEC, NULL, 0,
            NULL, HFILL }
        },
        { &hf_bmc_new_message_bitmap,
            { "New Message Bitmap", "bmc.new_message_bitmap",
            FT_BYTES, BASE_NONE, NULL, 0,
            NULL, HFILL }
        },
        { &hf_bmc_message_description_type,
            { "Message Description Type", "bmc.message_description_type",
            FT_UINT8, BASE_DEC, VALS(message_description_type_vals), 0,
            NULL, HFILL }
        },
        { &hf_bmc_offset_to_ctch_bs_index_of_first_transmission,
            { "Offset to CTCH BS index of first transmission", "bmc.offset_to_ctch_bs_index_of_first_transmission",
            FT_UINT8, BASE_DEC, NULL, 0,
            NULL, HFILL }
        },
        { &hf_bmc_broadcast_address,
            { "Broadcast Address", "bmc.broadcast_address",
            FT_BYTES, BASE_NONE, NULL, 0,
            NULL, HFILL }
        },
        { &hf_bmc_cb_data41,
            { "CB Data41", "bmc.cb_data41",
            FT_BYTES, BASE_NONE, NULL, 0,
            NULL, HFILL }
        },
        { &hf_bmc_future_extension_bitmap,
            { "Future Extension Bitmap", "bmc.future_extension_bitmap",
            FT_UINT8, BASE_DEC, NULL, 0,
            NULL, HFILL }
        },
        { &hf_bmc_length_of_serial_number_list,
            { "Length of Serial Number List", "bmc.length_of_serial_number_list",
            FT_UINT8, BASE_DEC, NULL, 0,
            NULL, HFILL }
        },
        { &hf_bmc_ctch_bs_index,
            { "CTCH BS Index", "bmc.ctch_bs_index",
            FT_UINT8, BASE_DEC, NULL, 0,
            NULL, HFILL }
        }
    };

    static int *ett[] = {
        &ett_bmc,
        &ett_bmc_message_description
    };

    proto_bmc = proto_register_protocol("Broadcast/Multicast Control", "BMC", "bmc");
    register_dissector("bmc", dissect_bmc, proto_bmc);

    proto_register_field_array(proto_bmc, hf, array_length(hf));
    proto_register_subtree_array(ett, array_length(ett));
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 4
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=4 tabstop=8 expandtab:
 * :indentSize=4:tabSize=8:noTabs=true:
 */
