/*
    XCruise - A directory browser
    Copyright (C) 1999  Yusuke Shinyama <euske@cl.cs.titech.ac.jp>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 *  zone.c
 *	Manage a directory tree
 */


#include "xcruise.h"

#include <sys/stat.h>
#include <dirent.h>

/* mysterious constants */
#define FileUnitSize	4096
#define WormHoleR	0.5
#define ChildMaxR	0.9
#define RotDiff1	0.4
#define GalaxyR		1.0
#define SwitchSpeedRate	2.0

/* #define DEBUG_DIR */


/*  Scan a directory
 */
static int chaselink(char* path, char* linkpath)
{
    int len;
#ifdef DEBUG_DIR
    fprintf(stderr, "link: %s\n", path);
#endif
    if ((len = readlink(path, linkpath, PathLen-1)) < 0)
	return(-1);
    linkpath[len] = '\0';
    return(0);
}

static void getDirectory(galaxyinfo* g)
{
    DIR* dirp;
    static star starbuff[MaxStars];
    
    g->nst = 0;
    g->children = NULL;
    
    if (g->info.stat == S_OK && (dirp = opendir(g->path))) {
	star* s1 = starbuff;
	struct dirent* ent;
	/* read directory */
	while(ent = readdir(dirp)) {
	    char path1[PathLen];
	    struct stat buf;

	    /* star's name */
	    if (FileNameLen-1 < strlen(ent->d_name))
		ent->d_name[FileNameLen-1] = '\0';
	    strcpy(s1->i.name, ent->d_name);
	    sprintf(path1, "%s%s", g->path, s1->i.name);
	    if (ent->d_name[0] == '.') {
		if (!viewallflag ||
		    !strcmp (ent->d_name,".") ||
		    !strcmp (ent->d_name,".."))
		    continue;
	    }
	    
	    lstat(path1, &buf);
#ifdef DEBUG_DIR
	    fprintf(stderr, "file: %s: %x\n", path1, buf.st_mode);
#endif
	    if ((buf.st_mode & S_IFLNK) == S_IFLNK) {
		/* wormhole */
		s1->i.type = T_Wormhole;
		s1->i.stat = S_OK;
		if (!chaselink(path1, s1->w.linkpath)) {
		    g->nst++, s1++;
		}
	    } else if ((buf.st_mode & S_IFDIR) == S_IFDIR) {
		/* galaxy */
		s1->i.type = T_Galaxy;
	    	s1->i.stat = ((buf.st_mode & S_IROTH) && 
			      (buf.st_mode & S_IXOTH))?
		    S_OK : S_NoRead;
		s1->g.expanded = 0;
		g->nst++, s1++;
	    } else if ((buf.st_mode & S_IFREG) == S_IFREG &&
		       (buf.st_mode & S_IFBLK) != S_IFBLK &&
		       (buf.st_mode & S_IFCHR) != S_IFCHR) {
		/* planet */
		if (viewallflag || (star*)g != &universe) {
		    s1->i.type = T_Planet;
		    s1->i.stat = (buf.st_mode & S_IROTH)? S_OK : S_NoRead;
		    s1->p.size = buf.st_size / FileUnitSize;
		    g->nst++, s1++;
		}
	    }
	    
	    if (MaxStars <= g->nst) {
		fprintf(stderr, "getDirectory: too many stars: %d in %s\n",
			g->nst, path1);
		g->nst = MaxStars-1;
		break;
	    }
	}
	closedir(dirp);
    }
    
#ifdef DEBUG
    fprintf(stderr, "scandir: %s (%d files)\n", g->path, g->nst);
#endif
    if (g->nst) {
	g->children = (star*)xalloc(sizeof(star) * g->nst);
	memcpy((char*)g->children,
	       (char*)starbuff, 
	       sizeof(star) * g->nst);
    }
}

/*  Crop the part of a path name
 */
static char* get1word(char* wrd, char* path)
{
    if (*path == '/')
	path++;
    if (*path == '\0')
	return(NULL);
    while(*path && *path != '/')
	*wrd++ = *path++;
    *wrd = 0;
    return(path);
}

/*  Place a star
 */

/* calc the galaxy direction from a name */
static void calcNamePos(Point3D* p, char* name, double r)
{
    int			i, len = strlen(name);
    int			rot1 = 0, rot2 = 0;
    double		r1, r2;
    
    for(i = 0; i < len; i++)
	if (i % 2)
	    rot2 += *name++;
	else
	    rot1 += *name++;
    r1 = rot1 * 0.01;
    r2 = rot2 * 0.01;
    p->x =-r * sin(r1) * cos(r2);
    p->y = r * cos(r1) * cos(r2);
    p->z = r * sin(r2);
}

/* calc the star position from a name */
static int calcP1Pos(star* stp, Point3D* px, Point3D* py)
{
    char* s;
    int   i, len = strlen(stp->i.name);
    int   r0;
    double r1, r2, r;
    
    if (stp->i.type == T_Galaxy) {
	calcNamePos(&stp->i.px, stp->i.name, 1.0);
	stp->i.py.x = stp->i.px.z - stp->i.px.y;
	stp->i.py.y = stp->i.px.x - stp->i.px.z;
	stp->i.py.z = stp->i.px.y - stp->i.px.x;
	formalize(stp->i.py);
    }

    for(s = stp->i.name, r1 = 0, r = 0.1, i = 0;
	i < len; i += 2, s += 2, r *= RotDiff1) {
	r1 += (*s & 0x3F) * r;
    }
    for(s = stp->i.name+1, r2 = 0, r = 0.1, i = 1;
	i < len; i += 2, s += 2, r *= RotDiff1) {
	r2 += (*s & 0x3F) * r;
    }

    if (10 < len) len = 10;
    r0 = (stp->i.name[0] % 0xF) + len*3 - 5;
    stp->i.pos.x = (double)-r0 * sin(r1) * cos(r2);
    stp->i.pos.y = (double) r0 * cos(r1) * cos(r2);
    stp->i.pos.z = (double) r0 * sin(r2);

    return(r0);
}


/*  Expand a galaxy
 */
void expandGalaxy(galaxyinfo* g)
{
    int		i, r0, rmax = 1;
    double	r1;
    star*	s1;
    double	r;
    Point3D	pos;
    
    g->expanded = 1;
    g->marked = 0;
    
    if ((star*)g == &universe) {
	strcpy(g->path, "/");
	g->info.parent = NULL;
    } else {
	sprintf(g->path, "%s%s/", g->info.parent->g.path, g->info.name);
    }
    
    alert("Scanning...", -1);
    alert(g->path, 1);
    
#ifdef DEBUG
    fprintf(stderr, "expand: %s\n", g->path);
#endif

    getDirectory(g);
    
    pos = g->info.pos;
    r = g->info.r;
    if (g->nst) {
	g->view = (ZStar*)xalloc(sizeof(ZStar) * g->nst);
    } else {
	g->view = NULL;
    }
    
    s1 = g->children;
    for(i = 0; i < g->nst; i++, s1++) {
	/* place a child */
	s1->i.parent = (star*)g;
	switch(s1->i.type) {
	case T_Planet:
	    s1->i.r = (curt(s1->p.size)+1.0) * r / CurrentSize;
	    s1->i.col = (s1->i.stat == S_OK)?
		((unsigned char)s1->i.name[1] % PlanetColors) +1 :
		    PlanetColors+1;
	    break;

	case T_Galaxy:
	    s1->i.r = (curt(s1->g.nst)+1.0) * GalaxyR * r / CurrentSize;
	    s1->i.col = (s1->i.stat == S_OK)?
		((unsigned char)s1->i.name[1] % GalaxyColors)+
		NoReadPlanet+PlanetColors+1 :
		GalaxyColors+NoReadPlanet+PlanetColors+1;
	    s1->g.marked = 0;
	    break;

	case T_Wormhole:
	    s1->i.r = WormHoleR * r / CurrentSize;
	    s1->i.col = (s1->i.stat == S_OK)?
		((unsigned char)s1->i.name[1] % WormholeColors) +
		NoReadGalaxy+GalaxyColors+NoReadPlanet+PlanetColors+1 :
		WormholeColors+NoReadGalaxy+GalaxyColors+
		NoReadPlanet+PlanetColors+1;
	    break;
	}
	r0 = calcP1Pos(s1, &g->info.px, &g->info.py);
	rmax = (rmax < r0)? r0 : rmax;
    }
    
    r1 = ChildMaxR * r / rmax;
    s1 = g->children;
    for(i = 0; i < g->nst; i++, s1++) {
	s1->i.pos.x = s1->i.pos.x * r1 + pos.x;
	s1->i.pos.y = s1->i.pos.y * r1 + pos.y;
	s1->i.pos.z = s1->i.pos.z * r1 + pos.z;
    }

    lnupdate = true;
}


/*  Collapse a galaxy
 */
void collapseGalaxy(galaxyinfo* g)
{
    star* s1;
    int	i;
    
    if (g->expanded) {
#ifdef DEBUG
	fprintf(stderr, "collapse: %s\n", g->path);
#endif	
	g->expanded = 0;
	s1 = g->children;
	for(i = 0; i < g->nst; i++, s1++) {
	    if (s1->i.type == T_Galaxy)
		collapseGalaxy(&s1->g);
	}
	if (g->children) {
	    xfree(g->children);
	}
	if (g->view) {
	    xfree(g->view);
	}

	lnupdate = true;
    }
}


/*  Collapse galaxies which have been marked (recursively)
 */
void cleanupGalaxy(galaxyinfo* g)
{
    if (g->expanded) {
	if (g->marked) {
	    g->marked = 0;
	    collapseGalaxy(g);
	} else {
	    star* s1 = g->children;
	    int i;
	    for(i = 0; i < g->nst; i++, s1++) {
		if (s1->i.type == T_Galaxy)
		    cleanupGalaxy(&s1->g);
	    }
	}
    }
}


/*  Enter/Leave a galaxy
 */
static void switchIn0(star* s, double scal, Point3D* p0)
{
    s->i.pos.x = (s->i.pos.x - p0->x) * scal;
    s->i.pos.y = (s->i.pos.y - p0->y) * scal;
    s->i.pos.z = (s->i.pos.z - p0->z) * scal;
    s->i.r *= scal;
    
    if (s->i.type == T_Galaxy && 
	s->g.expanded &&
	s->g.children) {
	int i;
	star* s1 = s->g.children;
	for(i = 0; i < s->g.nst; i++, s1++)
	    switchIn0(s1, scal, p0);
    }
}

static void switchOut0(star* s, double scal, Point3D* p0)
{
    s->i.pos.x = s->i.pos.x * scal + p0->x;
    s->i.pos.y = s->i.pos.y * scal + p0->y;
    s->i.pos.z = s->i.pos.z * scal + p0->z;
    s->i.r *= scal;
    
    if (s->i.type == T_Galaxy && 
	s->g.expanded &&
	s->g.children) {
	int i;
	star* s1 = s->g.children;
	for(i = 0; i < s->g.nst; i++, s1++)
	    switchOut0(s1, scal, p0);
    }
}

static void switchInZone(star* s)
{
    Point3D	p0 = (s->g.oldpos = s->i.pos);	
    double	scal = CurrentSize / (s->g.oldr = s->i.r);
    viewp.x = (viewp.x - p0.x) * scal;
    viewp.y = (viewp.y - p0.y) * scal;
    viewp.z = (viewp.z - p0.z) * scal;
    vspeed *= (scal/SwitchSpeedRate);

    switchIn0(&universe, scal, &p0);
    curzone = s;
}

static void switchOutZone(void)
{
    Point3D	p0 = curzone->g.oldpos;
    double	scal = curzone->g.oldr / CurrentSize;
    viewp.x = viewp.x * scal + p0.x;
    viewp.y = viewp.y * scal + p0.y;
    viewp.z = viewp.z * scal + p0.z;
    vspeed *= scal/(SwitchSpeedRate);
    
    switchOut0(&universe, scal, &p0);
    curzone = curzone->i.parent;
}


/*  Check switch
 */
void checkSwitch(void)
{
    if (stpswin) {
	switchInZone(stpswin);
	lnupdate = true;
    } else if (curzone != &universe && CurrentSize < vlength(viewp)) {
	switchOutZone();
	lnupdate = true;
    }
}


/*  Update wormholes
 */
static int updateLink1(wormholeinfo* w)
{
    char	file[FileNameLen];
    star*	nearest;
    char*	sp = w->linkpath;
    
    Point3D* pts;
    Point3D	a, b, v0, v1, difc;
    double	t, dist;
    int		i;

    if (sp[0] == '/') {
	nearest = &universe;
    } else {
	nearest = w->info.parent;
    }
    
    /* search the nearest star */
    while((sp = get1word(file, sp)) &&
	  nearest->i.type == T_Galaxy &&
	  nearest->g.expanded) {
	if (!strcmp(file, ".")) {
	    ;
	} else if (!strcmp(file, "..") && nearest->i.parent) {
	    nearest = nearest->i.parent;
	} else {
	    star* s1 = nearest->g.children;
	    for(i = 0; i < nearest->g.nst; i++, s1++) {
		if (!strcmp(file, s1->i.name)) {
		    nearest = s1;
		    break;
		}
	    }
	}
    }
    
    if (nearest != &universe) {
	/* calc a bezier curve */
#ifdef DEBUG
	fprintf(stderr, "update-link: %s->%s\n",
		w->linkpath, nearest->i.name);
#endif
	if (numlinks < MaxLinks) {
	    curlinks[numlinks++] = w;
	}
	
	difc.x = nearest->i.pos.x - w->info.pos.x;
	difc.y = nearest->i.pos.y - w->info.pos.y;
	difc.z = nearest->i.pos.z - w->info.pos.z;
	dist = vlength(difc)/2;
	
	calcNamePos(&v0, w->info.name, dist);
	calcNamePos(&v1, w->linkpath, dist);
	
	a.x =-2*(nearest->i.pos.x - w->info.pos.x) + v0.x + v1.x;
	a.y =-2*(nearest->i.pos.y - w->info.pos.y) + v0.y + v1.y;
	a.z =-2*(nearest->i.pos.z - w->info.pos.z) + v0.z + v1.z;
	b.x = 3*(nearest->i.pos.x - w->info.pos.x) - 2*v0.x - v1.x;
	b.y = 3*(nearest->i.pos.y - w->info.pos.y) - 2*v0.y - v1.y;
	b.z = 3*(nearest->i.pos.z - w->info.pos.z) - 2*v0.z - v1.z;
	
	pts = w->wpt;
	for(i = 1; i < WormPoints-1; i++, pts++) {
	    t = (double)i / (double)WormPoints;
	    pts->x = ((a.x*t + b.x)*t + v0.x)*t + w->info.pos.x;
	    pts->y = ((a.y*t + b.y)*t + v0.y)*t + w->info.pos.y;
	    pts->z = ((a.z*t + b.z)*t + v0.z)*t + w->info.pos.z;
	}
	pts->x = nearest->i.pos.x;
	pts->y = nearest->i.pos.y;
	pts->z = nearest->i.pos.z;
    }

    return(0);
}

/*  Update wormholes in a galaxy (recursively)
 */
static void updateGalaxyLinks(star* s)
{
    star* s1 = s->g.children;
    int i;
    if (s->g.expanded && s1) {
	for(i = 0; i < s->g.nst; i++, s1++) {
	    switch(s1->i.type) {
	    case T_Galaxy:
		updateGalaxyLinks(s1);
		break;
	    case T_Wormhole:
		updateLink1(&s1->w);
		break;
	    }
	}
    }
}

void updateLink(void)
{
    numlinks = 0;
    updateGalaxyLinks(&universe);
}
