/*  xhangglider
 *  Copyright (C) 1999 Yasuhiro Take
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#define STR_VERSION "0.92e"
#define STR_DATE "Sun Jul 25 1999"

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>

#include "def.h"
#include "xhang.h"

extern PolygonData polygons_load, polygons_orig[], polygons_shown;
extern char datadir[];
extern char backcolor[];
extern int pilot_num, thermal_num;

extern double relative_speed[8];
extern char nobankflag[8];
extern double max_y, viewpoint_z;

extern char Transient, Vario, DrawingMode;
extern char Pylon, Verbose;
extern float Frequency;

extern int scr_width, scr_height, scr_x, scr_y;
extern double thermal_x[5], thermal_y[5], pylon_x[5], pylon_y[5];

int into_data_dir()
{
  int i = 0;

  if (datadir[0] == '~') {
    chdir(getenv("HOME"));
    if (strlen(datadir) > 2)
      i = chdir(&(datadir[2]));
  } else {
    i = chdir(datadir);
  }
  
  if (i) {
    fprintf(stderr, "polygon data path %s not exist.\n", datadir);
    fprintf(stderr, "To specify the path, describe it in the deffile like\n");
    fprintf(stderr, "PolygonDataPath /usr/share/xhangglider\n\n");
  }

  return 0;
}

int load_file(char *fname, PolygonData *pd)
{
  int ret, i;
  FILE *fp;
  char name[20], colorname[20];
  float x[3], y[3], z[3];
  Polygon *pol;

  if (pd->pol)
    memset(pd->pol, 0, sizeof(Polygon) * pd->n_pol);
  
  if ((fp = fopen(fname, "r")) != NULL) {
    for (;;) {
      ret = fscanf(fp, "%s %s %f %f %f %f %f %f %f %f %f\n", name, colorname,
		   &(x[0]), &(y[0]), &(z[0]),
		   &(x[1]), &(y[1]), &(z[1]),
		   &(x[2]), &(y[2]), &(z[2]));

      pol = new_polygon(pd);
      strcpy(pol->name, name);
      strcpy(pol->color.name, colorname);
      for (i = 0; i < 3; i++) {
	pol->vertix[0][i] = x[i];
	pol->vertix[1][i] = y[i];
	pol->vertix[2][i] = z[i];
      }
      
      if (ret == EOF)
	break;
    }

    fclose(fp);
  } else {
    fprintf(stderr, "Polygon data file %s not exist in %s abort...\n",fname,
	    datadir);
    exit(1);
  }

  return 0;
}

int optimize(PolygonData *pd)
{
  double same_x[2], same_y[2], same_z[2];
  double ns_x[2], ns_y[2], ns_z[2];
  int same, ns, i, j, k, slen;

  for (i = 0; i < pd->n_pol; i++) {
    slen = strlen(pd->pol[i].name);

    if (pd->pol[i].name[0] == '-')
      pd->pol[i].type = T_LINE;
    if (pd->pol[i].name[0] == '*')
      pd->pol[i].type = T_BALL;

    if (pd->pol[i].name[slen - 1] == '1' && pd->pol[i].name[slen - 2] == '.'
	&& pd->pol[i].type != T_UNUSED) {
      same = ns = 0;
      for (j = 0; j < 3; j++) {
	for (k = 0; k < 3; k++) {
	  if (pd->pol[i].vertix[0][j] == pd->pol[i + 1].vertix[0][k] &&
	      pd->pol[i].vertix[1][j] == pd->pol[i + 1].vertix[1][k] &&
	      pd->pol[i].vertix[2][j] == pd->pol[i + 1].vertix[2][k]) {
	    same_x[same] = pd->pol[i].vertix[0][j];
	    same_y[same] = pd->pol[i].vertix[1][j];
	    same_z[same] = pd->pol[i].vertix[2][j];

	    same++;
	  }
	}
      }

      if (same < 2) {
	fprintf(stderr, "%s not same coord. aborting...\n", pd->pol[i].name);
	exit(1);
      }

      for (j = 0; j < 3; j++) {
	if ((pd->pol[i].vertix[0][j] == same_x[0] &&
	     pd->pol[i].vertix[1][j] == same_y[0] &&
	     pd->pol[i].vertix[2][j] == same_z[0]) ||
	    (pd->pol[i].vertix[0][j] == same_x[1] &&
	     pd->pol[i].vertix[1][j] == same_y[1] &&
	     pd->pol[i].vertix[2][j] == same_z[1]))
	  continue;

	ns_x[0] = pd->pol[i].vertix[0][j];
	ns_y[0] = pd->pol[i].vertix[1][j];
	ns_z[0] = pd->pol[i].vertix[2][j];
      }

      for (j = 0; j < 3; j++) {
	if ((pd->pol[i + 1].vertix[0][j] == same_x[0] &&
	     pd->pol[i + 1].vertix[1][j] == same_y[0] &&
	     pd->pol[i + 1].vertix[2][j] == same_z[0]) ||
	    (pd->pol[i + 1].vertix[0][j] == same_x[1] &&
	     pd->pol[i + 1].vertix[1][j] == same_y[1] &&
	     pd->pol[i + 1].vertix[2][j] == same_z[1]))
	  continue;

	ns_x[1] = pd->pol[i + 1].vertix[0][j];
	ns_y[1] = pd->pol[i + 1].vertix[1][j];
	ns_z[1] = pd->pol[i + 1].vertix[2][j];
      }

      for (j = 0; j < 2; j++) {
	pd->pol[i].vertix[0][j * 2] = same_x[j];
	pd->pol[i].vertix[1][j * 2] = same_y[j];
	pd->pol[i].vertix[2][j * 2] = same_z[j];

	pd->pol[i].vertix[0][j * 2 + 1] = ns_x[j];
	pd->pol[i].vertix[1][j * 2 + 1] = ns_y[j];
	pd->pol[i].vertix[2][j * 2 + 1] = ns_z[j];
      }

      pd->pol[i + 1].type = T_UNUSED;
      pd->pol[i].type = T_QUAD;
      pd->pol[i].name[slen - 1] = '\0';
    }
  }

  return 0;
}

char *get_nth_arg(char *in, char *out, int n)
{
  char *p, *p2;
  int i;
  
  p = in;
  i = 0;
  
  while (*p) {
    if (isspace(*p) == 0) {
      p2 = out;
      while (isspace(*p) == 0)
	*p2++ = *p++;
      *p2 = '\0';
      i++;
      if (i == n)
	return out;
    }

    p++;
  }

  out[0] ='\0';
  return NULL;
}

int atoi_range(char *str, int min, int max)
{
  int i;

  i = atoi(str);
  if (i < min)
    i = min;
  if (i > max)
    i = max;
  return i;
}

int parse_deffile_glider(FILE *fp, int n, int line)
{
  char buf[256], buf2[256], pattern[256], color[256];
  int fileloadedflag = 0;
  int mode = 0;
  int i;
  PolygonData *pol;

  pol = &polygons_load;
  
  while (feof(fp) == 0) {
    if (fgets(buf, 256, fp)) {
      line++;
      
      if (buf[0] != '#') {
	get_nth_arg(buf, buf2, 1);

	if (strcmp(buf2, "File") == 0) {
	  if (get_nth_arg(buf, buf2, 2)) {
	    into_data_dir();
	    load_file(buf2, pol);
	    fileloadedflag = 1;
	  } else {
	    fprintf(stderr, "in deffile, line %d: No file name.\n", line);
	    exit(1);
	  }
	} else if (strcmp(buf2, "RelativeSpeed") == 0) {
	  if (get_nth_arg(buf, buf2, 2)) {
	    relative_speed[n] = atof(buf2);
	  }
	} else if (strcmp(buf2, "NoBank") == 0) {
	  nobankflag[n] = 1;
	} else if (fileloadedflag) {
	  if (strcmp(buf2, "Optimize") == 0)
	    optimize(pol);
	  else if (strcmp(buf2, "ChangeColor") == 0) {
	    get_nth_arg(buf, buf2, 2);

	    if (strcmp(buf2, "-exact") == 0)
	      mode = 1;
	    else if (strcmp(buf2, "-substr") == 0)
	      mode = 2;
	    else
	      fprintf(stderr, "in deffile, line %d: Invalid search mode.\n",
		      line);

	    if (get_nth_arg(buf, pattern, 3) && get_nth_arg(buf, color, 4)) {
	      for (i = 0; i < pol->n_pol; i++) {
		if ((mode == 1 && strcmp(pattern, pol->pol[i].name) == 0) ||
		    (mode == 2 && strstr(pol->pol[i].name, pattern) != NULL)) {
		  strncpy(pol->pol[i].color.name, color, 20);
		  pol->pol[i].color.name[19] = '\0';
		}
	      }
	    } else {
	      fprintf(stderr, "in defflie, line %d: too few arguments.\n",
		      line);
	    }
	  } else if (strcmp(buf2, "End") == 0) {
	    copy_polygons(pol, &(polygons_orig[n]), COPY_EACH);
	    return line;
	  }
	}
      }
    }
  }

  fprintf(stderr, "in deffile: reached end of file before 'End'.\n");
  exit(1);
  
  return -1;
}
	    
int parse_deffile_toplevel(FILE *fp)
{
  char buf[256], buf2[256];
  int line = 0, i;

  while (feof(fp) == 0) {
    if (fgets(buf, 256, fp)) {
      line++;
      
      if (buf[0] != '#') {
	get_nth_arg(buf, buf2, 1);
	
	if (strcmp(buf2, "Begin") == 0) {
	  if (get_nth_arg(buf, buf2, 2) == NULL)
	    fprintf(stderr, "in deffile, line %d: no glider number.\n", line);
	  else {
	    i = atoi(buf2);
	    if (i < 0 || i > 7)
	      fprintf(stderr, "in deffile, line %d: invalid glider number.\n",
		      line);
	    else
	      line = parse_deffile_glider(fp, i, line);
	  }
	} else if (strcmp(buf2, "PolygonDataPath") == 0) {
	  if (get_nth_arg(buf, buf2, 2) != NULL)
	    strncpy(datadir, buf2, 255);
	} else if (strcmp(buf2, "BackgroundColor") == 0) {
	  if (get_nth_arg(buf, buf2, 2) != NULL)
	    strncpy(backcolor, buf2, 79);
	} else if (strcmp(buf2, "DisplayArea") == 0) {
	  if (get_nth_arg(buf, buf2, 5) != NULL) {
	    scr_height = atoi(buf2);
	    scr_width = atoi(get_nth_arg(buf, buf2, 4));
	    scr_y = atoi(get_nth_arg(buf, buf2, 3));
	    scr_x = atoi(get_nth_arg(buf, buf2, 2));
	  }
	} else if (strcmp(buf2, "NumOfPilots") == 0) {
	  if (get_nth_arg(buf, buf2, 2) != NULL)
	    pilot_num = atoi_range(buf2, 1, 8);
	} else if (strcmp(buf2, "NumOfThermal") == 0) {
	  if (get_nth_arg(buf, buf2, 2) != NULL)
	    thermal_num = atoi_range(buf2, 0, 5);
	} else if (strcmp(buf2, "UpdateFrequency") == 0) {
	  if (get_nth_arg(buf, buf2, 2) != NULL)
	    Frequency = atof(buf2);
	} else if (strcmp(buf2, "MaxY") == 0) {
	  if (get_nth_arg(buf, buf2, 2) != NULL)
	    max_y = atof(buf2);
	} else if (strcmp(buf2, "ViewpointZ") == 0) {
	  if (get_nth_arg(buf, buf2, 2) != NULL)
	    viewpoint_z = atof(buf2);
	} else if (strcmp(buf2, "NoFlicker") == 0)
	  DrawingMode = DRAW_NOFLICKER;
	else if (strcmp(buf2, "UseClipMask") == 0)
	  DrawingMode = DRAW_USECLIP;
	else if (strcmp(buf2, "Transient") == 0)
	  Transient = 1;
	else if (strcmp(buf2, "ShowVariometer") == 0)
	  Vario = 1;
	else if (strcmp(buf2, "PylonRace") == 0)
	  Pylon = 1;
	else if (strcmp(buf2, "NoPylon") == 0)
	  fprintf(stderr, "NoPylon is now invalid.\n");
      }
    }
  }

  return 0;
}
	
int read_deffile(char *fname)
{
  FILE *fp = NULL;
  char *home, *cwd;
  
  cwd = getcwd(NULL, 0);

  if (fname) {
    if ((fp = fopen(fname, "r")) == NULL) {
      fprintf(stderr, "The deffile %s not exist. Abort.\n", fname);
      exit(1);
    }
  } else {
    home = getenv("HOME");
    chdir(home);
    
    if ((fp = fopen(".xhangglider", "r")) == NULL) {
      into_data_dir();
      if ((fp = fopen("glider.def", "r")) == NULL) {
	fprintf(stderr, "No deffile. Abort...\n");
	exit(1);
      }
    }
  }

  chdir(cwd);
  free(cwd);

  parse_deffile_toplevel(fp);
  fclose(fp);
  
  return 0;
}

int init()
{
  int i;
  
  srand((unsigned int)time(0));

  for (i = 0; i < thermal_num; i++) {
    thermal_x[i] = rand() % 600 - 300;
    thermal_y[i] = rand() % ((int)max_y - 50) + 50;
  }

  pylon_x[0] = -180; pylon_y[0] = 200;
  pylon_x[1] = 280; pylon_y[1] = 300;
  pylon_x[2] = -50; pylon_y[2] = 60;
  pylon_x[3] = 90; pylon_y[3] = 100;
  pylon_x[4] = 0; pylon_y[4] = 150;

  return 0;
}

int parse_option(int argc, char **argv)
{
  int i;

  for (i = 1; i < argc; i++) {
    if (strcmp(argv[i], "-x") == 0 && i + 1 < argc)
      scr_x = atoi(argv[++i]);
    else if (strcmp(argv[i], "-y") == 0 && i + 1 < argc)
      scr_y = atoi(argv[++i]);
    else if (strcmp(argv[i], "-width") == 0 && i + 1 < argc)
      scr_width = atoi(argv[++i]);
    else if (strcmp(argv[i], "-height") == 0 && i + 1 < argc)
      scr_height = atoi(argv[++i]);
    else if (strcmp(argv[i], "-pilot") == 0 && i + 1 < argc)
      pilot_num = atoi_range(argv[++i], 1, 8);
    else if (strcmp(argv[i], "-thermal") == 0 && i + 1 < argc)
      thermal_num = atoi_range(argv[++i], 0, 5);
    else if (strcmp(argv[i], "-frequency") == 0 && i + 1 < argc)
      Frequency = atof(argv[++i]);
    else if (strcmp(argv[i], "-maxy") == 0 && i + 1 < argc)
      max_y = atof(argv[++i]);
    else if (strcmp(argv[i], "-viewpointz") == 0 && i + 1 < argc)
      viewpoint_z = atof(argv[++i]);
    else if (strcmp(argv[i], "-noflicker") == 0)
      DrawingMode = DRAW_NOFLICKER;
    else if (strcmp(argv[i], "-trans") == 0)
      Transient = 1;
    else if (strcmp(argv[i], "-useclip") == 0)
      DrawingMode = DRAW_USECLIP;
    else if (strcmp(argv[i], "-vario") == 0)
      Vario = 1;
    else if (strcmp(argv[i], "-pylonrace") == 0)
      Pylon = 1;
    else if (strcmp(argv[i], "-color") == 0 && i + 1 < argc)
      strncpy(backcolor, argv[++i], 79);
  }

  return 0;
}
  
int print_options()
{
  fprintf(stderr, "xhangglider %s %s\n", STR_VERSION, STR_DATE);
  fprintf(stderr, "Usage: xhangglider [options...]\n");
  fprintf(stderr, "\t[-x int][-y int] ... origin coord of the display area\n");
  fprintf(stderr, "\t                     only valid with '-noflicker'\n");
  fprintf(stderr, "\t[-width int][-height int] ... size of the display area.\n");
  fprintf(stderr, "\t                          ... only valid with '-noflicker'\n");
  fprintf(stderr, "\t[-maxy double] ... max Y coord. (default 450.0)\n");
  fprintf(stderr, "\t        ... Y coord is the depth.\n");
  fprintf(stderr, "\t[-viewpointz double] ... Z coord of the viewpoint. (default 0)\n");
  fprintf(stderr, "\t              ... Z coord is the height.\n");
  fprintf(stderr, "\t[-pilot num] ... num of the pilot (default 1)\n");
  fprintf(stderr, "\t[-thermal num] ... num of thermals (default 5)\n");
  fprintf(stderr, "\t[-frequency sec] ... update frequency. (default 0.1)\n");
  fprintf(stderr, "\t[-noflicker] ... no flicker\n");
  fprintf(stderr, "\t[-useclip] ... use clipmask\n");
  fprintf(stderr, "\t[-trans] ... transient\n");
  fprintf(stderr, "\t[-vario] ... show vario\n");
  fprintf(stderr, "\t[-nopylon] ... do not pylon-race\n");
  fprintf(stderr, "\t[-deffile file] ... definition file\n");
  fprintf(stderr, "\t[-color colorname] ... background color name (default DeepSkyBlue)\n");

  return 0;
}
