/* $Header: /home/yav/catty/xmagv/RCS/xmagv.c,v 1.11 1995/11/17 18:06:16 yav Exp $
 *
 * xmagv - X Window System MAG Viewer
 *
 * Author :
 *   yav (OeRSTED Inc., UHD98984@pcvan.or.jp)
 * How to make :
 *  % cc -o xmagv xmagv.c fblib.c error.c -lX11
 *  or
 *  % cc -I/usr/X11R6/include -L/usr/X11R6/lib -o xmagv xmagv.c fblib.c error.c -lX11
 * Special thanks :
 *   T.Ishii - X-Window MAG LOADER
 *   Woody RINN - MAG loader Specifications
 */

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>

#include "config.h"
#include "headers.h"
#include "fb.h"
#include "error.h"

#ifndef MAX_COLOR
#define MAX_COLOR 256		/* 256 or 16 */
#endif
#ifndef MAX_FILE_Y
#define MAX_FILE_Y 4096
#endif
#ifndef MAX_FILE_X
#define MAX_FILE_X 4096		/* 8 boundary */
#endif

#ifndef MAXPATH
#define MAXPATH 1024
#endif

#define BRIGHT_G 151		/* .59 * 256 */
#define BRIGHT_R  77		/* .30 * 256 */
#define BRIGHT_B  28		/* .11 * 256 */
#define TONE_100 257		/* 0xff(color max) * TONE_100
				 *   = 65535(X color max) */

#ifndef SCRDOTS
#define SCRDOTS 16		/* scroll dot */
#endif
#ifndef SCRTIMER
#define SCRTIMER 60
#endif
#ifndef SCRDIV
#define SCRDIV 4		/* scroll acceleration 1/SCRDIV dot */
#endif

#define DEFALT_SCROLL_SLEEP 573	/* millisec. root scroll sleep time default */

#ifndef ZPIPE
#define ZPIPE "gzip"
#endif

#define MAGSM_256COLOR	0x80
#define MAGSM_DIGITAL	0x04
#define MAGSM_8COLOR	0x02
#define MAGSM_ASPECT	0x01

typedef struct {
  unsigned char top;		/* padding , every $00 */
  unsigned char MachineCode;
  unsigned char MachineFlag;
  unsigned char ScreenMode;
  unsigned short xtop;
  unsigned short ytop;
  unsigned short xend;
  unsigned short yend;
  long flagAofs;
  long flagBofs;
  long flagBlen;
  long pixdataofs;
  long pixlen;
} MagFileHeader;		/* Header structure in MAG file */
#define MAGHDSIZE 32

typedef struct {
  MagFileHeader f;
  unsigned long flagAlen;
  int xsize, ysize, colsize;
  unsigned short *pix_buffer;
  unsigned pixbuf_xsize;
  unsigned char *flag_a, *flag_b, *pixel_data;
  char *image_data;		/* [x*y*Depth] ??? */
  FILE *fp;
  int wx, wy;			/* window pos */
  unsigned wxs, wys;		/* window size */
  int imgx, imgy;		/* image pos offset */
  int title_w, title_h;
  long fptr, hdofs;
  char filename[MAXPATH];	/* mag file name */
  char dispname[MAXPATH];	/* window title name */
  char comment[1024];
} MagWin;

static struct {int x,y;} postbl[16]={
  {0,0},{1,0},{2,0},{4,0},{0,1},{1,1},{0,2},{1,2},
  {2,2},{0,4},{1,4},{2,4},{0,8},{1,8},{2,8},{0,16}};

#define DUMP_OFF	0
#define DUMP_PIXEL	1
#define DUMP_BITMAP	2
#define DUMP_PPM	3
#define DUMP_PPM6	4

#define OPTDEFAULT (-1)

char xmagv_id[] = "$Id: xmagv.c,v 1.11 1995/11/17 18:06:16 yav Exp $";
char rcsrev[] = "$Revision: 1.11 $";
char *myname = NULL;
int argumentc;
char **argumentv;

XColor col[MAX_COLOR];		/* use [colsize] */
static unsigned char colaltbl[MAX_COLOR];
static int debug_f = 0;
static int quick_f = OPTDEFAULT;
static int gray_f = OPTDEFAULT;
static int info_f = 0;		/* print information only */
static int tone_v = 100;	/* 0 - 100 */
static int scroll_f = OPTDEFAULT;
static int root_f = 0;
static int quit_f = 0;
static int verbose_f = OPTDEFAULT;
static int dump_f = 0;		/* 0:Off 1:Pixel 2:Bitmap */
static int dump_width = 78;
static unsigned char bitmap_fg[MAX_COLOR];
#include "icon.xbm"
#define ICON_W icon_width
#define ICON_H icon_height
#define ICON_DATA icon_bits
static int internal_icon_w = ICON_W;
static int internal_icon_h = ICON_H;
static char *internal_icon_data = ICON_DATA;
static char *str_icon = NULL;	/* icon bitmap file name */
static char *str_fg = NULL, *str_bg = NULL;
static int pictnum, pictnummax, pictnumnext;
static int scracc = 0, scrdx = 0, scrdy = 0;
extern int imgx, imgy;
static char stdinfile[] = "-";
static int rootx = 0;		/* root offset-x */
static int rooty = 0;		/* root offset-y */
static int rootdx = -1;		/* root scroll x-dots */
static int rootdy = -1;		/* root scroll y-dots */
static int sleeptime = OPTDEFAULT;
static int nowait_f = 0;
static int rootscroll_f = 0;
static int cmap_mode = 1;

static char optusage[] = "\
  -display DISPLAY NAME\n\
  -geometry WIDTHxHEIGHT[{+-}XOFFSET{+-}YOFFSET]\n\
  -info\n\
       print information in MAG file\n\
  -quick\n\
       put image line by line\n\
  -gray\n\
       for gray scale monitor\n\
  -tone TONE\n\
       0% < TONE <= 100%\n\
  -scroll\n\
       mouse left button scroll\n\
  -root\n\
       set root window background pixmap\n";

void usage(fp)
     FILE *fp;
{
  char **p;

  fprintf(fp, "* X MAG Viewer %s *\n", rcsrev);
  fprintf(fp, "written by yav (UHD98984@pcvan.or.jp)\n");
  fprintf(fp, "usage : %s [options] filename[.mag]\n", myname);
  fprintf(fp, "option :\n%s", optusage);
  if (debug_f) {
    fprintf(fp, " %s %s\n", xmagv_id, rcsrev);
    putc('\n', fp);
    fprintf(fp, " max_color:%d, max_file_x:%d, max_file_y:%d\n",
	    MAX_COLOR, MAX_FILE_X, MAX_FILE_Y);
    fprintf(fp, " %s %s\n", fblib_id, fblibRCSrevision);
    for (p = fblibproginfo; *p != NULL; p++)
      fprintf(fp," %s", *p);
    putc('\n', fp);
  }
}

/* print usage to stdout and exit */
void f_usage()
{
  usage(stdout);
  exit(0);
}

char *getmyname(str)
     char *str;
{
  char *p;

  return ((p = strrchr(str, '/')) != NULL) ? p+1 : str;
}

char *getdispname(str)
     char *str;
{
  char *p, *p0;
  char buf[MAXPATH];
  
  strcpy(buf, str);
  p0 = getmyname(buf);
  if ((p = strrchr(p0, '.')) != NULL)
    *p = '\0';
  return strcpy(str, p0);
}

void disp_info(p)
     MagWin *p;
{
  printf("%-12s %4d x%4d %3d %s\n", getmyname(p->filename),
	 p->xsize, p->ysize, p->colsize, p->comment);
  if (debug_f) {
    printf(" header offset %3d\n", p->hdofs);
    printf(" top %d, %d\n", p->f.xtop, p->f.ytop);
    printf(" end %d, %d\n", p->f.xend, p->f.yend);
    printf(" machine $%02x, $%02x\n", p->f.MachineCode, p->f.MachineFlag);
    printf(" screen mode $%02x %s %s %s %s\n", p->f.ScreenMode,
	   p->f.ScreenMode & MAGSM_256COLOR ? "256col" : "",
	   p->f.ScreenMode & MAGSM_DIGITAL ? "digital" : "",
	   p->f.ScreenMode & MAGSM_8COLOR ? "8col" : "",
	   p->f.ScreenMode & MAGSM_ASPECT ? "aspect2" : "");
    printf(" flag A %10ld, %10ld\n", p->f.flagAofs, p->flagAlen);
    printf(" flag B %10ld, %10ld\n", p->f.flagBofs, p->f.flagBlen);
    printf(" pixel  %10ld, %10ld\n", p->f.pixdataofs, p->f.pixlen);
  }
}

#ifdef ZPIPE
int is_compressed_file(name)
     char *name;
{
  char *p;
  
  p = strrchr(name, '.');
  if (p != NULL) {
    if (!strcmp(p, ".Z") || !strcmp(p, ".z") || !strcmp(p, ".gz"))
      return 1;
  }
  return 0;
}
#endif /* ZPIPE */

/* open MAG file */
FILE *open_mag_file(p, str)
     MagWin *p; char *str;
{
  int i;
  int fd[2];
  
  if (!strcmp(str, stdinfile)) {
    p->fp = stdin;
    strcpy(p->filename, "<stdin>");
  } else {
    strcpy(p->filename, str);
#ifdef ZPIPE
    if (is_compressed_file(p->filename)) {
      pipe(fd);
      i = fork();
      if (i < 0) {
	error("cannot to fork child process!");
	return NULL;
      }
      if (i == 0) {
	close(1);
	dup(fd[1]);
	close(fd[1]);
	close(fd[0]);
	execlp(ZPIPE, "zcat", p->filename,(char *) 0);
	exit(0);
      }
      p->fp = fdopen(fd[0], "rb");
      close(fd[1]);
      *(strrchr(p->filename, '.')) = '\0';
    } else
#endif
      if ((p->fp = fopen(p->filename, "rb")) == NULL) {
	strcat(p->filename, ".mag");
	if ((p->fp = fopen(p->filename, "rb")) == NULL) {
	  strcpy(p->filename, str);
	  strcat(p->filename, ".MAG");
	  p->fp = fopen(p->filename, "rb");
	}
      }
  }
  strcpy(p->dispname, p->filename);
  getdispname(p->dispname);
  p->fptr = 0;
  return p->fp;
}

/* byte swap macros */
#define GETSHORT(p) ((*((p)+1)<<8)| *(p))
#define GETLONG(p) ((*((p)+3)<<24)|(*((p)+2)<<16)|(*((p)+1)<<8)| *(p))

#define HDERR_NOMAG 1
#define HDERR_NOSUP 2
#define HDERR_LARGE 3
int ReadHeader(fp, p)
     FILE *fp; MagWin *p;
{
  int c, i;
  char buf[MAGHDSIZE];
  unsigned char *pp;
  
  /* read ID */
  if ((fread(buf, 1, 8, fp) != 8) || strncmp(buf, "MAKI02  ", 8))
    return HDERR_NOMAG;
  p->fptr += 8;
  /* read comment */
  i = 0;
  while ((c=getc(fp)) != 0x1a) {
    p->fptr++;
    if (c == EOF)
      return HDERR_NOMAG;
    if (i < sizeof(p->comment))
      p->comment[i++] = c;
  }
  p->fptr++;
  if (i >= sizeof(p->comment))
    i = sizeof(p->comment) - 1;
  p->comment[i] = '\0';
  /* read header */
  p->hdofs = p->fptr;		/* store header top position */
  if (fread(buf, 1, MAGHDSIZE, fp) != MAGHDSIZE)
    return HDERR_NOMAG;
  p->fptr += MAGHDSIZE;
  pp = (unsigned char *)buf;
  p->f.top         = *pp++;
  p->f.MachineCode = *pp++;
  p->f.MachineFlag = *pp++;
  p->f.ScreenMode  = *pp++;
  p->f.xtop = GETSHORT(pp); pp += 2;
  p->f.ytop = GETSHORT(pp); pp += 2;
  p->f.xend = GETSHORT(pp); pp += 2;
  p->f.yend = GETSHORT(pp); pp += 2;
  p->f.flagAofs   = GETLONG(pp); pp += 4;
  p->f.flagBofs   = GETLONG(pp); pp += 4;
  p->f.flagBlen   = GETLONG(pp); pp += 4;
  p->f.pixdataofs = GETLONG(pp); pp += 4;
  p->f.pixlen     = GETLONG(pp); pp += 4;
  /* calc other parm */
  p->xsize = p->f.xend - p->f.xtop + 1;
  p->ysize = p->f.yend - p->f.ytop + 1;
  p->flagAlen = p->f.flagBofs - p->f.flagAofs;
  if (p->f.ScreenMode & MAGSM_256COLOR) {
    p->colsize = 256;
    p->pixbuf_xsize = MAX_FILE_X/2;
  } else {
    p->colsize = 16;
    p->pixbuf_xsize = MAX_FILE_X/4;
  }
  if (p->xsize > MAX_FILE_X || p->ysize > MAX_FILE_Y)
    return HDERR_LARGE;
  return 0;
}

int CheckPixlen(p)
     MagWin *p;
{
  if (p->fptr != p->hdofs + p->f.pixdataofs + p->f.pixlen)
    return error("``%s'' pixlen may be %ld not %ld!",
		 p->filename,
		 p->fptr - p->hdofs - p->f.pixdataofs,
		 p->f.pixlen);
  return 0;
}

typedef struct {unsigned char g,r,b;} MAKIPAL;
static MAKIPAL colbuf[MAX_COLOR];

int ReadColors(p)
     MagWin *p;
{
  int i;
  unsigned char rdbuf[3*MAX_COLOR], *pp;

  i = p->colsize * 3;
  if (fread(rdbuf, 1, i, p->fp) != i)
    return 1;
  p->fptr += i;
  pp = rdbuf;
  for (i = 0; i < p->colsize; i++) {
    colbuf[i].g = *pp++;
    colbuf[i].r = *pp++;
    colbuf[i].b = *pp++;
  }
  return 0;
}

/* return alloc miss count */
int AllocColors_sub(p)
     MagWin *p;
{
  int i, r;
  MAKIPAL *pp;
  Status s;
  unsigned tr,tg,tb;
  
  memset(colaltbl, 0, sizeof(colaltbl));
  pp = colbuf;
  if (gray_f) {
    tr = BRIGHT_R * tone_v;
    tg = BRIGHT_G * tone_v;
    tb = BRIGHT_B * tone_v;
  }
  r = 0;
  for (i = 0; i < p->colsize; i++) {
    if (gray_f) {
      col[i].green = col[i].red = col[i].blue =
	((pp->g*tg)>>8) + ((pp->r*tr)>>8) + ((pp->b*tb)>>8);
    } else {
      col[i].green = pp->g * tone_v;
      col[i].red = pp->r * tone_v;
      col[i].blue = pp->b * tone_v;
    }
    if (dump_f) {
      col[i].pixel = i;
    } else {
      s = XAllocColor(dsp, cmap, &col[i]);
      if (!s)
	r++;
      else
	colaltbl[i] = 1;
    }
    if (debug_f) {
      printf("%3d %d %3ld %04x %04x %04x\n",
	     i, s, col[i].pixel, col[i].red, col[i].green, col[i].blue);
    }
    pp++;
  }
  return r;
}

void FreeColors_sub(p)
     MagWin *p;
{
  int i,n;
  XPIX pxtbl[MAX_COLOR];

  n = 0;
  for (i = 0; i < p->colsize; i++) {
    if (colaltbl[i])
      pxtbl[n++] = col[i].pixel;
  }
  XFreeColors(dsp, cmap, pxtbl, n, 0);
}

static Colormap org_cmap;

void FreeColors(p)
     MagWin *p;
{
  FreeColors_sub(p);
  if (cmap_mode) {
    cmap = org_cmap;
    XSetWindowColormap(dsp, win, cmap);
  }
}

void AllocColors(p)
     MagWin *p;
{
  int i;
  
  if (cmap_mode)
    org_cmap = cmap;
  i = AllocColors_sub(p);
  if (!i)
    return;
  if (cmap_mode) {
    /* make new colormap */
    FreeColors_sub(p);
    cmap = XCopyColormapAndFree(dsp, cmap);
    XSetWindowColormap(dsp, win, cmap);
    i = AllocColors_sub(p);
  }
  if (i)
    error("alloc miss : %d colors.", i);
}

#ifndef SEEK_CUR
#define SEEK_CUR 1
#endif
/* seek MAG file
 * from header top
 */
int SeekMagFile(p, pos)
     MagWin *p; long pos;
{
  int i;
  long l;
  
  l = pos + p->hdofs - p->fptr;
  if (l < 0L) {
    if ((i = fseek(p->fp, l, SEEK_CUR)) == 0)
      p->fptr += l;
    return i;
  }
  /* dummy read seek for stdin (no use fseek) */
  while (l--) {
    if (getc(p->fp) == EOF)
      return -1;
    p->fptr++;
  }
  return 0;
}

int ReadMagFile(p)
     MagWin *p;
{
  /* read flag A */
  if (SeekMagFile(p, p->f.flagAofs))
    return 1;
  if (fread(p->flag_a, 1, p->flagAlen, p->fp) != p->flagAlen)
    return 1;
  p->fptr += p->flagAlen;
  /* read flag B (flag A len = flag B ofs - flag A ofs  ->  non seek) */
  if (fread(p->flag_b, 1, p->f.flagBlen, p->fp) != p->f.flagBlen)
    return 1;
  if (p->f.flagBlen & 1) {
    if (getc(p->fp) == EOF)
      return 1;
    p->fptr++;
  }
  p->fptr += p->f.flagBlen;
  /* read pixel data */
  if (SeekMagFile(p, p->f.pixdataofs))
    return 1;
  return 0;
}

int cancel_check()
{
  XEvent e;
  extern int keyboard();
  
  XCheckWindowEvent(dsp, win, ButtonPressMask|KeyPressMask, &e);
  switch (e.type) {
  case ButtonPress:
    if (e.xbutton.button == Button3)
      return 1;
    break;
  case KeyPress:
    if (keyboard(&e))
      return 1;
    break;
  }
  return 0;
}

void DumpColors(p)
     MagWin *p;
{
  int i;
  
  for (i = 0; i < p->colsize; i++)
    printf("# color: %2d %3d %3d %3d\n",
	   i, col[i].red>>8, col[i].green>>8, col[i].blue>>8);
}

PreDumpPixels(p)
     MagWin *p;
{
  int r;
  char *fn;

  fn = getmyname(p->filename);
  switch(dump_f) {
  case DUMP_PPM:
    printf("P3\n# %s %s %s\n", myname, fn, p->comment);
    DumpColors(p);
    printf("%d %d\n255\n", p->xsize, p->ysize);
    break;
  case DUMP_PPM6:
    printf("P6\n# %s %s %s\n", myname, fn, p->comment);
    DumpColors(p);
    printf("%d %d\n255\n", p->xsize, p->ysize);
    break;
  case DUMP_BITMAP:
    printf("/* %s %dx%d */\n", fn, p->xsize, p->ysize);
    printf("/*\n");
    DumpColors(p);
    printf("*/\n");
    printf("#define %s_width %d\n", p->dispname, p->xsize);
    printf("#define %s_height %d\n", p->dispname, p->ysize);
    printf("static char %s_bits[] = {\n", p->dispname);
    break;
  case DUMP_PIXEL:
    printf("# %s %dx%d %s\n", fn, p->xsize, p->ysize, p->comment);
    DumpColors(p);
    break;
  default:
    return 0;
  }
  return ferror(stdout);
}

PostDumpPixels(p)
     MagWin *p;
{
  int r;
  
  switch(dump_f) {
  case DUMP_BITMAP:
    r = puts("};") == EOF;
    break;
  default:
    r = 0;
    break;
  }
  return r;
}

DumpPpm6(wp, x, y, xs, n, buf, pp)
     MagWin *wp; char *buf; XPIX *pp;
{
  while (xs--) {
    if (fputc(col[*pp].red>>8, stdout) == EOF ||
	fputc(col[*pp].green>>8, stdout) == EOF ||
	fputc(col[*pp].blue>>8, stdout) == EOF)
      return 1;
    pp++;
  }
  return 0;
}

DumpPpm(wp, x, y, xs, n, buf, pp)
     MagWin *wp; char *buf; XPIX *pp;
{
  int i, cnt;
  char *p, *form;

  form = " %d %d %d";
  n /= 4*3;
  while (xs) {
    i = xs < n ? xs : n;
    p = buf;
    xs -= i;
    while (i--) {
      sprintf(p,form,col[*pp].red>>8,col[*pp].green>>8,col[*pp].blue>>8);
      p += strlen(p);
      pp++;
    }
    *p = '\0';
    if (puts(buf) == EOF)
      return 1;
  }
  return 0;
}

DumpPix(wp, x, y, xs, n, buf, pp)
     MagWin *wp; char *buf; XPIX *pp;
{
  int i, cnt;
  char *form, *p;
  
  form = (wp->colsize > 16) ? "%02x" : "%x";
  while (xs) {
    i = xs < n ? xs : n;
    p = buf;
    xs -= i;
    while (i--) {
      sprintf(p, form, *pp++);
      p += strlen(p);
    }
    *p = '\0';
    if (puts(buf) == EOF)
      return 1;
  }
  return 0;
}

#define isdumpbit1(c) bitmap_fg[c]

DumpBit(wp, x, y, xs, n, buf, pp)
     MagWin *wp; char *buf; XPIX *pp;
{
  int i, cnt;
  char *p;
  unsigned char d;
  static unsigned char bittbl[8] = {0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80};
  static int outf = 0;
  
  n &= ~7;
  while (xs) {
    i = xs < n ? xs : n;
    p = buf;
    xs -= i;
    d = cnt = 0;
    while (i--) {
      if (isdumpbit1(*pp))
	d |= bittbl[cnt];
      if (++cnt == 8) {
	sprintf(p, " 0x%02x,", d);
	p += strlen(p);
	d = cnt = 0;
      }
      pp++;
    }
    if (cnt) {
      sprintf(p, " 0x%02x,", d);
      p += strlen(p);
      cnt = 0;
    }
    *--p = '\0';
    if (outf) {
      if (fputs(",\n", stdout) == EOF)
	return 1;
    } else {
      outf = 1;
    }
    if (fputs(buf, stdout) == EOF)
      return 1;
  }
  return 0;
}

DumpPixels(wp, x, y, xs, pp)
     MagWin *wp; XPIX *pp;
{
  int n, r;
  char buf[MAX_FILE_X+2];
  
  if (wp->colsize > 16)
    n = (sizeof(buf)-2)/2;
  else
    n = sizeof(buf)-2;
  n = n > dump_width ? dump_width : n;
  switch(dump_f) {
  case DUMP_PPM:
    r = DumpPpm(wp, x, y, xs, n, buf, pp);
    break;
  case DUMP_PPM6:
    r = DumpPpm6(wp, x, y, xs, n, buf, pp);
    break;
  case DUMP_BITMAP:
    r = DumpBit(wp, x, y, xs, n, buf, pp);
    break;
  case DUMP_PIXEL:
    r = DumpPix(wp, x, y, xs, n, buf, pp);
    break;
  default:
    r = 0;
    break;
  }
  return r;
}

/* read pixel data from file */
int readpixel(p, px)
     MagWin *p; unsigned short *px;
{
  int c;
  
  if ((c = getc(p->fp)) == EOF)
    return 1;
  p->fptr++;
  *px = c << 8;
  if ((c = getc(p->fp)) == EOF)
    return 1;
  p->fptr++;
  *px |= c & 0xff;
  return 0;
}

int DecodePixGroup(p, pbuf, y, x2, f)
     MagWin *p; XPIX pbuf[];
{
  unsigned short px;
  
  if (f)
    px = p->pix_buffer[((y-postbl[f].y)%16)*p->pixbuf_xsize + x2-postbl[f].x];
  else
    if (readpixel(p, &px))
      return 1;			/* decode error */
  p->pix_buffer[(y%16)*p->pixbuf_xsize + x2] = px;
  if (p->colsize == 256) {
    pbuf[0] = col[(px>>8)&0xff].pixel;
    pbuf[1] = col[px&0xff].pixel;
  } else {
    pbuf[0] = col[(px>>12)&0xf].pixel;
    pbuf[1] = col[(px>>8)&0xf].pixel;
    pbuf[2] = col[(px>>4)&0xf].pixel;
    pbuf[3] = col[px&0xf].pixel;
  }
  return 0;
}

int DecodeMag(p)
     MagWin *p;
{
  int x,y,x2, y2, aspf, dx;
  unsigned char a_bit,flag,*pf, *pp, *pa, *pb;
  XPIX pbuf[MAX_FILE_X+4];
  unsigned char flag_buffer[(MAX_FILE_X+3)/4]; /* [MAX_FILE_X/8] (16 color) */
  
  if (dump_f)
    if (PreDumpPixels(p))
      return error("dump error!");
  /* pixel data decode */
  memset(flag_buffer, 0, sizeof(flag_buffer));
  a_bit=0x80;
  pa = p->flag_a;
  pb = p->flag_b;
  aspf = p->f.ScreenMode & MAGSM_ASPECT;
  dx = p->colsize == 256 ? 2 : 4;
  for (y2 = y = 0; y < p->ysize; y++) {
    pf = flag_buffer;
    for (x = x2 = 0; x < p->xsize; pf++) {
      if (*pa & a_bit)
 	*pf ^= *pb++;
      if (!(a_bit >>= 1)) {
 	a_bit = 0x80;
 	pa++;
      }
      /* left */
      if (DecodePixGroup(p, pbuf+x, y, x2++, *pf >> 4))
	return 1;		/* decode error */
      x += dx;
      /* right */
      if (DecodePixGroup(p, pbuf+x, y, x2++, *pf & 0xf))
	return 1;		/* decode error */
      x += dx;
    }
    if (dump_f) {
      if (DumpPixels(p, 0, y, p->xsize, pbuf))
	return error("dump error!");
    } else {
      FB_putline2(0, y2++, p->xsize, pbuf);
      if (aspf)
	FB_putline2(0, y2++, p->xsize, pbuf);
    }
    if (cancel_check()) {
      XSync(dsp, True);
      return -1;
    }
  }
  if (dump_f && PostDumpPixels(p))
    return error("dump error!");
  return 0;
}


/* allocate work area and set pointer */
AllocMemories(p)
     MagWin *p;
{
  if (debug_f) {
    fprintf(stderr, "malloc %ld + %ld + %ld\n",
	    sizeof(unsigned short)*16*p->pixbuf_xsize,
	    p->flagAlen, p->f.flagBlen);
  }
  p->pix_buffer = (unsigned short*)malloc(sizeof(unsigned short)*16*p->pixbuf_xsize);
  p->flag_a = (unsigned char *)malloc(p->flagAlen?p->flagAlen:1);
  p->flag_b = (unsigned char *)malloc(p->f.flagBlen?p->f.flagBlen:1);
  return p->pix_buffer == NULL || p->flag_a == NULL || p->flag_b == NULL;
}

void FreeMemories(p)
     MagWin *p;
{
  if (p->pix_buffer != NULL)
    free(p->pix_buffer);
  if (p->flag_a != NULL)
    free(p->flag_a);
  if (p->flag_b != NULL)
    free(p->flag_b);
}

/***** option analyze *****/
void optstr(ac, av, opt)
     int *ac; char ***av; char **opt;
{
  if (--*ac)
    *opt = *++*av;
}

optnum(ac, av, n, max)
     int *ac; char ***av; int *n; int max;
{
  long i;
  char *p;

  if (--*ac) {
    ++*av;
    i = strtol(**av, &p, 0);
    if ((p != **av)&&(i < max)) {
      *n = i;
      return 1;
    }
  }
  return 0;
}

static Pixmap root_pix;
static Pixmap tile_pix;

void tilepixmap(src, dst, dstx, dsty)
     Pixmap src;
     Pixmap dst;
     int dstx;
     int dsty;
{
  int dx, dy;
  
  dx = wxs - dstx;
  dy = wys - dsty;
  XCopyArea(dsp, src, dst, gc, dx, dy, dstx, dsty, 0, 0);
  XCopyArea(dsp, src, dst, gc, 0, dy, dx, dsty, dstx, 0);
  XCopyArea(dsp, src, dst, gc, dx, 0, dstx, dy, 0, dsty);
  XCopyArea(dsp, src, dst, gc, 0, 0, dx, dy, dstx, dsty);
}

void make_root_pix()
{
  extern XImage *img;

  tile_pix = XCreatePixmap(dsp, win, wxs, wys, DefaultDepth(dsp, scr));
  root_pix = XCreatePixmap(dsp, win, wxs, wys, DefaultDepth(dsp, scr));
  XPutImage(dsp, tile_pix, gc, img, 0, 0, 0, 0, wxs, wys);
}

void free_root_pix()
{
  XFreePixmap(dsp, tile_pix);
  XFreePixmap(dsp, root_pix);
}

/* set root window background pixmap */
void setroot()
{
  XEvent ev;
  
  make_root_pix();
  do {
    tilepixmap(tile_pix, root_pix, rootx, rooty);
    XSetWindowBackgroundPixmap(dsp, win, root_pix);
    XClearWindow(dsp, win);
    rootx += rootdx;
    if (rootx < 0)
      rootx += wxs;
    else if (rootx >= wxs)
      rootx -= wxs;
    rooty += rootdy;
    if (rooty < 0)
      rooty += wys;
    else if (rooty >= wys)
      rooty -= wys;
    if (sleeptime) {
      XFlush(dsp);
#ifdef HAVE_USLEEP
      usleep(sleeptime);	/* microsec. */
#else
      sleep(sleeptime);		/* sec. */
#endif
      while (XEventsQueued(dsp, QueuedAfterFlush)) {
	XNextEvent(dsp, &ev);
	/* ignore all events */
	if (debug_f)
	  fprintf(stderr, "rootev %d.\n", ev.type);
      }
      XSync(dsp, False);
    }
  } while (rootscroll_f);
  free_root_pix();
}


/*
 * options
 */

f_bit1(ac, av)
     int *ac; char ***av;
{
  int i;
  
  if (!*ac)
    return 1;
  i = strtol(**av, NULL, 0);
  --*ac;
  ++*av;
  if (i >= 0 && i < MAX_COLOR)
    bitmap_fg[i] = 1;
  return 0;
}

#define OPT_FUNC	1
#define OPT_SET_INT	2
#define OPT_GET_INT	3
#define OPT_GET_STR	4

typedef struct {
  char *name;
  VOIDPTR parm;
  char type;
  char setvalue;		/* for OPT_SET_INT */
} OPTION;

#define OPTTBL(name,type,ptr,val) {name, (VOIDPTR)ptr, type, val}

static OPTION opttbl[] = {
  OPTTBL("-help",	OPT_FUNC,	f_usage,	0),
  OPTTBL("-?",		OPT_FUNC,	f_usage,	0),
  OPTTBL("-V",		OPT_FUNC,	f_usage,	0),
  OPTTBL("-h",		OPT_FUNC,	f_usage,	0),
  OPTTBL("-display",	OPT_GET_STR,	&str_display,	0),
  OPTTBL("-geometry",	OPT_GET_STR,	&str_geometry,	0),
  OPTTBL("-geom",	OPT_GET_STR,	&str_geometry,	0),
  OPTTBL("-info",	OPT_SET_INT,	&info_f,	1),
  OPTTBL("-quick",	OPT_SET_INT,	&quick_f,	1),
  OPTTBL("-gray",	OPT_SET_INT,	&gray_f,	1),
  OPTTBL("-tone",	OPT_GET_INT,	&tone_v,	0),
  OPTTBL("-debug",	OPT_SET_INT,	&debug_f,	1),
  OPTTBL("-fn",		OPT_GET_STR,	&fb_font,	0),
  OPTTBL("-scroll",	OPT_SET_INT,	&scroll_f,	1),
  OPTTBL("-icon",	OPT_GET_STR,	&str_icon,	0),
  OPTTBL("-ppm",	OPT_SET_INT,	&dump_f,	DUMP_PPM),
  OPTTBL("-ppm6",	OPT_SET_INT,	&dump_f,	DUMP_PPM6),
  OPTTBL("-dump",	OPT_SET_INT,	&dump_f,	DUMP_PIXEL),
  OPTTBL("-bitmap",	OPT_SET_INT,	&dump_f,	DUMP_BITMAP),
  OPTTBL("-width",	OPT_GET_INT,	&dump_width,	0),
  OPTTBL("-bit1",	OPT_FUNC,	f_bit1,		0),
  OPTTBL("-root",	OPT_SET_INT,	&root_f,	1),
  OPTTBL("-noscroll",	OPT_SET_INT,	&scroll_f,	0),
  OPTTBL("-verbose",	OPT_SET_INT,	&verbose_f,	1),
  OPTTBL("-silent",	OPT_SET_INT,	&verbose_f,	0),
  OPTTBL("-bg",		OPT_GET_STR,	&str_bg,	0),
  OPTTBL("-fg",		OPT_GET_STR,	&str_fg,	0),
  OPTTBL("-rootx",	OPT_GET_INT,	&rootx,		0),
  OPTTBL("-rooty",	OPT_GET_INT,	&rooty,		0),
  OPTTBL("-rx",		OPT_GET_INT,	&rootx,		0),
  OPTTBL("-ry",		OPT_GET_INT,	&rooty,		0),
  OPTTBL("-rootdx",	OPT_GET_INT,	&rootdx,	0),
  OPTTBL("-rootdy",	OPT_GET_INT,	&rootdy,	0),
  OPTTBL("-rdx",	OPT_GET_INT,	&rootdx,	0),
  OPTTBL("-rdy",	OPT_GET_INT,	&rootdy,	0),
  OPTTBL("-sleep",	OPT_GET_INT,	&sleeptime,	0),
  OPTTBL("-nowait",	OPT_SET_INT,	&nowait_f,	1),
  OPTTBL("-rootscroll",	OPT_SET_INT,	&rootscroll_f,	1),
  OPTTBL("-rs",		OPT_SET_INT,	&rootscroll_f,	1),
  OPTTBL("-cmap",	OPT_SET_INT,	&cmap_mode,	1),
  OPTTBL("-nocmap",	OPT_SET_INT,	&cmap_mode,	0),
  OPTTBL(NULL,		0,		NULL,		0)
};

int parse_option(ac, av)
     int *ac;
     char ***av;
{
  OPTION *p;
  int i;
  typedef int (*FUNCPTR)();
  
  for (p = opttbl; p->name != NULL; p++) {
    if (strcmp(**av, p->name) == 0) {
      --*ac;
      ++*av;
      switch(p->type) {
      case OPT_FUNC:
	return ((FUNCPTR)(p->parm))(ac, av);
      case OPT_SET_INT:
	*((int *)(p->parm)) = p->setvalue;
	break;
      case OPT_GET_INT:
	if (*ac) {
	  *((int *)(p->parm)) = strtol(**av, NULL, 0);
	  ++*av;
	  --*ac;
	}
	break;
      case OPT_GET_STR:
	if (*ac) {
	  *((char **)(p->parm)) = **av;
	  ++*av;
	  --*ac;
	}
	break;
      }
      return 0;
    }
  }
  return 1;
}


void default_onoff(str, f)
     char *str; int *f;
{
  char *p;

  if (*f == OPTDEFAULT)
    *f = ((p = XGetDefault(dsp, myname, str)) != NULL) && !strcmp(p, "on");
}

void default_string(str, f)
     char *str; char **f;
{
  if (*f == NULL)
    *f = XGetDefault(dsp, myname, str);
}

void getxdefault()
{
  default_onoff("gray", &gray_f);
  default_onoff("quick", &quick_f);
  default_string("geometry", &str_geometry);
  default_string("icon", &str_icon);
  default_string("foreground", &str_fg);
  default_string("background", &str_bg);
  default_onoff("verbose", &verbose_f);
  default_string("font", &fb_font);
  default_onoff("scroll", &scroll_f);
}

void adjust_options()
{
  int i;
  extern int fb_root;
  
  tone_v *= TONE_100;
  tone_v /= 100;
  if (root_f)
    quit_f = 1;
  if (rootscroll_f)
    root_f = 1;
  if (root_f)
    fb_root = 1;
  if (quick_f)
    fb_realtime = 1;
  if (sleeptime == OPTDEFAULT) {
    /* sleeptime is not specified by user */
    sleeptime = rootscroll_f ? DEFALT_SCROLL_SLEEP : 0;
  }
#ifdef HAVE_USLEEP
  sleeptime *= 1000;		/* sleeptime unit millisec. -> microsec. */
#else
  sleeptime = sleeptime/1000	/* millisec. -> sec. */
    + !!(sleeptime%1000);
#endif
  if (dump_f) {
    for (i = 0; i < MAX_COLOR; i++) {
      if (bitmap_fg[i])
	break;
    }
    if (i >= MAX_COLOR)
      bitmap_fg[0] = 1;
  }
}

/****************************************************************************
 * event
 */

void Scroll(dx, dy)
     int dx, dy;
{
  int x0, y0, x1, y1, adx, ady;
  
  if (imgx-dx < -fb_xsize)
    dx = imgx + fb_xsize;
  if (imgx-dx > fb_xsize)
    dx = imgx - fb_xsize;
  if (imgy-dy < -fb_ysize)
    dy = imgy + fb_ysize;
  if (imgy-dy > fb_ysize)
    dy = imgy - fb_ysize;
  if (!dx && !dy)
    return;
  imgx -= dx;
  imgy -= dy;
  x0 = x1 = y0 = y1 = 0;
  if (dx < 0)
    x0 = adx = -dx;
  else
    x1 = adx = dx;
  if (dy < 0)
    y0 = ady = -dy;
  else
    y1 = ady = dy;
  XCopyArea(dsp, win, win, gc, x0, y0, wxs-adx, wys-ady, x1, y1);
  if (dx)
    FB_redrawimage((dx<0?wxs-adx:0), 0, adx, wys);
  if (dy)
    FB_redrawimage(0, (dy<0?wys-ady:0), wxs, ady);
}

int prevpict()
{
  int r;
  
  if ((r = pictnum) != 0)
    pictnumnext = pictnum - 1;
  return r;
}

int mouseop(f, x, y)
     int f, x, y;
{
  int dx, dy;

  dx = dy = 0;
  switch(f) {
  case 1:
    if (scroll_f) {
      if (x < wxs/3)
	scrdx = SCRDIV;
      else if (x > (wxs*2)/3)
	scrdx = -SCRDIV;
      if (y < wys/3)
	scrdy = SCRDIV;
      else if (y > (wys*2)/3)
	scrdy = -SCRDIV;
      scracc = 1;
    }
    break;
  case 0x81:
    scracc = scrdx = scrdy = 0;
    break;
  case 3:
    {
      return -1;
    }
  }
  return 0;
}

int scraccel(n)
     int n;
{
  if (n > 0 && n < SCRDOTS*SCRDIV)
    ++n;
  if (n < 0 && n > -SCRDOTS*SCRDIV)
    --n;
  return n;
}

int mainevent(f)
     int f;
{
  static int cnt = 0;
  
  if (f)
    return cnt = 0;
  if (nowait_f)
    return -1;
  if (++cnt > SCRTIMER) {
    cnt = 0;
    if (scrdx || scrdy) {
      Scroll(scrdx/SCRDIV, scrdy/SCRDIV);
      if (scracc) {
	scrdx = scraccel(scrdx);
	scrdy = scraccel(scrdy);
      } else {
	scrdx = scrdy = 0;
      }
    }
  }
  return cnt;
}


#define CMD_none 0
#define CMD_quit 1
#define CMD_next 2
#define CMD_prev 3
#define CMD_left 4
#define CMD_right 5
#define CMD_up 6
#define CMD_down 7

int keyop(p, sym)
     char *p; KeySym *sym;
{
  int r, cmd, fct, *cp;
  static int cmdtbl[] = {
    'H', CMD_left, 'h', CMD_left, 'L', CMD_right, 'l', CMD_right,
    'J', CMD_down, 'j', CMD_down, 'K', CMD_up, 'k', CMD_up,
    'b', CMD_prev, 'p', CMD_prev, 'f', CMD_next, 'n', CMD_next,
    ' ', CMD_next, 'q', CMD_quit, 0x1b, CMD_quit,
    0, CMD_none};

  fct = SCRDOTS;
  r = 1;
  cmd = CMD_none;
  switch(*sym) {
  case XK_KP_Up:
  case XK_Up:
    cmd = CMD_up;
    break;
  case XK_KP_Down:
  case XK_Down:
    cmd = CMD_down;
    break;
  case XK_KP_Left:
  case XK_Left:
    cmd = CMD_left;
    break;
  case XK_KP_Right:
  case XK_Right:
    cmd = CMD_right;
    break;
  }
  if (cmd == CMD_none) {
    if (*p == 'H' || *p == 'L' || *p == 'J' || *p == 'K')
      fct = 1;
    cp = cmdtbl;
    while (*cp != 0) {
      if (*p == *cp)
	break;
      cp += 2;
    }
    cmd = *++cp;
  }
  switch(cmd) {
  case CMD_left:
    scrdx = SCRDIV*fct;
    break;
  case CMD_right:
    scrdx = -SCRDIV*fct;
    break;
  case CMD_up:
    scrdy = SCRDIV*fct;
    break;
  case CMD_down:
    scrdy = -SCRDIV*fct;
    break;
  case CMD_prev:
    if (prevpict())
      r = -1;
    break;
  case CMD_next:
    if (pictnum != pictnummax-1)
      r = -1;
    break;
  case CMD_quit:
    pictnumnext = -1;		/* to exit */
    r = -1;
    break;
  }
  return r;
}

void SetIcon()
{
  extern unsigned int fb_icon_w, fb_icon_h;
  extern char *fb_icon_d;
  
  if (str_icon == NULL) {
    fb_icon_w = internal_icon_w;
    fb_icon_h = internal_icon_h;
    fb_icon_d = internal_icon_data;
  } else {
    fb_icon_w = fb_icon_h = 0;
    fb_icon_d = str_icon;
  }
}

void SetBGcolors()
{
  XColor color;
  
  if (str_bg != NULL) {
    if (XParseColor(dsp, cmap, str_bg, &color) &&
	XAllocColor(dsp, cmap, &color)) {
      bgpix = color.pixel;
    } else {
      error("bg color ``%s'' alloc error!", str_bg);
      str_bg = NULL;
    }
  }
  if (str_fg != NULL) {
    if (XParseColor(dsp, cmap, str_fg, &color) &&
	XAllocColor(dsp, cmap, &color)) {
      fgpix = color.pixel;
    } else {
      error("fg color ``%s'' alloc error!", str_fg);
      str_fg = NULL;
    }
  }
}

disp_mag_file(path)
     char *path;		/* MAG file pathname */
{
  int i, r;
  MagWin hdr;
  
  if (open_mag_file(&hdr, path) == NULL) {
    error("``%s'' open error!", hdr.filename);
    return 1;
  }
  switch (ReadHeader(hdr.fp, &hdr)) {
  case HDERR_NOMAG:
    error("``%s'' is not MAG file!", hdr.filename);
    return 1;
  case HDERR_NOSUP:
    error("``%s'' is not supported MAG format!", hdr.filename);
    return 1;
  case HDERR_LARGE:
    error("``%s'' is too large!", hdr.filename);
    return 1;
  default:
    break;
  }
  r = 0;
  if (verbose_f||info_f)
    disp_info(&hdr);
  if (!info_f) {
    if (str_geometry == NULL)
      wxs = wys = 0;
    fb_xsize = hdr.xsize;
    fb_ysize = hdr.ysize;
    if (hdr.f.ScreenMode & MAGSM_ASPECT)
      fb_ysize *= 2;
    fb_title = hdr.dispname;
    SetIcon();
    SetBGcolors();
    if (FB_open(argumentc, argumentv, hdr.dispname))
      exit(1);
    if (!dump_f)
      FB_disp(3);
    FB_setmousecursor(XC_watch);
    if (AllocMemories(&hdr)) {
      error("more core!");
      r = 1;
    } else {
      /* read and alloc colors */
      if (!ReadColors(&hdr) && !ReadMagFile(&hdr)) {
	AllocColors(&hdr);
	FB_setevent(mainevent);
	FB_setmouseevent(mouseop);
	FB_setkeyevent(keyop);
	imgx = (fb_xsize-(int)wxs)/2;
	imgy = (fb_ysize-(int)wys)/2;
	XSync(dsp, False);
	if ((i = DecodeMag(&hdr)) > 0)
	  error("``%s'' decode error!", hdr.filename);
	if (i >= 0) {
	  CheckPixlen(&hdr);
	  if (!quit_f && !dump_f && !rootscroll_f) {
	    FB_setmousecursor(XC_top_left_arrow);
	    FB_event();
	  }
	  FB_setmousecursor(FBMC_OFF);
	  if (root_f)
	    setroot();
	}
      }
    }
    FB_clear();
    FB_close();
    if (!root_f)
      FreeColors(&hdr);
    FreeMemories(&hdr);
  }
  if (hdr.fp != stdin)
    fclose(hdr.fp);
#ifdef ZPIPE
  wait(&i);
#endif
  return r;
}

int main(argc, argv)
     int argc; char **argv;
{
  int i, ac;
  char **av;
  
  argumentc = argc;
  argumentv = argv;
  memset(bitmap_fg, 0, sizeof(bitmap_fg));
  errname = myname = getmyname(*argv++);
  argc--;
  while (argc > 0) {
    if (strcmp(*argv, stdinfile) == 0)
      break;
    if (parse_option(&argc, &argv)) {
      if (**argv == '-') {
	error("unknown option ``%s''.", *argv);
	usage(stderr);
	exit(1);
      }
      break;
    }
  }
  if (!argc) {
#ifdef OLD_COMPATIBLE
    usage(stderr);
    exit(1);
#else
    argc = 1;
    *argv = stdinfile;
#endif
  }
  if (FB_libinit(NULL) < 0) {
    error("Cannot access graphic resource!");
    exit(1);
  }
  getxdefault();
  adjust_options();
  pictnummax = argc;
  pictnum = 0;  
  while (pictnummax > 0 && pictnum >= 0 && pictnum < pictnummax) {
    pictnumnext = pictnum+1;	/* default look next picture */
    if (disp_mag_file(*(argv+pictnum))) {
      /* can't display this picture, delete from list */
      for (i = pictnum; i < pictnummax - 1; i++)
	*(argv+i) = *(argv+i+1);
      --pictnummax;
    }
    pictnum = pictnumnext;
  }
  if (!info_f && root_f)
    XSetCloseDownMode(dsp, RetainTemporary);
  FB_libend();
  exit(0);
}

/* End of file */
