/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   atom.c								*
*									*
*	Implements X server Atom stuff.					*
*									*
************************************************************************/
#include <stdlib.h>
#define NEED_REPLIES
#include <X11/Xproto.h>
#include <X11/X.h>
#include "xmx.h"
#include "incl/atom.pvt.h"

typedef struct _amap_t {
	char *			name;
	int			len;
	atom_t			atom;
	struct _amap_t *	next;
}amap_t;

static amap_t predefatoms[] = {
        {0,0,0,0},
/*  1 */{ "PRIMARY",		7,	1,	0	},
/*  2 */{ "SECONDARY",		9,	2,	0	},
/*  3 */{ "ARC",		3,	3,	0	},
/*  4 */{ "ATOM",		4,	4,	0	},
/*  5 */{ "BITMAP",		6,	5,	0	},
/*  6 */{ "CARDINAL",		8,	6,	0	},
/*  7 */{ "COLORMAP",		8,	7,	0	},
/*  8 */{ "CURSOR",		6,	8,	0	},
/*  9 */{ "CUT_BUFFER0",	11,	9,	0	},
/* 10 */{ "CUT_BUFFER1",	11,	10,	0	},
/* 11 */{ "CUT_BUFFER2",	11,	11,	0	},
/* 12 */{ "CUT_BUFFER3",	11,	12,	0	},
/* 13 */{ "CUT_BUFFER4",	11,	13,	0	},
/* 14 */{ "CUT_BUFFER5",	11,	14,	0	},
/* 15 */{ "CUT_BUFFER6",	11,	15,	0	},
/* 16 */{ "CUT_BUFFER7",	11,	16,	0	},
/* 17 */{ "DRAWABLE",		8,	17,	0	},
/* 18 */{ "FONT",		4,	18,	0	},
/* 19 */{ "INTEGER",		7,	19,	0	},
/* 20 */{ "PIXMAP",		6,	20,	0	},
/* 21 */{ "POINT",		5,	21,	0	},
/* 22 */{ "RECTANGLE",		9,	22,	0	},
/* 23 */{ "RESOURCE_MANAGER",	16,	23,	0	},
/* 24 */{ "RGB_COLOR_MAP",	13,	24,	0	},
/* 25 */{ "RGB_BEST_MAP",	12,	25,	0	},
/* 26 */{ "RGB_BLUE_MAP",	12,	26,	0	},
/* 27 */{ "RGB_DEFAULT_MAP",	15,	27,	0	},
/* 28 */{ "RGB_GRAY_MAP",	12,	28,	0	},
/* 29 */{ "RGB_GREEN_MAP",	13,	29,	0	},
/* 30 */{ "RGB_RED_MAP",	11,	30,	0	},
/* 31 */{ "STRING",		6,	31,	0	},
/* 32 */{ "VISUALID",		8,	32,	0	},
/* 33 */{ "WINDOW",		6,	33,	0	},
/* 34 */{ "WM_COMMAND",		10,	34,	0	},
/* 35 */{ "WM_HINTS",		7,	35,	0	},
/* 36 */{ "WM_CLIENT_MACHINE",	17,	36,	0	},
/* 37 */{ "WM_ICON_NAME",	12,	37,	0	},
/* 38 */{ "WM_ICON_SIZE",	12,	38,	0	},
/* 39 */{ "WM_NAME",		7,	39,	0	},
/* 40 */{ "WM_NORMAL_HINTS",	14,	40,	0	},
/* 41 */{ "WM_SIZE_HINTS",	12,	41,	0	},
/* 42 */{ "WM_ZOOM_HINTS",	12,	42,	0	},
/* 43 */{ "MIN_SPACE",		9,	43,	0	},
/* 44 */{ "NORM_SPACE",		10,	44,	0	},
/* 45 */{ "MAX_SPACE",		9,	45,	0	},
/* 46 */{ "END_SPACE",		9,	46,	0	},
/* 47 */{ "SUPERSCRIPT_X",	13,	47,	0	},
/* 48 */{ "SUPERSCRIPT_Y",	13,	48,	0	},
/* 49 */{ "SUBSCRIPT_X",	11,	49,	0	},
/* 50 */{ "SUBSCRIPT_Y",	11,	50,	0	},
/* 51 */{ "UNDERLINE_POSITION",	18,	51,	0	},
/* 52 */{ "UNDERLINE_THICKNESS",19,	52,	0	},
/* 53 */{ "STRIKEOUT_ASCENT",	16,	53,	0	},
/* 54 */{ "STRIKEOUT_DESCENT",	17,	54,	0	},
/* 55 */{ "ITALIC_ANGLE",	12,	55,	0	},
/* 56 */{ "X_HEIGHT",		8,	56,	0	},
/* 57 */{ "QUAD_WIDTH",		10,	57,	0	},
/* 58 */{ "WEIGHT",		6,	58,	0	},
/* 59 */{ "POINT_SIZE",		10,	59,	0	},
/* 60 */{ "RESOLUTION",		10,	60,	0	},
/* 61 */{ "COPYRIGHT",		9,	61,	0	},
/* 62 */{ "NOTICE",		6,	62,	0	},
/* 63 */{ "FONT_NAME",		9,	63,	0	},
/* 64 */{ "FAMILY_NAME",	11,	64,	0	},
/* 65 */{ "FULL_NAME",		9,	65,	0	},
/* 66 */{ "CAP_HEIGHT",		10,	66,	0	},
/* 67 */{ "WM_CLASS",		8,	67,	0	},
/* 68 */{ "WM_TRANSIENT_FOR",	16,	68,	0	},
	/*
	** X Logical Font Description font property atoms.
	*/
/* 69 */{ "FOUNDRY",		7,	69,	0	},
/* 70 */{ "FAMILY_NAME",	11,	70,	0	},
/* 71 */{ "WEIGHT_NAME",	11,	71,	0	},
/* 72 */{ "SLANT",		5,	72,	0	},
/* 73 */{ "SETWIDTH_NAME",	13,	73,	0	},
/* 74 */{ "ADD_STYLE_NAME",	14,	74,	0	},
/* 75 */{ "PIXEL_SIZE",		10,	75,	0	},
/* 76 */{ "POINT_SIZE",		10,	76,	0	},
/* 77 */{ "RESOLUTION_X",	12,	77,	0	},
/* 78 */{ "RESOLUTION_Y",	12,	78,	0	},
/* 79 */{ "SPACING",		7,	79,	0	},
/* 80 */{ "AVERAGE_WIDTH",	13,	80,	0	},
/* 81 */{ "CHARSET_REGISTRY",	16,	81,	0	},
/* 82 */{ "CHARSET_ENCODING",	16,	82,	0	},
/* 83 */{ "MIN_SPACE",		9,	83,	0	},
/* 84 */{ "NORM_SPACE",		10,	84,	0	},
/* 85 */{ "MAX_SPACE",		9,	85,	0	},
/* 86 */{ "END_SPACE",		9,	86,	0	},
/* 87 */{ "AVG_CAPITAL_WIDTH",	17,	87,	0	},
/* 88 */{ "AVG_LOWERCASE_WIDTH",19,	88,	0	},
/* 89 */{ "QUAD_WIDTH",		10,	89,	0	},
/* 90 */{ "FIGURE_WIDTH",	12,	90,	0	},
/* 91 */{ "SUPERSCRIPT_X",	13,	91,	0	},
/* 92 */{ "SUPERSCRIPT_Y",	13,	92,	0	},
/* 93 */{ "SUBSCRIPT_X",	11,	93,	0	},
/* 94 */{ "SUBSCRIPT_Y",	11,	94,	0	},
/* 95 */{ "SUPERSCRIPT_SIZE",	16,	95,	0	},
/* 96 */{ "SUBSCRIPT_SIZE",	14,	96,	0	},
/* 97 */{ "SMALL_CAP_SIZE",	14,	97,	0	},
/* 98 */{ "UNDERLINE_POSITION",	18,	98,	0	},
/* 99 */{ "UNDERLINE_THICKNESS",19,	99,	0	},
/*100 */{ "STRIKEOUT_ASCENT",	16,	100,	0	},
/*101 */{ "STRIKEOUT_DESCENT",	17,	101,	0	},
/*102 */{ "ITALIC_ANGLE",	12,	102,	0	},
/*103 */{ "CAP_HEIGHT",		10,	103,	0	},
/*104 */{ "X_HEIGHT",		8,	104,	0	},
/*105 */{ "RELATIVE_SETWIDTH",	17,	105,	0	},
/*106 */{ "RELATIVE_WEIGHT",	15,	106,	0	},
/*107 */{ "WEIGHT",		6,	107,	0	},
/*108 */{ "RESOLUTION",		10,	108,	0	},
/*109 */{ "FACE_NAME",		9,	109,	0	},
/*110 */{ "COPYRIGHT",		9,	110,	0	},
/*111 */{ "NOTICE",		6,	111,	0	},
/*112 */{ "DESTINATION",	11,	112,	0	}
};

#define INIT	512
#define INCR	128

/*
**   Not fancy, but surprisingly good, at least for the predefined ones.
*/
#define HASH(key, size, name, len) {\
   register int k;\
   key = 0;\
   for (k=0; k<(len); k++)\
      key += (name)[k];\
   key = key % (size);\
}

static amap_t **atoms;			/* name->atom map */
static amap_t **names;			/* atom->name map */
static atom_t curmax = MAX_KNOWN_ATOM;	/* current max atom */
static int size = 0;			/* current size */

/************************************************************************
*									*
*   atom_max								*
*									*
************************************************************************/
atom_t
atom_max
   VOID
{
   return curmax;
}

/************************************************************************
*									*
*   atom_reset								*
*									*
*	Initialize, or clear existing atom tables and re-initialize.	*
*	Returns zero for success, -1 for failure.			*
*									*
************************************************************************/
int
atom_reset
   VOID
{
   register int i;

   if (size) {
      for (i=MAX_KNOWN_ATOM+1; i<=curmax; i++)
         free(names[i]);
      free(names);
      free(atoms);
      size = 0;
      curmax = MAX_KNOWN_ATOM;
   }
   return bigger();
}

/************************************************************************
*									*
*   atom_intern								*
*									*
*	Handle an InternAtom request.					*
*									*
************************************************************************/
void
atom_intern
   AL((cp, p))
   DB client_t *cp
   DD xInternAtomReq *p
   DE
{
   register atom_t atom;

   if (atom = atom_store((char *)(p+1), p->nbytes, p->onlyIfExists))
      proto_InternAtomReply(cp, atom);
   else if (p->onlyIfExists)
      proto_InternAtomReply(cp, 0);
   else
      proto_Error(cp, BadAlloc, 0,0, X_InternAtom);
}

/************************************************************************
*									*
*   atom_store								*
*									*
*	Do an InternAtom, essentially.  If the tables are getting too	*
*	full, make'em bigger.  Returns the atom for success, or zero	*
*	(an impossible atom) for failure.				*
*									*
************************************************************************/
atom_t
atom_store
   AL((name, len, onlyifexists))
   DB char *name
   DD int len
   DD int onlyifexists
   DE
{
   register amap_t *ap;
   register atom_t key;

   HASH(key, size, name, len);

   for (ap=atoms[key]; ap; ap=ap->next)
      if (len == ap->len && bcmp(name, ap->name, len) == 0)
         return ap->atom;

   if (onlyifexists)
      return 0;

   if (curmax + INCR > size) {
      bigger();		/* invalidates hash key */
      HASH(key, size, name, len);
   }
   if (	MALLOC(ap, amap_t *, sizeof(amap_t)) ||
	MALLOC(ap->name, char *, len + 1)) {
      if (ap) free(ap);
      return 0;
   }
   ap->atom = ++curmax;
   bcopy(name, ap->name, len);
   ap->name[len] = '\0';
   ap->len = len;

   ap->next = atoms[key];
   atoms[key] = ap;

   names[ap->atom] = ap;

   return ap->atom;
}

/************************************************************************
*									*
*   atom_get								*
*									*
*	Get an atom if it exists.					*
*									*
************************************************************************/
atom_t
atom_get
   AL((name, len))
   DB char *name
   DD int len
   DE
{
   register amap_t *ap;
   register atom_t key;

   HASH(key, size, name, len);

   for (ap=atoms[key]; ap; ap=ap->next)
      if (len == ap->len && bcmp(name, ap->name, len) == 0)
         return ap->atom;
   return 0;
}

/************************************************************************
*									*
*   atom_getname							*
*									*
*	Handle a GetAtomName request.					*
*									*
************************************************************************/
void
atom_getname
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   int len;
   char *name;

   if (atom_name(p->id, &name, &len))
      proto_Error(cp, BadAtom, p->id, 0, X_GetAtomName);
   else
      proto_GetAtomNameReply(cp, name, len);
}

/************************************************************************
*									*
*   atom_name								*
*									*
*	Do a GetAtomName.  Returns zero for success, -1 for failure.	*
*									*
************************************************************************/
int
atom_name
   AL((atom, name, len))
   DB atom_t atom
   DD char **name
   DD int *len
   DE
{
   if (atom >= 1 && atom <= curmax) {
      if (name) *name = names[atom]->name;
      if (len) *len = names[atom]->len;
      return 0;
   }
   else {
      if (name) *name = 0;
      if (len) *len = 0;
      return -1;
   }
}

/*
**   bigger
**
**	Initialize or increase the size of the atom map tables.
**	Returns zero for success, -1 for failure.
*/
static int
bigger
   VOID
{
   register int i;
   register atom_t key;
   register int nsize;
   register amap_t **nnames;
   register amap_t **natoms;

   nsize = size ? size + INCR : INIT;

   if (CALLOC(nnames, amap_t **, nsize, sizeof(amap_t *)))
      return -1;
   if (CALLOC(natoms, amap_t **, nsize, sizeof(amap_t *))) {
      free(nnames);
      return -1;
   }
   if (size) {
      for (i=1; i<=curmax; i++)
         nnames[i] = names[i];
      free(names);
      free(atoms);
   }
   else
      for (i=1; i<=curmax; i++)
         nnames[i] = &predefatoms[i];

   size = nsize;
   names = nnames;
   atoms = natoms;

   for (i=1; i<=curmax; i++) {
      HASH(key, size, names[i]->name, names[i]->len);
      names[i]->next = atoms[key];
      atoms[key] = names[i];
   }
   return 0;
}
