/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   buf.c								*
*									*
*	Manage buffers.							*
*									*
************************************************************************/
#include <errno.h>

#include "xmx.h"
#include "df.h"
#include "rs.h"
#include "ptc.h"
#include "incl/buf.pvt.h"

#define MIN_BUF_SIZE	2048
#define MIN_READ_SIZE	1024	/* half of MIN_BUF_SIZE to minimize mallocs */

static buffer_t *bfree;		/* buffer free list */
static chunk_t *chfree;		/* chunk free list */

/************************************************************************
*									*
*   buf_new								*
*									*
*	Allocate and initialize a new buffer.				*
*									*
*	Buffers _always_ have a current chunk.				*
*									*
************************************************************************/
buffer_t *
buf_new
   AL((type))
   DB u8_t type
   DE
{
   register buffer_t *bp;
   register chunk_t *chp;

   if (bfree) {
      bp = bfree;
      bfree = bfree->next;
      chp = bp->head;
   }
   else {
      if (MALLOC(bp, buffer_t *, sizeof(buffer_t)))
         return (buffer_t *)err(0, "malloc returned zero");
      if ((chp = new_chunk()) == 0) {
         free(bp);
         return 0;
      }
      bp->sz = 0;
      bp->tfree = 0;
      bp->bp = 0;
      bp->head = chp;
      bp->tail = chp;

      chp->bp = bp;
      chp->prev = 0;
      chp->next = 0;
   }
   bp->type = type;
   bp->clinum = 0;

   chp->type = P_NONE;
   chp->dptr = (void *)0;
   chp->refs = 0;	/* the "current" chunk always has zero refs */
   chp->off = 0;
   chp->sz = 0;
   chp->lpp = chp->tpp = 0;
   chp->curp = bp->bp;

   DEBUG3(D_BUFFERS, "buf_new: %s -> bp 0x%x chunk [0x%x]\n",
				debug_buf_type_str(type), bp, chp);
   D_CALL1(D_VERIFY, buf_verify, bp);
   return bp;
}

/************************************************************************
*									*
*   buf_free								*
*									*
*	Free a buffer...but...						*
*									*
*	...if any of the buffer's chunks are referenced, mark the	*
*	buffer free-on-write, and keep it around.  Assuming the chunk	*
*	is queued somewhere, it will eventually be written out or the	*
*	queue will be freed.  In either case, the chunk will be		*
*	cleared and the buffer freed.					*
*									*
************************************************************************/
void
buf_free
   AL((bp))
   DB buffer_t *bp
   DE
{
   register chunk_t *chp, *last;

   D_CALL1(D_VERIFY, buf_verify, bp);
   DEBUG2(D_BUFFERS, "buf_free: bp 0x%x current chunk [0x%x]\n", bp, bp->head);

   for (chp=bp->head; last=chp;) {	/* free extra chunks */
      if ((chp = chp->next) == 0)
         break;
      if (last->refs) {			/* don't free a referenced chunk */
         last->prev = 0;			/* reconnect */
         bp->head = last;
         bp->type = B_FREEONWRITE;		/* cut buffer loose */
         DEBUG3(D_BUFFERS, "buf_free: bp 0x%x bp->head [0x%x] %s\n",
				bp, last, "converted to FREEONWRITE");
         return;				/* EXIT */
      }
      if (last->lpp)
         pp_free(last->lpp);
      if (last->tpp)
         pp_free(last->tpp);
      last->curp = 0;
      free_chunk(last);
   }
   if (bp->tail->lpp)
      pp_free(bp->tail->lpp);
   if (bp->tail->tpp)
      pp_free(bp->tail->tpp);
   if (bp->tail->dptr) {
      warn("buf_free: metadata not cleared, hmmmm...\n");
      bp->tail->dptr = 0;
   }
   bp->tail->prev = 0;
   bp->head = bp->tail;
   bp->next = bfree;
   bfree = bp;
}

/************************************************************************
*									*
*   buf_free_freelist							*
*									*
************************************************************************/
void
buf_free_freelist
   VOID
{
   register buffer_t *bp, *last;

   for (bp=bfree; last=bp;) {
      bp = bp->next;
      free_chunk(last->tail);
      free(last);
   }
   free_chunk_freelist();
   bfree = 0;
}

/************************************************************************
*									*
*   buf_split								*
*									*
*	Split a chunk into two.						*
*									*
************************************************************************/
chunk_t *
buf_split
   AL((bp, off))
   DB buffer_t *bp	/* buffer to split */
   DD uint_t off	/* offset of new chunk == new size of old chunk */
   DE
{
   uint_t nsz;
   register chunk_t *chp, *nchp;

   D_CALL1(D_VERIFY, buf_verify, bp);

   chp = bp->tail;
#ifdef DEBUG
   if (chp->refs)
      quit(-1, "buf_split: chunk [0x%x] has refs [%d]\n", chp, chp->refs);
#endif
   if (off)
      if (off > chp->sz)
         quit(-1, "buf_split: sanity check - negative split (new)\n");
      else
         nsz = chp->sz - off;

   else {
      off = chp->sz;
      nsz = 0;
   }
   nchp = new_chunk();	/* what if error? */
   nchp->type = P_NONE;
   nchp->dptr = (void *)0;
   nchp->refs = 0;	/* current chunk always has zero refs */
   nchp->bp = bp;
   nchp->off = chp->off + off;
   nchp->curp = bp->bp + nchp->off;
   nchp->sz = nsz;
   nchp->lpp = 0;
   nchp->tpp = 0;
   nchp->prev = chp;
   nchp->next = 0;

   chp->refs = 1;
   chp->sz = off;
   chp->next = nchp;
#ifdef DEBUG
   if (chp->lpp) {
      warn("buf_split: current chunk [0x%x] had leading partial [0x%x]!\n",
							chp, chp->lpp);
      chp->lpp = 0;
   }
#endif

   bp->tail = nchp;

   D_CALL1(D_VERIFY, buf_verify, bp);
   DEBUG3(D_BUFFERS, "buf_split: bp 0x%x old chunk 0x%x new chunk 0x%x\n",
						bp, chp, nchp);
   return chp;
}

/************************************************************************
*									*
*   buf_clear								*
*									*
*	Clear a chunk - fold it into any adjacent empty chunks.		*
*									*
************************************************************************/
void
buf_clear
   AL((chp))
   DB chunk_t *chp
   DE
{
   register buffer_t *bp;
   register chunk_t *pchp, *nchp;

   D_CALL1(D_VERIFY, buf_verify, chp->bp);
   DEBUG3(D_BUFFERS, "buf_clear: bp 0x%x chunk 0x%x refs %d\n",
						chp->bp, chp, chp->refs);
#ifdef DEBUG
   if (chp->valid == 0)
      quit(-1, "buf_clear: invalid chunk 0x%x\n", chp);
#endif
   switch (chp->refs) {	/* refs is unsigned */
      case 0:
         warn("buf_clear: chunk [0x%x] has zero refs\n", chp);
#ifdef DEBUG
         abort();
#endif
         break;

      case 1:		/* really clear it */
         bp = chp->bp;

         pchp = chp->prev;
         nchp = chp->next;
#ifdef DEBUG
         if (nchp == 0) {
            warn("buf_clear: clear current chunk [0x%x] bp [0x%x]\n", chp, bp);
            abort();
         }
#endif
         nchp->prev = pchp;
         if (pchp) {
            pchp->next = nchp;
            if (nchp->sz == 0) {
               nchp->off = pchp->off + pchp->sz;
               nchp->curp = bp->bp + nchp->off;
               if (bp->tail != nchp)
                  bp->tfree += chp->sz;
            }
            else
               bp->tfree += chp->sz;
         }
         else {			/* current chunk is head */
            bp->head = nchp;
            if (nchp->sz == 0) {
               nchp->off = 0;
               nchp->curp = bp->bp;
               bp->tfree = 0;
            }
            else
               bp->tfree += chp->sz;
         }
         chp->sz = 0;
         chp->refs = 0;
         chp->curp = 0;
         free_chunk(chp);

         D_CALL1(D_VERIFY, buf_verify, bp);
         if (bp->type==B_FREEONWRITE && bp->head==bp->tail && bp->head->sz==0)
            buf_free(bp);
         break;

      default:
         chp->refs--;
         break;
   }
}

/************************************************************************
*									*
*   buf_read								*
*									*
*	Attempt to fill the buffer with data read from fd.  The buffer	*
*	is expanded as needed.  This routine returns immediately.	*
*									*
*	Returns -1 if error, 0 if EOF, otherwise a positive value.	*
*	A positive return value does not imply that new data was read.	*
*									*
************************************************************************/
int
buf_read
   AL((fd, bp))
   DB int fd
   DD buffer_t *bp
   DE
{
   register int r;
   register uint_t t, n;
   register char *cp;
   register chunk_t *chp;

   D_CALL1(D_VERIFY, buf_verify, bp);

   chp = buf_chunk(bp);
   t = buf_avail(bp);

   if (t >= MIN_READ_SIZE)
      n = t;
   else {
      n = MIN_READ_SIZE;
      buf_adjust(bp, n);	/* adjust buffer size */
   }
   cp = bp->bp + chp->off + chp->sz;

   for (t=0; t < n; t+=(uint_t)r)
      if ((r = read(fd, cp+t, n-t)) < 0)
         if (errno == EWOULDBLOCK) {
            r = 1;
            break;
         }
         else {
            r = -1;
            break;
         }
      else if (r == 0)
         break;			/* EOF */

   chp->sz += t;
#ifdef DEBUG
   if (chp->sz > bp->sz) {
      warn("buf_read: sanity check, tail[0x%x]->sz[%d] > bp[0x%x]->sz[%d]\n",
					chp, chp->sz, bp, bp->sz);
      abort();
   }
#endif

   DEBUG3(D_BUFFERS, "buf_read: bp 0x%x chunk 0x%x n %d\n", bp, chp, t);

   return r;
}

/************************************************************************
*									*
*   buf_put								*
*									*
*	Copy n bytes of raw data into the current chunk of a buffer	*
*	block.								*
*									*
************************************************************************/
void
buf_put
   AL((bp, sp, n))
   DB buffer_t *bp
   DD char *sp
   DD uint_t n
   DE
{
   register char *dp;
   register chunk_t *chp;

   D_CALL1(D_VERIFY, buf_verify, bp);
   DEBUG2(D_BUFFERS, "buf_put: bp 0x%x n %d\n", bp, n);

   chp = buf_chunk(bp);
   if (buf_avail(bp) < n)
      buf_adjust(bp, n);

   dp = buf_next(bp);
   bcopy(sp, dp, n);
   chp->sz += n;
#ifdef DEBUG
   if (chp->sz > bp->sz) {
      warn("buf_put: sanity check, tail[0x%x]->sz[%d] > bp[0x%x]->sz[%d]\n",
					chp, chp->sz, bp, bp->sz);
      abort();
   }
#endif
}

/************************************************************************
*									*
*   buf_adjust								*
*									*
*	Reallocate buffer block so that it has at least n bytes of	*
*	free space.							*
*									*
*	First try to free the space by compacting the buffer.  If	*
*	that is insufficient, allocate a new, larger buffer, and	*
*	copy over just the active data, compacting as we go.		*
*									*
*	Two dangerous things:						*
*		1) if this routine is called while a pointer into	*
*		   the buffer is active, that pointer may become	*
*		   invalid, causes ugliness.				*
*		2) the compaction scheme implemented below forces	*
*		   chunks to be word-aligned.  If a chunk is		*
*		   deliberately not word aligned (as in a partial	*
*		   system write), it will be after this.		*
*									*
************************************************************************/
void
buf_adjust
   AL((bp, n))
   DB buffer_t *bp
   DD uint_t n		/* free space needed */
   DE
{
#define register /**/
   register uint_t i, off, nfree, sz;
   register char *dp, *sp, *tp;
   register chunk_t *chp, *tchp;
#undef register

   D_CALL1(D_VERIFY, buf_verify, bp);

   DEBUG6(D_BUFFERS, "buf_adjust: bp 0x%x sz %d tail 0x%x off %d sz %d, n %d\n",
		bp, bp->sz, bp->tail, bp->tail->off, bp->tail->sz, n);

   bp->tfree = 0;	/* this routine gets rid of this */

   chp = buf_chunk(bp);
   nfree = bp->sz - chp->off - chp->sz;

   if (chp->off) {		/* try compacting chunks first */
      /*
      **  is there enough space?
      */
      off = 0;
      for (tchp=bp->head; tchp; tchp=tchp->next) {
         if (tchp->off > off)
            nfree += RDN(tchp->off - off, 4);	/* round each block down */
         off = tchp->off + tchp->sz;
      }
      if (nfree >= n) {
         /*
         **  there is, slide everything left
         */
         off = 0;
         for (tchp=bp->head; tchp; tchp=tchp->next) {
            sp = bp->bp + tchp->off;
            dp = bp->bp + off;
            if (dp != sp) {
               for (i=0; i<tchp->sz; i++)
                  *dp++ = *sp++;
               tchp->off = off;
               tchp->curp = bp->bp + tchp->off;
            }
            off = RUP(tchp->off + tchp->sz, 4);
         }
         return;	/* all done */
      }
   }
   /*
   **  allocate a bigger buffer
   */
   sz = bp->sz - nfree + n; /* total size needed (may allocate more) */
   if (sz < MIN_BUF_SIZE)
      bp->sz = MIN_BUF_SIZE;
   else
      bp->sz = RUP(sz, 2*MIN_BUF_SIZE);

   tp = bp->bp;
   if (MALLOC(bp->bp, char *, bp->sz))
      quit(-1, "buf_adjust: fatal memory allocation error\n");

   if (tp) {
      /*
      **  compact chunks as they're copied
      */
      off = 0;
      for (tchp=bp->head; tchp; tchp=tchp->next) {	/* preserve chunks */
         sp = tp + tchp->off;
         dp = bp->bp + off;

         tchp->off = off;
         if (tchp->sz) {
            bcopy(sp, dp, tchp->sz);
            off += RUP(tchp->sz, 4);
         }
         tchp->curp = bp->bp + tchp->off;
      }
      free(tp);
   }
   else
      bp->tail->curp = bp->bp;
   DEBUG2(D_BUFFERS, "buf_adjust: bp[0x%x] realloc'd to sz %d\n", bp, bp->sz);
   D_CALL1(D_VERIFY, buf_verify, bp);
}

/************************************************************************
*									*
*   buf_verify								*
*									*
*	Verify internal consistency of a buffer structure.		*
*	Dumps core if anything is amiss.				*
*									*
************************************************************************/
void
buf_verify
   AL((bp))
   DB buffer_t *bp
   DE
{
   int off = 0;
   chunk_t *chp, *lchp = 0;

   if (bp)
      if (bp->type == B_STATIC || bp->type == B_FREEONWRITE)
         if (bp->sz)
            if (bp->head)
               if (bp->tail)
                  if (bp->bp) {
                     for (chp=bp->head; chp; chp=chp->next)
                        if (chp->bp == bp)
                           if (off <= (int)chp->off) {
                              off = (int)(chp->off + chp->sz);
/* TODO
                              buf_verify_chunk(chp);
*/
                              if (lchp != chp->prev) {
                                 warn("buf_verify: chunk[0x%x]->prev[0x%x]\n",
							chp, chp->prev);
                                 break;
                              }
                              lchp = chp;
                           }
                           else {
                              warn("buf_verify: offset overlap [%d > %d]\n",
							off, chp->off);
                              break;
                           }
                        else {
                           warn("buf_verify: bp [0x%x] != chp->bp [0x%x]\n",
							bp, chp->bp);
                           break;
                        }
                     if (chp == 0)
                        if (lchp == bp->tail)
                           return;		/* okay */
                        else
                           warn("buf_verify: tail [0x%x] not eolist [0x%x]\n",
							bp->tail, lchp);
                  }
                  else
                     warn("buf_verify: buffer is null, sz is %d\n", bp->sz);
               else
                  warn("buf_verify: no tail\n");
            else
               warn("buf_verify: no head\n");
         else if (bp->bp == 0)
            return;		/* okay */
         else
            warn("buf_verify: sz is zero, but buffer is 0x%x\n", bp->bp);
      else
         warn("buf_verify: type is %d\n", bp->type);
   else
      warn("buf_verify: null\n");

   abort();
}

#ifdef REWRITE_ME_TODO
/************************************************************************
*									*
*   buf_verify_chunk							*
*									*
*	Verify internal consistency of a chunk structure.		*
*	Dumps core if anything is amiss.				*
*									*
************************************************************************/
void
buf_verify_chunk
   AL((chp))
   DB chunk_t *chp
   DE
{
   if (chp)
      if (chp->valid)
         if (chp->bp)
            if ((chp->flags & (CF_SWAPPED | CF_IMAGE)) == chp->flags)
               if (chp->off + chp->sz <= chp->bp->sz)
                  if (chp->lstp)
                     return;	/* okay */
                  else
                     warn("buf_verify_chunk: no list\n");
               else
                  warn("buf_verify_chunk: off[%d]+sz[%d] > buffer size [%d]\n",
					chp->off, chp->sz, chp->bp->sz);
            else
               warn("buf_verify_chunk: bad flags [0x%x]\n", chp->flags);
         else
            warn("buf_verify_chunk: no buffer!\n");
      else
         warn("buf_verify_chunk: invalid!\n");
   else
      warn("buf_verify_chunk: null\n");

   abort();
}
#endif

void
buf_print
   AL((bp))
   DB buffer_t *bp
   DE
{
   register chunk_t *chp;

   warn("buffer[0x%x/%d] type[%s]\n", bp, bp, debug_buf_type_str(bp->type));
   warn("\tsz[%d] bp[0x%x/%d]\n", bp->sz, bp->bp, bp->bp);
   warn("\tCHUNKLIST->\n");
   for (chp=bp->head; chp; chp=chp->next)
      buf_print_chunk(chp);
   warn("\t<-CHUNKLIST\n");
}

void
buf_print_chunk
   AL((chp))
   DB chunk_t *chp
   DE
{
   warn("chunk[0x%x/%d] valid[%d] refs[%d]\n", chp, chp->valid, chp, chp->refs);
   warn("\toff[%d] sz[%d]\n", chp->off, chp->sz);
   warn("\ttype[%s]\n", debug_proto_type_str(chp->type));
   warn("\tlpp[0x%x/%d] tpp[0x%x/%d]\n", chp->lpp,chp->lpp, chp->tpp,chp->tpp);
   warn("\tdptr[0x%x/%d]\n", chp->dptr, chp->dptr);
}

/*
**	Chunk management
*/

/*
**  new_chunk
**
**	Grab a chunk off the free list, or malloc a new one.
**
**	Chunks _always_ have a list.
*/
static chunk_t *
new_chunk
   VOID
{
   register chunk_t *chp;

   if (chfree) {
      chp = chfree;
      chfree = chfree->next;
      DEBUG1(D_CHUNK, "new_chunk(0x%x) [freelist]\n", chp);
   }
   else {
      if (MALLOC(chp, chunk_t *, sizeof(chunk_t)))
         return 0;
      DEBUG1(D_CHUNK, "new_chunk(0x%x) [malloc]\n", chp);
   }
#ifdef DEBUG
   chp->valid = 1;
#endif
   return chp;
}

/*
**  free_chunk
**
**	Put a chunk on the free list.
*/
static void
free_chunk
   AL((chp))
   DB chunk_t *chp
   DE
{
#ifdef DEBUG
   register int i, j, cnt;
   register chunk_t *nchp;

   if (chp->refs) {
      warn("free_chunk: chunk [0x%x] has %d refs\n", chp, chp->refs);
      abort();
   }
   if (chp->valid == 0) {
      warn("free_chunk: chunk not valid, in free list?...");
      for (nchp=chfree; nchp; nchp=nchp->next)
         if (chp == nchp) {
            warn("yes! very bad - aborting.\n");
            abort();
         }
      warn("no, freeing it.\n");
   }
#endif
   /*
   **  free metadata
   */
   switch (chp->type) {
      case P_NONE:
         break;
      case P_REQUEST:
         ptc_unset(chp);
         break;
      case P_REPLY:
         rs_unmark(chp);
         break;
      case P_ERROR:
         break;
      case P_EVENT:
         break;
      case P_IMAGE:
         image_free(chp);
         break;
   }
   if (chp->lpp) {
      pp_free(chp->lpp);
      chp->lpp = 0;
   }
   if (chp->tpp) {
      pp_free(chp->tpp);
      chp->tpp = 0;
   }
   chp->valid = 0;
   chp->next = chfree;
   chfree = chp;
   DEBUG1(D_CHUNK, "free_chunk(0x%x)\n", chp);
}

/*
**  free_chunk_freelist
**
**	Actually free all the unused chunks.
*/
static void
free_chunk_freelist
   VOID
{
   register chunk_t *chp, *last;

   for (chp=chfree; last=chp;) {
      chp = chp->next;
      free(last);
   }
   chfree = 0;
}

void
buf_verify_chunks
   VOID
{
   register int i, j, k;
   register chunk_t *chp;

   for (i=0; i<num_serv; i++) {
      for (j=0; j<(int)servers[i]->qp->qnel; j++)
         for (chp=chfree; chp; chp=chp->next)
            if (chp == servers[i]->qp->queue[j]) {
               warn("buf_chunk_verify: chunk [0x%x] queued to server [%d]\n",
   								chp, i);
               abort();
            }
      for (j=0; j<(int)servers[i]->qp->wnel; j++)
         for (chp=chfree; chp; chp=chp->next)
            if (chp == servers[i]->qp->wait[j]) {
               warn("buf_chunk_verify: chunk [0x%x] wait for server [%d]\n",
   								chp, i);
               abort();
            }
   }
   for (i=k=1; k<num_clients; i++)
      if (clients[i]) {
         k++;
         for (j=0; j<(int)clients[i]->qp->qnel; j++)
            for (chp=chfree; chp; chp=chp->next)
               if (chp == clients[i]->qp->queue[j]) {
                  warn("buf_chunk_verify: chunk [0x%x] queued to client [%d]\n",
   								chp, i);
                  abort();
               }
         for (j=0; j<(int)clients[i]->qp->wnel; j++)
            for (chp=chfree; chp; chp=chp->next)
               if (chp == clients[i]->qp->wait[j]) {
                  warn("buf_chunk_verify: chunk [0x%x] wait for client [%d]\n",
   								chp, i);
                  abort();
               }
      }
}
