/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*    client.c								*
*									*
*	Routines to manage X client list.				*
*									*
************************************************************************/
#include <sys/time.h>
#include <sys/types.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#include <stdlib.h>
#include <X11/X.h>
#include "xmx.h"
#include "res.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "incl/client.pvt.h"
#include "fd.h"

			/* the zero'th element must always be zero */
static client_t *	clientlist[NOCLIENTS];

/************************************************************************
*									*
*   client_reset							*
*									*
*	Clear client vector, freeing as we go.				*
*									*
************************************************************************/
void
client_reset
   VOID
{
   register int i;

   if (clients == 0)
      clients = clientlist;

   for (i=1; num_clients; i++)
      if (clients[i]) {
         zb_client_death(clients[i]);
         client_free(clients[i]);	/* decrements num_clients */
      }
   num_clients = 1;	/* client zero is always there */
}

/************************************************************************
*									*
*    client_alloc							*
*									*
*	Allocate a client structure and link it into the clients list.	*
*	Returns the client structure.					*
*									*
************************************************************************/
client_t *
client_alloc
   AL((fd, ap))
   DB int fd
   DD hostaddr_t *ap
   DE
{
   register int i;
   register client_t *cp;
   static rid_t id = 0;
   static u8_t clinum = 0;

   if (++clinum == NOCLIENTS)
      clinum = 1;

   for (i=(int)clinum; i<NOCLIENTS;) {
      if (clients[i] == 0)
         break;
      if (++i == NOCLIENTS)
         i = 1;
      if (i == (int)clinum)
         return 0;
   }
   clinum = (u8_t)i;

   if (MALLOC(cp, client_t *, sizeof(client_t)))
      return (client_t *)0;

   cp->fd = fd;
   cp->qp = queue_new(Q_XCLIENT, (char *)cp);
   cp->swap = 0;
   cp->clinum = clinum;
   cp->state = K_NEW;
   cp->closedownmode = DestroyAll;
   cp->id = ++id;
   cp->seqno = 0;	/* last seqno */
   cp->oseqno = 0;
   cp->refs = 0;
   cp->rxqp = rx_new();
   cp->eslp = list_new();
   cp->addr.family = ap->family;
   cp->addr.length = ap->length;
   if (ap->length) {
      if (MALLOC(cp->addr.address, char *, ap->length)) {
         free(cp);
         return (client_t *)0;
      }
      bcopy(ap->address, cp->addr.address, ap->length);
   }
   else
      cp->addr.address = 0;

   cp->base = (mask_t)clinum << RID_MASKBITS;
   cp->smap = 0;
   cp->pp = 0;
#ifdef DEBUG
   /*LINTED constant in conditional okay */
   if (sizeof(client_t) != 52)
      warn("client_alloc: not updated since client_t last changed\n");
#endif
   num_clients++;
   return clients[clinum] = cp;
}

int
client_drop
   VOID
{
   printf("client_drop does not exist\n");
   return 0;
}

/************************************************************************
*									*
*   client_free								*
*									*
*	Freeing a dead client, or a "live" client with closedownmode	*
*	DestroyAll really frees it.  Otherwise if any of it's resources	*
*	are still around, it is simply marked "dead".			*
*									*
************************************************************************/
int
client_free
   AL((cp))
   DB client_t *cp
   DE
{
   register int i;

   if (cp->pp) {
      /*
      **  If this happens, we're screwed - an incomplete request is
      **  jammed into all the server queues with no hope of ever getting
      **  the rest of it.  Xmx will just hang.  It would be pretty
      **  slick to generate the appropriate number of bytes of garbage
      **  to send to the servers to clear their queues so we could
      **  continue.  (TODO)
      */
      warn("client_free: I will hang now\n");
      pp_free(cp->pp);
      cp->pp = 0;
      fd_unpartial(cp->fd);
   }
   if (cp->state == K_DEAD || cp->closedownmode == DestroyAll){
      if (cp->refs) {
         warn("client_free: WARNING - client has refs [0x%x,%d] [%d]\n",
						cp, cp->clinum, cp->refs);
#ifdef DEBUG
         abort();
#endif
      }
      if (cp->rxqp)	/* replies will be tossed out as they come in */
         rx_free(cp->rxqp);

      i = cp->base >> RID_MASKBITS;
      if (clients[i])
         clients[i] = 0;
      else
         return err(-1, "client_free: can't free freed client! [0x%x,%d]\n",
								cp, i);
      if (cp->qp)
         queue_free(cp->qp);
      if (cp->addr.address)
         free(cp->addr.address);
      if (cp->smap)
         free(cp->smap);
      num_clients--;
      free(cp);
   }
   else
      cp->state = K_DEAD;

   return 0;
}
