/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   color.c								*
*									*
************************************************************************/
#include <sys/types.h>
#include <sys/time.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#include <stdlib.h>
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>
#include <X11/X.h>
#include <X11/Xatom.h>

#include "xmx.h"
#include "am.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "res.h"
#include "sv.h"
#include "zb.h"
#include "fd.h"
#include "incl/color.pvt.h"

static colormap_t *colormaps;
static colormap_t *last_cmap;	/* last colormap installed */

/************************************************************************
*									*
*   color_root_cmap							*
*									*
************************************************************************/
colormap_t *
color_root_cmap
   AL((id, vp, whitep, blackp, nhide, shellroot, wid, depth))
   DB rid_t id
   DD visual_t *vp
   DD pixel_t *whitep
   DD pixel_t *blackp
   DD int nhide
   DD rid_t shellroot
   DD rid_t wid
   DD u8_t depth
   DE
{
   colormap_t *mp;
   int new;
   u16_t t;

   if (MALLOC(mp, colormap_t *, sizeof(colormap_t)))
      return 0;
   mp->cmap = cmap_create(	vp->class,
				vp->nentries,
				vp->rmask,
				vp->gmask,
				vp->bmask,
				0,
				nhide);
   if (mp->cmap == 0) {
      free(mp);
      return 0;
   }
   cmap_alloc_white_black(mp->cmap, whitep, blackp);

   mp->res.client = 0;		/* client zero is xmx */
   mp->installed = 1;
   mp->bpc = vp->bprgb;
   mp->mappixels = opt.owncmap == 0 && (vp->class == GrayScale ||
					vp->class == PseudoColor ||
					vp->class == DirectColor);
   mp->depth = depth;
   mp->cid = mp->vid = id;
   mp->wid = wid;
   mp->vp = vp;
   mp->winlist = plist_add(0, vscreen.wp);
#ifdef DEBUG
   /*LINTED constant in conditional okay */
   if (sizeof(colormap_t) != 40)
      warn("color_root_cmap: not updated since colormap_t last changed\n");
#endif
   mp->last = 0;
   mp->next = colormaps;
   if (mp->next)
      mp->next->last = mp;
   colormaps = mp;
   last_cmap = mp;
   /*
   **  Some magic here.  We put shellroot at the end of the list
   **  (it grows like a stack) so that window managers will ignore
   **  the top-level virtual root window colormap attribute and
   **  pay attention to the rest of the list.  Shellroot never
   **  actually heads the list.  These two never get removed.
   */
   cmapwin_install(shellroot);
   cmapwin_install(mp->wid);

   hash_add(vmap, id, id, (char *)mp);

   return mp;
}

/************************************************************************
*									*
*   color_create_cmap							*
*									*
************************************************************************/
int
color_create_cmap
   AL((cp, p, chpp))
   DB client_t *cp
   DD xCreateColormapReq *p
   DD chunk_t **chpp
   DE
{
   window_t *wp;
   colormap_t *mp;
   visual_t *vp;
   u32_t vals[2];

   if ((wp = (window_t *)hash_data(vmap, p->window)) == 0) {
      proto_Error(cp, BadWindow, p->window, 0, X_CreateColormap);
      return -1;
   }
   if ((vp = (visual_t *)hash_data(vmap, p->visual)) == 0) {
      proto_Error(cp, BadValue, p->visual, 0, X_CreateColormap);
      return -1;
   }
   if (MALLOC(mp, colormap_t *, sizeof(colormap_t))) {
      proto_Error(cp, BadAlloc, 0, 0, X_CreateColormap);
      return -1;
   }
   mp->cmap = cmap_create(	vp->class,
				vp->nentries,
				vp->rmask,
				vp->gmask,
				vp->bmask,
				p->alloc,
				0);
   if (mp->cmap == 0) {
      free(mp);
      proto_Error(cp, BadAlloc, 0, 0, X_CreateColormap);
      return -1;
   }
   if ((mp->vid = hash_add_client_id(p->mid, (char *)mp)) == 0) {
      cmap_free(mp->cmap);
      free(mp);
      proto_Error(cp, BadIDChoice, p->mid, 0, X_CreateColormap);
      return -1;
   }
   mp->wid = util_new_client_id();
   hash_add(vmap, mp->wid, mp->wid, (char *)0);
   mp->res.client = cp;
   mp->installed = 0;
   mp->bpc = vp->bprgb;
   mp->mappixels = 0;	/* only default root colormaps */
   mp->depth = vp->depth;
   mp->cid = p->mid;
   mp->vp = vp;
   mp->winlist = 0;		/* initially, no windows use it */
#ifdef DEBUG
   /*LINTED constant in conditional okay */
   if (sizeof(colormap_t) != 40)
      warn("color_create_cmap: not updated since colormap_t last changed\n");
#endif

   mp->last = 0;
   mp->next = colormaps;
   if (mp->next)
      mp->next->last = mp;
   colormaps = mp;

   cp->refs++;

   switch (mp->vp->class) {
      case GrayScale:
      case PseudoColor:
      case DirectColor:
         p->alloc = 1;	/* all allocable colormaps are allocated */
         break;
   }
   /*
   **  Create an unmapped, 1x1, dummy window to be used to refer to
   **  this colormap in the WM_COLORMAP_WINDOWS property.
   */
   zb_dest(ZD_NONE, 0);
   vals[0] = vscreen.blackPixel;
   vals[1] = mp->cid;
   proto_CreateWindow(	mp->depth,
			mp->wid,
			vscreen.shellroot,
			0, 0,
			1, 1,
			0,
			InputOutput,
			mp->vp->cid,
			CWBorderPixel | CWColormap,
			vals,
			0);
   *chpp = zb_split();

   return 0;
}

/************************************************************************
*									*
*   color_free_cmap							*
*									*
************************************************************************/
int
color_free_cmap
   AL((cp, p, chpp))
   DB client_t *cp
   DD xResourceReq *p
   DD chunk_t **chpp
   DE
{
   colormap_t *mp;

   if ((mp = (colormap_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadColor, p->id, 0, X_FreeColormap);
      return -1;
   }
   if (mp->res.client == 0)		/* it's a default colormap */
      return 0;

   if (mp->installed) {
      uninstall_colormap(mp);

      zb_dest(ZD_NONE, 0);
      color_update_winvec();
      proto_DestroyWindow(mp->wid);
      *chpp = zb_split();
   }
   else
      colormap_change(vscreen.wp, mp, ColormapUninstalled);

   free_colormap(mp);

   return 0;
}

/************************************************************************
*									*
*   color_copy_cmap_and_free						*
*									*
************************************************************************/
int
color_copy_cmap_and_free
   AL((cp, p, chpp))
   DB client_t *cp
   DD xCopyColormapAndFreeReq *p
   DD chunk_t **chpp
   DE
{
   register int n;
   register chunk_t *chp;
   colormap_t *omp, *nmp;
   xColorItem cbuf[256];
   u32_t vals[2];

   if ((omp = (colormap_t *)hash_data(vmap, p->srcCmap)) == 0) {
      proto_Error(cp, BadColor, p->srcCmap, 0, X_CopyColormapAndFree);
      return -1;
   }
   if ((nmp = (colormap_t *)hash_data(vmap, p->mid)) != 0) {
      proto_Error(cp, BadIDChoice, p->mid, 0, X_CopyColormapAndFree);
      return -1;
   }
   if (MALLOC(nmp, colormap_t *, sizeof(colormap_t))) {
      proto_Error(cp, BadAlloc, 0, 0, X_CopyColormapAndFree);
      return -1;
   }
   if ((nmp->cmap = cmap_copy_and_free(omp->cmap, cp->clinum)) == 0) {
      free(nmp);
      proto_Error(cp, BadAlloc, 0, 0, X_CopyColormapAndFree);
      return -1;
   }
   if ((nmp->vid = hash_add_client_id(p->mid, (char *)nmp)) == 0) {
      cmap_free(nmp->cmap);
      free(nmp);
      proto_Error(cp, BadIDChoice, p->mid, 0, X_CopyColormapAndFree);
      return -1;
   }
   nmp->wid = util_new_client_id();
   hash_add(vmap, nmp->wid, nmp->wid, (char *)0);
   nmp->res.client = cp;
   nmp->installed = 0;
   nmp->bpc = omp->bpc;
   nmp->mappixels = 0;	/* only default root colormaps */
   nmp->cid = p->mid;
   nmp->vp = omp->vp;
#ifdef DEBUG
   /*LINTED constant in conditional okay */
   if (sizeof(colormap_t) != 40)
      warn("color_copy_cmap_and_free: not updated since colormap_t last changed\n");
#endif

   nmp->last = 0;
   nmp->next = colormaps;
   if (nmp->next)
      nmp->next->last = nmp;
   colormaps = nmp;

   cp->refs++;

   zb_dest(ZD_NONE, 0);
   proto_CreateColormap(	nmp->cid,
				nmp->vp->cid,
				vscreen.wp->cid,
				xTrue);
   vals[0] = vscreen.blackPixel;
   vals[1] = nmp->cid;
   proto_CreateWindow(	nmp->depth,
			nmp->wid,
			vscreen.shellroot,
			0, 0,
			1, 1,
			0,
			InputOutput,
			nmp->vp->cid,
			CWBorderPixel | CWColormap,
			vals,
			0);

   while (n = cmap_get_cells(nmp->cmap, CS_ALLOC, 256, cbuf))
      proto_StoreColors(nmp->cid, n, cbuf, nmp->mappixels);

   *chpp = zb_split();

   return 0;
}

/************************************************************************
*									*
*   color_install_cmap							*
*									*
************************************************************************/
int
color_install_cmap
   AL((cp, p, chpp))
   DB client_t *cp
   DD xResourceReq *p
   DD chunk_t **chpp
   DE
{
   register colormap_t *mp;

   if ((mp = (colormap_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadColor, p->id, 0, X_InstallColormap);
      return -1;
   }
   if (mp->installed == 0) {
      install_colormap(mp);

      zb_dest(ZD_NONE, 0);
      color_update_winvec();
      *chpp = zb_split();
   }
   return 0;
}

/************************************************************************
*									*
*   color_uninstall_cmap						*
*									*
************************************************************************/
int
color_uninstall_cmap
   AL((cp, p, chpp))
   DB client_t *cp
   DD xResourceReq *p
   DD chunk_t **chpp
   DE
{
   register colormap_t *mp;

   if ((mp = (colormap_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadColor, p->id, 0, X_UninstallColormap);
      return -1;
   }
   if (mp->installed) {
      uninstall_colormap(mp);

      zb_dest(ZD_NONE, 0);
      color_update_winvec();
      *chpp = zb_split();
   }
   return 0;
}

/************************************************************************
*									*
*   color_list_cmaps							*
*									*
************************************************************************/
void
color_list_cmaps
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p	/* not used TODO */
   DE
{
   colormap_t *mp;
   register list_t *lstp;

   lstp = list_new();
   for (mp=colormaps; mp; mp=mp->next)
      if (mp->installed)
         list_put(lstp, mp->cid);

   proto_ListInstalledColormapsReply(cp, lstp);
   list_free(lstp);
}

/************************************************************************
*									*
*   color_alloc								*
*									*
************************************************************************/
int
color_alloc
   AL((cp, p, chpp))
   DB client_t *cp
   DD xAllocColorReq *p
   DD chunk_t **chpp
   DE
{
   register colormap_t *mp;
   int n, new;
   register sv_t *svp;
   static xColorItem c;

   if ((mp = (colormap_t *)hash_data(vmap, p->cmap)) == 0) {
      proto_Error(cp, BadColor, p->cmap, 0, X_AllocColor);
      return -1;
   }
   if (cmap_alloc_read(mp->cmap, cp->clinum, p->red, p->green, p->blue,
				&c.pixel, &c.red, &c.green, &c.blue, &new)) {
      DEBUG4(D_COLOR, "color_alloc: BadAlloc [0x%x] [%x.%x.%x]\n",
					mp->cid, p->red, p->green, p->blue);
      proto_Error(cp, BadAlloc, 0, 0, X_AllocColor);
      return -1;
   }
   if (new) {
      if (mp->mappixels) {	/* shared colormap, really allocate */
         svp = sv_new();
         rx_begin_repeat(cp->rxqp);
         df_put_p((void *)svp);
         df_put_p((void *)mp);
         df_put_i((int)c.pixel);
         rx_queue(cp->rxqp, X_AllocColor, 0, cp->seqno, alloc_color_reply);
         rx_end_repeat(cp->rxqp);

         df_put_i((int)cp->clinum);
         df_put_p((void *)mp->cmap);
         df_put_p((void *)svp);
         df_put_i((int)cp->seqno);
         df_put_p((void *)&c);
         cx_push(cp->rxqp, alloc_color_cont);

         fd_reset(-1);
         return 0;		/* forward request to servers... */
      }
      c.flags = DoRed | DoGreen | DoBlue;
      zb_dest(ZD_NONE, 0);
      proto_StoreColors(mp->cid, 1, &c, mp->mappixels);
      *chpp = zb_split();
   }
   proto_AllocColorReply(cp, c.pixel, c.red, c.green, c.blue);
   DEBUG5(D_COLOR, "color_alloc: alloc'd [0x%x] pixel %d [%x.%x.%x]\n",
				mp->cid, c.pixel, c.red, c.green, c.blue);
   return 1;
}

void
alloc_color_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp                                                 
   DD client_t *cp                                                    
   DD u8_t major                                                      
   DD u16_t minor
   DD u16_t seqno                                                        
   DD chunk_t *chp                                           
   DE                         
{
   register int i;
   register chunk_t *nchp;
   pixel_t real_pixel;
   xGenericReply *rp = (xGenericReply *)buf_data(chp);
   register sv_t *svp = (sv_t *)df_get_p(dfp);
   register colormap_t *mp = (colormap_t *)df_get_p(dfp);
   pixel_t pixel = (pixel_t)df_get_i(dfp);

   if (rp->type == X_Error) {
      if (!sv_failed(svp)) {	/* first error? */
         if (sv_some_set(svp)) {	/* any previous successes? */
            /*
            **  roll back earlier allocations
            */
            zb_dest(ZD_NONE, 0);
            proto_FreeColors(mp->cid, 0, 1, &pixel, mp->mappixels);
            nchp = zb_split();
   
            for (i=0; i<num_serv; i++)
               if (sv_isset(svp, servers[i]->fd)) {	/* succeeded? */
                  queue_add(servers[i]->qp, nchp);
                  sv_clear(svp, servers[i]->fd);
               }
   
            buf_clear(nchp);
         }
         /*
         **  send this error to client (first come, first served)
         */
         if (cp)
            queue_add(cp->qp, chp);
         sv_fail(svp);
      }
   }
   else { /* X_Reply */
      if (major == X_AllocColor)
         real_pixel = ((xAllocColorReply *)rp)->pixel;
      else      /* X_AllocNamedColor */
         real_pixel = ((xAllocNamedColorReply *)rp)->pixel;

      if (sv_failed(svp)) {
         /*
         **  roll back this allocation
         */
         zb_dest(ZD_SERV, sp);
         proto_FreeColors(mp->cid, 0, 1, &real_pixel, mp->mappixels);
      }
      else
         pmap_map(sp, mp, 1, &pixel, 0, &real_pixel, 0);
   }
}

void
alloc_color_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   register u8_t clinum = (u8_t)df_get_i(dfp);
   register cmap_t *cmp = (cmap_t *)df_get_p(dfp);
   register sv_t *svp = (sv_t *)df_get_p(dfp);
   register u16_t seqn = (u16_t)df_get_i(dfp);
   register xColorItem *cip = (xColorItem *)df_get_p(dfp);

   if (sv_failed(svp)) {
      cmap_free_cells(cmp, clinum, 1, &cip->pixel, 0);
   }
   else
      proto_AllocColorReply(	client_ptr(clinum),
				cip->pixel,
				cip->red,
				cip->green,
				cip->blue);
   fd_unreset();
}

/************************************************************************
*									*
*   color_alloc_named							*
*									*
************************************************************************/
int
color_alloc_named
   AL((cp, p, chpp))
   DB client_t *cp
   DD xAllocNamedColorReq *p
   DD chunk_t **chpp
   DE
{
   register colormap_t *mp;
   register sv_t *svp;
   int n, new;
   static xColorItem c, rc;

   if ((mp = (colormap_t *)hash_data(vmap, p->cmap)) == 0) {
      proto_Error(cp, BadColor, p->cmap, 0, X_AllocNamedColor);
      return -1;
   }
   if (rgb_find((char *)(p+1), p->nbytes, &rc.red, &rc.green, &rc.blue)) {
      proto_Error(cp, BadName, 0, 0, X_AllocNamedColor);
      return -1;
   }
   if (cmap_alloc_read(mp->cmap, cp->clinum, rc.red, rc.green, rc.blue,
				&c.pixel, &c.red, &c.green, &c.blue, &new)) {
      DEBUG4(D_COLOR, "color_alloc_named: BadAlloc [0x%x] [%x.%x.%x]\n",
					mp->cid, rc.red, rc.green, rc.blue);
      proto_Error(cp, BadAlloc, 0, 0, X_AllocNamedColor);
      return -1;
   }
   if (new) {
      if (mp->mappixels) {	/* shared colormap, really allocate */
         svp = sv_new();
         rx_begin_repeat(cp->rxqp);
         df_put_p((void *)svp);
         df_put_p((void *)mp);
         df_put_i((int)c.pixel);
         rx_queue(cp->rxqp, X_AllocNamedColor, 0, cp->seqno, alloc_color_reply);
         rx_end_repeat(cp->rxqp);

         df_put_i((int)cp->clinum);
         df_put_p((void *)mp->cmap);
         df_put_p((void *)svp);
         df_put_i((int)cp->seqno);
         df_put_p((void *)&rc);
         df_put_p((void *)&c);
         cx_push(cp->rxqp, alloc_named_color_cont);

         fd_reset(-1);
         return 0;         /* forward request to servers... */
      }
      c.flags = DoRed | DoGreen | DoBlue;
      zb_dest(ZD_NONE, 0);
      proto_StoreColors(mp->cid, 1, &c, mp->mappixels);
      *chpp = zb_split();
   }
   proto_AllocNamedColorReply(cp, c.pixel, rc.red, rc.green, rc.blue,
						c.red, c.green, c.blue);
   DEBUG5(D_COLOR, "color_alloc_named: alloc'd [0x%x] pixel %d [%x.%x.%x]\n",
				mp->cid, c.pixel, c.red, c.green, c.blue);
   return 1;
}

void
alloc_named_color_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   register u8_t clinum = (u8_t)df_get_i(dfp);
   register cmap_t *cmp = (cmap_t *)df_get_p(dfp);
   register sv_t *svp = (sv_t *)df_get_p(dfp);
   register u16_t seqn = (u16_t)df_get_i(dfp);
   register xColorItem *rcip = (xColorItem *)df_get_p(dfp);
   register xColorItem *cip = (xColorItem *)df_get_p(dfp);

   if (sv_failed(svp))
      cmap_free_cells(cmp, clinum, 1, &cip->pixel, 0);
   else
      proto_AllocNamedColorReply(	client_ptr(clinum),
					cip->pixel,
					rcip->red, rcip->green, rcip->blue,
					cip->red, cip->green, cip->blue);
   fd_unreset();
}

/************************************************************************
*									*
*   color_alloc_cells							*
*									*
************************************************************************/
int
color_alloc_cells
   AL((cp, p))
   DB client_t *cp
   DD xAllocColorCellsReq *p
   DE
{
   register colormap_t *mp;
   register sv_t *svp;
   static pixel_t *pixelp;
   static mask_t rmask, gmask, bmask;

   if ((mp = (colormap_t *)hash_data(vmap, p->cmap)) == 0) {
      proto_Error(cp, BadColor, p->cmap, 0, X_AllocColorCells);
      return -1;
   }
   if (cmap_alloc_cells(	mp->cmap,
				cp->clinum,
				p->colors,
				p->planes,
				p->contiguous,
				&pixelp,
				&rmask, &gmask, &bmask)) {
         DEBUG3(D_COLOR, "color_alloc_cells: BadAlloc [0x%x] [%d.%d]\n",
						mp->cid, p->colors, p->planes);
         proto_Error(cp, BadAlloc, 0, 0, X_AllocColorCells);
         return -1;
   }
   if (mp->mappixels) {		/* shared colormap, really allocate */
      DEBUG3(D_COLOR, "color_alloc_cells: *really* alloc'ing [0x%x] [%d.%d]\n",
					mp->cid, p->colors, p->planes);
      /*
      **  note that we put some static data on the queues below, which
      **  is okay in this case only because of the fd_reset below, that
      **  insures that we won't ever do two of these simultaneously.
      */
      svp = sv_new();
      rx_begin_repeat(cp->rxqp);
      df_put_p((void *)svp);
      df_put_p((void *)mp);
      df_put_i((int)p->colors);
      df_put_p((void *)pixelp);
      df_put_i((int)(rmask | gmask | bmask));	/* composite mask */
      rx_queue(cp->rxqp, X_AllocColorCells, 0, cp->seqno, alloc_cells_reply);
      rx_end_repeat(cp->rxqp);

      df_put_i((int)cp->clinum);
      df_put_p((void *)mp->cmap);
      df_put_p((void *)svp);
      df_put_i((int)cp->seqno);
      df_put_i((int)p->colors);
      df_put_i((int)p->planes);
      df_put_p((void *)pixelp);
      df_put_i((int)rmask);
      df_put_i((int)gmask);
      df_put_i((int)bmask);
      cx_push(cp->rxqp, alloc_cells_cont);

      fd_reset(-1);
      return 0;		/* forward request to servers... */
   }
   DEBUG3(D_COLOR, "color_alloc_cells: alloc'd [0x%x] [%d.%d]\n",
					mp->cid, p->colors, p->planes);
   proto_AllocColorCellsReply(	cp,
				p->colors,
				p->planes,
				pixelp,
				rmask, gmask, bmask);
   return 1;
}

void
alloc_cells_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register int i;
   register chunk_t *nchp;
   register int npixels, nmasks;
   pixel_t *real_pixels;
   mask_t real_mask, *maskp;
   xGenericReply *rp = (xGenericReply *)buf_data(chp);
   register sv_t *svp = (sv_t *)df_get_p(dfp);
   register colormap_t *mp = (colormap_t *)df_get_p(dfp);
   register int ncolors = df_get_i(dfp);
   register pixel_t *pixelp = (pixel_t *)df_get_p(dfp);
   register mask_t mask = (mask_t)df_get_i(dfp);

   if (rp->type == X_Error) {
      DEBUG3(D_COLOR, "alloc_cells_reply: X_Error [0x%x] [%d] %s\n",
		mp->cid, ncolors, sv_some_set(svp) ? "rolling back" : "");
      if (!sv_failed(svp)) {		/* first error? */
         if (sv_some_set(svp)) {	/* any previous successes? */
            /*
            **  roll back earlier allocations
            */
            zb_dest(ZD_NONE, 0);
            proto_FreeColors(mp->cid, mask, ncolors, pixelp, mp->mappixels);
            nchp = zb_split();
   
            for (i=0; i<num_serv; i++)
               if (sv_isset(svp, servers[i]->fd)) {	/* succeeded? */
                  queue_add(servers[i]->qp, nchp);
                  sv_clear(svp, servers[i]->fd);
               }
   
            buf_clear(nchp);
         }
         sv_fail(svp);
         /*
         **  send this error to client (first come, first served)
         */
         queue_add(cp->qp, chp);
      }
   }
   else {	/* X_Reply */
      if (major == X_AllocColorCells) {
         npixels = ((xAllocColorCellsReply *)rp)->nPixels;
         nmasks = ((xAllocColorCellsReply *)rp)->nMasks;
         real_pixels = (pixel_t *)((xAllocColorCellsReply *)rp + 1);
         maskp = (mask_t *)(real_pixels[npixels]);
         real_mask = 0;
         for (i=0; i<nmasks; i++)
            real_mask |= maskp[i];
      }
      else {    /* X_AllocColorPlanes */
         npixels = ((xAllocColorPlanesReply *)rp)->nPixels;
         real_pixels = (pixel_t *)((xAllocColorPlanesReply *)rp + 1);
         real_mask =	((xAllocColorPlanesReply *)rp)->redMask |
			((xAllocColorPlanesReply *)rp)->greenMask |
			((xAllocColorPlanesReply *)rp)->blueMask;
      }
      DEBUG3(D_COLOR, "alloc_cells_reply: X_Reply [0x%x] [%d] %s\n",
		mp->cid, ncolors, sv_failed(svp) ? "rolling back" : "okay");
      if (sv_failed(svp)) {
         /*
         **  roll back this allocation
         */
         zb_dest(ZD_SERV, sp);
         proto_FreeColors(mp->cid, real_mask, npixels, real_pixels,
							mp->mappixels);
      }
      else
         pmap_map(sp, mp, ncolors, pixelp, mask, real_pixels, real_mask);
   }
}

void
alloc_cells_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   register u8_t clinum = (u8_t)df_get_i(dfp);
   register cmap_t *cmp = (cmap_t *)df_get_p(dfp);
   register sv_t *svp = (sv_t *)df_get_p(dfp);
   register u16_t seqn = (u16_t)df_get_i(dfp);
   register u16_t npixels = df_get_i(dfp);
   register u16_t nmasks = df_get_i(dfp);
   register pixel_t *pixelp = (pixel_t *)df_get_p(dfp);
   register mask_t rmask = (mask_t)df_get_i(dfp);
   register mask_t gmask = (mask_t)df_get_i(dfp);
   register mask_t bmask = (mask_t)df_get_i(dfp);

   DEBUG3(D_COLOR, "alloc_cells_cont: [%d.%d] %s\n",
	npixels, nmasks, sv_failed(svp) ? "failed" : "success - replying");
   if (sv_failed(svp))
      cmap_free_cells(cmp, clinum, npixels, pixelp, rmask | gmask | bmask);
   else
      proto_AllocColorCellsReply(	client_ptr(clinum),
					npixels, nmasks,
					pixelp,
					rmask, gmask, bmask);
   fd_unreset();
}

/************************************************************************
*									*
*   color_alloc_planes							*
*									*
************************************************************************/
int
color_alloc_planes
   AL((cp, p))
   DB client_t *cp
   DD xAllocColorPlanesReq *p
   DE
{
   register colormap_t *mp;
   register sv_t *svp;
   static pixel_t *pixelp;
   static mask_t rmask, gmask, bmask;

   if ((mp = (colormap_t *)hash_data(vmap, p->cmap)) == 0) {
      proto_Error(cp, BadColor, p->cmap, 0, X_AllocColorPlanes);
      return -1;
   }
   if (cmap_alloc_planes(	mp->cmap,
				cp->clinum,
				p->contiguous,
				p->colors,
				p->red, p->green, p->blue,
				&pixelp,
				&rmask, &gmask, &bmask)) {
      DEBUG5(D_COLOR, "color_alloc_planes: BadAlloc [0x%x] [%d,%x.%x.%x]\n",
				mp->cid, p->colors, p->red, p->green, p->blue);
      proto_Error(cp, BadAlloc, 0, 0, X_AllocColorPlanes);
      return -1;
   }
   if (mp->mappixels) {		/* shared colormap, really allocate */
      DEBUG5(D_COLOR,
		"color_alloc_planes: *really* alloc'ing [0x%x] [%d/%x.%x.%x]\n",
		mp->cid, p->colors, p->red, p->green, p->blue);
      /*
      **  note that we put some static data on the queues below, which
      **  is okay in this case only because of the fd_reset below, that
      **  insures that we won't ever do two of these simultaneously.
      */
      svp = sv_new();
      rx_begin_repeat(cp->rxqp);
      df_put_p((void *)svp);
      df_put_i((int)p->colors);
      df_put_p((void *)pixelp);
      df_put_i((int)(rmask | gmask | bmask)); /* composite mask */
      rx_queue(cp->rxqp, X_AllocColorPlanes, 0, cp->seqno, alloc_cells_reply);
      rx_end_repeat(cp->rxqp);
 
      df_put_i((int)cp->clinum);
      df_put_p((void *)mp->cmap);
      df_put_p((void *)svp);
      df_put_i((int)cp->seqno);
      df_put_i((int)p->colors);
      df_put_p((void *)pixelp);
      df_put_i((int)rmask);
      df_put_i((int)gmask);
      df_put_i((int)bmask);
      cx_push(cp->rxqp, alloc_planes_cont);
 
      fd_reset(-1);
      return 0;         /* forward request to servers... */
   }
   DEBUG5(D_COLOR, "color_alloc_planes: alloc'd [0x%x] [%d,%x.%x.%x]\n",
				mp->cid, p->colors, p->red, p->green, p->blue);
   proto_AllocColorPlanesReply(	cp,
				p->colors,
				rmask, gmask, bmask,
				pixelp);
   return 1;
}

void
alloc_planes_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   register u8_t clinum = (u8_t)df_get_i(dfp);
   register cmap_t *cmp = (cmap_t *)df_get_p(dfp);
   register sv_t *svp = (sv_t *)df_get_p(dfp);
   register u16_t seqn = (u16_t)df_get_i(dfp);
   register int npixels = df_get_i(dfp);
   register pixel_t *pixelp = (pixel_t *)df_get_p(dfp);
   register mask_t rmask = (mask_t)df_get_i(dfp);
   register mask_t gmask = (mask_t)df_get_i(dfp);
   register mask_t bmask = (mask_t)df_get_i(dfp);

   DEBUG5(D_COLOR, "alloc_planes_cont: [%d/%x.%x.%x] %s\n",
			npixels, rmask, gmask, bmask,
			sv_failed(svp) ? "failed" : "success - replying");
   if (sv_failed(svp))
      cmap_free_cells(cmp, clinum, npixels, pixelp, rmask | gmask | bmask);
   else
      proto_AllocColorPlanesReply(	client_ptr(clinum),
					npixels,
					rmask, gmask, bmask,
					pixelp);
   fd_unreset();
}

/************************************************************************
*									*
*   color_free								*
*									*
************************************************************************/
colormap_t *
color_free
   AL((cp, p))
   DB client_t *cp
   DD xFreeColorsReq *p
   DE
{
   register colormap_t *mp;

   if ((mp = (colormap_t *)hash_data(vmap, p->cmap)) == 0) {
      proto_Error(cp, BadColor, p->cmap, 0, X_FreeColors);
      return 0;
   }
   if (cmap_free_cells(	mp->cmap,
			cp->clinum,
			p->length - sz_xFreeColorsReq/4,
			(pixel_t *)(p+1),
			p->planeMask)) {
      proto_Error(cp, BadValue, 0, 0, X_FreeColors);/* not good TODO */
      return 0;
   }
   if (mp->mappixels) {		/* shared colormap, really free */
      DEBUG2(D_COLOR, "color_free: *really* freeing [0x%x] [%d]\n", mp->cid,
					p->length - sz_xFreeColorsReq/4);
      return mp;
   }
   DEBUG2(D_COLOR, "color_free: freed [0x%x] [%d]\n", mp->cid,
					p->length - sz_xFreeColorsReq/4);
   return 0;
}

/************************************************************************
*									*
*   color_store								*
*									*
************************************************************************/
colormap_t *
color_store
   AL((cp, p))
   DB client_t *cp
   DD xStoreColorsReq *p
   DE
{
   register int n;
   register xColorItem *c;
   register colormap_t *mp;

   if ((mp = (colormap_t *)hash_data(vmap, p->cmap)) == 0) {
      proto_Error(cp, BadColor, p->cmap, 0, X_StoreColors);
      return 0;
   }
   n = (int)(p->length - sz_xStoreColorsReq/4) / (sz_xColorItem/4);
   for (c=(xColorItem *)(p+1); n; c++, n--)
      if (cmap_store(	mp->cmap,
			cp->clinum,
			c->pixel,
			c->red, c->green, c->blue,
			c->flags&DoRed, c->flags&DoGreen, c->flags&DoBlue)) {
         proto_Error(cp, BadAccess, c->pixel, 0, X_StoreColors);
         return 0;
      }
   if (mp->mappixels) {		/* shared colormap, really free */
      DEBUG5(D_COLOR, "color_store: *really* [0x%x] %d [%x.%x.%x]\n", mp->cid,
					c->pixel, c->red, c->green, c->blue);
      return mp;
   }
   DEBUG5(D_COLOR, "color_store: [0x%x] %d [%x.%x.%x]\n", mp->cid,
					c->pixel, c->red, c->green, c->blue);
   return 0;
}

/************************************************************************
*									*
*   color_store_named							*
*									*
************************************************************************/
colormap_t *
color_store_named
   AL((cp, p))
   DB client_t *cp
   DD xStoreNamedColorReq *p
   DE
{
   register colormap_t *mp;
   u16_t max, red, green, blue;

   if ((mp = (colormap_t *)hash_data(vmap, p->cmap)) == 0) {
      proto_Error(cp, BadColor, p->cmap, 0, X_StoreNamedColor);
      return 0;
   }
   if (rgb_find((char *)(p+1), p->nbytes, &red, &green, &blue)) {
      proto_Error(cp, BadName, 0, 0, X_StoreNamedColor);
      return 0;
   }
/* unnecessary  deleteme TODO
   max = (1 << mp->bpc) - 1;
   red = (u16_t)(((u32_t)red * MAXCOLVAL) / max);
   green = (u16_t)(((u32_t)green * MAXCOLVAL) / max);
   blue = (u16_t)(((u32_t)blue * MAXCOLVAL) / max);
*/
   if (cmap_store(	mp->cmap,
			cp->clinum,
			p->pixel,
			red, green, blue,
			p->flags&DoRed, p->flags&DoGreen, p->flags&DoBlue)) {
      proto_Error(cp, BadAccess, 0, 0, X_StoreNamedColor);
      return 0;
   }
   DEBUG5(D_COLOR, "color_store_named: [0x%x] %d [%x.%x.%x]\n", mp->cid,
					p->pixel, red, green, blue);
   return mp;
}

/************************************************************************
*									*
*   color_query								*
*									*
************************************************************************/
int
color_query
   AL((cp, p))
   DB client_t *cp
   DD xQueryColorsReq *p
   DE
{
   register colormap_t *mp;
   register list_t *lstp;

   if ((mp = (colormap_t *)hash_data(vmap, p->cmap)) == 0) {
      proto_Error(cp, BadColor, p->cmap, 0, X_QueryColors);
      return -1;
   }
   lstp = list_new();
   if (cmap_query(mp->cmap, p->length-2, (pixel_t *)(p+1), lstp)) {
      proto_Error(cp, BadValue, list_get(lstp), 0, X_QueryColors);
      list_free(lstp);
      return -1;
   }
   proto_QueryColorsReply(cp, lstp);
   list_free(lstp);

   return 0;
}

/************************************************************************
*									*
*   color_lookup							*
*									*
************************************************************************/
int
color_lookup
   AL((cp, p))
   DB client_t *cp
   DD xLookupColorReq *p
   DE
{
   register colormap_t *mp;
   u16_t max, exr, exg, exb;
   u16_t r, g, b;

   if ((mp = (colormap_t *)hash_data(vmap, p->cmap)) == 0) {
      proto_Error(cp, BadColor, p->cmap, 0, X_LookupColor);
      return -1;
   }
   if (rgb_find((char *)(p+1), p->nbytes, &exr, &exg, &exb)) {
      proto_Error(cp, BadName, 0, 0, X_LookupColor);
      return -1;
   }
/* unnecessary  deleteme TODO
   max = (1 << mp->bpc) - 1;
   exr = (u16_t)(((u32_t)exr * MAXCOLVAL) / max);
   exg = (u16_t)(((u32_t)exg * MAXCOLVAL) / max);
   exb = (u16_t)(((u32_t)exb * MAXCOLVAL) / max);
*/
   cmap_lookup(mp->cmap, exr,exg,exb, &r,&g,&b);

   proto_LookupColorReply(cp, exr, exg, exb, r, g, b);

   return 0;
}

/************************************************************************
*									*
*   color_winlist_add							*
*									*
************************************************************************/
void
color_winlist_add
   AL((mp, wp))
   DB colormap_t *mp
   DD window_t *wp
   DE
{
   if (mp)
      mp->winlist = plist_add(mp->winlist, wp);
}

/************************************************************************
*									*
*   color_winlist_delete						*
*									*
************************************************************************/
void
color_winlist_delete
   AL((mp, wp))
   DB colormap_t *mp
   DD window_t *wp
   DE
{
   if (mp)
      mp->winlist = plist_delete(mp->winlist, wp, 0);
}

/************************************************************************
*									*
*   color_client_death							*
*									*
************************************************************************/
void
color_client_death
   AL((cp))
   DB client_t *cp
   DE
{
   register colormap_t *mp, *nmp;

   if (colormaps) {
      for (nmp=colormaps; nmp;) {
         mp = nmp;
         nmp = nmp->next;
         if (client_cmp(cp, mp->res.client)) {
            proto_DestroyWindow(mp->wid);
            proto_FreeColormap(mp->cid);
            if (mp->installed)
               uninstall_colormap(mp);
            free_colormap(mp);
         }
         else	/* this is an exhaustive search...yuk - TODO */
            cmap_free_client(mp->cmap, cp);
      }
      color_update_winvec();
   }
}

/************************************************************************
*									*
*   color_reset								*
*									*
************************************************************************/
void
color_reset
   VOID
{
   register colormap_t *mp, *nmp;

   if (colormaps) {
      for (nmp=colormaps; mp=nmp;) {
         nmp = nmp->next;
         proto_DestroyWindow(mp->wid);
         proto_FreeColormap(mp->cid);
         if (mp->installed)
            uninstall_colormap(mp);
         free_colormap(mp);
      }
      color_update_winvec();
   }
}

/************************************************************************
*									*
*   color_ketchup							*
*									*
************************************************************************/
void
color_ketchup
   VOID
{
   register int n, pseudo;
   register colormap_t *mp;
   xColorItem cbuf[256];
   u32_t vals[2];

   for (mp=colormaps; mp; mp=mp->next) {
      switch (mp->vp->class) {
         case GrayScale:
         case PseudoColor:
         case DirectColor:
            pseudo = 1;
            break;
         case StaticGray:
         case StaticColor:
         case TrueColor:
            pseudo = 0;
            break;
      }
      if (mp == vscreen.colormap) {	/* default colormap */
         if (mp->mappixels)
            color_alloc_root_ketchup(mp);
      }
      else {
         proto_CreateColormap(	mp->cid,
				mp->vp->cid,
				vscreen.vservroot,
				pseudo);
	 vals[0] = vscreen.blackPixel;
	 vals[1] = mp->cid;
         proto_CreateWindow(	mp->depth,
				mp->wid,
				vscreen.shellroot,
				0, 0,
				1, 1,
				0,
				InputOutput,
				mp->vp->cid,
				CWBorderPixel | CWColormap,
				vals, 0);
         if (pseudo)
            while (n = cmap_get_cells(mp->cmap, CS_ALLOC, 256, cbuf))
               proto_StoreColors(mp->cid, n, cbuf, mp->mappixels);
      }
   }
}

/************************************************************************
*									*
*   color_alloc_root_ketchup						*
*									*
*	Allocate read-only and read/write cells in a shared root	*
*	colormap.							*
*									*
*	There's a lot going on here.  The basic idea is to allocate	*
*	read-only and read/write cells in the root colormaps of each	*
*	new server, to match those that have already been allocated	*
*	in other servers' colormaps during the session so far.		*
*	This involves a lot of server dialog, as each allocation	*
*	generates a reply, and each reply must be noted in a		*
*	pixel-map, which is later used to massage pixel values on a	*
*	per-server basis.						*
*									*
*	As if that wasn't enough, we gracefully handle servers that	*
*	fail with allocation errors - hey, you can't expect everyone	*
*	have empty colormaps, can you?  So, if an allocation fails in	*
*	a shared root colormap on a particular server, we allocate a	*
*	new colormap there, and attempt to make all the allocations	*
*	in it instead.  That must succeed or something is really	*
*	bizarre (and bad things will ensue).				*
*									*
************************************************************************/
void
color_alloc_root_ketchup
   AL((mp))
   DB colormap_t *mp
   DE
{
   register int i, n;
   register sv_t *svp;
   xColorItem cbuf[256];
   xColorItem *cip;

   rx_begin_repeat(zrxqp);
   svp = sv_new();
   while (n = cmap_get_cells(mp->cmap, CS_READ, 256, cbuf))
      for (i=0; i<n; i++) {
         proto_AllocColor(mp->cid, cbuf[i].red, cbuf[i].green, cbuf[i].blue);
         df_put_p((void *)svp);
         df_put_p((void *)mp);
         df_put_i(cbuf[i].pixel);
         df_put_i((int)cbuf[i].red);
         df_put_i((int)cbuf[i].green);
         df_put_i((int)cbuf[i].blue);
         df_put_i(0);	/* try 0 */
         rx_queue(zrxqp, X_AllocColor, 0, zb_seqno(),
						alloc_color_ketchup_reply);
      }

   for (;;) {
      if (MALLOC(cip, xColorItem *, 256 * sizeof(xColorItem))) {
         warn("color_alloc_root_ketchup/color.c: can't alloc writable cells\n");
         break;		/* loop exit */
      }
      if (n = cmap_get_cells(mp->cmap, CS_WRITE, 256, cip)) {
         proto_AllocColorCells(mp->cid, n, 0, 0);
         df_put_p((void *)svp);
         df_put_p((void *)mp);
         df_put_i(n);
         df_put_p((void *)cip);
         df_put_i(0);	/* try 0 */
         rx_queue(zrxqp, X_AllocColorCells, 0, zb_seqno(),
						alloc_cells_ketchup_reply);
      }
      else {
         free(cip);
         break;		/* loop exit */
      }
   }
   proto_GetInputFocus();
   df_put_p((void *)svp);
   rx_queue(zrxqp, X_GetInputFocus, 0, zb_seqno(), ketchup_sync_reply);
   rx_end_repeat(zrxqp);
}

/*
**  alloc_color_ketchup_reply
**
**	Respond to an AllocColor reply.  If everything is hunky-dory,
**	just map the pixel.  If this or a previous request failed, though,
**	it's time for plan B, allocate a new colormap and use it instead.
**
*/
static void
alloc_color_ketchup_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp                                                 
   DD client_t *cp                                                    
   DD u8_t major                                                      
   DD u16_t minor
   DD u16_t seqno                                                        
   DD chunk_t *chp                                           
   DE                         
{
   register int i;
   register chunk_t *nchp;
   pixel_t real_pixel;
   xGenericReply *rp = (xGenericReply *)buf_data(chp);
   register sv_t *svp = (sv_t *)df_get_p(dfp);
   register colormap_t *mp = (colormap_t *)df_get_p(dfp);
   pixel_t pixel = (pixel_t)df_get_i(dfp);
   u16_t red = (u16_t)df_get_i(dfp);
   u16_t green = (u16_t)df_get_i(dfp);
   u16_t blue = (u16_t)df_get_i(dfp);
   int try = (u16_t)df_get_i(dfp);

   if (rp->type == X_Error) {
      if (try) {		/* very, very bad */
         warn("alloc_color_ketchup_reply: alloc failed on 2nd round!\n");
         return;
      }
      if (!sv_isset(svp, sp->fd)) {	/* first error from this server */
         if (sp->smap.root.flags & RF_OWNCMAP) {
            warn("%s: alloc error in own cmap during ketchup, server [%s/%d]\n",
			"alloc_color_ketchup_reply/color.c", sp->tag, sp->fd);
            dprx_error((xError *)rp);
            return;					/* exit */
         }
         zb_dest(ZD_SERV, sp);
         proto_CopyColormapAndFree(vscreen.cmapvid, mp->cid);
         sv_set(svp, sp->fd);		/* mark server as failed */
      }
      /*
      **  reissue alloc, but from new colormap
      **
      **	(note that by the time the reply to this is processed, mp->cid
      **	will refer to the *new* colormap, not the old.
      */
      proto_AllocColor(vscreen.cmapvid, red, green, blue);
      df_put_p((void *)svp);
      df_put_p((void *)mp);
      df_put_i((int)pixel);
      df_put_i((int)red);
      df_put_i((int)green);
      df_put_i((int)blue);
      df_put_i(1);	/* second try */
      rx_queue(zrxqp, X_AllocColor, 0, zb_seqno(), alloc_color_ketchup_reply);
   }
   else { /* X_Reply */
      /*
      **  there are three cases here, all valid:
      **	no errors - this just allocates from the first colormap
      **	previous errors - this allocation is preserved later by
      **			CopyColormapAndFree
      **	realloc - this is a reallocation from a new colormap
      */
      real_pixel = ((xAllocColorReply *)rp)->pixel;
      pmap_map(sp, mp, 1, &pixel, 0, &real_pixel, 0);
   }
}

/*
**  alloc_cells_ketchup_reply
**
**	Just like the preceding routine, but for AllocColorCells replies.
**
*/
static void
alloc_cells_ketchup_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register int i;
   register chunk_t *nchp;
   register int npixels, nmasks;
   pixel_t *real_pixels;
   mask_t real_mask, *maskp;
   xGenericReply *rp = (xGenericReply *)buf_data(chp);
   register sv_t *svp = (sv_t *)df_get_p(dfp);
   register colormap_t *mp = (colormap_t *)df_get_p(dfp);
   register int ncolors = df_get_i(dfp);
   register xColorItem *cip = (xColorItem *)df_get_p(dfp);
   int try = (u16_t)df_get_i(dfp);

   if (rp->type == X_Error) {
      if (try) {		/* very, very bad */
         warn("alloc_cells_ketchup_reply: alloc failed on 2nd round!\n");
         return;
      }
      if (!sv_isset(svp, sp->fd)) {	/* first error from this server */
         if (sp->smap.root.flags & RF_OWNCMAP) {
            warn("%s: alloc error in own cmap during ketchup, server [%s/%d]\n",
			"alloc_cells_ketchup_reply/color.c", sp->tag, sp->fd);
            free(cip);
            return;                                     /* exit */
         }
         zb_dest(ZD_SERV, sp);
         proto_CopyColormapAndFree(vscreen.cmapvid, mp->cid);
         sv_set(svp, sp->fd);           /* mark server as failed */
      }
      /*
      **  reissue alloc, but from new colormap
      **
      **	(note that by the time the reply to this is processed, mp->cid
      **	will refer to the *new* colormap, not the old.
      */
      proto_AllocColorCells(vscreen.cmapvid, ncolors, 0, 0);
      df_put_p((void *)svp);
      df_put_p((void *)mp);
      df_put_i(ncolors);
      df_put_p((void *)cip);
      rx_queue(zrxqp, X_AllocColorCells, 0, zb_seqno(),
						alloc_cells_ketchup_reply);
   }
   else { /* X_Reply */
      /*
      **  always valid - see comment in alloc_color_ketchup_reply
      */
      npixels = ((xAllocColorCellsReply *)rp)->nPixels;
      nmasks = ((xAllocColorCellsReply *)rp)->nMasks;
      real_pixels = (pixel_t *)((xAllocColorCellsReply *)rp + 1);

      if (npixels != ncolors)
         warn("alloc_cells_ketchup_reply/color.c: npixels != ncolors!!!\n");
      if (nmasks)
         warn("alloc_cells_ketchup_reply/color.c: got masks, didn't want'em\n");

      pmap_map_coloritems(sp, mp, ncolors, cip, real_pixels);
      zb_dest(ZD_KETCHUP, 0);
      proto_StoreColors(mp->cid, ncolors, cip, mp->mappixels);
      free(cip);
   }
}

/*
**  ketchup_sync_reply
**
**	If plan A (the initial round of allocations) succeeded, then this
**	merely serves to sync the servers in preparation for the next
**	phase of a ketchup (which is the big picture).  If plan A failed,
**	then a new colormap has already been created, and it is here
**	that we switch, so that all further references to this server's
**	default root colormap go to it.  We also reset the sync vector
**	(svp) because we're starting fresh.  We're just in time, too,
**	because the next thing that happens is all the replies to our
**	plan B allocations start coming in, and the routines above snap
**	into action servicing them, and they expect the default root to
**	be correct.
*/
static void
ketchup_sync_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp                                                 
   DD client_t *cp                                                    
   DD u8_t major                                                      
   DD u16_t minor
   DD u16_t seqno                                                        
   DD chunk_t *chp                                           
   DE                         
{
   register sv_t *svp = (sv_t *)df_get_p(dfp);

   /*
   **  if sharing failed and we set up our own colormap,
   **  make the final switch here
   */
   if (sv_isset(svp, sp->fd)) {
      if ((sp->smap.root.flags & RF_OWNCMAP) == 0) {
         hash_remove(sp->smap.mp, vscreen.rootcmap);
         hash_add(	sp->smap.mp,
			vscreen.rootcmap, 
			mkrid(sp, vscreen.cmapvid),
			0);
         hash_add(	sp->smap.mp,
			vscreen.vservcmap,
			sp->smap.root.realcmap,
			0);
         sp->smap.root.flags |= RF_OWNCMAP;
      }
      sv_clear(svp, sp->fd);	/* unmark failure */
   }
}

/************************************************************************
*									*
*   color_update_winvec							*
*									*
*	Set WM_COLORMAP_WINDOWS property on shell root window.		*
*									*
************************************************************************/
void
color_update_winvec
   VOID
{
   rid_t *winvec;
   int nwins;
   
   nwins = cmapwin_winvec(&winvec);
   proto_ChangeProperty(vscreen.shellroot,
			_WM_COLORMAP_WINDOWS,
			XA_WINDOW,
			PropModeReplace,
			32,
			nwins,
			(char *)winvec);
}

/*
**  free_colormap
**
**	Assumes colormap has been uninstalled.
*/
static void
free_colormap
   AL((mp))
   DB colormap_t *mp
   DE
{
   register plist_t *lp;

   for (lp=mp->winlist; lp; lp=lp->next) /* this could be abstracted TODO */
      window_unset_colormap((window_t *)lp->ptr);
   plist_free(mp->winlist);

   if (mp->last)
      mp->last->next = mp->next;
   if (colormaps == mp)
      colormaps = mp->next;
   if (mp->next)
      mp->next->last = mp->last;
   hash_mark(vmap, mp->wid);
   hash_mark(vmap, mp->cid);
   mp->res.client->refs--;
   cmap_free(mp->cmap);
   free(mp);
}

/*
**   install_colormap
**
**	Assumes colormap is not installed.
*/
static void
install_colormap
   AL((mp))
   DB colormap_t *mp
   DE
{
   last_cmap = mp;

   mp->installed = 1;
   cmapwin_install(mp->wid);
   colormap_change(vscreen.wp, mp, ColormapInstalled);
}

/*
**   uninstall_colormap
**
**	Assumes colormap is installed.
*/
static void
uninstall_colormap
   AL((mp))
   DB colormap_t *mp
   DE
{
   last_cmap = vscreen.wp->mp;

   mp->installed = 0;
   cmapwin_uninstall(mp->wid);
   colormap_change(vscreen.wp, mp, ColormapUninstalled);
}

/*
**   colormap_change
*/
static void
colormap_change
   AL((wp, mp, state))
   DB window_t *wp
   DD colormap_t *mp
   DD u8_t state
   DE
{
   if (wp->nextsib)
      colormap_change(wp->nextsib, mp, state);
   if (wp->child)
      colormap_change(wp->child, mp, state);

   if (wp->mp == mp)
      event_ColormapNotify(wp, 0, state);
}
