/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   connect.c								*
*									*
*	Handle X protocol connection-block processing.			*
*									*
************************************************************************/
#include <sys/time.h>
#include <sys/types.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#include <stdlib.h>
#include <errno.h>
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>

#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "zb.h"
#include "incl/connect.pvt.h"
#include "fd.h"

#define X_MAJOR_VERSION	11
#define X_MINOR_VERSION	0

#define	DENY_VERSION	"Protocol version mismatch"
#define	DENY_AUTH	"Client is not authorized to connect to Server"
#define DENY_NOTREADY	"Server not ready"

/************************************************************************
*									*
*   connect_client_in							*
*									*
*	Handle an incoming client connection block.			*
*									*
************************************************************************/
int
connect_client_in
   AL((cp, bp))
   DB client_t *cp
   DD buffer_t *bp
   DE
{
   register int len, rv = -1;
   register chunk_t *chp;
   register xConnClientPrefix *prefix;

   chp = buf_chunk(bp);
   if (buf_chunksize(chp) < sz_xConnClientPrefix)
      return 0;		/* no error, just wait for more */

   prefix = (xConnClientPrefix *)buf_data(chp);

   if (cp->swap = (prefix->byteOrder == endian) ? 0 : 1)
      swap_ConnClientPrefix(prefix);

   len = sz_xConnClientPrefix +
		prefix->nbytesAuthProto + PAD(prefix->nbytesAuthProto) +
		prefix->nbytesAuthString + PAD(prefix->nbytesAuthString);

   if (buf_chunksize(chp) < len) {
      if (cp->swap)
         swap_ConnClientPrefix(prefix);	/* swap it back again... */
      return 0;
   }
   chp = buf_split(bp, len);	/* bite it off */

   time_end(cp->fd);

   D_CALL1(D_PROTO1, dprx_ConnClientPrefix, prefix);
   if (	access_get() == 0 ||
	access_check(cp->addr.family, cp->addr.length, cp->addr.address) ||
	auth_check_local(	prefix->nbytesAuthProto,
				(char *)(prefix + 1),
				prefix->nbytesAuthString,
				(char *)(prefix + 1) +
					prefix->nbytesAuthProto +
					PAD(prefix->nbytesAuthProto))) {
      if (	prefix->majorVersion == X_MAJOR_VERSION &&
		prefix->minorVersion <= X_MINOR_VERSION) {
         if (vcstate == VC_FIXED) {
            vconf_write(cp);
            cp->state = K_READY;	/* advance client to "ready" state */
            rv = 0;
         }
         else
            connect_server_deny(cp, DENY_NOTREADY);
      }
      else
         connect_server_deny(cp, DENY_VERSION);
   }
   else
      connect_server_deny(cp, DENY_AUTH);

   buf_clear(chp);

   return rv;
}

/************************************************************************
*									*
*   connect_server_deny							*
*									*
*	Generate an outgoing server connection denial.			*
*									*
************************************************************************/
void
connect_server_deny
   AL((cp, reason))
   DB client_t *cp
   DD char *reason
   DE
{
   int len;

   len = strlen(reason);
   proto_ConnSetup_deny(cp, len, reason);
}

/************************************************************************
*									*
*   connect_as_client							*
*									*
*	Initiate a client connection.					*
*									*
************************************************************************/
int
connect_as_client
   AL((sp))
   DB server_t *sp
   DE
{
   if (sp->state != S_NAMED)
      return err(-1, "connect_as_client: server [%s] in state [%s]\n",
				sp->tag, debug_server_state_str(sp->state));

   if ((socket_as_client(	sp->addr.family,
				sp->addr.length,
				sp->addr.address,
				sp->display,
				&sp->fd)) < 0) {
      if (errno == EINPROGRESS && opt.to) {
         sp->qp->fd = sp->fd;		/* this is somewhat ugly */
         server_state(sp, S_INPROGRESS);
         fd_write(sp->fd);	/* select for connection completion */
         return 0;
      }
      else {
         pwarn("cannot connect to display %s ", sp->tag);
         return -1;
      }
   }
   sp->qp->fd = sp->fd;		/* this is somewhat ugly */

   return 1;
}

/************************************************************************
*									*
*   connect_client_out							*
*									*
*	Generate an outgoing client connection block.			*
*									*
************************************************************************/
int
connect_client_out
   AL((sp))
   DB server_t *sp
   DE
{
   u16_t nlen, dlen;
   char *name, *data;
   chunk_t *chp;

   if (sp->cookie) {
      nlen = sp->cookie->namelen;
      name = sp->cookie->name;
      dlen = sp->cookie->datalen;
      data = sp->cookie->data;
   }
   else {
      nlen = dlen = 0;
      (void) auth_get(	sp->addr.family,
			sp->addr.length,
			sp->addr.address,
			sp->display,
			&nlen, &name, &dlen, &data);
   }
   proto_ConnClientPrefix(nlen, name, dlen, data);
   /*
   **  throw out cookie once it's used
   */
   if (sp->cookie) {
      cookie_free(sp->cookie);
      sp->cookie = 0;
   }
   return 0;
}

/************************************************************************
*									*
*   connect_server_in							*
*									*
*	Handle an incoming server connection block.  A return value	*
*	of zero indicates either that the entire block has not yet	*
*	arrived, or it has, the connection was successful and a		*
*	ListExtensions request has been sent back.  A return value	*
*	of -1 indicates either connection refusal, or a local error.	*
*									*
************************************************************************/
int
connect_server_in
   AL((sp, chp))
   DB server_t *sp
   DD chunk_t *chp
   DE
{
   register int rv, len;
   register xConnSetupPrefix *prefix;
   char rbuf[256];
   char *reason;

   prefix = (xConnSetupPrefix *)buf_data(chp);

   D_CALL1(D_PROTO3, dprx_connsetup, (char *)prefix);

   if (prefix->success) {
      len = buf_chunksize(chp);
      if (MALLOC(sp->block, char *, len))
         return -1;
      bcopy((char *)prefix, sp->block, len);
      rv = 0;
   }
   else {
      reason = (char *)(prefix + 1);
      if (prefix->lengthReason < (u8_t)(prefix->length * 4))
         reason[prefix->lengthReason] = '\0';
      else {	/* so sue me */
         len = prefix->lengthReason;
         bcopy(reason, rbuf, len);
         rbuf[len] = '\0';
         reason = rbuf;
      }
      warn("connection to \"%s\" refused by server\n\t%s\n", sp->tag, reason);
      rv = -1;
   }
   return rv;
}
