/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   cx.c								*
*									*
*	"Continue" context stack routines.  Continue contexts contain	*
*	routines to execute after a reply context queue is exhausted.	*
*	They typically "finish" an operation that involved an extended	*
*	server i/o dialog.						*
*									*
************************************************************************/
#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "incl/cx.pvt.h"

static cx_t *cxfree;		/* free list */

/************************************************************************
*									*
*   cx_push								*
*									*
*	Push a continue context onto the stack.				*
*									*
************************************************************************/
void
cx_push
   AL((rxqp, fptr))
   DB rxq_t *rxqp
   DD cont_proc_t fptr
   DE
{
   register cx_t *cxp;

   cxp = new_cx();
   cxp->fptr = fptr;
   cxp->dfp = df_current();

   cxp->next = rxqp->cstk;
   rxqp->cstk = cxp;

   /*
   **  if there is nothing on the reply stack, this routine may
   **  never get called, so call it right now
   */
   if (rxqp->out == 0)
      cx_continue(rxqp);
}

/************************************************************************
*									*
*   cx_continue								*
*									*
*	Pop the next continue context off the stack and execute the	*
*	the function contained therein.  No checking is done either	*
*	that there is a next context or that it has a valid function	*
*	pointer.							*
*									*
************************************************************************/
void
cx_continue
   AL((rxqp))
   DB rxq_t *rxqp
   DE
{
   register cx_t *cxp;

   cxp = rxqp->cstk;
   rxqp->cstk = cxp->next;

   if (cxp->dfp)
      df_reset(cxp->dfp);

   (*(cxp->fptr))(cxp->dfp);

   if (cxp->dfp)
      df_free(cxp->dfp);
   free_cx(cxp);
}

/*
**  new_cx	- allocate a context
*/
static cx_t *
new_cx
   VOID
{
   register cx_t *cxp;

   if (cxfree) {
      cxp = cxfree;
      cxfree = cxfree->next;
   }
   else if (MALLOC(cxp, cx_t *, sizeof(cx_t)))
      return (cx_t *)err(0, "malloc returned zero");

   return cxp;
}

/*
**  free_cx	- put context on free list
*/
void
free_cx
   AL((cxp))
   DB cx_t *cxp
   DE
{
   cxp->next = cxfree;
   cxfree = cxp;
}

/************************************************************************
*									*
*   cx_free_freelist							*
*									*
************************************************************************/
void
cx_free_freelist
   VOID
{
   register cx_t *cxp, *last;

   for (cxp=cxfree; last=cxp;) {
      cxp = cxp->next;
      free(last);
   }
   cxfree = 0;
}
