/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   expos.c								*
*									*
*	Routines to delay exposure processing to allow filtering of	*
*	redundant Expose events.					*
*									*
************************************************************************/
#define NEED_EVENTS
#include <X11/Xproto.h>
#include <X11/X.h>

#include "xmx.h"
#include "incl/expos.pvt.h"

typedef struct _expvec_t {
   window_t *		wp;
   int			n;
   chunk_t *		chv[EXPOS_Q_LEN];
   int			nv[EXPOS_Q_LEN];
   struct _expvec_t *	wnext;
   struct _expvec_t *	prev;
   struct _expvec_t *	next;
}expvec_xxx;		/* silences ansi compilers - declared in xmx.h */

static expvec_t *exphead;
static expvec_t *exptail;

/************************************************************************
*									*
*   expos_queue								*
*									*
*	If there is only one server, deliver the exposure immediately	*
*	and exit.							*
*									*
*	Otherwise, inspect the queue of pending exposures for the	*
*	given window and determine if the new exposure is redundant.	*
*	If so, exit and allow it to be ignored.				*
*									*
*	If the new exposure is new, queue it for delivery at a later	*
*	time.								*
*									*
*	Note: compares exposures on a chunk-by-chunk basis.  A chunk	*
*	here represents a block of exposure events as received from	*
*	an X server.  It is guaranteed that the events in such blocks	*
*	were generated by single window, and that they are contiguous.	*
*	Such a (logical) block may span multiple chunks, however.	*
*									*
************************************************************************/
void
expos_queue
   AL((wp, chp, n))
   DB window_t *wp
   DD chunk_t *chp
   DD int n
   DE
{
   register int nc, i, j, nevents;
   register xEvent *nep, *sep;	/* "next" event, "source" event */
   register expvec_t *evp;

   if (num_serv == 1) {
      event_send(chp, n, wp, ExposureMask, 0);
      return;
   }
   sep = 0;
   if (wp->expos_q) {	/* there are pending exposures for this window */
      nep = (xEvent *)buf_data(chp);
      for (evp=wp->expos_q; ; evp=evp->wnext) {
         for (i=0, nc=evp->n; i<nc; i++) {
            sep = (xEvent *)buf_data(evp->chv[i]);
            if (evp->nv[i] == n) {
               for (j=0; j<n; j++)
                  if (	sep[j].u.expose.x != nep[j].u.expose.x ||
			sep[j].u.expose.y != nep[j].u.expose.y ||
			sep[j].u.expose.width != nep[j].u.expose.width ||
			sep[j].u.expose.height != nep[j].u.expose.height)
                     break;
               if (j == n) {		/* chunk matched */
                  DEBUG0(D_EXPOS, "expos_queue: duplicate chunk - ignored\n");
                  return;		/* EXIT (allows chunk reclamation) */
               }
            }
         }
         if (evp->wnext == 0)	/* loop exit */
            break;
      }
      if (i == EXPOS_Q_LEN) {
         if ((evp = evp->wnext = new_expvec()) == 0)
            return;
         evp->n = 1;
         evp->wnext = 0;
         i = 0;
      }
      else
         evp->n++;
   }
   else {
      if ((evp = wp->expos_q = new_expvec()) == 0)
         return;
      evp->n = 1;
      evp->next = 0;
      evp->wnext = 0;
      evp->prev = exptail;
      if (exptail)
         exptail->next = evp;
      else
         exphead = evp;
      exptail = evp;
      i = 0;
   }

   /* this exposure is a keeper - queue it */

   DEBUG2(D_EXPOS, "expos_queue: queueing %d events for window 0x%x\n", n, wp);
   evp->wp = wp;
   evp->chv[i] = chp;
   evp->nv[i] = n;
   chp->refs++;			/* prevent it from being reclaimed */

   exposures = 1;		/* mark global boolean */

   /*
   **	tie previous exposure(s) for this window to current one(s)
   */
   if (sep)
      sep->u.expose.count = 1;
}

/************************************************************************
*									*
*   expos_timeout							*
*									*
*	Process all pending exposures.					*
*									*
*	This routine is abit tricky.  It walks the expvec list from	*
*	window to window, pausing to walk the expvec list for each.	*
*	As each expvec struct is encountered, it is processed and	*
*	removed - the entire data structure is destroyed as it is	*
*	traversed in a single pass.					*
*									*
*	The list is traversed in natural list order, to preserve, at	*
*	least partially, the order in which the exposures were		*
*	received.							*
*									*
************************************************************************/
void
expos_timeout
   VOID
{
   register int i, n;
   register expvec_t *evp, *last, *wevp, *wlast;

   DEBUG0(D_EXPOS, "expos_timeout: invoked\n");
   for (evp=exphead; last=evp;) {
      evp = evp->next;
      last->wp->expos_q = 0;
      for (wevp=last; wlast=wevp;) {
         wevp = wevp->wnext;
         for (i=0, n=wlast->n; i<n; i++) {
            DEBUG2(D_EXPOS, "expos_timeout: sending %d events to window 0x%x\n",
						wlast->nv[i], wlast->wp);
            event_send(wlast->chv[i], wlast->nv[i], wlast->wp, ExposureMask, 0);
            buf_clear(wlast->chv[i]);
         }
         free_expvec(wlast);
      }
   }
   exphead = exptail = 0;
   exposures = 0;	/* clear global boolean */
}

/************************************************************************
*									*
*   expos_window_death							*
*									*
*	Remove pending exposures from a window.  The window's		*
*	reference isn't cleared, since we assume it's about to be	*
*	nuked anyway.							*
*									*
*	Note: The window must have pending exposures.			*
*									*
************************************************************************/
void
expos_window_death
   AL((wp))
   DB window_t *wp
   DE
{
   register int i;
   register expvec_t *evp, *last;

   evp = wp->expos_q;

   if (evp->prev)
      evp->prev->next = evp->next;
   else
      exphead = evp->next;

   if (evp->next)
      evp->next->prev = evp->prev;
   else
      exptail = evp->prev;

   while (last=evp) {
      evp = evp->wnext;
      for (i=0; i<last->n; i++)
         buf_clear(last->chv[i]);
      free_expvec(last);
   }
}

/*
**	expvec alloc/free routines
*/
static expvec_t *evfree;

static expvec_t *
new_expvec
   VOID
{
   register expvec_t *evp;

   if (evfree) {
      evp = evfree;
      evfree = evfree->next;
   }
   else if (MALLOC(evp, expvec_t *, sizeof(expvec_t)))
      return 0;

   return evp;
}

static void
free_expvec
   AL((evp))
   DB expvec_t *evp
   DE
{
   evp->next = evfree;
   evfree = evp;
}

static void
free_expvec_freelist
   VOID
{
   register expvec_t *evp, *last;

   for (evp=evfree; last=evp;) {
      evp = evp->next;
      free(last);
   }
   evfree = 0;
}
