/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   inp.c								*
*									*
*	Maintain the virtual pointer and keyboard devices.		*
*									*
*	Careful when reading the grab code.  I use "active" and		*
*	"passive" differently than does the protocol spec.  In the	*
*	spec, an active grab is one that is in effect, a passive grab	*
*	is a grab waiting to happen.  A passive grab becomes an active	*
*	grab when it is triggered.					*
*									*
*	What I call active is a protocol-request-initiated grab - one	*
*	that was explicitely called for by a client application.  All	*
*	other grabs are initiated by events, and I lump them together	*
*	and call them passive.  In my parlance, a passive grab never	*
*	becomes active and vice versa - they are disjoint.		*
*									*
*	This is confusing, even to me, and a global search and replace	*
*	is in order as soon as I can think of appropriate ways to	*
*	rename them.							*
*									*
************************************************************************/
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>
#include <X11/X.h>
#include <xmc.h>
#include <xmcp.h>
#include "xmx.h"
#include "df.h"
#include "zb.h"
#include "res.h"
#include "kpm.h"
#include "incl/inp.pvt.h"

#define NBUTTONS	6
#define NKEYCODES	256
#define BUTTONMASKS\
	(Button1Mask|Button2Mask|Button3Mask|Button4Mask|Button5Mask)
#define MODMASKS\
	(ShiftMask|LockMask|ControlMask|Mod1Mask|Mod2Mask|Mod3Mask|Mod4Mask|Mod5Mask)

/*
**	masks for frozen input devices
*/
#define POINTER_GRAB	0x01
#define KEYBOARD_GRAB	0x02

/*
**	passive grabs
*/
typedef struct _grab_t {
   u8_t			detail;		/* key or button */
   u8_t			ownev;		/* owner_events */
   u8_t			kbdsync;	/* keyboard mode is sync? */
   u8_t			ptrsync;	/* pointer mode is sync? */
   u16_t		mods;		/* keymasks or AnyModifier */ 
   u16_t		evmask;		/* event mask (pointer only) */
   window_t *		wp;		/* grab window */
   window_t *		cwp;		/* "confine to" window (pointer only) */
   cursor_t *		cursp;		/* cursor (pointer only) */
   client_t *		cp;		/* grabbing client */
   struct _grab_t *	next;
}grab_xxx;	/* grab_t defined in xmx.h */

typedef struct {
   server_t *		sp;
   window_t *		wp;
}spwp_t;

/*
**	virtual input devices
*/
static struct {
   s16_t                rootx, rooty;   /* current virtual root location */
   s16_t                eventx, eventy; /* current location in top window */
   timestamp_t          time;           /* time of last event */
   window_t *		grabwp;		/* grab window */
   client_t *		grabcp;		/* grabbing client */
   u8_t			ownev;		/* owner_events */
   int			active;		/* active or passive? */
   int			sync;		/* mode sync or async */
   uint_t		frozen;		/* if set, bitmask identifies cause */
   chunk_t *		echp;		/* chunk with event that froze it */
   etime_t		grabtime;	/* time of grab */
   mask_t		mask;		/* event mask */
   window_t *		cwp;		/* confine_to window */
   cursor_t *		cursp;		/* grab cursor */
}vptr;

static struct {
   timestamp_t          time;           /* time of last event */
   window_t *		grabwp;		/* grab window */
   client_t *		grabcp;		/* grabbing client */
   u8_t			ownev;		/* owner_events */
   u8_t			keycode;	/* key pressed if passive */
   int			active;		/* active or passive? */
   int			sync;		/* mode sync or async */
   uint_t		frozen;		/* if set, bitmask identifies cause */
   chunk_t *		echp;		/* chunk with event that froze it */
   etime_t		grabtime;	/* time of grab */
}vkbd;

static int curmo;		/* current motion state (index into mobuf) */
static xTimecoord mobuf[MOBUFSIZE];
static buffer_t *ebp;		/* event buffer (for synchronous grabs) */

static u16_t vkbstat;		/* virtual key/button state */
static int keysdown;		/* number of keys pressed */
static u8_t vkeymap[32];	/* virtual keymap */

/************************************************************************
*									*
*   inp_init								*
*									*
*	Initialize default pointer.					*
*									*
************************************************************************/
int
inp_init
   VOID
{
   ebp = buf_new(B_STATIC);
   return 0;
}

/************************************************************************
*									*
*   inp_motion_init							*
*									*
*	Create initial motion buffer event.				*
*									*
************************************************************************/
void
inp_motion_init
   AL((wp))
   DB window_t *wp
   DE
{
   register s16_t x, y;

   x = wp->dwb.width/2;
   y = wp->dwb.height/2;

   mobuf[curmo].time = time_stamp(vtime);	/* current time */
   mobuf[curmo].x = x;
   mobuf[curmo].y = y;

   ptrwp = wp;
   vptr.rootx = x;
   vptr.rooty = y;
   vptr.eventx = x;
   vptr.eventy = y;
}

#define kbp_chroot(sp, wp, ep)\
   ep->u.keyButtonPointer.root = vscreen.wp->cid;\
   if (sp->smap.root.ok == 0) {\
      register window_t *_tw;\
      register int _x, _y;\
\
      _x = ep->u.keyButtonPointer.eventX;\
      _y = ep->u.keyButtonPointer.eventY;\
      for (_tw=wp; _tw->parent; _tw=_tw->parent) {\
         _x += XOFFSET(_tw);\
         _y += YOFFSET(_tw);\
      }\
      sp->smap.root.xoff = ep->u.keyButtonPointer.rootX - _x;\
      sp->smap.root.yoff = ep->u.keyButtonPointer.rootY - _y;\
      sp->smap.root.ok = 1;\
   }\
   ep->u.keyButtonPointer.rootX -= sp->smap.root.xoff;\
   ep->u.keyButtonPointer.rootY -= sp->smap.root.yoff

/************************************************************************
*									*
*   inp_device								*
*									*
*	Process a device (keyboard/pointer) event.			*
*									*
************************************************************************/
void
inp_device
   AL((sp, wp, chp, type))
   DB server_t *sp
   DD window_t *wp
   DD chunk_t *chp
   DD u8_t type
   DE
{
   register xEvent *p = (xEvent *)buf_data(chp);

   if (sp->mode == Floor) {
      switch (type) {
         case KeyPress:
         case KeyRelease:
            if (focuswp) {
               kbp_chroot(sp, wp, p);
               time_update(p->u.keyButtonPointer.time);
               p->u.keyButtonPointer.time = vkbd.time = time_stamp(vtime);

               if (vkbd.frozen) {
                  ebuf_store(sp, chp);
                  return;
               }
               else
                  inp_key(sp, wp, chp, type);
            }
            break;
         case ButtonPress:
         case ButtonRelease:
         case MotionNotify:
            kbp_chroot(sp, wp, p);
            time_update(p->u.keyButtonPointer.time);
            p->u.keyButtonPointer.time = vkbd.time = time_stamp(vtime);

            if (vptr.frozen) {
               ebuf_store(sp, chp);
               return;
            }
            else
               inp_pointer(sp, wp, chp, type);
            break;
      }
   }
   else {	/* Seat or View */
      kbp_chroot(sp, wp, p);

      if (sp->tpp && sp->tpp->show && num_serv > 1) {
         switch (type) {
            case ButtonPress:
            case ButtonRelease:
            case MotionNotify:
               tptr_move(	sp->tpp,
				sp,
				wp,
				p->u.keyButtonPointer.rootX,
				p->u.keyButtonPointer.rootY);
               break;
         }
      }
      if (sp->mode == Seat) {
         if (type == MotionNotify)
            return;
         time_update(p->u.keyButtonPointer.time);
         p->u.keyButtonPointer.time = vkbd.time = time_stamp(vtime);
         xmc_xevent(	sp,
			wp,
			p->u.u.type,
			p->u.u.detail,
			p->u.keyButtonPointer.time,
			p->u.keyButtonPointer.event,
			p->u.keyButtonPointer.child,
			p->u.keyButtonPointer.rootX,
			p->u.keyButtonPointer.rootY,
			p->u.keyButtonPointer.eventX,
			p->u.keyButtonPointer.eventY,
			p->u.keyButtonPointer.state);
      }
   }
}

/************************************************************************
*									*
*   inp_key								*
*									*
*	Respond to a key press/release event.  Focuswp must be set	*
*	whenever this routine is called.				*
*									*
************************************************************************/
void
inp_key
   AL((sp, wp, chp, type))
   DB server_t *sp
   DD window_t *wp
   DD chunk_t *chp
   DD u8_t type
   DE
{
   register int propagate;
   register mask_t mask;
   etime_t etime;
   register s16_t x, y;
   register window_t *tw;
   register grab_t *gp;
   register xEvent *p = (xEvent *)buf_data(chp);

   switch (type) {
      case KeyPress:
         p->u.u.detail = kpm_map_key(sp->smap.ktmp, p->u.u.detail);
         km_set(p->u.u.detail);
         if (vkbd.grabcp) {
            /*
            **  KeyPress re-freezes a synchronous thawed grab.
            */
            if (vkbd.sync && !vkbd.frozen) {
               vkbd.frozen |= KEYBOARD_GRAB;
               vkbd.echp = clone_event_chunk(chp);
            }
         }
         else if (gp=pgrabkey(	p->u.u.detail,
				p->u.keyButtonPointer.state & MODMASKS)) {
            time_convert(p->u.keyButtonPointer.time, &etime);
            DEBUG3(D_GRABS,
			"inp_key: (grab) client [%d] wp [0x%x] ownev [%d]\n",
			gp->cp->clinum, wp, gp->ownev);
            grab_keyboard(	gp->wp, gp->cp, 0, gp->ownev, chp,
				gp->ptrsync, gp->kbdsync, p->u.u.detail,
				&etime);
            /*
            **  When a key grab activates, the KeyPress event is
            **  always forwarded to the grabbing client and always
            **  relative to the grab window, irrespective of any event
            **  masks or the owner-events flag.
            */
            p->u.keyButtonPointer.event = gp->wp->cid;
            p->u.keyButtonPointer.sameScreen = 1;
            p->u.keyButtonPointer.child = get_child_and_translate(wp, gp->wp,
					p->u.keyButtonPointer.eventX,
					p->u.keyButtonPointer.eventY,
					&p->u.keyButtonPointer.eventX,
					&p->u.keyButtonPointer.eventY);
            event_sendto(chp, gp->cp);
            /*
            **  play queued events
            */
            if (ebuf_ready())
               ebuf_play();
            return;
         }
         mask = KeyPressMask;
         break;
      case KeyRelease:
         p->u.u.detail = kpm_map_key(sp->smap.ktmp, p->u.u.detail);
         km_clr(p->u.u.detail);
         mask = KeyReleaseMask;
         if (vkbd.grabcp && vkbd.sync && !vkbd.frozen &&
			(vkbd.active || p->u.u.detail != vkbd.keycode)) {
            /*
            **  KeyRelease re-freezes a synchronous thawed grab only
            **  if it does not release the grab (it's active or the key
            **  is still pressed).
            */
            vkbd.frozen |= KEYBOARD_GRAB;
            vkbd.echp = clone_event_chunk(chp);
         }
	 break;
      default:
         warn("inp_key: bad event '%s' [%d]\n", dprx_event_str(type), type);
	 return;
   }
   propagate = 1;
   if (focuswp->parent) {	/* if not PointerRoot focus */
      for (tw=wp; tw->level > focuswp->level; tw=tw->parent);
      if (tw != focuswp) {	/* focusify */
         p->u.keyButtonPointer.event = focuswp->cid;
         p->u.keyButtonPointer.sameScreen = 1;
         p->u.keyButtonPointer.child = get_child_and_translate(wp, focuswp,
					p->u.keyButtonPointer.eventX,
					p->u.keyButtonPointer.eventY,
					&p->u.keyButtonPointer.eventX,
					&p->u.keyButtonPointer.eventY);
         wp = focuswp;
         propagate = 0;
      }
   }
   if (vkbd.grabcp) {
      if (vkbd.ownev == 0)
         event_sendto(chp, vkbd.grabcp);
      else
         event_send(chp, 1, wp, mask, propagate);

      if (vkbd.active == 0 && type == KeyRelease &&
					p->u.u.detail == vkbd.keycode)
         if (ungrab_keyboard(0))
            if (ebuf_ready())
               ebuf_play();
   }
   else
      event_send(chp, 1, wp, mask, propagate);
}

/************************************************************************
*									*
*   inp_pointer								*
*									*
*	Respond to a pointer event.					*
*									*
************************************************************************/
void
inp_pointer
   AL((sp, wp, chp, type))
   DB server_t *sp
   DD window_t *wp
   DD chunk_t *chp
   DD u8_t type
   DE
{
   s16_t x, y;
   register mask_t mask;
   register grab_t *gp;
   etime_t etime;
   register xEvent *p = (xEvent *)buf_data(chp);

   if (sp->tpp && sp->tpp->show && num_serv > 1)
      tptr_move(	sp->tpp,
			sp,
			wp,
			p->u.keyButtonPointer.rootX,
			p->u.keyButtonPointer.rootY);

   vptr.rootx = p->u.keyButtonPointer.rootX;
   vptr.rooty = p->u.keyButtonPointer.rootY;
   vptr.eventx = p->u.keyButtonPointer.eventX;
   vptr.eventy = p->u.keyButtonPointer.eventY;
   /*
   **	NO - throw out inappropriate state changes TODO
   */
   vkbstat = p->u.keyButtonPointer.state;
   switch (type) {
      case MotionNotify:
         curmo = (curmo + 1) % MOBUFSIZE;
         mobuf[curmo].time = vptr.time;
         mobuf[curmo].x = vptr.rootx;
         mobuf[curmo].y = vptr.rooty;
         if (ptrwp != wp) {	/* Leave/Enter Window */
            leave_enter_events(ptrwp, wp, NotifyNormal);
            ptrwp = wp;
         }
         mask = vkbstat & BUTTONMASKS;	/* yep, this works */
         if (mask)	/* if any button */
            mask |= ButtonMotionMask;
         mask |= PointerMotionMask;
         /* do MotionHint here (TODO) */
         break;
      case ButtonPress:
         p->u.u.detail = kpm_map_button(sp->smap.ktmp, p->u.u.detail);
         mask = ButtonPressMask;
         vkbstat |= 1 << (7 + p->u.u.detail);
         if (vptr.grabcp) {
            /*
            **  ButtonPress re-freezes a synchronous thawed grab.
            */
            if (vptr.sync && !vptr.frozen) {
               vptr.frozen |= POINTER_GRAB;
               vptr.echp = clone_event_chunk(chp);
            }
         }
         else if (gp = pgrabbutton(wp,p->u.u.detail,vkbstat & MODMASKS)) {
			/* need to handle ReplayPointer here by saving event */
            time_convert(vptr.time, &etime);
            DEBUG2(D_GRABS, "inp_pointer: (grab) client [%d] wp [0x%x] ",
							gp->cp->clinum, wp);
            DEBUG2(D_GRABS, "ownev [%d] evmask [%s]\n",
				gp->ownev, dprx_event_mask_str(gp->evmask));
            grab_pointer(	gp->wp, gp->cwp, gp->cp, 0, gp->ownev, 
				chp, gp->ptrsync, gp->kbdsync, &etime,
				gp->evmask, gp->cursp);
            /*
            **  When a button grab activates, the ButtonPress event is
            **  always forwarded to the grabbing client and always
            **  relative to the grab window, irrespective of any event
            **  masks or the owner-events flag.
            */
            p->u.keyButtonPointer.event = gp->wp->cid;
            p->u.keyButtonPointer.sameScreen = 1;
            p->u.keyButtonPointer.child = get_child_and_translate(
					wp, gp->wp,
					p->u.keyButtonPointer.eventX,
					p->u.keyButtonPointer.eventY,
					&p->u.keyButtonPointer.eventX,
					&p->u.keyButtonPointer.eventY);
            event_sendto(chp, gp->cp);
            /*
            **  play queued events
            */
            if (ebuf_ready())
               ebuf_play();
            return;				/* EXIT */
         }
         break;
      case ButtonRelease:
         p->u.u.detail = kpm_map_button(sp->smap.ktmp, p->u.u.detail);
         mask = ButtonReleaseMask;
         vkbstat &= ~(1 << (7 + p->u.u.detail));
         if (vptr.grabcp && vptr.sync && !vptr.frozen &&
				(vptr.active || vkbstat & BUTTONMASKS)) {
            /*
            **  ButtonRelease re-freezes a synchronous thawed grab only
            **  if it does not release the grab (it's active or there are
            **  still be buttons pressed).
            */
            vptr.frozen |= POINTER_GRAB;
            vptr.echp = clone_event_chunk(chp);
         }
         break;
   }
   if (vptr.grabcp) {
      if (vptr.ownev == 0 || event_send(chp, 1, wp, mask, 1) == 0)
         if (mask & vptr.mask) {
            if (wp != vptr.grabwp) {	/* grabify */
               p->u.keyButtonPointer.event = vptr.grabwp->cid;
               p->u.keyButtonPointer.sameScreen = 1;
               p->u.keyButtonPointer.child = get_child_and_translate(
					wp, vptr.grabwp,
					p->u.keyButtonPointer.eventX,
					p->u.keyButtonPointer.eventY,
					&p->u.keyButtonPointer.eventX,
					&p->u.keyButtonPointer.eventY);
            }
            if (vptr.cwp) {	/* do confineto */
               if (vptr.cwp == vptr.grabwp) {
                  x = p->u.keyButtonPointer.eventX;
                  y = p->u.keyButtonPointer.eventY;
               }			/* we have correct root coords TODO */
               if (x < 0)
                  p->u.keyButtonPointer.eventX -= x;
               else if (x >= (s16_t)vptr.cwp->dwb.width)
                  p->u.keyButtonPointer.eventX -= (x - vptr.cwp->dwb.width+1);
               if (y < 0)
                  p->u.keyButtonPointer.eventY -= y;
               else if (y >= (s16_t)vptr.cwp->dwb.height)
                  p->u.keyButtonPointer.eventY -= (y - vptr.cwp->dwb.height+1);
            }
            event_sendto(chp, vptr.grabcp);
         }

      if ((vkbstat & BUTTONMASKS) == 0)
         if (vptr.active == 0)	/* includes autograbs */
            if (ungrab_pointer())
               if (ebuf_ready())
                  ebuf_play();
   }
   else
      event_send(chp, 1, wp, mask, 1);
}
#undef kbp_chroot

/************************************************************************
*									*
*   inp_adjust								*
*									*
*	Find the pointer window the hard way.  For window mapping and	*
*	unmapping.							*
*									*
************************************************************************/
void
inp_adjust
   VOID
{
   register window_t *wp, *tw;
   register s16_t x, y;

   wp = vscreen.wp;
   x = vptr.rootx;
   y = vptr.rooty;

   for (tw=wp; tw;)
      for (tw=wp->child; tw; tw=tw->nextsib)
         if (	tw->mapped &&
		x >= LEFTEDGE(tw) && x < LEFTEDGE(tw) + FULLWIDTH(tw) &&
		y >= TOPEDGE(tw) && y < TOPEDGE(tw) + FULLHEIGHT(tw)) {
            x -= XOFFSET(tw);
            y -= YOFFSET(tw);
            wp = tw;
            break;
         }

   if (ptrwp != wp) {
      time_update(0);
      leave_enter_events(ptrwp, wp, NotifyNormal);
      ptrwp = wp;
      vptr.eventx = x;
      vptr.eventy = y;
   }
}

/*
**  inp_root_translate
**
**	Find the topmost, mapped window that contains the given point.
*/
void
inp_root_translate
   AL((rootx, rooty, wpp, xp, yp))
   DB s16_t rootx
   DD s16_t rooty
   DD window_t **wpp
   DD s16_t *xp
   DD s16_t *yp
   DE
{
   register window_t *wp, *twp;
   register s16_t x, y;

   wp = vscreen.wp;
   x = rootx;
   y = rooty;

   for (twp=wp; twp;)
      for (twp=wp->child; twp; twp=twp->nextsib)
         if (	twp->mapped &&
		x >= LEFTEDGE(twp) && x < LEFTEDGE(twp) + FULLWIDTH(twp) &&
		y >= TOPEDGE(twp) && y < TOPEDGE(twp) + FULLHEIGHT(twp)) {
            x -= XOFFSET(twp);
            y -= YOFFSET(twp);
            wp = twp;
            break;
         }
   *wpp = wp;
   *xp = x;
   *yp = y;
}

/************************************************************************
*									*
*   inp_inside								*
*									*
*	Is the pointer located within the same rectangular area of the	*
*	root window occupied by the window.  Ignores whether the	*
*	window is mapped and assumes that ptrwp may be wrong.		*
*									*
************************************************************************/
int
inp_inside
   AL((wp))
   DB window_t *wp
   DE
{
   register s16_t x, y, w, h;

   w = FULLWIDTH(wp);
   h = FULLHEIGHT(wp);
   for (x=y=0; wp->parent; wp=wp->parent) {
      x += XOFFSET(wp);
      y += YOFFSET(wp);
   }
   return (int) (	vptr.rootx >= x && vptr.rootx < x + w &&
			vptr.rooty >= y && vptr.rooty < y + h);
}

/************************************************************************
*									*
*   inp_query_pointer							*
*									*
************************************************************************/
void
inp_query_pointer
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   register u8_t ss;
   register rid_t child;
   register s16_t winx, winy;
   register window_t *wp, *tw;

   if ((wp = (window_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadWindow, p->id, 0, X_QueryPointer);
      return;
   }
   child = (ptrwp->parent == wp) ? ptrwp->cid : 0;
   winx = vptr.rootx;
   winy = vptr.rooty;
   for (tw=wp; tw->parent; tw=tw->parent) {
      winx -= XOFFSET(tw);
      winy -= YOFFSET(tw);
   }
   proto_QueryPointerReply(	cp,
				vscreen.wp->cid,
				child,
				vptr.rootx,
				vptr.rooty,
				winx, winy,
				vkbstat);
}

/************************************************************************
*									*
*   inp_query_keymap							*
*									*
************************************************************************/
void
inp_query_keymap
   AL((cp, p))
   DB client_t *cp
   DD xReq *p
   DE
{
	/* this is bogus, but should we select on all key events?? TODO */
   proto_QueryKeymapReply(cp, vkeymap);
}

/************************************************************************
*									*
*   inp_get_motion							*
*									*
************************************************************************/
void
inp_get_motion
   AL((cp, p))
   DB client_t *cp
   DD xGetMotionEventsReq *p
   DE
{
   register int i, j;
   register s16_t xorig, yorig, left, top, right, bottom;
   register window_t *wp, *tw;
   etime_t start, stop, etime;
   xTimecoord mbuf[MOBUFSIZE];

   if ((wp = (window_t *)hash_data(vmap, p->window)) == 0) {
      proto_Error(cp, BadWindow, p->window, 0, X_GetMotionEvents);
      return;
   }
   for (i=(curmo+1)%MOBUFSIZE; mobuf[i].time == 0; i=(i+1)%MOBUFSIZE);

   time_convert(p->start, &start);
   time_convert(p->stop, &stop);
   xorig = yorig = 0;
   for (tw=wp; tw->parent; tw=tw->parent) {
      xorig += XOFFSET(tw);
      yorig += YOFFSET(tw);
   }
   left = xorig - (s16_t)wp->borderwidth;
   top = yorig - (s16_t)wp->borderwidth;
   right = left + FULLWIDTH(wp) - 1;
   bottom = top + FULLHEIGHT(wp) - 1;
   for (j=0; i!=curmo; i=(i+1)%MOBUFSIZE)
      if (	mobuf[i].x >= left &&
		mobuf[i].x <= right &&
		mobuf[i].y >= top &&
		mobuf[i].y <= bottom) {
         time_convert(mobuf[i].time, &etime);
         if (	time_cmp(start, etime) <= 0 &&
		time_cmp(etime, stop) <= 0) {
            mbuf[j].time = mobuf[i].time;
            mbuf[j].x -= xorig;
            mbuf[j].y -= yorig;
            j++;
         }
      }

   proto_GetMotionEventsReply(cp, j, mbuf);
}

/************************************************************************
*									*
*   inp_translate_coords						*
*									*
************************************************************************/
void
inp_translate_coords
   AL((cp, p))
   DB client_t *cp
   DD xTranslateCoordsReq *p
   DE
{
   s16_t x, y;
   register rid_t child;
   register window_t *swp, *dwp, *wp;

   if ((swp = (window_t *)hash_data(vmap, p->srcWid)) == 0) {
      proto_Error(cp, BadWindow, p->srcWid, 0, X_TranslateCoords);
      return;
   }
   if ((dwp = (window_t *)hash_data(vmap, p->dstWid)) == 0) {
      proto_Error(cp, BadWindow, p->dstWid, 0, X_TranslateCoords);
      return;
   }
   if (swp != dwp)
      inp_translate(swp, dwp, p->srcX, p->srcY, &x, &y);
   else {
      x = p->srcX;
      y = p->srcY;
   }
   child = 0;

   for (wp=dwp->child; wp; wp=wp->nextsib)
      if (	wp->mapped &&
		x >= LEFTEDGE(wp) &&
		x < LEFTEDGE(wp) + FULLWIDTH(wp) &&
		y >= TOPEDGE(wp) &&
		y < TOPEDGE(wp) + FULLHEIGHT(wp)) {
         child = wp->cid;
         break;
      }
   proto_TranslateCoordsReply(cp, child, x, y);
}

/************************************************************************
*									*
*   inp_warp								*
*									*
*	This is probably not necessary, as WarpPointer requests		*
*	result in PointerMotion events, which we can catch on the	*
*	rebound.							*
*									*
************************************************************************/
int
inp_warp
   AL((cp, p))
   DB client_t *cp
   DD xWarpPointerReq *p
   DE
{
   register s16_t xmax, ymax;
   register window_t *wp;

   if (p->srcWid) {
      if ((wp = (window_t *)hash_data(vmap, p->srcWid)) == 0) {
         proto_Error(cp, BadWindow, p->srcWid, 0, X_WarpPointer);
         return -1;
      }
      xmax = p->srcWidth == 0 ? wp->dwb.width : p->srcX + p->srcWidth;
      ymax = p->srcHeight == 0 ? wp->dwb.height : p->srcY + p->srcHeight;
      if (	wp != ptrwp ||
		vptr.eventx < p->srcX || vptr.eventx >= xmax ||
		vptr.eventy < p->srcY || vptr.eventy >= ymax)
         return -1;
   }
   /* TODO have to do a regular pointer motion here... */
   if (vptr.grabcp && vptr.cwp) {
   }
   vptr.rootx += p->dstX;
   vptr.rooty += p->dstY;
   vptr.eventx += p->dstX;
   vptr.eventy += p->dstY;
   return 0;
}

/************************************************************************
*									*
*   inp_grab_pointer							*
*									*
************************************************************************/
void
inp_grab_pointer
   AL((cp, p))
   DB client_t *cp
   DD xGrabPointerReq *p
   DE
{
   register window_t *wp, *cwp = 0;
   register cursor_t *cursp = 0;
   etime_t reqtime;

   if ((wp = (window_t *)hash_data(vmap, p->grabWindow)) == 0) {
      proto_Error(cp, BadWindow, p->grabWindow, 0, X_GrabPointer);
      return;
   }
   if (wp->mapped == 0) {	/* need check for _visible_ TODO */
      proto_GrabPointerReply(cp, GrabNotViewable);
      return;
   }
   if (p->confineTo) {
      if ((cwp = (window_t *)hash_data(vmap, p->confineTo)) == 0) {
         proto_Error(cp, BadWindow, p->confineTo, 0, X_GrabPointer);
         return;
      }
      if (cwp->mapped == 0) {	/* need check for _visible_ TODO */
         proto_GrabPointerReply(cp, GrabNotViewable);
         return;
      }
   }
   if (p->cursor)
      if ((cursp = (cursor_t *)hash_data(vmap, p->cursor)) == 0) {
         proto_Error(cp, BadCursor, p->cursor, 0, X_GrabPointer);
         return;
      }
   if (vptr.grabcp != cp)
      if (vptr.grabcp) {
         proto_GrabPointerReply(cp, AlreadyGrabbed);
         return;
      }
      else if (vptr.sync && vkbd.grabcp && vkbd.grabcp != cp) {
         proto_GrabPointerReply(cp, GrabFrozen);
         return;
      }
   time_update(0);
   if (p->time) {
      time_convert(p->time, &reqtime);
      if (	time_cmp(reqtime, vptr.grabtime) < 0 ||
		time_cmp(reqtime, vtime) > 0) {
         proto_GrabPointerReply(cp, GrabInvalidTime);
         return;
      }
   }
   DEBUG2(D_GRABS, "inp_grab_pointer: client [%d] wp [0x%x] ",
							cp->clinum, wp);
   DEBUG2(D_GRABS, "ownev [%d] evmask [%s]\n",
			p->ownerEvents, dprx_event_mask_str(p->eventMask));
   proto_GrabPointerReply(cp, GrabSuccess);

   grab_pointer(	wp, cwp, cp, 1, p->ownerEvents, 0,
			(p->pointerMode == GrabModeSync),
			(p->keyboardMode == GrabModeSync),
			p->time ? &reqtime : &vtime,
			p->eventMask, cursp);
   /*
   **  play queued events
   */
   if (ebuf_ready())
      ebuf_play();
}

/************************************************************************
*									*
*  inp_ptr_grab_client							*
*									*
************************************************************************/
client_t *
inp_ptr_grab_client
   VOID
{
   return vptr.grabcp;
}

/*
**  inp_autograb
**
**	Initiate an automatic grab as the result of a ButtonPress that
**	caused no other grab.  The grab is automatically released when
**	all buttons are up.
*/
void
inp_autograb
   AL((wp, cp, mask))
   DB window_t *wp
   DD client_t *cp
   DD mask_t mask
   DE
{
   register int ownev;

   if (vptr.grabcp)
      return;

   ownev = mask & OwnerGrabButtonMask ? 1 : 0;

   DEBUG2(D_GRABS, "inp_autograb: client [%d] wp [0x%x]\n", cp->clinum, wp);
   grab_pointer(wp, 0, cp, 0, ownev, 0, 0, 0, &vtime, mask, 0);
   /*
   **  play queued events
   */
   if (ebuf_ready())
      ebuf_play();
}

u8_t *
inp_keymap
   VOID
{
   return &vkeymap[0];
}

/*
**  km_set
**  km_clr
**  km_zero
**
**	Set or clear a key in the keymap, or zero it.  The keymap is a
**	bit vector representation of the current state of the keyboard.
**	Bits on represent keys pressed.
*/
static void
km_set
   AL((keycode))
   DB u8_t keycode
   DE
{
   register int ix, shift;

   if (keycode > 7) {
      ix = keycode / 8;
      shift = keycode % 8;

      if ((vkeymap[ix] & 1 << shift) == 0) {
         vkeymap[ix] |= 1 << shift;
         keysdown++;
      }
   }
}

static void
km_clr
   AL((keycode))
   DB u8_t keycode
   DE
{
   register int ix, shift;

   if (keycode > 7) {
      ix = keycode / 8;
      shift = keycode % 8;

      if (vkeymap[ix] & 1 << shift) {
         vkeymap[ix] &= ~(1 << shift);
         keysdown--;
      }
   }
}

static void
km_zero
   VOID
{
   register int i;

   for (i=1; i<32; i++)
      vkeymap[i] = 0;

   keysdown = 0;
}

static void
grab_pointer
   AL((wp, cwp, cp, active, ownev, echp, ptrsync, kbdsync, tp, mask, cursp))
   DB window_t *wp
   DD window_t *cwp
   DD client_t *cp
   DD int active
   DD u8_t ownev
   DD chunk_t *echp	/* event that caused grab, if passive */
   DD int ptrsync
   DD int kbdsync
   DD etime_t *tp
   DD mask_t mask
   DD cursor_t *cursp
   DE
{
   vptr.grabwp = wp;
   vptr.grabcp = cp;
   vptr.active = active;
   vptr.ownev = ownev;
   vptr.sync = ptrsync;

   if (ptrsync) {
      if (active == 0 && echp)
         /*
         **  Save event for possible replay.
         */
         vptr.echp = clone_event_chunk(echp);

      vptr.frozen |= POINTER_GRAB;
   }
   else if (vptr.frozen & POINTER_GRAB)
      vptr.frozen &= ~POINTER_GRAB;

   if (kbdsync) {
      if (active == 0 && echp)
         /*
         **  Save event for possible replay.
         */
         vkbd.echp = clone_event_chunk(echp);

      vkbd.frozen |= POINTER_GRAB;
   }
   time_assign(*tp, vptr.grabtime);
   vptr.mask = mask;
   vptr.cwp = cwp;
   vptr.cursp = cursp;

   if (ptrwp != wp)
      leave_enter_events(ptrwp, wp, NotifyGrab);
}

/************************************************************************
*									*
*   inp_change_pointer_grab						*
*									*
************************************************************************/
void
inp_change_pointer_grab
   AL((cp, p))
   DB client_t *cp
   DD xChangeActivePointerGrabReq *p
   DE
{
   register cursor_t *cursp = 0;
   etime_t reqtime;

   if (cp != vptr.grabcp || vptr.active == 0)
      return;

   if (p->cursor)
      if ((cursp = (cursor_t *)hash_data(vmap, p->cursor)) == 0) {
         proto_Error(cp, BadCursor, p->cursor, 0, X_ChangeActivePointerGrab);
         return;
      }
   if (p->time) {
      time_update(0);
      time_convert(p->time, &reqtime);
      if (	time_cmp(reqtime, vptr.grabtime) < 0 ||
		time_cmp(reqtime, vtime) > 0)
         return;
   }
   DEBUG2(D_GRABS, "inp_change_pointer_grab: cursp [0x%x] eventmask [%s]\n",
				p->cursor ? cursp : 0,
				dprx_event_mask_str(p->eventMask));
   vptr.cursp = cursp;
   vptr.mask = p->eventMask;
}

/************************************************************************
*									*
*   inp_ungrab_pointer							*
*									*
************************************************************************/
void
inp_ungrab_pointer
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   etime_t reqtime;

   if (cp != vptr.grabcp)
      return;

   if (p->id) {
      time_update(0);
      time_convert(p->id, &reqtime);
      if (	time_cmp(reqtime, vptr.grabtime) < 0 ||
		time_cmp(reqtime, vtime) > 0)
         return;
   }
   DEBUG1(D_GRABS, "inp_ungrab_pointer: vptr.sync [%d]\n", vptr.sync);
   if (ungrab_pointer())
      if (ebuf_ready())
         ebuf_play();
}

/*
**  ungrab_pointer
**
**	Actually ungrab the pointer.  
**
**	Note that an ungrab_pointer is done in inp_allow_events
**	under ReplayPointer, by hand, without this routine.  So
**	if you are fixing ungrab code, you must visit there, too.
*/
static int
ungrab_pointer
   VOID
{
   register int unfroze = 0;
   /*
   **  This invalidates the rest of the grab state.  Play fast and
   **  loose here and don't bother zeroing the rest out.
   */
   vptr.grabcp = 0;

   if (vptr.frozen & POINTER_GRAB) {
      vptr.frozen &= ~POINTER_GRAB;
      vptr.echp = 0;
      unfroze = 1;
   }
   if (vkbd.frozen & POINTER_GRAB) {
      vkbd.frozen &= ~POINTER_GRAB;
      vkbd.echp = 0;
      unfroze = 1;
   }
   if (vptr.grabwp != ptrwp)
      leave_enter_events(vptr.grabwp, ptrwp, NotifyUngrab);
   
   return unfroze;
}

/************************************************************************
*									*
*   inp_grab_button							*
*									*
************************************************************************/
void
inp_grab_button
   AL((cp, p))
   DB client_t *cp
   DD xGrabButtonReq *p
   DE
{
   register window_t *wp, *cwp = 0;
   register cursor_t *cursp = 0;
   register grab_t *gp;

   if ((wp = (window_t *)hash_data(vmap, p->grabWindow)) == 0) {
      proto_Error(cp, BadWindow, p->grabWindow, 0, X_GrabButton);
      return;
   }
   if (p->confineTo)
      if ((cwp = (window_t *)hash_data(vmap, p->confineTo)) == 0) {
         proto_Error(cp, BadWindow, p->confineTo, 0, X_GrabButton);
         return;
      }
   if (p->cursor)
      if ((cursp = (cursor_t *)hash_data(vmap, p->cursor)) == 0) {
         proto_Error(cp, BadCursor, p->cursor, 0, X_GrabButton);
         return;
      }

   for (gp=wp->buttons; gp; gp=gp->next)
      if (gp->detail == p->button && gp->mods == p->modifiers)
         if (gp->cp == cp)	/* client's grab */
            break;
         else {			/* someone else's */
            proto_Error(cp, BadAccess, 0, 0, X_GrabButton);
            return;
         }
      else if (	(gp->detail == AnyButton || p->button == AnyButton ||
				gp->detail == p->button) &&
		(gp->mods == AnyModifier || p->modifiers == AnyModifier ||
				gp->mods == p->modifiers)) {
         proto_Error(cp, BadAccess, 0, 0, X_GrabButton);
         return;
      }
   if (gp == 0) {
      if (MALLOC(gp, grab_t *, sizeof(grab_t))) {
         proto_Error(cp, BadAlloc, 0, 0, X_GrabButton);
         return;
      }
      gp->next = wp->buttons;
      wp->buttons = gp;
   }
   gp->detail = p->button;
   gp->ownev = (p->ownerEvents != 0);
   gp->kbdsync = (p->keyboardMode == GrabModeSync);
   gp->ptrsync = (p->pointerMode == GrabModeSync);
   gp->mods = p->modifiers;
   gp->evmask = p->eventMask;
   gp->wp = wp;
   gp->cwp = cwp;
   gp->cursp = cursp;
   gp->cp = cp;
#ifdef DEBUG
   if (sizeof(grab_t) != 28)
      warn("inp_grab_button: not updated since grab_t last changed\n");
#endif
   DEBUG4(D_GRABS,
	"inp_grab_button: client [%d] ownev [%d] evmask [%s] window [0x%x]\n",
	cp->clinum, gp->ownev, dprx_event_mask_str(gp->evmask), wp);
}

/************************************************************************
*									*
*   inp_ungrab_button							*
*									*
************************************************************************/
void
inp_ungrab_button
   AL((cp, p))
   DB client_t *cp
   DD xUngrabButtonReq *p
   DE
{
   register window_t *wp;
   register grab_t *gp, *lp;

   if ((wp = (window_t *)hash_data(vmap, p->grabWindow)) == 0) {
      proto_Error(cp, BadWindow, p->grabWindow, 0, X_UngrabButton);
      return;
   }
   for (lp=0, gp=wp->buttons; gp; lp=gp, gp=gp->next)
      if (	gp->detail == p->button &&
		gp->mods == p->modifiers &&
		gp->cp == cp) {
         if (lp)
            lp->next = gp->next;
         else
            wp->buttons = gp->next;
         DEBUG2(D_GRABS, "inp_ungrab_button: wp [0x%x] client [%d]\n", wp,
							gp->cp->clinum);
         free(gp);
         break;
      }
}

/************************************************************************
*									*
*   inp_grab_keyboard							*
*									*
************************************************************************/
void
inp_grab_keyboard
   AL((cp, p))
   DB client_t *cp
   DD xGrabKeyboardReq *p
   DE
{
   register window_t *wp;
   etime_t reqtime;

   if ((wp = (window_t *)hash_data(vmap, p->grabWindow)) == 0) {
      proto_Error(cp, BadWindow, p->grabWindow, 0, X_GrabKeyboard);
      return;
   }
   /* grabWindow not viewable ? GrabNotViewable TODO */

   if (vkbd.grabcp != cp)
      if (vkbd.grabcp) {
         proto_GrabKeyboardReply(cp, AlreadyGrabbed);
         return;
      }
      else if (vkbd.sync && vptr.grabcp && vptr.grabcp != cp) {
         proto_GrabKeyboardReply(cp, GrabFrozen);
         return;
      }
   time_update(0);
   if (p->time) {
      time_convert(p->time, &reqtime);
      if (	time_cmp(reqtime, vkbd.grabtime) < 0 ||
		time_cmp(reqtime, vtime) > 0) {
         proto_GrabKeyboardReply(cp, GrabInvalidTime);
         return;
      }
   }
   DEBUG3(D_GRABS, "inp_grab_keyboard: client [%d] wp [0x%x] ownev [%a]\n",
					cp->clinum, wp, p->ownerEvents);
   proto_GrabKeyboardReply(cp, GrabSuccess);

   grab_keyboard(	wp, cp, 1, p->ownerEvents, 0,
			(p->pointerMode == GrabModeSync),
			(p->keyboardMode == GrabModeSync),
			0,
			p->time ? &reqtime : &vtime);
   /*
   **  play queued events
   */
   if (ebuf_ready())
      ebuf_play();
}

static void
grab_keyboard
   AL((wp, cp, active, ownev, echp, ptrsync, kbdsync, keycode, tp))
   DB window_t *wp
   DD client_t *cp
   DD int active
   DD u8_t ownev
   DD chunk_t *echp	/* event that caused grab, if passive */
   DD int ptrsync
   DD int kbdsync
   DD u8_t keycode
   DD etime_t *tp
   DE
{
   vkbd.grabwp = wp;
   vkbd.grabcp = cp;
   vkbd.active = active;
   vkbd.ownev = ownev;
   vkbd.keycode = keycode;	/* only valid if active == 0 */
   vkbd.sync = kbdsync;

   if (ptrsync) {
      if (active == 0 && echp)
         /*
         **  Save event for possible replay.
         */
         vptr.echp = clone_event_chunk(echp);
      
      vptr.frozen |= KEYBOARD_GRAB;
   }

   if (kbdsync) {
      if (active == 0 && echp)
         /*
         **  Save event for possible replay.
         */
         vkbd.echp = clone_event_chunk(echp);

      vkbd.frozen |= KEYBOARD_GRAB;
   }
   else if (vkbd.frozen & KEYBOARD_GRAB)
      vkbd.frozen &= ~KEYBOARD_GRAB;

   time_assign(*tp, vkbd.grabtime);

   focus_grab(wp, tp);
}

/*
**  clone_event_chunk
**
**	A bit of a hack.  Just make an exact copy of a chunk, and mark
**	it as an event - it had better be!
*/
static chunk_t *
clone_event_chunk
   AL((chp))
   DB chunk_t *chp
   DE
{
   buf_put(ebp, buf_data(chp), buf_chunksize(chp));
   es_mark(ebp);
   return buf_split(ebp, 0);
}

/************************************************************************
*									*
*   inp_ungrab_keyboard							*
*									*
************************************************************************/
void
inp_ungrab_keyboard
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   etime_t reqtime;

   if (cp != vkbd.grabcp)
      return;

   time_update(0);
   if (p->id) {
      time_convert(p->id, &reqtime);
      if (	time_cmp(reqtime, vkbd.grabtime) < 0 ||
		time_cmp(reqtime, vtime) > 0)
         return;
   }
   DEBUG1(D_GRABS, "inp_ungrab_keyboard: vkbd.sync [%d]\n", vkbd.sync);
   if (ungrab_keyboard(p->id ? &reqtime: &vtime))
      if (ebuf_ready())
         ebuf_play();
}

/*
**  ungrab_keyboard
**
**	Actually ungrab the keyboard.  
**
**	Note that an ungrab_keyboard is done in inp_allow_events
**	under ReplayKeyboard, by hand, without this routine.  So
**	if you are fixing ungrab code, you must visit there, too.
*/
static int
ungrab_keyboard
   AL((tp))
   DB etime_t *tp
   DE
{
   register int unfroze = 0;
   /*
   **  This invalidates the rest of the grab state.  Play fast and
   **  loose here and don't bother zeroing the rest out.
   */
   vkbd.grabcp = 0;

   if (vptr.frozen & KEYBOARD_GRAB) {
      vptr.frozen &= ~KEYBOARD_GRAB;
      vptr.echp = 0;
      unfroze = 1;
   }
   if (vkbd.frozen & KEYBOARD_GRAB) {
      vkbd.frozen &= ~KEYBOARD_GRAB;
      vkbd.echp = 0;
      unfroze = 1;
   }
   focus_ungrab(tp);
   
   return unfroze;
}

/************************************************************************
*									*
*   inp_grab_key							*
*									*
************************************************************************/
void
inp_grab_key
   AL((cp, p))
   DB client_t *cp
   DD xGrabKeyReq *p
   DE
{
   register window_t *wp;
   register grab_t *gp;

   if ((wp = (window_t *)hash_data(vmap, p->grabWindow)) == 0) {
      proto_Error(cp, BadWindow, p->grabWindow, 0, X_GrabKey);
      return;
   }
   for (gp=wp->keys; gp; gp=gp->next)
      if (gp->detail == p->key && gp->mods == p->modifiers)
         if (gp->cp == cp)	/* client's grab */
            break;
         else {			/* someone else's */
            proto_Error(cp, BadAccess, 0, 0, X_GrabKey);
            return;
         }
      else if (	(gp->detail == AnyKey || p->key == AnyKey ||
				gp->detail == p->key) &&
		(gp->mods == AnyModifier || p->modifiers == AnyModifier ||
				gp->mods == p->modifiers)) {
         proto_Error(cp, BadAccess, 0, 0, X_GrabKey);
         return;
      }
   if (gp == 0) {
      if (MALLOC(gp, grab_t *, sizeof(grab_t))) {
         proto_Error(cp, BadAlloc, 0, 0, X_GrabKey);
         return;
      }
      gp->next = wp->keys;
      wp->keys = gp;
   }
   gp->detail = p->key;
   gp->ownev = (p->ownerEvents != 0);
   gp->kbdsync = (p->keyboardMode == GrabModeSync);
   gp->ptrsync = (p->pointerMode == GrabModeSync);
   gp->mods = p->modifiers;
   gp->wp = wp;
   gp->cp = cp;
#ifdef DEBUG
   if (sizeof(grab_t) != 28)
      warn("inp_grab_key: not updated since grab_t last changed\n");
#endif
   DEBUG3(D_GRABS, "inp_grab_key: client [%d] ownev [%d] window [0x%x]\n",
						cp->clinum, gp->ownev, wp);
}

/************************************************************************
*									*
*   inp_ungrab_key							*
*									*
************************************************************************/
void
inp_ungrab_key
   AL((cp, p))
   DB client_t *cp
   DD xUngrabKeyReq *p
   DE
{
   register window_t *wp;
   register grab_t *gp, *lp;

   if ((wp = (window_t *)hash_data(vmap, p->grabWindow)) == 0) {
      proto_Error(cp, BadWindow, p->grabWindow, 0, X_UngrabKey);
      return;
   }
   for (lp=0, gp=wp->keys; gp; lp=gp, gp=gp->next)
      if (	gp->detail == p->key &&
		gp->mods == p->modifiers &&
		gp->cp == cp) {
         if (lp)
            lp->next = gp->next;
         else
            wp->keys = gp->next;
         DEBUG2(D_GRABS, "inp_ungrab_key: wp [0x%x] client [%d]\n", wp,
							gp->cp->clinum);
         free(gp);
         break;
      }
}

/************************************************************************
*									*
*   inp_playmask							*
*									*
*	Return a mask indicating which devices are unfrozen.  Called	*
*	from ebuf_play only, most likely.				*
*									*
************************************************************************/
uint_t
inp_playmask
   VOID
{
   return (uint_t)((vptr.frozen ? 0 : ButtonPressMask) |
				(vkbd.frozen ? 0 : KeyPressMask));
}

/************************************************************************
*									*
*   inp_allow_events							*
*									*
*	Unfreeze a frozen device, releasing events.  Except for the	*
*	"Replay" modes, this is pretty straightforward.			*
*									*
************************************************************************/
void
inp_allow_events
   AL((cp, p))
   DB client_t *cp
   DD xAllowEventsReq *p
   DE
{
   register grab_t *gp;
   etime_t reqtime;
   window_t *ewp;
   register xEvent *ep;

   if (p->time) {
      time_update(0);
      time_convert(p->time, &reqtime);
      if (	time_cmp(reqtime, vptr.grabtime) < 0 ||
		time_cmp(reqtime, vtime) > 0)
         return;					/* EXIT */
   }
   DEBUG2(D_GRABS, "inp_allow_events: client fd [%d] %s\n",
				cp->fd, dprx_allowevents_mode_str(p->mode));
   /*
   **  Have we got the pointer grabbed?
   */
   if (vptr.grabcp == cp) {
      switch (p->mode) {
         case AsyncPointer:
            vptr.frozen &= ~POINTER_GRAB;
            vptr.sync = 0;
            break;
         case SyncPointer:
            vptr.frozen &= ~POINTER_GRAB;
            break;
         case ReplayPointer:
            /*
            **  If the pointer was frozen our our behalf by an event...
            */
            if (vptr.frozen & POINTER_GRAB && vptr.echp) {
               /*
               **  ...then release the grab and replay that event.
               */
               vptr.grabcp = 0;
               vptr.frozen = 0;	/* guessing we thaw for all here */
               if (vptr.grabwp != ptrwp)
                  leave_enter_events(vptr.grabwp, ptrwp, NotifyUngrab);
               ep = (xEvent *)buf_data(vptr.echp);
               /*
               **  The old event window could be gone, or moved.  Only
               **  the root coords are guaranteed to be correct, so
               **  recalculate the event stuff.
               */
               inp_root_translate(	ep->u.keyButtonPointer.rootX,
					ep->u.keyButtonPointer.rootY,
					&ewp,
					&ep->u.keyButtonPointer.eventX,
					&ep->u.keyButtonPointer.eventY);
               ep->u.keyButtonPointer.event = ewp->cid;
               switch (ep->u.u.type & 0x7f) {
                  case ButtonPress:
                     gp = pgrabbutton(ewp, ep->u.u.detail, vkbstat & MODMASKS);
				/*
				**  Checking levels, as below, is okay as
				**  long as the new event window is either
				**  a child of the old grab window or on
				**  its path to the root.  That is probably
				**  always the case, but if not, there could
				**  be trouble here.  The same comment goes
				**  for ReplayKeyboard, further below.
				*/
                     if (gp && gp->wp->level > vptr.grabwp->level) {
                        /*
                        **  The replayed event triggers a passive grab.
                        */
                        ep->u.keyButtonPointer.event = gp->wp->cid;
                        ep->u.keyButtonPointer.sameScreen = 1;
                        ep->u.keyButtonPointer.child = get_child_and_translate(
                                        ewp, gp->wp,
                                        ep->u.keyButtonPointer.eventX,
                                        ep->u.keyButtonPointer.eventY,
                                        &ep->u.keyButtonPointer.eventX,
                                        &ep->u.keyButtonPointer.eventY);
                        event_sendto(vptr.echp, gp->cp);

                        time_convert(vptr.time, &reqtime);
                        grab_pointer(	gp->wp, gp->cwp, gp->cp, 0, gp->ownev,
					vptr.echp, gp->ptrsync, gp->kbdsync,
					&reqtime, gp->evmask, gp->cursp);
                     }
                     else
                        event_send(vptr.echp, 1, ewp, ButtonPressMask, 1);
                     break;
                  case ButtonRelease:
                     event_send(vptr.echp, 1, ewp, ButtonReleaseMask, 1);
                     break;
               }
               buf_clear(vptr.echp);	/* this gets zero'd below */
            }
            break;
         case AsyncKeyboard:	/* keyboard frozen by pointer grab? */
            vkbd.frozen &= ~POINTER_GRAB;
            break;
         case SyncKeyboard:	/* keyboard frozen by pointer grab? */
	    vkbd.frozen &= ~POINTER_GRAB;
            break;
         case AsyncBoth:
            if (vptr.frozen & POINTER_GRAB && vkbd.frozen & POINTER_GRAB) {
               vkbd.frozen &= ~POINTER_GRAB;
               vptr.frozen &= ~POINTER_GRAB;
               vptr.sync = 0;
            }
            else if (vkbd.grabcp == cp && vptr.frozen && vkbd.frozen) {
               vkbd.frozen = 0;
               vptr.frozen = 0;
               vptr.sync = 0;
            }
            break;
         case SyncBoth:
            if (vptr.frozen & POINTER_GRAB && vkbd.frozen & POINTER_GRAB) {
               vptr.frozen &= ~POINTER_GRAB;
               vkbd.frozen &= ~POINTER_GRAB;
            }
            else if (vkbd.grabcp == cp && vptr.frozen && vkbd.frozen) {
               vptr.frozen = 0;
               vkbd.frozen = 0;
            }
      }
   }
   /*
   **  Have we got the keyboard grabbed?
   */
   if (vkbd.grabcp == cp) {
      switch (p->mode) {
         case AsyncPointer:	/* pointer frozen by keyboard grab? */
            vptr.frozen &= ~KEYBOARD_GRAB;
            break;
         case SyncPointer:	/* pointer frozen by keyboard grab? */
            vptr.frozen &= ~KEYBOARD_GRAB;
            break;
         case AsyncKeyboard:
            vkbd.frozen &= ~KEYBOARD_GRAB;
            vkbd.sync = 0;
            break;
         case SyncKeyboard:
            vkbd.frozen &= ~KEYBOARD_GRAB;
            break;
         case ReplayKeyboard:
            /*
            **  If the pointer was frozen our our behalf by an event...
            */
            if (vkbd.frozen & KEYBOARD_GRAB && vkbd.echp) {
               /*
               **  ...then release the grab and replay that event.
               */
               vkbd.grabcp = 0;
               vkbd.frozen = 0;	/* guessing we thaw for all here */
               focus_ungrab(0);	/* guessing time is now */
               ep = (xEvent *)buf_data(vkbd.echp);
               /*
               **  The old event window could be gone, or moved.  Only
               **  the root coords are guaranteed to be correct, so
               **  recalculate the event stuff.
               */
               inp_root_translate(	ep->u.keyButtonPointer.rootX,
					ep->u.keyButtonPointer.rootY,
					&ewp,
					&ep->u.keyButtonPointer.eventX,
					&ep->u.keyButtonPointer.eventY);
               ep->u.keyButtonPointer.event = ewp->cid;
               switch (ep->u.u.type & 0x7f) {
                  case KeyPress:
                     gp = pgrabkey(ep->u.u.detail,
				ep->u.keyButtonPointer.state & MODMASKS);
				/*
				**  See the comment for ReplayPointer above
				*/
                     if (gp && gp->wp->level > vkbd.grabwp->level) {
                        /*
                        **  The replayed event triggers a passive grab.
                        */
                        ep->u.keyButtonPointer.event = gp->wp->cid;
                        ep->u.keyButtonPointer.sameScreen = 1;
                        ep->u.keyButtonPointer.child = get_child_and_translate(
                                        ewp, gp->wp,
                                        ep->u.keyButtonPointer.eventX,
                                        ep->u.keyButtonPointer.eventY,
                                        &ep->u.keyButtonPointer.eventX,
                                        &ep->u.keyButtonPointer.eventY);
                        event_sendto(vkbd.echp, gp->cp);

                        time_convert(vkbd.time, &reqtime);
                        grab_keyboard(	gp->wp, gp->cp, 0, gp->ownev,
					vkbd.echp, gp->ptrsync, gp->kbdsync,
					ep->u.u.detail, &reqtime);
                     }
                     else
                        event_send(vkbd.echp, 1, ewp, KeyPressMask, 1);
                     break;
                  case KeyRelease:
                     event_send(vkbd.echp, 1, ewp, KeyReleaseMask, 1);
                     break;
               }
               buf_clear(vkbd.echp);	/* this gets zero'd below */
            }
            break;
         case AsyncBoth:
            if (vptr.frozen & KEYBOARD_GRAB && vkbd.frozen & KEYBOARD_GRAB) {
               vptr.frozen &= ~KEYBOARD_GRAB;
               vkbd.frozen &= ~KEYBOARD_GRAB;
               vkbd.sync = 0;
            }
            break;
         case SyncBoth:
            if (vptr.frozen & KEYBOARD_GRAB && vkbd.frozen & KEYBOARD_GRAB) {
               vptr.frozen &= ~KEYBOARD_GRAB;
               vkbd.frozen &= ~KEYBOARD_GRAB;
            }
      }
   }
   if (vptr.frozen == 0)
      vkbd.echp = 0;

   if (vkbd.frozen == 0)
      vkbd.echp = 0;
   /*
   **  play queued events
   */
   if (ebuf_ready())
      ebuf_play();
}

/************************************************************************
*									*
*   inp_client_death							*
*									*
************************************************************************/
void
inp_client_death
   AL((cp))
   DB client_t *cp
   DE
{
   register int unfroze = 0;

   free_client_grabs(vscreen.wp, cp);

   if (vptr.grabcp == cp) {
      DEBUG2(D_GRABS,
		"inp_client_death: (release ptr grab) client [%d] wp [0x%x]\n",
		cp->clinum, vptr.grabwp);
      unfroze += ungrab_pointer();
   }
   if (vkbd.grabcp == cp) {
      DEBUG2(D_GRABS,
		"inp_client_death: (release kbd grab) client [%d] wp [0x%x]\n",
		cp->clinum, vptr.grabwp);
      unfroze += ungrab_keyboard(0);
   }
   if (unfroze)
      if (ebuf_ready())
         ebuf_play();
}

/************************************************************************
*									*
*   inp_window_invis							*
*									*
*	What to do when a window becomes invisible.			*
*									*
************************************************************************/
void
inp_window_invis
   AL((wp))
   DB window_t *wp
   DE
{
   register int unfroze = 0;

   if (vptr.grabwp == wp || vptr.cwp == wp) {
      DEBUG1(D_GRABS, "inp_window_invis: wp [0x%x], freeing ptr grab\n", wp);
      unfroze += ungrab_pointer();
   }
   if (vkbd.grabwp == wp) {
      DEBUG1(D_GRABS, "inp_window_invis: wp [0x%x], freeing kbd grab\n", wp);
      unfroze += ungrab_keyboard(0);
   }
   if (unfroze)
      if (ebuf_ready())
         ebuf_play();
}

/************************************************************************
*									*
*   inp_free_grabs							*
*									*
*	Free all passive grabs on a window without side effects.	*
*									*
************************************************************************/
void
inp_free_grabs
   AL((wp))
   DB window_t *wp
   DE
{
   register grab_t *lp, *gp;

   DEBUG1(D_GRABS, "inp_free_grabs: (only on) wp [0x%x]\n", wp);
   for (gp=wp->buttons; lp=gp;) {
      gp = gp->next;
      free(lp);
   }
   for (gp=wp->keys; lp=gp;) {
      gp = gp->next;
      free(lp);
   }
}

/************************************************************************
*									*
*   inp_reset								*
*									*
************************************************************************/
void
inp_reset
   VOID
{
   register chunk_t *chp;

   curmo = 0;

   km_zero();
   bzero(&vptr, sizeof(vptr));
   bzero(&vkbd, sizeof(vkbd));

   if (buf_active(ebp)) {	/* discard buffer'd events */
      chp = buf_split(ebp, 0);
      buf_clear(chp);
   }
}

/************************************************************************
*									*
*   inp_translate							*
*									*
*	Translate x and y values from one window's frame of reference	*
*	to another's.							*
*									*
************************************************************************/
void
inp_translate
   AL((swp, dwp, sx, sy, dx, dy))
   DB window_t *swp
   DD window_t *dwp
   DD s16_t sx
   DD s16_t sy
   DD s16_t *dx
   DD s16_t *dy
   DE
{
   for (; swp->level > dwp->level; swp=swp->parent) {
      sx += XOFFSET(swp);
      sy += YOFFSET(swp);
   }
   for (; dwp->level > swp->level; dwp=dwp->parent) {
      sx -= XOFFSET(dwp);
      sy -= YOFFSET(dwp);
   }
   for (; swp != dwp; swp=swp->parent, dwp=dwp->parent) {
      sx += XOFFSET(swp);
      sy += YOFFSET(swp);
      sx -= XOFFSET(dwp);
      sy -= YOFFSET(dwp);
   }
   *dx = sx;
   *dy = sy;
}

/*
**  get_child_and_translate
**
**	Translates coordinates as in inp_translate, but also returns
**	a child of the destination window if it is either the source
**	window or one of the source window's ancestors.
**	
*/
static rid_t
get_child_and_translate
   AL((swp, dwp, sx, sy, dx, dy))
   DB window_t *swp
   DD window_t *dwp
   DD s16_t sx
   DD s16_t sy
   DD s16_t *dx
   DD s16_t *dy
   DE
{
   register window_t *child = 0;

   for (; swp->level > dwp->level; swp=swp->parent) {
      sx += XOFFSET(swp);
      sy += YOFFSET(swp);
      child = swp;
   }
   for (; dwp->level > swp->level; dwp=dwp->parent) {
      sx -= XOFFSET(dwp);
      sy -= YOFFSET(dwp);
   }
   if (swp != dwp) {
      child = 0;
      for (; swp != dwp; swp=swp->parent, dwp=dwp->parent) {
         sx += XOFFSET(swp);
         sy += YOFFSET(swp);
         sx -= XOFFSET(dwp);
         sy -= YOFFSET(dwp);
      }
   }
   *dx = sx;
   *dy = sy;

   return child ? child->cid : 0;
}

static void
free_client_grabs
   AL((wp, cp))
   DB window_t *wp
   DD client_t *cp
   DE
{
   register grab_t *ngp, *gp, *pgp;
   register chunk_t *chp;

   if (wp->nextsib)
      free_client_grabs(wp->nextsib, cp);

   if (wp->child)
      free_client_grabs(wp->child, cp);

   if (imask_remove(wp->xmask, (char *)cp) && wp->dwb.res.client != cp) {
      MkCompMask(wp);
      zb_dest(ZD_ALLSERV, 0);
      proto_ChangeWindowAttributes(wp->cid, CWEventMask, &wp->compmask,
						wp->mp->mappixels);
   }
   for (pgp=0, ngp=wp->buttons; gp=ngp;) {
      ngp = ngp->next;
      if (gp->cp == cp) {
         if (pgp)
            pgp->next = ngp;
         else
            wp->buttons = ngp;
         free(gp);
      }
      else
         pgp = gp;
   }
   for (pgp=0, ngp=wp->keys; gp=ngp;) {
      ngp = ngp->next;
      if (gp->cp == cp) {
         if (pgp)
            pgp->next = ngp;
         else
            wp->keys = ngp;
         free(gp);
      }
      else
         pgp = gp;
   }
}

/*
**	look for a passive button grab on a window
*/
static grab_t *
pgrabbutton
   AL((wp, detail, mods))
   DB window_t *wp
   DD u8_t detail
   DD u16_t mods
   DE
{
   register grab_t *gp;

   if (wp->parent && (gp = pgrabbutton(wp->parent, detail, mods)))
      return gp;

   for (gp=wp->buttons; gp; gp=gp->next)
      if (	(gp->detail == AnyButton || gp->detail == detail) &&
		(gp->mods == AnyModifier || gp->mods == mods))
         return gp;

   return 0;
}

/*
**  pgrabkey
**
**	Look for a passive key grab.  We need to look at two paths
**	through the window hierarchy: the path between the root and the
**	focus window, and if the pointer is a descendent of the focus
**	window, the path between them.  This routine gets called a lot
**	- with every keystroke - so some effort is made here to
**	minimize the recursion.
**
**	Assumes focuswp is non-zero.
*/
static grab_t *
pgrabkey
   AL((detail, mods))
   DB u8_t detail
   DD u16_t mods
   DE
{
   register grab_t *gp;
   int gotfocus, sawptr;

   sawptr = 0;
   if ((gp = pkeytoroot(focuswp, detail, mods, &sawptr)) == 0 && sawptr == 0) {
      gotfocus = 0;
      gp = pkeytofocus(ptrwp, detail, mods, &gotfocus);
   }

   return gp;
}

/*
**  pkeytoroot
**
**	Ascend window hierarchy to the root window.
**	Search for passive key grabs on the way back.
**	If no grabs are found, sawptrp points to a boolean indicating
**	if any windows encountered contained the pointer.
*/
static grab_t *
pkeytoroot
   AL((wp, detail, mods, sawptrp))
   DB window_t *wp
   DD u8_t detail
   DD u16_t mods
   DD int *sawptrp
   DE
{
   register grab_t *gp;

   if (wp->parent && (gp = pkeytoroot(wp->parent, detail, mods, sawptrp)))
      return gp;

   for (gp=wp->keys; gp; gp=gp->next)
      if (	(gp->detail == AnyButton || gp->detail == detail) &&
		(gp->mods == AnyModifier || gp->mods == mods))
         return gp;

   if (wp == ptrwp)
      *sawptrp = 1;

   return 0;
}

/*
**  pkeytofocus
**
**	Ascend window hierarchy to just below the focus window.
**	If the focus window is found, search for passive key grabs
**	on the way back.
*/
static grab_t *
pkeytofocus
   AL((wp, detail, mods, gotfocusp))
   DB window_t *wp
   DD u8_t detail
   DD u16_t mods
   DD int *gotfocusp
   DE
{
   register grab_t *gp;

   if (wp->level <= focuswp->level)	/* not a descendent of focuswp */
      return 0;

   if (wp->parent == focuswp)	/* end recursive climb */
      *gotfocusp = 1;
   else if (gp = pkeytofocus(wp->parent, detail, mods, gotfocusp))
      return gp;

   if (*gotfocusp)
      for (gp=wp->keys; gp; gp=gp->next)
         if (	(gp->detail == AnyButton || gp->detail == detail) &&
		(gp->mods == AnyModifier || gp->mods == mods))
            return gp;

   return 0;
}

/*
**	generate window leave/enter events
*/
static void
leave_enter_events
   AL((fromwp, towp, mode))
   DB window_t *fromwp
   DD window_t *towp
   DD int mode
   DE
{
   register int direction;
   register u8_t ofocus, nfocus;
   register window_t *wp;

   /*
   **  are fromwp or towp the focus or an inferior of it?
   */
   if (focuswp) {
      for (wp=fromwp; wp->level > focuswp->level; wp=wp->parent);
      ofocus = (wp == focuswp);
      for (wp=towp; wp->level > focuswp->level; wp=wp->parent);
      nfocus = (wp == focuswp);
   }
   else
      ofocus = nfocus = 0;
   /*
   **  linear or nonlinear?	(positive is up, negative is down)
   */
   if (fromwp->level > towp->level) {
      for (wp=fromwp; wp->level > towp->level; wp=wp->parent);
      direction = (wp == towp) ? 1 : 0;
   }
   else {
      for (wp=towp; wp->level > fromwp->level; wp=wp->parent);
      direction = (wp == fromwp) ? -1 : 0;
   }
   enterleave(fromwp, towp, 0, 0, ofocus, nfocus, 0, 0, mode, direction);
}

/*
**	Recursively generate leave/enter events in accordance with
**	the mysterious and magical rules set forth in the X protocol spec.
**
**	This routine climbs the window heirarchy beginning at both of
**	the two windows involved.  As "leave" windows are encountered,
**	LeaveWindow events are generated.  As the recursion unwinds,
**	previously encountered "enter" windows generate EnterWindow events.
**	In this way a single recursive traversal generates all events in the
**	correct order.
**
**	Assumes both the old and new windows are non-zero.
*/
static void
enterleave
   AL((ow, nw, ochild, nchild, ofocus, nfocus, leaves, enters, mode, direction))
   DB window_t *ow
   DD window_t *nw
   DD window_t *ochild
   DD window_t *nchild
   DD u8_t ofocus
   DD u8_t nfocus
   DD int leaves
   DD int enters
   DD u8_t mode
   DD int direction
   DE
{
   if (ow->level > nw->level) {
      if (ow->xmask->mask & LeaveWindowMask)
         leave(ow, ochild, ofocus, mode, leaves, direction);
      enterleave(	ow->parent, nw, ow, nchild,
			ow == focuswp ? 0 : ofocus, nfocus,
			leaves+1, enters, mode, direction);
   }
   else if (ow->level < nw->level) {
      enterleave(	ow, nw->parent, ochild, nw,
			ofocus, nw == focuswp ? 0 : nfocus,
			leaves, enters+1, mode, direction);
      if (nw->xmask->mask & EnterWindowMask)
         enter(nw, nchild, nfocus, mode, enters, direction);
   }
   else if (ow != nw) {
      if (ow->xmask->mask & LeaveWindowMask)
         leave(ow, ochild, ofocus, mode, leaves, direction);
      enterleave(	ow->parent, nw->parent, ow, nw,
			ow == focuswp ? 0 : ofocus,
			nw == focuswp ? 0 : nfocus,
			leaves+1, enters+1, mode, direction);
      if (nw->xmask->mask & EnterWindowMask)
         enter(nw, nchild, nfocus, mode, enters, direction);
   }
   else {
      if (leaves == 0)
         leave(ow, ochild, ofocus, mode, leaves, direction);
      else if (enters == 0)
         enter(nw, nchild, nfocus, mode, enters, direction);
   }
}

static void
leave
   AL((wp, child, focus, mode, cnt, direction))
   DB window_t *wp
   DD window_t *child
   DD u8_t focus
   DD u8_t mode
   DD int cnt
   DD int direction
   DE
{
   register u8_t detail;
   register chunk_t *chp;
   s16_t x, y;

   if (cnt)
      if (direction)
         detail = NotifyVirtual;
      else
         detail = NotifyNonlinearVirtual;
   else if (direction < 0)
      detail = NotifyInferior;
   else if (direction > 0)
      detail = NotifyAncestor;
   else
      detail = NotifyNonlinear;
   inp_translate(	vscreen.wp, wp,
			vptr.rootx,
			vptr.rooty,
			&x, &y);
   event_LeaveNotify(	wp,
			child,
			time_stamp(vtime),
			vptr.rootx, vptr.rooty,
			x, y,
			vkbstat,
			mode,
			focus,
			detail);
}

static void
enter
   AL((wp, child, focus, mode, cnt, direction))
   DB window_t *wp
   DD window_t *child
   DD u8_t focus
   DD u8_t mode
   DD int cnt
   DD int direction
   DE
{
   register u8_t detail;
   s16_t x, y;
   register chunk_t *chp;

   if (cnt)
      if (direction)
         detail = NotifyVirtual;
      else
         detail = NotifyNonlinearVirtual;
   else if (direction < 0)
      detail = NotifyAncestor;
   else if (direction > 0)
      detail = NotifyInferior;
   else
      detail = NotifyNonlinear;
   inp_translate(	vscreen.wp, wp,
			vptr.rootx,
			vptr.rooty,
			&x, &y);
   event_EnterNotify(	wp,
			child,
			time_stamp(vtime),
			vptr.rootx, vptr.rooty,
			x, y,
			vkbstat,
			mode,
			focus,
			detail);
}
