/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   window.c								*
*									*
************************************************************************/
#include <X11/X.h>
#define NEED_EVENTS
#include <X11/Xproto.h>
#include <X11/Xatom.h>

#include "xmx.h"
#include "df.h"
#include "zb.h"
#include "res.h"
#include "incl/window.pvt.h"

#define Undef	0

/************************************************************************
*									*
*   window_reset							*
*									*
*	Destroy the client window heirarchy with no side effects.	*
*	Assumes all other resources have been or will be reset.		*
*									*
************************************************************************/
void
window_reset
   VOID
{
   window_t *wp;

   if (vscreen.wp) {
      for (wp=vscreen.wp->child; wp; wp=wp->nextsib);
      for (; wp; wp=wp->prevsib)	/* destroy bottom-to-top */
         proto_DestroyWindow(wp->cid);
      reset_tree(vscreen.wp->child);
      vscreen.wp->child = 0;
   }
}

/************************************************************************
*									*
*   window_client_death							*
*									*
*	Cleanup client resources and generate protocol requests		*
*	to be sent to servers to remove the buggers.			*
*									*
************************************************************************/
void
window_client_death
   AL((cp))
   DB client_t *cp
   DE
{
   register window_t *wp;

   wp = vscreen.wp;
   if (imask_remove(wp->xmask, (char *)cp)) {
      MkCompMask(wp);
      proto_ChangeWindowAttributes(	wp->cid,
					CWEventMask,
					&wp->compmask,
					wp->mp->mappixels);
   }
   if (wp->child) {
      if (cp)
         reparent_savesets(wp->child, cp);
      if (client_window_unmap(wp->child, cp))
         inp_adjust();
      client_window_death(wp->child, cp);
   }
}

/*
**	unmap client-owned window subtrees
*/
static int
client_window_unmap
   AL((wp, cp))
   DB window_t *wp
   DD client_t *cp
   DE
{
   register int didptr = 0;

   if (wp->nextsib)
      if (client_window_unmap(wp->nextsib, cp))
         didptr = 1;

   if (client_cmp(cp, wp->dwb.res.client)) {
      if (wp->child) {
         if (unmap_tree(wp->child, 0))
            didptr = 1;
      }
      if (unmap_window(wp, 0))
         didptr = 1;
   }
   else if (wp->child)
      if (client_window_unmap(wp->child, cp))
         didptr = 1;

   return didptr;
}

/*
**	destroy client-owned window subtrees and generate protocol
**	to DestroyWindow them on the servers.
*/
static void
client_window_death
   AL((wp, cp))
   DB window_t *wp
   DD client_t *cp
   DE
{
   if (wp->nextsib)
      client_window_death(wp->nextsib, cp);

   if (client_cmp(cp, wp->dwb.res.client)) {
      proto_DestroyWindow(wp->cid);
      unlink_window(wp);
      if (wp->child)
         free_tree(wp->child);
      free_window(wp);
   }
   else {
      if (imask_remove(wp->xmask, (char *)cp)) {
         MkCompMask(wp);
         proto_ChangeWindowAttributes(	wp->cid,
					CWEventMask,
					&wp->compmask,
					wp->mp->mappixels);
      }
      if (wp->child)
         client_window_death(wp->child, cp);
   }
}

/************************************************************************
*									*
*   window_xmc_client_death						*
*									*
*	Cleanup xmc client input masks.					*
*									*
************************************************************************/
void
window_xmc_client_death
   AL((xp))
   DB xmc_t *xp
   DE
{
   purge_xmc_client(vscreen.wp, xp);
}

static void
purge_xmc_client
   AL((wp, xp))
   DB window_t *wp
   DD xmc_t *xp
   DE
{
   if (wp->nextsib)
      purge_xmc_client(wp->nextsib, xp);

   (void)imask_remove(wp->xmcmask, (char *)xp);

   if (wp->child)
      purge_xmc_client(wp->child, xp);
}

/************************************************************************
*									*
*   window_create							*
*									*
************************************************************************/
window_t *
window_create
   AL((cp, p, chpp))
   DB client_t *cp
   DD xCreateWindowReq *p
   DD chunk_t **chpp
   DE
{
   register window_t *wp, *parent;
   register visual_t *vp;
   register chunk_t *chp;
   register u16_t class;
   register u8_t depth;

   if ((parent = (window_t *)hash_data(vmap, p->parent)) == 0) {
      proto_Error(cp, BadWindow, p->parent, 0, X_CreateWindow);
      return 0;
   }
   if (p->visual == CopyFromParent)
      vp = parent->vp;
   else if ((vp = (visual_t *)hash_data(vmap, p->visual)) == 0) {
      proto_Error(cp, BadIDChoice, p->visual, 0, X_CreateWindow);
      return 0;
   }
   class = (p->class == CopyFromParent) ? parent->class : p->class;

   /*
   ** look for BadMatch errors (and set depth)
   */
   switch (class) {
      case InputOutput:
         depth = p->depth ? p->depth : parent->dwb.depth;

         if (parent->class == InputOnly) {
            proto_Error(cp, BadMatch, 0, 0, X_CreateWindow);
            return 0;
         }
         if (sres_match(depth, vp)) {
            /* really should return depth or vid as appropriate here TODO */
            proto_Error(cp, BadMatch, 0, 0, X_CreateWindow);
            return 0;
         }
         break;
      case InputOnly:
         depth = p->depth;

         if (p->depth) {	/* depth must be zero */
            proto_Error(cp, BadMatch, p->depth, 0, X_CreateWindow);
            return 0;
         }
         /* check visual here */
         if (p->borderWidth) {
            proto_Error(cp, BadMatch, p->borderWidth, 0, X_CreateWindow);
            return 0;
         }
         if (p->mask & ~(	CWWinGravity |	/* only these attributes */
				CWOverrideRedirect |
				CWEventMask |
				CWDontPropagate |
				CWCursor) ) {
            proto_Error(cp, BadMatch, p->mask, 0, X_CreateWindow);
            return 0;
         }
         break;
      default:
         proto_Error(cp, BadValue, p->class, 0, X_CreateWindow);
         return 0;
   }
   /*
   ** look for BadValue errors
   */
   if (p->width == 0 || p->height == 0) {
      proto_Error(cp, BadValue, 0, 0, X_CreateWindow);
      return 0;
   }
   /*
   ** so far so good
   */
   wp = window_new(	cp,
			depth,
			p->wid,
			0,
			p->x, p->y,
			p->width, p->height,
			p->borderWidth,
			class,
			vp);
   if (wp == 0) {
      proto_Error(cp, BadAlloc, 0, 0, X_CreateWindow);
      return 0;
   }
   if ((wp->vid = hash_add_client_id(p->wid, (char *)wp)) == 0) {
      imask_free(wp->xmask);
      imask_free(wp->xmcmask);
      free(wp);
      proto_Error(cp, BadIDChoice, p->wid, 0, X_CreateWindow);
      return 0;
   }
   link_window(parent, wp);
   wp->mp = 0;
   if (process_attrs(cp, X_CreateWindow, wp, p->mask,
						(u32_t *)(p+1), win_defs)) {
      unlink_window(wp);
      imask_free(wp->xmask);
      imask_free(wp->xmcmask);
      free(wp);
      return 0;
   }
   if ((p->mask & CWEventMask) == 0) {
      zb_dest(QC_NONE, 0);
      proto_ChangeWindowAttributes(	wp->cid,
					CWEventMask,
					&wp->compmask,
					wp->mp->mappixels);
      *chpp = zb_split();
   }
   color_winlist_add(wp->mp, wp);
   event_CreateNotify(wp);
   cp->refs++;

   return wp;
}

/************************************************************************
*									*
*   window_new								*
*									*
*	Allocate a new window_t structure.  This routine DOES NOT set	*
*	the following fields:						*
*									*
*		vid		- virtual id (obsolete?)		*
*		atts[15]	- all attributes			*
*		mp		- colormap				*
*		parent		- parent window				*
*		prevsib		- upper sibling				*
*		nextsib		- lower sibling				*
*									*
************************************************************************/
window_t *
window_new
   AL((cp, depth, cid, mapped, x, y, w, h, bw, class, vp))
   DB client_t *cp	/* client owner */
   DD u8_t depth	/* depth */
   DD rid_t cid		/* client resource id */
   DD u8_t mapped	/* initially mapped? */
   DD s16_t x		/* screen location */
   DD s16_t y
   DD u16_t w		/* sizes */
   DD u16_t h
   DD u16_t bw
   DD u16_t class	/* window class */
   DD visual_t *vp	/* visual */
   DE
{
   register window_t *wp;

   if (MALLOC(wp, window_t *, sizeof(window_t)))
      return 0;

   wp->dwb.res.client = cp;
   wp->dwb.type = W_WINDOW;
   wp->dwb.depth = depth;
   wp->cid = cid;
   wp->level = 0;
   wp->mapped = mapped;
   wp->x = x;
   wp->y = y;
   wp->dwb.width = w;
   wp->dwb.height = h;
   wp->borderwidth = bw;
   wp->class = class;
   wp->vp = vp;
   wp->cursp = 0;
   wp->bgpxp = 0;
   wp->bdpxp = 0;
   wp->attmask = 0;

   wp->xmask = imask_alloc();
   wp->xmcmask = imask_alloc();
   MkCompMask(wp);

   wp->props = 0;
   wp->keys = wp->buttons = 0;
   wp->expos_q = 0;
   wp->saveset = 0;
   wp->child = 0;

#ifdef DEBUG
   if (sizeof(window_t) != 164)
      warn("window_new: not updated since window_t last changed\n");
#endif

   return wp;
}

/************************************************************************
*									*
*   window_change_attributes						*
*									*
************************************************************************/
window_t *
window_change_attributes
   AL((cp, p, chpp))
   DB client_t *cp
   DD xChangeWindowAttributesReq *p
   DD chunk_t **chpp
   DE
{
   register int i, j;
   register mask_t mask;
   register window_t *wp;
   register chunk_t *chp;
   register colormap_t *mp;
   u32_t pixels[3];
   static int mskix[3] = {IWBackPixel, IWBorderPixel, IWBackingPixel};

   if ((wp = (window_t *)hash_data(vmap, p->window)) == 0) {
      proto_Error(cp, BadWindow, p->window, 0, X_ChangeWindowAttributes);
      return 0;
   }
   mp = wp->mp;
   if (process_attrs(cp, X_ChangeWindowAttributes, wp, p->valueMask,
							(u32_t *)(p+1), 0))
      return 0;

   if (p->valueMask & CWColormap && mp != wp->mp) {
      color_winlist_delete(mp, wp);
      color_winlist_add(wp->mp, wp);
      /*
      ** only update pixel values if they are being used
      */
      mask = wp->attmask & (CWBackPixel | CWBorderPixel | CWBackingPixel);
      for (i=j=0; i<3; i++)
         if (mask & (1L<<mskix[i])) {
            pixels[j] = wp->atts[mskix[i]];
            j++;
         }
      if (mask) {
         zb_dest(ZD_NONE, 0);
         proto_ChangeWindowAttributes(wp->cid, mask, pixels, wp->mp->mappixels);
         *chpp = zb_split();
      }
      event_ColormapNotify(wp, 1, wp->mp->installed ?
					ColormapInstalled :
					ColormapUninstalled);
   }
   return wp;
}

void
window_unset_colormap
   AL((wp))
   DB window_t *wp
   DE
{
   wp->mp = 0;
   wp->atts[IWColormap] = 0;
}

/*
**	process_attrs
**
**	Note: wp->mp must be set coming into this routine.
*/
static int
process_attrs
   AL((cp, op, wp, mask, attrs, dattrs))
   DB client_t *cp
   DD u16_t op
   DD window_t *wp
   DD mask_t mask
   DD u32_t *attrs
   DD u32_t *dattrs
   DE
{
   register int i, j;
   register rid_t id;
   register mask_t tm;
   register colormap_t *mp;
   register cursor_t *cursp;
   register pixmap_t *pxp;
   client_t *tc;

   wp->attmask |= mask;

   if (mask & CWColormap) {	/* do this first */
      tm = mask & (CWColormap - 1);
      NUMONES(tm, j);
      wp->atts[IWColormap] = attrs[j];
      if (attrs[j] == CopyFromParent) {
         wp->attmask &= ~CWColormap;	/* CopyFromParent is the default */
         if (wp->parent) {
            wp->mp = wp->parent->mp;
            wp->atts[IWColormap] = (u32_t)wp->mp->cid;
         }
         else
            wp->mp = vscreen.colormap;	/* root default */
      }
      else if (mp = (colormap_t *)hash_data(vmap, attrs[j]))
         wp->mp = mp;
      else {
         proto_Error(cp, BadColor, attrs[j], 0, op);
         return -1;
      }
   }
   else if (dattrs) {
      wp->atts[IWColormap] = dattrs[IWColormap];
      wp->mp = wp->parent->mp;
   }
   for (i=0; i<IWCount; i++) {
      switch (mask & 1<<i) {
         case 0:
            if (dattrs)
               wp->atts[i] = dattrs[i];
            continue;			/* short circuit */
         case CWBackPixmap:
            pxp = 0;
            if (*attrs == None || *attrs == ParentRelative) {
               if (wp->parent == 0) {	/* root pixmap reset */
                  wp->atts[i] = vscreen.rootpixmap;
                  *attrs = vscreen.rootpixmap;
               }
               else if (*attrs == ParentRelative &&
				wp->dwb.depth != wp->parent->dwb.depth) {
                     proto_Error(cp, BadMatch, *attrs, 0, op);
                     return -1;
               }
               else
                  wp->atts[i] = *attrs;

               if (*attrs == None)
                  wp->attmask &= ~CWBackPixmap;	/* None is the default */
            }
            else if (pxp = (pixmap_t *)hash_data(vmap, *attrs)) {
               if (wp->dwb.depth != pxp->dwb.depth) {
                  proto_Error(cp, BadMatch, *attrs, 0, op);
                  return -1;
               }
               wp->atts[i] = *attrs;
            }
            else {
               proto_Error(cp, BadPixmap, *attrs, 0, op);
               return -1;
            }
            pixmap_assign(&wp->bgpxp, pxp);
            wp->attmask &= ~CWBackPixel;	/* cancels pixel background */
            break;
         case CWBorderPixmap:
            pxp = 0;
            if (*attrs == CopyFromParent) {
               if (wp->parent == 0)	/* root border pixmap reset */
                  wp->atts[i] = vscreen.borderpixmap;

               else if (wp->dwb.depth != wp->parent->dwb.depth) {
                  proto_Error(cp, BadMatch, *attrs, 0, op);
                  return -1;
               }
               else {
                  wp->atts[i] = *attrs;
                  pxp = wp->parent->bdpxp;
               }
               wp->attmask &= ~CWBorderPixmap;	/* CopyFromParent is default */
            }
            else if (pxp = (pixmap_t *)hash_data(vmap, *attrs)) {
               if (wp->dwb.depth != pxp->dwb.depth) {
                  proto_Error(cp, BadMatch, *attrs, 0, op);
                  return -1;
               }
               wp->atts[i] = *attrs;
            }
            else {
               proto_Error(cp, BadPixmap, *attrs, 0, op);
               return -1;
            }
            pixmap_assign(&wp->bdpxp, pxp);
            wp->attmask &= ~CWBorderPixel;	/* cancels pixel border */
            break;
         case CWBackingPixel:
            if (*attrs == 0)
               wp->attmask &= ~CWBackingPixel;	/* zero is default pixel */
            /** FALL THROUGH **/
         case CWBackPixel:
         case CWBorderPixel:
            wp->atts[i] = *attrs;
            break;
         case CWBitGravity:
            wp->atts[i] = *attrs;
            if (*attrs == ForgetGravity)
               wp->attmask &= ~CWBitGravity;	/* Forget is default */
            break;
         case CWWinGravity:
            wp->atts[i] = *attrs;
            if (*attrs == NorthWestGravity)
               wp->attmask &= ~CWWinGravity;	/* NorthWest is default */
            break;
         case CWBackingStore:
            wp->atts[i] = *attrs;
            if (*attrs == NotUseful)
               wp->attmask &= ~CWBackingStore;	/* NotUseful is default */
            break;
         case CWBackingPlanes:
            wp->atts[i] = *attrs;
            tm = (mask_t)( (1<<wp->dwb.depth) - 1);
            if (((mask_t)*attrs & tm) == tm)
               wp->attmask &= ~CWBackingPlanes;	/* all-ones is default */
            break;
         case CWOverrideRedirect:
            wp->atts[i] = *attrs;
            if (*attrs == xFalse)
               wp->attmask &= ~CWOverrideRedirect;	/* False is default */
            break;
         case CWSaveUnder:
            wp->atts[i] = *attrs;
            if (*attrs == xFalse)
               wp->attmask &= ~CWSaveUnder;	/* False is default */
            break;
         case CWEventMask:
            if (*attrs == 0) {
               if (imask_remove(wp->xmask, (char *)cp) && wp->xmask->mask == 0)
                  wp->attmask &= ~CWEventMask;	/* empty mask is the default */
            }
            else {
               tm = *attrs & (SubstructureRedirectMask | ResizeRedirectMask |
							ButtonPressMask);
               if (tm & wp->xmask->mask)
                  for (	j=imask_first(wp->xmask, tm, (char **)&tc);
			j;
			j=imask_next((char **)&tc))
                     if (tc != cp) {
                        proto_Error(cp, BadAccess, 0, 0, op);
                        return -1;
                     }
               imask_put(wp->xmask, (char *)cp, *attrs);
            }
            MkCompMask(wp);
            wp->atts[i] = wp->xmask->mask;
            *attrs = wp->compmask;
            break;
         case CWDontPropagate:
            wp->atts[i] = *attrs;
            if (*attrs == 0)
               wp->attmask &= ~CWDontPropagate;	/* propagate is the default */
            break;
         case CWColormap:
/*
		MUST BE:		TODO
                same root and visual as window
                if CopyFromParent, window must have same visual type as
                        parent & parent must not have Colormap of None
*/
            /* do nothing */
            break;
         case CWCursor:
            cursp = 0;
            if (*attrs == None) {
               if (wp->parent == 0)	/* root cursor reset */
                  wp->atts[i] = vscreen.cursor;
               else
                  wp->atts[i] = *attrs;
               wp->attmask &= ~CWCursor;	/* None is the default */
            }
            else if (cursp=(cursor_t *)hash_data(vmap, *attrs))
               wp->atts[i] = *attrs;
            else {
               proto_Error(cp, BadCursor, *attrs, 0, op);
               return -1;
            }
            cursor_assign(&wp->cursp, cursp);
            break;
         default:
            wp->atts[i] = *attrs;
            break;
      }
      attrs++;
   }
   return 0;
}

/************************************************************************
*									*
*   window_get_attributes						*
*									*
************************************************************************/
int
window_get_attributes
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   register window_t *wp, *ap;
   register u8_t mapstate;

   if ((wp = (window_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadWindow, p->id, 0, X_GetWindowAttributes);
      return -1;
   }
   if (wp->mapped) {
      mapstate = IsViewable;
      for (ap=wp->parent; ap; ap=ap->parent)
         if (ap->mapped == 0) {
            mapstate = IsUnviewable;
            break;
         }
   }
   else
      mapstate = IsUnmapped;

   proto_GetWindowAttributesReply(	cp,
					wp->vp->cid,
					wp->class,
					wp->mp->installed,
					mapstate,
					imask_get(wp->xmask, (char *)cp),
					wp->atts);
   return 0;
}

/************************************************************************
*									*
*   window_destroy							*
*									*
************************************************************************/
int
window_destroy
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   register int didptr = 0;
   register window_t *wp;

   if ((wp = (window_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadWindow, p->id, 0, X_DestroyWindow);
      return -1;
   }
   if (wp->parent == 0)			/* root window */
      return -1;

   if (wp->child)
      didptr = unmap_tree(wp->child, 0);
   if (unmap_window(wp, 0) || didptr)
      inp_adjust();
   unlink_window(wp);
   if (wp->child)
      free_tree(wp->child);
   free_window(wp);

   return 0;
}

/************************************************************************
*									*
*   window_destroy_subwindows						*
*									*
************************************************************************/
int
window_destroy_subwindows
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   register window_t *wp;

   if ((wp = (window_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadWindow, p->id, 0, X_DestroySubwindows);
      return -1;
   }
   if (wp->child) {
      if (unmap_subwindows(wp->child, 0))
         inp_adjust();
      free_subwindows(wp->child);
      wp->child = 0;
   }
   return 0;
}

/************************************************************************
*									*
*   window_change_save_set						*
*									*
************************************************************************/
int
window_change_save_set
   AL((cp, p))
   DB client_t *cp
   DD xChangeSaveSetReq *p
   DE
{
   register window_t *wp;
   int found;

   if ((wp = (window_t *)hash_data(vmap, p->window)) == 0) {
      proto_Error(cp, BadWindow, p->window, 0, X_ChangeSaveSet);
      return -1;
   }
   if (wp->dwb.res.client == cp) {
      proto_Error(cp, BadMatch, p->window, 0, X_ChangeSaveSet);
      return -1;
   }
   if (p->mode == 0) {					/* Insert */
      if (plist_find(wp->saveset, (void *)cp) == 0)
         if ((wp->saveset = plist_add(wp->saveset, (void *)cp)) == 0) {
            proto_Error(cp, BadAlloc, 0, 0, X_ChangeSaveSet);
            return -1;
         }
   }
   else {						/* Delete */
      wp->saveset = plist_delete(wp->saveset, (void *)cp, &found);
      if (!found) {
         proto_Error(cp, BadValue, p->window, 0, X_ChangeSaveSet);
         return -1;
      }
   }
   return 0;
}

/************************************************************************
*									*
*   window_reparent							*
*									*
************************************************************************/
int
window_reparent
   AL((cp, p))
   DB client_t *cp
   DD xReparentWindowReq *p
   DE
{
   register window_t *wp, *parent;

   if ((wp = (window_t *)hash_data(vmap, p->window)) == 0) {
      proto_Error(cp, BadWindow, p->window, 0, X_ReparentWindow);
      return -1;
   }
   if (wp->parent == 0)		/* can't reparent root window */
      return -1;
   if ((parent = (window_t *)hash_data(vmap, p->parent)) == 0) {
      proto_Error(cp, BadWindow, p->parent, 0, X_ReparentWindow);
      return -1;
   }
   wp->x = p->x;
   wp->y = p->y;
   reparent_window(parent, wp);

   return 0;
}

/************************************************************************
*									*
*   window_map								*
*									*
*	Returns zero if the window is mapped.  Returns non-zero		*
*	otherwise.							*
*									*
************************************************************************/
int
window_map
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   register window_t *wp;
   client_t *ecp;

   if ((wp = (window_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadWindow, p->id, 0, X_MapWindow);
      return -1;
   }
   if (wp->mapped)
      return -1;
   if (wp->atts[IWOverrideRedirect] == 0)
      if (wp->parent)
         if (	wp->parent->xmask->mask & SubstructureRedirectMask &&
		imask_first(	wp->parent->xmask,
				SubstructureRedirectMask,
				(char **)&ecp) &&
		cp != ecp) {
            event_MapRequest(ecp, wp);
            return -1;
         }

   map_window(wp);
   inp_adjust();
   return 0;
}

/************************************************************************
*									*
*   window_map_subwindows						*
*									*
*	Returns zero if at least one window was mapped.  Returns	*
*	non-zero otherwise.						*
*									*
************************************************************************/
int
window_map_subwindows
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   register didone = 0;
   register window_t *wp, *np;
   client_t *ecp;

   if ((wp = (window_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadWindow, p->id, 0, X_MapSubwindows);
      return -1;
   }
   if (	wp->xmask->mask & SubstructureRedirectMask &&
	imask_first(wp->xmask, SubstructureRedirectMask, (char **)&ecp) &&
	cp != ecp) {
      for (np=wp->child; np; np=np->nextsib)
         if (np->mapped == 0)
            event_MapRequest(ecp, np);
      return -1;
   }
   for (np=wp->child; np; np=np->nextsib)
      if (np->mapped == 0) {
         didone = 1;
         map_window(np);
      }
   if (didone) {
      inp_adjust();
      return 0;
   }
   else
      return -1;
}

/************************************************************************
*									*
*   window_unmap							*
*									*
************************************************************************/
int
window_unmap
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   register window_t *wp;

   if ((wp = (window_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadWindow, p->id, 0, X_UnmapWindow);
      return -1;
   }
   if (wp->mapped == 0)
      return -1;

   if (unmap_window(wp, 0))
      inp_adjust();

   return 0;
}

/************************************************************************
*									*
*   window_unmap_subwindows						*
*									*
************************************************************************/
int
window_unmap_subwindows
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   register window_t *wp;

   if ((wp = (window_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadWindow, p->id, 0, X_UnmapSubwindows);
      return -1;
   }
   if (unmap_subwindows(wp, 0))
      inp_adjust();

   return 0;
}

/************************************************************************
*									*
*   window_configure							*
*									*
************************************************************************/
int
window_configure
   AL((cp, p))
   DB client_t *cp
   DD xConfigureWindowReq *p
   DE
{
   register s16_t x, y;
   register u16_t width, height, borderwidth;
   register int dx, dy, dwidth, dheight;
   register u8_t stackmode = Above;	/* default */
   client_t *evcp;
   register window_t *wp, *sibp = 0;
   register mask_t m, tm;
   register u32_t *vp;
   uint_t adjptr;

   if ((wp = (window_t *)hash_data(vmap, p->window)) == 0) {
      proto_Error(cp, BadWindow, p->window, 0, X_ConfigureWindow);
      return -1;
   }
   x = wp->x;
   y = wp->y;
   width = wp->dwb.width;
   height = wp->dwb.height;
   borderwidth = wp->borderwidth;
   m = CWSibling | CWStackMode;
   if (p->mask & CWSibling && (p->mask & CWStackMode) == 0) {
      proto_Error(cp, BadMatch, 0, 0, X_ConfigureWindow);
      return -1;
   }
   vp = (u32_t *)(p + 1);
   for (m=1, tm=p->mask; tm; m<<=1, tm>>=1)
      if (tm & 1) {
         switch (m) {
            case CWX:
               x = (s16_t)*vp;
               break;
            case CWY:
               y = (s16_t)*vp;
               break;
            case CWWidth:
               if ((width = (u16_t)*vp) == 0) {
                  proto_Error(cp, BadValue, 0, 0, X_ConfigureWindow);
                  return -1;
               }
               break;
            case CWHeight:
               if ((height = (u16_t)*vp) == 0) {
                  proto_Error(cp, BadValue, 0, 0, X_ConfigureWindow);
                  return -1;
               }
               break;
            case CWBorderWidth:
               borderwidth = (u16_t)*vp;
               break;
            case CWSibling:
               if ((sibp = (window_t *)hash_data(vmap, (rid_t)*vp)) == 0) {
                  proto_Error(cp, BadWindow, *vp, 0, X_ConfigureWindow);
                  return -1;
               }
               if (sibp->parent != wp->parent) {
                  proto_Error(cp, BadMatch, *vp, 0, X_ConfigureWindow);
                  return -1;
               }
               break;
            case CWStackMode:
               stackmode = (u8_t)*vp;
         }
         vp++;
      }
		/* should we actually do anything? */
   if (wp->atts[IWOverrideRedirect] == 0)
      if (wp->parent)
         if (	wp->parent->xmask->mask & SubstructureRedirectMask &&
		imask_first(	wp->parent->xmask,
				SubstructureRedirectMask,
				(char **)&evcp) &&
		cp != evcp) {
            event_ConfigureRequest(	evcp,
					wp->parent->cid,
					p->window,
					sibp ? sibp->cid : 0,
					x, y, width, height, borderwidth,
					stackmode,
					p->mask);
            return -1;
         }
   dx = x - wp->x;
   dy = y - wp->y;
   dwidth = width - wp->dwb.width;
   dheight = height - wp->dwb.height;

   if (	(dwidth || dheight) &&
	wp->xmask->mask & ResizeRedirectMask &&
	imask_first(wp->xmask, ResizeRedirectMask, (char **)&evcp) &&
	cp != evcp) {
      event_ResizeRequest(evcp, p->window, width, height);
      width = wp->dwb.width;
      height = wp->dwb.height;
      dwidth = dheight = 0;
   }
		/* if we got this far, there are no errors, so */
		/* go ahead and modify the window */
   adjptr = inp_inside(wp) ? 0x1 : 0x0;
   for (m=1, tm=p->mask; tm; m<<=1, tm>>=1)
      if (tm & 1)
         switch (m) {
            case CWX:
               wp->x = x;
               break;
            case CWY:
               wp->y = y;
               break;
            case CWWidth:
               wp->dwb.width = width;
               break;
            case CWHeight:
               wp->dwb.height = height;
               break;
            case CWBorderWidth:
               wp->borderwidth = borderwidth;
               break;
         }
   if (p->mask & (CWX | CWY | CWWidth | CWHeight))
      adjptr |= (inp_inside(wp) ? 0x2 : 0x0);
   if (p->mask & CWStackMode) {
      if (adjptr)
         adjptr = 0x1;
      switch (stackmode) {
         case Above:
            if (sibp)
               sib_above(wp, sibp);
            else
               sib_to_top(wp);
            break;
         case Below:
            if (sibp)
               sib_below(wp, sibp);
            else
               sib_to_bottom(wp);
            break;
         case TopIf:
            if (util_occludes(wp, sibp) == -1)
               sib_to_top(wp);
            break;
         case BottomIf:
            if (util_occludes(wp, sibp) == 1)
               sib_to_bottom(wp);
            break;
         case Opposite:
            switch (util_occludes(wp, sibp)) {
               case -1:
                  sib_to_top(wp);
                  break;
               case 1:
                  sib_to_bottom(wp);
                  break;
            }
            break;
      }
   }
   event_ConfigureNotify(wp);
   if (dwidth || dheight) {
      for (sibp=wp->child; sibp; sibp=sibp->nextsib)
         gravity_reconfig(sibp, dx, dy, dwidth, dheight, &adjptr);
   }
			/* did we slither out from under the pointer? */
   if (adjptr == 0x1 || adjptr == 0x2)
      inp_adjust();
   return 0;
}

/************************************************************************
*									*
*   window_circulate							*
*									*
************************************************************************/
int
window_circulate
   AL((cp, p))
   DB client_t *cp
   DD xCirculateWindowReq *p
   DE
{
   register window_t *wp, *ew, *tw;
   client_t *evcp;
   register u8_t place;

   if ((wp = (window_t *)hash_data(vmap, p->window)) == 0) {
      proto_Error(cp, BadWindow, p->window, 0, X_CirculateWindow);
      return -1;
   }
   if (wp->parent == 0)	/* quietly ignore root window request */
      return -1;

   if (wp->prevsib == 0 && wp->nextsib == 0)	/* no sibs */
      return -1;

   if (wp->atts[IWOverrideRedirect] == 0)
      if (	wp->xmask->mask & SubstructureRedirectMask &&
		imask_first(	wp->xmask,
				SubstructureRedirectMask,
				(char **)&evcp) &&
		cp != evcp) {
         event_CirculateRequest(evcp, wp->parent->cid, wp->cid,
		p->direction == RaiseLowest ? PlaceOnTop : PlaceOnBottom);
         return -1;
      }

   switch (p->direction) {
      case RaiseLowest:
         place = PlaceOnTop;
         for (ew=wp->child; ew->nextsib; ew=ew->nextsib);
         /*
         **  search for lowest occluded sibling
         */
         while (ew && util_occludes(ew, 0) >= 0)
            ew = ew->prevsib;
         if (ew == 0)
            return -1;

         ew->prevsib->nextsib = ew->nextsib;
         if (ew->nextsib)
            ew->nextsib->prevsib = ew->prevsib;
         ew->nextsib = ew->parent->child;
         ew->nextsib->prevsib = ew;
         ew->parent->child = ew;
         ew->prevsib = 0;
         break;
      case LowerHighest:
         place = PlaceOnBottom;
         ew = wp->child;
         /*
         **  search for highest occluding sibling
         */
         while (ew && util_occludes(ew, 0) <= 0)
            ew = ew->nextsib;
         if (ew == 0)
            return -1;

         ew->nextsib->prevsib = ew->prevsib;
         if (ew->prevsib)
            ew->prevsib->nextsib = ew->nextsib;
         else
            ew->parent->child = ew->nextsib;
         for (tw=ew->nextsib; tw->nextsib; tw=tw->nextsib);
         tw->nextsib = ew;
         ew->prevsib = tw;
         break;
      default:
         proto_Error(cp, BadValue, p->direction, 0, X_CirculateWindow);
         return -1;
   }
   event_CirculateNotify(wp, ew->cid, place);
   return 0;
}

/************************************************************************
*									*
*   window_get_geometry							*
*									*
************************************************************************/
int
window_get_geometry
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   drawable_t *dp;

   if ((dp = (drawable_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadDrawable, p->id, 0, X_GetGeometry);
      return -1;
   }
   switch (dp->type) {
      case W_WINDOW:
         proto_GetGeometryReply(	cp,
					dp->depth,
					vscreen.wp->cid,
					((window_t *)dp)->x,
					((window_t *)dp)->y,
					dp->width,
					dp->height,
					((window_t *)dp)->borderwidth);
         break;
      case W_PIXMAP:
         proto_GetGeometryReply(	cp,
					dp->depth,
					vscreen.wp->cid,
					0, 0,
					dp->width,
					dp->height, 0);
         break;
   }
   return 0;
}

/************************************************************************
*									*
*   window_query_tree							*
*									*
************************************************************************/
int
window_query_tree
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   window_t *wp, *np;
   register list_t *lstp;

   if ((wp = (window_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadWindow, p->id, 0, X_QueryTree);
      return -1;
   }
   if (np=wp->child)
      for (; np->nextsib; np=np->nextsib);	/* wind to end */

   lstp = list_new();
   for (; np; np=np->prevsib) {
      list_put(lstp, np->cid);
   }
   proto_QueryTreeReply(	cp,
				vscreen.wp->cid,
				wp->parent ? wp->parent->cid : None,
				lstp);
   list_free(lstp);

   return 0;
}

/************************************************************************
*									*
*   window_ketchup							*
*									*
************************************************************************/
void
window_ketchup
   VOID
{
   ketchup(vscreen.wp);
}

/*
**	ketchup  -  recursive window traversal
*/
static void
ketchup
   AL((wp))
   DB window_t *wp
   DE
{
   register int i, j;
   register rid_t pid, vid;
   register mask_t mask;
   u32_t atts[IWCount];

   if (wp->nextsib)		/* build back-to-front */
      ketchup(wp->nextsib);
   /*
   **  all windows get event masks
   */
   mask = wp->attmask | CWEventMask;

   if (wp->parent == 0) {
      /*
      **  root window defaults differ slightly
      */
      mask |= (CWBackPixmap | CWBorderPixmap | CWColormap | CWCursor);

      pid = vscreen.shellroot;
      vid = wp->vp->cid;
   }
   else {
      pid = wp->parent->cid;
      vid = (wp->vp != wp->parent->vp) ? wp->vp->cid : CopyFromParent;
   }
   for (i=j=0; i<IWCount; i++) {
      if (mask & (1L<<i)) {
         switch (i) {
            case IWEventMask:
               atts[j] = wp->compmask;
               break;
            default:
               atts[j] = wp->atts[i];
               break;
         }
         j++;
      }
   }
   proto_CreateWindow(	wp->dwb.depth,
			wp->cid,
			pid,
			wp->x, wp->y,
			wp->dwb.width, wp->dwb.height,
			wp->borderwidth,
			wp->class,
			vid,
			mask,
			atts,
			wp->mp->mappixels);
   if (wp->mapped)
      proto_MapWindow(wp->cid);

   if (wp->child)		/* parents first */
      ketchup(wp->child);
}

/************************************************************************
*									*
*   window_expose_all							*
*									*
*	Generate expose events on all mapped client windows.		*
*									*
************************************************************************/
void
window_expose_all
   VOID
{
   if (vscreen.wp->child)
      expose(vscreen.wp->child);
}

/*
**	expose  -  recursive traversal
*/
static void
expose
   AL((wp))
   DB window_t *wp
   DE
{
   if (wp->nextsib)
      expose(wp->nextsib);
   if (wp->child)
      expose(wp->child);
   if (wp->mapped)
      event_Expose(wp);
}

/************************************************************************
*									*
*   window_touch_emask							*
*									*
*	Generate ChangeWindowAttributes requests to reset the event	*
*	mask of a window (or all windows).				*
*									*
************************************************************************/
void
window_touch_emask
   AL((wp))
   DB window_t *wp
   DE
{
   proto_ChangeWindowAttributes(wp->cid, CWEventMask, &wp->compmask,
							wp->mp->mappixels);
}

/************************************************************************
*									*
*   window_touch_all_emasks						*
*									*
*	Generate ChangeWindowAttributes requests to reset the event	*
*	mask of all windows.  This effects an input mode change.	*
*									*
************************************************************************/
void
window_touch_all_emasks
   VOID
{
   touch_emasks(vscreen.wp);
}

static void
touch_emasks
   AL((wp))
   DB window_t *wp
   DE
{
   if (wp->nextsib)
      touch_emasks(wp->nextsib);

   if (wp->child)
      touch_emasks(wp->child);

   proto_ChangeWindowAttributes(wp->cid, CWEventMask, &wp->compmask,
							wp->mp->mappixels);
}

/************************************************************************
*									*
*   window_print							*
*									*
************************************************************************/
void
window_print
   AL((wp))
   DB window_t *wp
   DE
{
   warn("window tree:\n");
   printwin(wp, 0);
   warn("\n");
}

void
printwin
   AL((wp, depth))
   DB window_t *wp
   DD int depth
   DE
{
   int i;
   char buf[128];

   if (wp->nextsib && wp->nextsib->prevsib != wp)
      warn("%08x*", wp);
   else if (wp->prevsib && wp->prevsib->nextsib != wp)
      warn("%08x*", wp);
   else if (wp->child && wp->child->parent != wp)
      warn("%08x*", wp);
   else
      warn("%08x ", wp);

   if (wp->child)
      printwin(wp->child, depth+1);

   if (wp->nextsib) {
      for (i=0; i<depth*9; i++)
         buf[i] = ' ';
      buf[i] = '\0';
      warn("\n%s", buf);
      printwin(wp->nextsib, depth);
   }
}

static void
map_window
   AL((wp))
   DB window_t *wp
   DE
{
   wp->mapped = 1;
   event_MapNotify(wp);
}

/*
**	unmap siblings bottom-up
*/
static int
unmap_subwindows
   AL((wp, fromconf))
   DB window_t *wp
   DD u8_t fromconf
   DE
{
   register int didptr = 0;

   if (wp->nextsib)
      if (unmap_subwindows(wp->nextsib, fromconf))
         didptr = 1;

   if (unmap_window(wp, fromconf))
      didptr = 1;

   return didptr;
}

/*
**	unmap an entire window heirarchy subtree (prior to destroy_window)
*/
static int
unmap_tree
   AL((wp, fromconf))
   DB window_t *wp
   DD u8_t fromconf
   DE
{
   register int didptr = 0;

   for (; wp; wp=wp->nextsib) {
      if (wp->child)
         if (unmap_tree(wp->child, fromconf))
            didptr = 1;
      if (unmap_window(wp, fromconf))
         didptr = 1;
   }
   return didptr;
}

/*
**	unmap a window
*/
static int
unmap_window
   AL((wp, fromconf))
   DB window_t *wp
   DD u8_t fromconf
   DE
{
   wp->mapped = 0;
   event_UnmapNotify(wp, fromconf);

   if (wp == focuswp)
      focus_revert();

   inp_window_invis(wp);

   return (wp == ptrwp);
}

static void
link_window
   AL((parent, wp))
   DB window_t *parent
   DD window_t *wp
   DE
{
   wp->prevsib = 0;
   wp->nextsib = parent->child;
   parent->child = wp;
   wp->parent = parent;
   if (wp->nextsib)
      wp->nextsib->prevsib = wp;
   if (wp->level != parent->level + 1) {
      wp->level = parent->level + 1;
      if (wp->child)
         update_level(wp->child, wp->level + 1);
   }
}

/*
**	recursively adjust "level" fields of window subheirarchy
*/
static void
update_level
   AL((wp, level))
   DB window_t *wp
   DD u16_t level
   DE
{
   wp->level = level;
   if (wp->child)
      update_level(wp->child, level + 1);
   if (wp->nextsib)
      update_level(wp->nextsib, level);
}

static void
unlink_window
   AL((wp))
   DB window_t *wp
   DE
{
   if (wp->parent->child == wp)
      if (wp->nextsib) {
         wp->parent->child = wp->nextsib;
         wp->nextsib->prevsib = 0;
      }
      else
         wp->parent->child = 0;
   else {
      if (wp->prevsib)
         wp->prevsib->nextsib = wp->nextsib;
      if (wp->nextsib)
         wp->nextsib->prevsib = wp->prevsib;
   }
}

/*
**	free siblings bottom-up
*/
static void
free_subwindows
   AL((wp))
   DB window_t *wp
   DE
{
   if (wp->nextsib)
      free_subwindows(wp->nextsib);

   free_tree(wp);
}

/*
**	free an entire window heirarchy subtree (destroy_window)
*/
static void
free_tree
   AL((wp))
   DB window_t *wp
   DE
{
   register window_t *lwp;

   while (lwp=wp) {
      wp = wp->nextsib;
      if (lwp->child)
         free_tree(lwp->child);
      free_window(lwp);
   }
}

/*
**	free a window
*/
static void
free_window
   AL((wp))
   DB window_t *wp
   DE
{
   hash_mark(vmap, wp->cid);

   free_savesets(wp);
   color_winlist_delete(wp->mp, wp);
   gc_fix_window(wp);	/* don't leave any gc's pointing at it */
				/* note: this window may be unlinked, */
				/* but it's parent should */
   event_DestroyNotify(wp);	/* still point to where it was...  */

   imask_free(wp->xmask);
   imask_free(wp->xmcmask);

   wp->dwb.res.client->refs--;

   if (ptrwp == wp)
      inp_adjust();
   if (focuswp == wp)
      focus_revert();

   if (wp->buttons || wp->keys)
      inp_free_grabs(wp);

   if (wp->props)
      prop_free(wp);

   if (wp->expos_q)
      expos_window_death(wp);

   if (wp->cursp)
      cursor_deref(wp->cursp);
   if (wp->bgpxp)
      pixmap_deref(wp->bgpxp);
   if (wp->bdpxp)
      pixmap_deref(wp->bdpxp);

   free(wp);
}

static void
reset_tree
   AL((wp))
   DB window_t *wp
   DE
{
   if (wp->child)
      reset_tree(wp->child);
   if (wp->nextsib)
      reset_tree(wp->nextsib);
   free_savesets(wp);
   imask_free(wp->xmask);
   imask_free(wp->xmcmask);
   prop_free(wp);
   inp_free_grabs(wp);
   free(wp);
}

static void
free_savesets
   AL((wp))
   DB window_t *wp
   DE
{
   plist_free(wp->saveset);
}

/*
**   reparent_window_xy
**
**	reparent the window and preserve its screen location
*/
static void
reparent_window_xy
   AL((parent, wp))
   DB window_t *parent
   DD window_t *wp
   DE
{
   inp_translate(wp->parent, parent, wp->x, wp->y, &wp->x, &wp->y);
   reparent_window(parent, wp);
}

/*
**   reparent_window
*/
static void
reparent_window
   AL((parent, wp))
   DB window_t *parent
   DD window_t *wp
   DE
{
   register window_t *oparent;

   oparent = wp->parent;
   unlink_window(wp);
   link_window(parent, wp);
   event_ReparentNotify(oparent, wp);
}

static void
reparent_savesets
   AL((wp, cp))
   DB window_t *wp
   DD client_t *cp
   DE
{
   register window_t *parent, *oldest;

   if (wp->nextsib)
      reparent_savesets(wp->nextsib, cp);

   if (wp->saveset) {		/* if this window is part of a saveset */
      if (plist_find(wp->saveset, (void *)cp)) {	/*...our saveset */
         oldest = 0;
         for (parent=wp->parent; parent; parent=parent->parent)
            if (parent->dwb.res.client == cp)
               oldest = parent;
         if (oldest != 0) {	/* in save set AND an inferior of this client */
            reparent_window_xy(oldest->parent, wp);
            proto_ReparentWindow(	wp->cid,
					oldest->parent->cid,
					wp->x, wp->y);
            if (wp->mapped == 0) {
               proto_MapWindow(wp->cid);
               map_window(wp);
            }
            return;	/* is this right? TODO */
         }
      }
   }
   if (wp->child)
      reparent_savesets(wp->child, cp);
}

static void
sib_to_top
   AL((wp))
   DB window_t *wp
   DE
{
   if (wp->prevsib == 0)
      return;
   if (wp->nextsib)
      wp->nextsib->prevsib = wp->prevsib;
   wp->prevsib->nextsib = wp->nextsib;
   wp->nextsib = wp->parent->child;
   wp->nextsib->prevsib = wp;
   wp->parent->child = wp;
   wp->prevsib = 0;
}

static void
sib_to_bottom
   AL((wp))
   DB window_t *wp
   DE
{
   window_t *lwp;

   if (wp->nextsib == 0)
      return;
   for (lwp=wp->nextsib; lwp->nextsib; lwp=lwp->nextsib);
   if (wp->prevsib)
      wp->prevsib->nextsib = wp->nextsib;
   else
      wp->parent->child = wp->nextsib;

   wp->nextsib->prevsib = wp->prevsib;
   lwp->nextsib = wp;
   wp->prevsib = lwp;
   wp->nextsib = 0;
}

static void
sib_above
   AL((wp, sib))
   DB window_t *wp
   DD window_t *sib
   DE
{
   if (wp->prevsib)
      wp->prevsib->nextsib = wp->nextsib;
   if (wp->nextsib)
      wp->nextsib->prevsib = wp->prevsib;
   wp->nextsib = sib;
   wp->prevsib = sib->prevsib;
   sib->prevsib = wp;
   if (wp->prevsib)
      wp->prevsib->nextsib = wp;
   else					/* wp is now the top */
      wp->parent->child = wp;
}

static void
sib_below
   AL((wp, sib))
   DB window_t *wp
   DD window_t *sib
   DE
{
   if (wp->prevsib)
      wp->prevsib->nextsib = wp->nextsib;
   else						/* wp was the top */
      wp->parent->child = wp->nextsib;
   if (wp->nextsib)
      wp->nextsib->prevsib = wp->prevsib;
   wp->prevsib = sib;
   wp->nextsib = sib->nextsib;
   sib->nextsib = wp;
   if (wp->nextsib)
      wp->nextsib->prevsib = wp;
}

/*
**	recursively reconfigure child windows according to win-gravity
*/
static void
gravity_reconfig
   AL((wp, dx, dy, dwidth, dheight, adjptr))
   DB window_t *wp
   DD int dx
   DD int dy
   DD int dwidth
   DD int dheight
   DD uint_t *adjptr
   DE
{
   register window_t *cp;
   register u16_t ox, oy;

   for (cp=wp->child; cp; cp=cp->nextsib)
      gravity_reconfig(cp, dx, dy, dwidth, dheight, adjptr);

   ox = wp->x;
   oy = wp->y;
   switch (wp->atts[IWWinGravity]) {
      case UnmapGravity:
         if (unmap_window(wp, 1))
            *adjptr = 0x1;
         break;
      case NorthWestGravity:
         break;
      case NorthGravity:
         wp->x += dwidth / 2;
         break;
      case NorthEastGravity:
         wp->x += dwidth;
         break;
      case WestGravity:
         wp->y += dheight / 2;
         break;
      case CenterGravity:
         wp->x += dwidth / 2;
         wp->y += dheight / 2;
         break;
      case EastGravity:
         wp->x += dwidth;
         wp->y += dheight / 2;
         break;
      case SouthWestGravity:
         wp->y += dheight;
         break;
      case SouthGravity:
         wp->x += dwidth / 2;
         wp->y += dheight;
         break;
      case SouthEastGravity:
         wp->x += dwidth;
         wp->y += dheight;
         break;
      case StaticGravity:
         wp->x -= dx;
         wp->y -= dy;
         break;
   }
   if (ox != wp->x || oy != wp->y)
      event_GravityNotify(wp);
}
