/* XNibbles - A simple X11 snake game
   Copyright (C) 1998 Sean MacIsaac and Ian Peters
  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY of FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
  
   Specific questions about this program can be addressed to the authors,
   Sean Macisaac (sjm@acm.org), and Ian Peters (ipeters@acm.org). */

/* joystick support added 23/2/98 by Bevan Schroeder bevan@foo.net
   JOYSTICK_THRESHOLD and JOYSTICK_MAX are probably wrong if it
   isn't working for you (works great with my Gravis) */

#include "common.h"

Display *display;
Window win;
GC gc;

Cursor myCursor;
Colormap myColors;
Pixmap theMask;
Pixmap digitXpms[10];
XpmAttributes myXpmAttributes;

int screenNum, install, quiet, cursor = 0;
char *progName, objects[BWIDTH][BHEIGHT], displayName[20];

level *myLevel;
sideBar *mySideBar;

void getGC(Window win, GC *gc);
void setupX(Window& win, GC& gc, int argc, char **argv, long mask);
void parseCommandLine(int argc, char **argv, char *displayName, int& install, int& quiet, char *levelFile);
void printOptions(char **argv);
void setupStuff();
void destroyStuff();
void error(int errorCode);
void myQuit (int code);
void getLevel(char *levelFile, char objects[BWIDTH][BHEIGHT]);
void writeLevel(char *levelFile, char objects[BWIDTH][BHEIGHT]);

void uDelay (unsigned long usec) {
  struct timeval delay;

  delay.tv_sec = usec / 1000000L;
  delay.tv_usec = usec % 1000000L;

  select (0, NULL, NULL, NULL, &delay );
}

void printBanner() {
  fprintf(stderr, "\nXNL Editor 1.0 BETA 4, Copyright (C) 1998 Sean MacIsaac and Ian Peters\n");
}

int main(int argc, char **argv) {
  XEvent report;

  long mask = ButtonPressMask | ButtonReleaseMask | PointerMotionMask | FocusChangeMask | KeyPressMask;
  char levelFile[255];
  char keyPress[15];
  int mouseDown = 0;
  int mouseButton = Button1;
  int gridX = -1, gridY = -1;

  progName = argv[0];
  levelFile[0] = 0;

  parseCommandLine(argc, argv, displayName, install, quiet, levelFile);

  if (!quiet) {
    printBanner();
    fprintf(stderr,"\n");
  }

  setupX(win, gc, argc, argv, mask | ExposureMask);
  myCursor = XCreateFontCursor(display, XC_tcross);
  XDefineCursor(display, win, myCursor);
  setupStuff();
  if (!quiet)
    fprintf(stderr, "Window initialized...\n");

  if (install) {
    myColors = XCreateColormap (display, RootWindow (display, screenNum), DefaultVisual (display, screenNum), AllocNone);
    if (myColors != XDefaultColormap(display, screenNum)) {
      XSetWindowColormap (display, win, myColors);
      myXpmAttributes.colormap = myColors;
      myXpmAttributes.valuemask = XpmColormap;
      if (!quiet)
        fprintf (stderr, "Colormap installed...\n");
    } else {
      if (!quiet)
        fprintf (stderr, "Colormap NOT installed...\n");
    }
  } else {
    myXpmAttributes.colormap = DefaultColormap(display, DefaultScreen(display));
    myXpmAttributes.valuemask = XpmColormap;
  }

  XMapWindow(display, win);
  if (!quiet)
    fprintf(stderr, "Window mapped.\n\n");

  mySideBar = new sideBar;
  myLevel = new level(levelFile);
  mySideBar->draw();
  myLevel->draw();
 
  while (mySideBar->go()) {

    while (XCheckWindowEvent(display, win, ExposureMask, &report) == True);
    if (report.type == Expose) {
      mySideBar->draw();
      myLevel->draw();
    }

    while (XCheckWindowEvent(display, win, mask, &report) == True) {
      if (report.type == KeyPress) {
        strcpy(keyPress,XKeysymToString(XKeycodeToKeysym(display,report.xkey.keycode,0)));
        if (strcmp(keyPress, "q")==0) {
          mySideBar->buttonPress(5, 655);
          mySideBar->buttonRelease(5, 655);
        }
        if (strcmp(keyPress, "s")==0) {
          mySideBar->buttonPress(5, 635);
          mySideBar->buttonRelease(5, 635);
        }
        if (strcmp(keyPress, "l")==0) {
          mySideBar->buttonPress(5, 615);
          mySideBar->buttonRelease(5, 615);
        }
        if (strcmp(keyPress, "n")==0) {
          mySideBar->buttonPress(5, 595);
          mySideBar->buttonRelease(5, 595);
        }
        if (strcmp(keyPress, "space")==0) {
          mySideBar->setType((mySideBar->getType()-'b'+1)%18+'b');
        }
      }
      if (report.type == FocusOut) {
        mySideBar->setDispXY(0, 0);
      } else
      if (report.type == ButtonPress) {
        gridX = (report.xbutton.x - 30) / 10;
        gridY = report.xbutton.y / 10;
        if (report.xbutton.x >= 30) {
          mouseDown = 1;
          mouseButton = report.xbutton.button;
          if (mouseButton == Button1) {
            myLevel->buttonPress(gridX, gridY, mySideBar->getType());
          }
          else if (mouseButton == Button2) {
            mySideBar->setType(myLevel->getObject(gridX, gridY));
          }
          else {
            myLevel->buttonPress(gridX, gridY, 'a');
          }
        }
        else {
          mySideBar->buttonPress(report.xbutton.x, report.xbutton.y);
        }
      }
      else if (report.type == ButtonRelease) {
        mouseDown = 0;
        if (report.xbutton.x < 30) {
          mySideBar->buttonRelease(report.xbutton.x, report.xbutton.y);
        }
      }
      else if (report.xmotion.x >= 30) {
        if (!cursor) {
          cursor = 1;
          XFreeCursor(display, myCursor);
          myCursor = XCreateFontCursor(display, XC_tcross);
          XDefineCursor(display, win, myCursor);
          mySideBar->setDispXY(gridX, gridY);
        }
        if (gridX != (report.xmotion.x - 30) / 10 || gridY != report.xmotion.y / 10) {
          gridX = (report.xbutton.x - 30) / 10;
          gridY = report.xbutton.y / 10;
          mySideBar->setDispXY(gridX, gridY);
          if (mouseDown) {
            if (mouseButton == Button1) {
              myLevel->buttonPress(gridX, gridY, mySideBar->getType());
            }
            else {
              myLevel->buttonPress(gridX, gridY, 'a');
            }
          }
        }
      } else if (report.xmotion.x < 30 && cursor) {
        cursor = 0;
        XFreeCursor(display, myCursor);
        myCursor = XCreateFontCursor(display, XC_arrow);
        XDefineCursor(display, win, myCursor);
        mySideBar->setDispXY(0, 0);
      }
      if (mySideBar->shouldLoad()) {
        myLevel->load();
        XClearWindow(display, win);
        myLevel->draw();
        mySideBar->draw();
      }
      if (mySideBar->shouldSave()) {
        myLevel->write();
      }
      if (mySideBar->shouldNew()) {
        myLevel->loadTemplate();
        XClearWindow(display, win);
        myLevel->draw();
        mySideBar->draw();
      }
    }
    XFlush(display);
    uDelay(1000);
  }
  delete myLevel;
  delete mySideBar;
}

void getGC(Window win, GC *gc) {
  unsigned long valuemask = 0;
  XGCValues values;
 
  *gc = XCreateGC(display, win, valuemask, &values);
  XSetForeground(display, *gc, BlackPixel(display, screenNum));
}

void setupX(Window& win, GC& gc, int argc, char **argv, long mask)
{
  char *charWindowName = "XNibbles Level Editor";
  XSizeHints sizeHints;
  XWMHints wmHints;
  XClassHint classHints;
  XTextProperty windowName;
  XSetWindowAttributes winAtt;

  if ((display = XOpenDisplay(displayName)) == NULL)
  {
    fprintf(stderr, "%s: cannot connect to X server %s\n\n", progName, XDisplayName(displayName));
    exit(-1);
  }
  screenNum = DefaultScreen(display);
  win = XCreateSimpleWindow(display, RootWindow(display, screenNum), 0, 0, WIDTH, HEIGHT, 4, WhitePixel(display, screenNum), BlackPixel(display, screenNum));
  sizeHints.flags = PMinSize | PMaxSize;
  sizeHints.min_width = WIDTH;
  sizeHints.min_height = HEIGHT;
  sizeHints.max_width = WIDTH;
  sizeHints.max_height = HEIGHT;
  XStringListToTextProperty(&charWindowName, 1, &windowName);
  wmHints.initial_state = NormalState;
  wmHints.input = True;
  wmHints.flags = StateHint | InputHint;
  classHints.res_name = progName;
  classHints.res_class = charWindowName;
  XSetWMProperties(display, win, &windowName, &windowName, argv, argc, &sizeHints, &wmHints, &classHints);
  XSelectInput(display, win, mask);
  getGC(win, &gc);
  winAtt.cursor = None;
  XChangeWindowAttributes(display, win, CWCursor, &winAtt);
  XClearWindow(display, win);
  XUndefineCursor(display, win);
  XFlush(display);
}

void parseCommandLine(int argc, char **argv, char *displayName, int& install, int& quiet, char *levelFile) {
  int loop;
  char *displayEnv;

  displayEnv = getenv("DISPLAY");
  if (displayEnv != NULL)
    strcpy(displayName, displayEnv);
  else
    strcpy(displayName,":0");
  quiet = 0;
  install = 0;
  for (loop = 1; loop < argc; loop++) {
    if (strcmp(argv[loop], "-q") == 0)
      quiet = 1;
    else if (strcmp(argv[loop], "-i") == 0)
      install = 1;
    else if (strcmp(argv[loop], "-h") == 0 || strcmp(argv[loop], "--help") == 0) {
      printBanner();
      printOptions(argv);
      exit(-1);
    }
    else if (strcmp(argv[loop], "-display") == 0) {
      if (loop + 1 < argc) {
        strcpy(displayName, argv[++loop]);
      } else {
        printBanner();
        fprintf(stderr, "\n-display requires a display!\n");
        printOptions(argv);
        exit(-1);
      }
    }
    else if (loop == argc - 1) {
      if (strlen(argv[loop])<255) {
        strcpy(levelFile, argv[loop]);
      } else {
        fprintf(stderr, "Oops!  That filename's just too damn big!\n");
        exit(-1);
      }
    }
    else {
      printBanner();
      fprintf(stderr, "\nIllegal option: \"%s\"\n", argv[loop]);
      printOptions(argv);
      exit(-1);
    }
  }
  if (strlen(levelFile) == 0) {
    strcpy(levelFile, "newlevel.xnl\0");
  }
}

void printOptions(char **argv) {
  fprintf(stderr, "\nusage: %s [options] filename\n",argv[0]);
  fprintf(stderr, "   or: %s --help\n",argv[0]);
  fprintf(stderr, "   or: %s -h\n\n",argv[0]);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "   -i                    Install private colormap\n");
  fprintf(stderr, "   -q                    Run quietly (less text output)\n");
  fprintf(stderr, "   -display <display>    Set the display name\n\n");
}

void destroyStuff() {
  if (install)
    XFreeColormap (display, myColors);
  XFreeCursor(display, myCursor);
}

void error(int errorCode) {
  switch (errorCode) {
    case NOCOLORS:
      fprintf(stderr, "\nThere are not enough available colors.  Try quitting some color hogging\napplications, or run with -i to install colormap.\n\n");
      myQuit (-1);
    break;
  }
}

void myQuit(int code) {
  destroyStuff();
  XFreeGC(display, gc);
  delete myLevel;
  exit(code);
}

void setupStuff() {
  makePixmap(num0_xpm, &digitXpms[0]);
  makePixmap(num1_xpm, &digitXpms[1]);
  makePixmap(num2_xpm, &digitXpms[2]);
  makePixmap(num3_xpm, &digitXpms[3]);
  makePixmap(num4_xpm, &digitXpms[4]);
  makePixmap(num5_xpm, &digitXpms[5]);
  makePixmap(num6_xpm, &digitXpms[6]);
  makePixmap(num7_xpm, &digitXpms[7]);
  makePixmap(num8_xpm, &digitXpms[8]);
  makePixmap(num9_xpm, &digitXpms[9]);
}
