/*
 * Copyright 1991 The University of Newcastle upon Tyne
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation for any purpose other than its commercial exploitation
 * is hereby granted without fee, provided that the above copyright
 * notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that
 * the name of The University of Newcastle upon Tyne not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission. The University of
 * Newcastle upon Tyne makes no representations about the suitability of
 * this software for any purpose. It is provided "as is" without express
 * or implied warranty.
 * 
 * THE UNIVERSITY OF NEWCASTLE UPON TYNE DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF
 * NEWCASTLE UPON TYNE BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 * 
 * Author:  Gerry Tomlinson (gerry.tomlinson@newcastle.ac.uk)
 *          Computing Laboratory, University of Newcastle upon Tyne, UK
 */

/*
 * readfile.c - read a file for xp
 */

#include <stdio.h>
#include <X11/Xos.h>			/* get types.h */
#include <sys/stat.h>

extern char *malloc();
extern char *realloc();

/*
 * readinlump - read data from a file into a single buffer in 1 go
 *
 * pass back size of file in bytes if success or
 * -1	can't stat file
 * -2	can't malloc space
 * -3   can't read file
 */

static
char *readinlump (fp,size)
    FILE *fp;
    int *size;
{
    struct stat statbuf;
    char *cp;


    if (fstat (fileno(fp), &statbuf) != 0 || statbuf.st_size < 0) {
	*size = -1;
    	return NULL;
    }

    cp = malloc ((unsigned)statbuf.st_size + 1);
    if (!cp)  {
	*size = -2;
	return NULL;
    }

    if (fread (cp, 1, (int) statbuf.st_size, fp) != statbuf.st_size) {
	(void) free (cp);
 	*size = -3;
	return NULL;
    }

    cp[statbuf.st_size] = '\0';		/* since we allocated one extra */
    *size = statbuf.st_size;
    return cp;
}

/*
 * readinchunks 
 * read data into single buffer  realloc'ing as we go in 
 * BUFSIZ chunks
 */


char *readinchunks (fp,size)
    FILE *fp;
    int *size;
{
    int n;
    int sum = 0;

    char *buf, *bptr;

    buf = malloc(BUFSIZ + 1);
    if (!buf) return  ((char *)-2);
    
    bptr = buf;

    while ((n = fread (bptr, 1, BUFSIZ, fp)) > 0) {
        sum = sum + n;
	bptr +=n;
        if (bptr + BUFSIZ  >= buf )  {
	    buf = (char *)realloc(buf,(unsigned)(sum + BUFSIZ + 1));
            if (!buf)  return (char *)-2;
	    bptr = &buf[sum];
	}
    }
    buf[sum] = '\0';


    *size = sum;
    return buf;
}


/*
 * read_file - read data from indicated file and return pointer to malloced
 * buffer or NULL on error
 *
 *
 * Pass back number of bytes in size or
 *	   -1  if no such file.
 *	   -2  malloc error
 *	   -3  error reading file
 *
 * treat NULL filename as stdin - figure whether can read in
 * one lump by testing whether can seek or is a tty
 * this handles input from files,pipes & ttys
 *
 * if stdin then do NOT close - but clear error instead - this way
 * reopens will work
 */

char *readfile (filename, size)
    char *filename;
    int  *size;
{
    FILE *fp;
    char *data;

    if (strcmp(filename,"-") == 0)
	fp = stdin;
    else {
	fp = fopen (filename, "r");
	if (!fp) {
	    *size = -1;
     	     return  NULL;
	}
    }

    if (fp == stdin || lseek(fileno(fp),(off_t)0,0) == -1 || isatty(fileno(fp)))
        data = readinchunks(fp,size);
    else data = readinlump(fp,size);

    if (fp != stdin)
        fclose(fp);
    else
        clearerr(fp);

    return(data);
    
}

