/*

    xpuyopuyo - pgame.c       Copyright(c) 1999,2000 Justin David Smith
    justins(at)chaos2.org     http://chaos2.org/
    
    Main game control code.
    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


/* System includes */
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>


/* Local includes */
#include <config.h>
#include <xpuyopuyo.h>
#include <psnprintf.h>
#include <pwindow.h>
#include <pconfigm.h>
#include <ptime.h>
#include <pfield.h>
#include <pmanip.h>
#include <ppiece.h>
#include <pgame.h>
#include <ptournament.h>
#include <psound.h>
#include <pinfo.h>
#include <pnet.h>
#include <pai.h>



/***     Timing and State Functions    ***/



static inline void p_game_time(struct timeval *tv) {
/* p_game_time */
/* Used to return gametime in ms */
/* Now this function returns game time in usecs, using tv */

   gettimeofday(tv, NULL);
   return;

}


static inline void p_game_set_drop_time(pplayer *p, unsigned long delay) {
/* p_game_set_drop_time

   delay is an offset in milliseconds.  Here's the way this works.
   delay is added to the player's "state" time, and when the game
   time exceeds the player's state time, then the next state is
   processed.  If by some chance, the player's new state time after
   adding delay is ALREADY less than the current game time, then we'll
   set the new state time to the game time plus delay.  */

   struct timeval tv;
   struct timeval tvadd;
   
   TV_FROM_MILLISEC(tvadd, delay);
   TV_ADD(p->statetime, tvadd);
   p_game_time(&tv);
   if(TV_LESS(p->statetime, tv)) {
      TV_COPY(p->statetime, tv);
      TV_ADD(p->statetime, tvadd);
   }
   return;

}


static inline void p_game_set_state(pplayer *p, int state, unsigned long delay) {
/* p_game_set_state

   Like above, but this also sets a new state.  This is the state
   transition function.  */

   p->state = state;
   p_game_set_drop_time(p, delay);
   return;

}


void p_game_set_new_state(pplayer *p, int state) {
/* p_game_set_new_state

   Like above, but player's state game time is set to the current game time,
   this causing this transition to immediately be processed.  */

   p->state = state;
   p_game_time(&p->statetime);
   return;

}


static inline void p_game_resync_timer(pplayer *p) {
/* p_game_resync_timer

   Occasionally, a player's time may need resynching to the current game
   time (particularly after a pause).  This function takes care of that.  */
/* Note, I don't think this function is needed anymore now that we are
   only using struct timeval, and those values can only ever increase.. */

   p_game_time(&p->statetime);

}


void p_game_resync_timers(pconfig *c) {
/* p_game_resync_timers

   Same as above, but for both players.  Called after an unpause.  */

   p_game_resync_timer(P_PLAYER_0(c));
   p_game_resync_timer(P_PLAYER_1(c));

}



/***     Create a New Game    ***/



#if USE_NETWORK /* Only define the following function if network allowed */


static void p_game_new_network(pconfig *c) {
/* p_game_new_network

   Initialise the network connection for a new game.  */
   
   int neterror;           /* True if a network error occurred. */

   /* If network, send/receive configuration as confirmation to start game */
   neterror = 0;
   if(P_NETWORK(c)) {   /* Are we playing a network game? */
      if(P_NETWORK_SERVER(c)) {  /* Server or client? */
         /* server-- Send configuration */
         if(pnet_send_config(c) < 0) neterror = 1;
      } else /* Must be the client ... */ {
         /* client-- Wait to receive a configuration */
         if(pnet_recv_config(c) < 0) neterror = 1;
      } /* Server or client? */
   } /* Are we in network mode? */

   /* Did a network error occur? */
   if(neterror) {
      /* Drop out of network mode and error message */
      p_window_set_message(c->window, pnet_get_error());
      pnet_close(c, &c->socket);
   } /* Did an error occur? */
   
}


#endif /* Network? */


#if P_ALLOW_TOURNAMENT  /* Only define the following function if tournaments are allowed */


static void p_game_new_tournament(pconfig *c) {
/* p_game_new_tournament

   Initialise a tournament game.   */ 
 
   pplayer *p;
  
   if(P_TOURNAMENT(c)) {
      if(c->tcurround[1] == -1 && c->tcurround[0] != -1) {
         p_game_tournament_advance(c, c->tcurround[0]);
         c->tcurround[0] = -1;
      }
      if(c->tcurround[0] == -1) {
         p = P_PLAYER_0(c);
         p_game_tournament_next_round(c);
         p_game_tournament_randomize_current(c);
         p_game_tournament_clear_next(c);
         if(c->tournament == 1) {
            if(!c->quiet) printf("We have a winner!   %d, %s (%s)\n\n", c->tcurround[0], p_ai_name_of(c->airules, c->tcurround[0]), p_ai_desc_of(c->airules, c->tcurround[0]));
            if(P_TOURNAMENT_HYPER(c)) {
               p_game_tournament_init(c);
               p_game_tournament_randomize_current(c);
               p_game_tournament_clear_next(c);
            } else {
               c->restart = 0;
               return;
            }
         }
      }
      P_PLAYER_0(c)->airule = c->tcurround[0];
      P_PLAYER_1(c)->airule = c->tcurround[1];
      if(!c->quiet) {
         printf("This round is between   \"%s\"   and   \"%s\"\n", p_ai_name_of(c->airules, P_PLAYER_0(c)->airule), p_ai_name_of(c->airules, P_PLAYER_1(c)->airule));
         printf("   \"%s\":  \"%s\"\n", p_ai_name_of(c->airules, P_PLAYER_0(c)->airule), p_ai_rule_text_of(c->airules, P_PLAYER_0(c)->airule));
         printf("   \"%s\":  \"%s\"\n", p_ai_name_of(c->airules, P_PLAYER_1(c)->airule), p_ai_rule_text_of(c->airules, P_PLAYER_1(c)->airule));
      }
      p_game_tournament_remove_current(c);
   } /* A tournament game? */

}


#endif /* Allow tournament mode? */


void p_game_new(pconfig *c) {
/* p_game_new

   Initialise a new game.  */

   int i;
   int j;
   pplayer *p;

   /* If in tournament, select two AI's */
   /* WARNING: this must be done before other player data is setup! */
   #if P_ALLOW_TOURNAMENT
      p_game_new_tournament(c);
   #endif 
   
   /* Set AI/network modes (must be done before names are pulled up) */
   if(c->ai == 1 && c->networked == 1) {  
      P_PLAYER_0(c)->aiactive = 1;
      P_PLAYER_1(c)->aiactive = 0;
      P_PLAYER_0(c)->network = 0;
      P_PLAYER_1(c)->network = 1;
   } else {
      P_PLAYER_0(c)->aiactive = (c->ai >= 2);
      P_PLAYER_1(c)->aiactive = (c->ai >= 1);
      P_PLAYER_0(c)->network = 0;
      P_PLAYER_1(c)->network = (c->networked >= 1);
   }

   /* Load AI rules, if needed, to each player */
   P_PLAYER_0(c)->ai = p_ai_get_rule(c->airules, P_PLAYER_0(c)->airule);
   P_PLAYER_1(c)->ai = p_ai_get_rule(c->airules, P_PLAYER_1(c)->airule);
   
   /* Setup player names (must be done before network connection) */
   for(i = 0; i < 2; i++) {
      p = P_PLAYER(c, i);
      if(P_PLAYER_HUMAN(c, p)) {
         char *user = getenv("USER");
         #if USE_NETWORK
            char buf[P_PLAYER_NAME_SIZE];
            char host[0x100];
            char *hostp;
            if(!pnet_get_hostname(host, 0x100)) {
               p_player_set_name(p, user);
            } else {
               p_snprintf(buf, P_PLAYER_NAME_SIZE, "%s@%s", user, host);
               hostp = strchr(buf, '@');
               if(hostp != NULL) {
                  hostp = strchr(buf, '.');
                  if(hostp != NULL) *hostp = '\0';
               } /* Remove dot in hostname */
               p_player_set_name(p, buf);
            } /* Hostname available? */
         #else /* No network support */
            p_player_set_name(p, user);
         #endif /* Network support available? */
      } else if(P_PLAYER_AI(c, p)) {
         p_player_set_name(p, p_ai_get_name(p->ai));
      } /* Otherwise, must be a network connection; we already have the name */
   }
   
   /* If network, send/receive configuration as confirmation to start game */
   #if USE_NETWORK
      p_game_new_network(c);
   #endif /* Network? */
   
   /* Start sound playback? */
   #if USE_SOUND
      p_sound_start(c->sound, P_MUSIC_GAME);
   #endif /* Sound? */
   
   /* Setup default whose_turn */   
   c->whose_turn = 0;

   /* Clear any paused state */
   c->paused = 0;
   
   /* Setup accel */
   c->realspeed= c->dropspeed;
   
   /* Initialise counters and whatnot for each player */
   for(i = 0; i <= 1; i++) {
      p = P_PLAYER(c, i);
      p_player_resize_field(p, c->fieldwidth, c->fieldheight, c->numcolors);
      p->rocks = 0;
      p->accum = 0;
      p->score = 0;
      p->scorerocks = 0;
      p->num_at_once = 0;
      p->f_pieces = 0;
      p->f_rksent = 0;
      p->f_rkrecv = 0;
      p_player_calc_next(p, c->numcolors);
      p_field_clear(p->field);
      p_field_clear(p->field->piece);
      p_field_redraw_all(p->field);
      for(j = 0; j < P_PLAYER_NUM_CLUMPS; j++) {
         p->num_clumps[j] = 0;
      }
   }
   
   /* Clear each player's state time by setting statetime = gametime */
   p_game_time(&P_PLAYER_0(c)->statetime);
   p_game_time(&P_PLAYER_1(c)->statetime);
   
   /* Single-player game? */
   if(P_1_PLAYER(c)) {
      p_game_set_state(P_PLAYER_0(c), P_STATE_IDLE, P_IDLE_TIME * 2);
      p_game_set_state(P_PLAYER_1(c), P_STATE_DISABLED, P_LONG_TIME);
   } else {
      p_game_set_state(P_PLAYER_0(c), P_STATE_IDLE, P_IDLE_TIME * 2);
      p_game_set_state(P_PLAYER_1(c), P_STATE_IDLE, P_IDLE_TIME * 2);
   }
   
   /* Display player names */
   p_window_set_player_info(c->window, 0, P_PLAYER_0(c));
   if(P_1_PLAYER(c)) p_window_set_player_info(c->window, 1, NULL);
   else              p_window_set_player_info(c->window, 1, P_PLAYER_1(c));
   
   /* Check if a player needs to be switched to the network state */
   #if USE_NETWORK
      if(P_PLAYER_NETWORK(c, P_PLAYER_0(c))) p_game_set_state(P_PLAYER_0(c), P_STATE_NETWORK, P_NETWORK_TIME);
      if(P_PLAYER_NETWORK(c, P_PLAYER_1(c))) p_game_set_state(P_PLAYER_1(c), P_STATE_NETWORK, P_NETWORK_TIME);
   #endif /* Network mode allowed? */

   /* Clear timer update field */
   TV_ZERO(c->timerupdate);

   /* Clear status field */
   p_window_set_status(c->window, NULL);
   
   /* Set game start time */
   p_game_time(&c->t1);
   p_game_time(&c->t2);
   
}



/***     Game Advancement     ***/



static void p_game_advance_game_time(pconfig *c) {
/* p_game_advance_game_time */

   if(!P_GAME_INACTIVE(c) && !c->paused) {
      p_game_time(&c->t2);
      c->realspeed = c->dropspeed * (1 - c->accel * (c->t2.tv_sec - c->t1.tv_sec) / TV_SEC_IN_MIN);
      if(c->realspeed < P_FALL_TIME) c->realspeed = P_FALL_TIME;
   }
   
}


static void p_game_send_board(pconfig *c, pplayer *p) {

   #if USE_NETWORK
      if(P_NETWORK(c)) {
         if(pnet_send_board(c, p) < 0) p_window_set_message(c->window, pnet_get_error());
      }
   #endif /* Network? */

}


static void p_game_send_status(pconfig *c, pplayer *p) {

   #if USE_NETWORK
      if(P_NETWORK(c)) {
         if(pnet_send_status(c, p) < 0) p_window_set_message(c->window, pnet_get_error());
      }
   #endif /* Network? */

}


static void p_game_send_subboard(pconfig *c, pplayer *p, int x1, int y1, int x2, int y2) {

   #if USE_NETWORK
      if(P_NETWORK(c)) {
         if(pnet_send_subboard(c, p, x1, y1, x2, y2) < 0) p_window_set_message(c->window, pnet_get_error());
      }
   #endif /* Network? */

}


static void p_game_advance_drop(pconfig *c, pplayer *p, int speed) {
/* p_game_advance_drop */

   pfield *f;

   /* Redraw the rectangle surrounding the playing piece, including one row below it. */
   f = p->field;
   p_field_redraw_rect(f, f->piece->x, f->piece->y, f->piece->x + 1, f->piece->y + 2);

   /* Check for a collision with the ground or other pieces */
   if(p_piece_fall_collide(f)) {
      /* Piece is down! */
      #if USE_SOUND
         /* Piece landed */
         p_sound_play_effect(c->sound, P_SOUND_PLOP, p->number);
      #endif
      p_manip_lock(f);
      p_field_clear(f->piece);
      p_game_set_state(p, P_STATE_MATCH_FALL, P_FALL_TIME);
   } else {
      /* Nope, we're still falling ... */
      p_piece_fall(f);
      p_game_set_drop_time(p, speed);
   }

   /* Send board to network player */
   p_game_send_subboard(c, p, f->piece->x, f->piece->y - 1, f->piece->x + 1, f->piece->y + 1);

}



/***     Game States    ***/



static void p_game_advance_state_network(pconfig *c, pplayer *p) {
/* p_game_advance_state_network */

   #if USE_NETWORK

      pfield *f;
      int res;

      /* Receive network control packets for this player */
      res = pnet_recv_game(c, p);

      switch(res) {
      case P_NET_RES_LOST:
         /* This player just lost the game */
         #if USE_SOUND
            /* Player died */
            p_sound_play_effect(c->sound, P_SOUND_DEATH, p->number);
         #endif
         f = p->field;
         p_field_redraw_all(f);
         p_game_set_state(p, P_STATE_LOSER_, P_DRAIN_TIME);
         p_field_clear(f->piece);
         p_game_set_state(P_PLAYER(c, P_OTHER_PLAYER(p->number)), P_STATE_WINNER, P_LONG_TIME);
         p_statistics(c);
         break;
      
      case P_NET_RES_ENDGAME:
         /* End gambit */
         break;

      case -1:
         /* A network error occurred */
         p_window_set_message(c->window, pnet_get_error());
         break;
         
      default:
         /* Continue game */
         p_game_set_state(p, P_STATE_NETWORK, P_NETWORK_TIME);
      } /* Did this player lose? */
      
   #else /* Cannot do network mode */

      p_game_set_state(p, P_STATE_IDLE, P_IDLE_TIME);

   #endif /* Network mode allowed? */         

}


static void p_game_advance_state_aidrop(pconfig *c, pplayer *p) {
/* p_game_advance_state_aidrop */

   p_game_advance_drop(c, p, P_FALL_TIME);

}


static void p_game_advance_state_falling(pconfig *c, pplayer *p) {
/* p_game_advance_state_falling */

   pplayer *pother;

   /* If other player is an AI, and they are allowed to drop, then let them drop! */
   pother = P_PLAYER(c, P_OTHER_PLAYER(p->number));
   if(c->aicandrop && pother->state == P_STATE_AIFALL) {
      p_game_set_new_state(pother, P_STATE_AIDROP);
   }

   p_game_advance_drop(c, p, P_FALL_TIME);

}


static void p_game_advance_state_playing(pconfig *c, pplayer *p) {
/* p_game_advance_state_playing */

   p_game_advance_drop(c, p, (int)(c->realspeed));

}


static void p_game_advance_state_idle(pconfig *c, pplayer *p) {
/* p_game_advance_state_idle */

   pplayer *pother;
   pfield *f;
   
   /* Make sure timer is accurate */
   p_game_resync_timer(p);

   /* Redraw the entire field, top two rows */
   f = p->field;
   p_field_redraw_rect(f, 0, 0, c->fieldwidth, 2);
   
   /* Obtain a new next piece and unlock in current piece */
   p_piece_renew(f, &p->next);

   /* This is where AI control is asserted (also for tutorial) */
   if(P_PLAYER_AI(c, p) || c->tutorial) {
      /* AI control is asserted here */
      int cur[2];
      cur[0] = p->next.block[0];
      cur[1] = p->next.block[1];
      p_player_calc_next(p, c->numcolors);
      p_ai_decide(f, p->ai, cur, p->next.block, c->fieldmatch, c->numcolors);
      /* For tutorial, the score will be zero'd */
      if(!P_PLAYER_AI(c, p)) p->score = 0;      
   } else {
      /* Just calculate the next piece for a normal player */
      p_player_calc_next(p, c->numcolors);
   }
   
   /* Increment pieces count (statistical information) */
   p->f_pieces++;

   /* If we've _already_ collided, then the player has lost */
   if(p_piece_collided(f)) {

      /* Game over. Current player lost. */
      p_field_redraw_all(f);
      p_game_set_state(p, P_STATE_LOSER_, P_TOURNAMENT_HYPER(c) ? 0 : P_DRAIN_TIME);

      #if USE_SOUND
         /* Player died */
         p_sound_play_effect(c->sound, P_SOUND_DEATH, p->number);
      #endif

      #if USE_NETWORK
         /* Tell opponent about our gruesome defeat */
         if(P_NETWORK(c)) {
            if(pnet_send_defeat(c, p) < 0) p_window_set_message(c->window, pnet_get_error());
         }
      #endif /* Network game? */

      /* Clear the piecefield */
      p_field_clear(f->piece);
      
      /* If 2-player, other player's state should be set to winner */
      if(P_2_PLAYER(c)) {
         /* Set other player as the winner */
         pother = P_PLAYER(c, P_OTHER_PLAYER(p->number));
         p_game_set_state(pother, P_STATE_WINNER, P_LONG_TIME);
         #if P_ALLOW_TOURNAMENT
            /* In tournament, advance the winner */
            if(P_TOURNAMENT(c)) p_game_tournament_advance(c, p_ai_get_index(pother->ai));
         #endif
         
         /* If AI, then update the win/loss counts */
         if(P_PLAYER_AI(c, pother)) p_ai_inc_win_count(c->airules,  p_ai_get_index(pother->ai));
         if(P_PLAYER_AI(c, p))      p_ai_inc_loss_count(c->airules, p_ai_get_index(p->ai));
         pother->wins++;
         
         /* If AI-AI, then mutate the loser */
         #if USE_AIBREED
            if(P_CONFIG_IS_AI_AI(c)) p_ai_mutate(c->airules, p->ai, pother->ai, c->quiet);
         #endif
         
         /* Print out the statistical information */
         p_statistics(c);
      } /* If a 2-player game ... */
      
   } else {

      /* Game is still in progress; set to normal falling state */
      p_game_set_state(p, P_PLAYER_AI(c, p) ? P_STATE_AIFALL : P_STATE_PLAYING, (int)(c->realspeed));

   } /* Is the game over? */

   #if USE_NETWORK
      if(P_NETWORK(c)) {
         if(pnet_send_next(c, p) < 0) p_window_set_message(c->window, pnet_get_error());
      } /* Sending our next piece along */
   #endif /* network? */
   
   p_game_send_status(c, p);

}


static void p_game_advance_state_objects_fall(pconfig *c, pplayer *p) {

   pfield *f;

   f = p->field;
   if(p_manip_collapse_once(f)) {
      /* We can still collapse some more */
      p_game_set_drop_time(p, P_FALL_TIME);
   } else {
      /* We reached the bottom; update joins between blobs */
      p_manip_update_joins(f);
      p_game_set_state(p, 
         p->state == P_STATE_ROCKS_FALL ? P_STATE_ROCKS : P_STATE_MATCHING, 
         p->state == P_STATE_ROCKS_FALL ? P_FALL_TIME : P_MATCH_TIME);
   }

   /* Send board to network player */
   p_game_send_board(c, p);

}


static void p_game_advance_state_rocks(pconfig *c, pplayer *p) {

   pfield *f;

   if(p->rocks > 0) {

      /* Going to receive some rocks */
      /* We receive at most <fieldwidth> rocks in one pass */
      int k[c->fieldwidth];         /* 2  - Place an indestructible here */
                                    /* 1  - space available in this column */
                                    /* 0  - slot has been filled up */
                                    /* -1 - no space available in this column */
      int *dest;                    /* Destination row */
      int avail;                    /* Number of available slots */
      int ind;                      /* Number of indestructibles to dump */
      int j;                        /* Iterator variable */

      /* Setup a destination list consisting of the top row */
      /* (to see if any row is already filled up) */      
      f = p->field;
      dest = P_FIELD_XY_P(f, 0, 1);
      avail = c->fieldwidth;
      j = 0;
      while(j < c->fieldwidth) {
         /* Check if this column has space for a rock */
         if(P_IS_CLEAR(*dest)) k[j] = 1;     /* Yes; there is space */
         else k[j] = -1, avail--;            /* No space available */
         dest++;
         j++;
      }
      
      if(avail == 0) {
         /* Game over. Current player will lose when next piece is placed */
         p_game_set_state(p, P_STATE_IDLE, (int)(c->realspeed));
      } else {
         /* Count indestructibles, and decrement the rocks count */
         ind = 0;                /* How many indestr. to dump in this pass? */
         while(avail) {          /* Count up the indestructibles ... */
            if(c->indestructibles && p->rocks > 0 && p->rocks % P_INDESTRUCTIBLE_THRESHOLD == 0) ind++;
            p->rocks--;
            avail--;
         }
         
         /* Did we have more available slots than rocks to dump? */
         while(p->rocks < 0) {
            /* Mark some slots as already "filled" */
            j = rand() % c->fieldwidth;
            if(k[j] > 0) {
               /* Good, we can recover some of those overcounted rocks */
               k[j] = 0;
               p->rocks++;
            }
         }
         
         /* Figure out where we should put the indestructibles */
         while(ind > 0) {
            j = rand() % c->fieldwidth;
            if(k[j] < 2) {
               /* Looks like a good place to put an indestructible */
               k[j] = 2;
               ind--;
            }
         }

         /* Now, place those rocks! */
         dest = P_FIELD_XY_P(f, 0, 1);
         j = 0;
         while(j < c->fieldwidth) {
            /* if k == 2, place an indestructible rock */
            if(k[j] == 2) *dest = P_INDESTRUCTIBLE;

            /* if k == 1, place a normal rock */
            else if(k[j] == 1) *dest = P_ROCK;

            /* Update the statistical counts */
            if(k[j] > 0) {
               p->f_rkrecv++;
               P_PLAYER(c, P_OTHER_PLAYER(p->number))->f_rksent++;
            }

            /* Next column ... */
            dest++;
            j++;
         }

         /* Now, we need to drop all those rocks */
         p_game_set_state(p, P_STATE_ROCKS_FALL, P_TOURNAMENT_HYPER(c) ? 0 : P_FALL_TIME);

         #if USE_SOUND
            /* Got some rocks */
            p_sound_play_effect(c->sound, P_SOUND_AGONY_2, p->number);
         #endif

      } /* Did we kill the player with rocks? */

   } else {
   
      /* NO ROCKS to receive currently; update score, go to idle */
      if(!c->tutorial) switch(p->num_at_once) {
         default:
            p->score += 15000 * (p->num_at_once - 4);
         case 4:
            p->score += 4000;
         case 3:        
            p->score += 750;             
         case 2:
            p->score += 100;
         case 1:
            p->score += 10;
         case 0:
            break;
      } /* Update the score */

      /* Statistics on the number clumps */
      if(p->num_at_once > P_PLAYER_NUM_CLUMPS) p->num_at_once = P_PLAYER_NUM_CLUMPS;
      if(p->num_at_once > 0) {
         p->num_clumps[p->num_at_once - 1]++;
         p->num_at_once = 0;
      }

      /* Set to idle state */
      p_game_set_state(p, P_STATE_IDLE, (int)(c->realspeed));

   } /* Any more rocks to be received? */

}


static void p_game_advance_state_matching(pconfig *c, pplayer *p) {

   int j;

   /* Explode all the matches on playfield (replace with *) */
   /* This method returns the number of matches that were exploded */
   j = p_manip_explode_matches(p->field, c->fieldmatch);
   if(j) {

      #if USE_SOUND
         /* Got a match */
         p_sound_play_effect(c->sound, P_SOUND_BWOAP, p->number);
      #endif

      /* Some matches were exploded; continue */
      p->num_at_once += j;
      p_game_set_state(p, P_STATE_MATCHING_2, P_TOURNAMENT_HYPER(c) ? 0 : P_EXPL_TIME);

   } else {
   
      /* No more matches. Transfer the resultant rockpile */
      p->accum -= (c->fieldmatch - 1);
      if(p->accum < 0) p->accum = 0;
      p->scorerocks += p->accum;       /* A statistical field */

      if(c->nullifyrocks) {
         /* We can nullify the rocks the opponent has racked up against us */
         if(p->accum > p->rocks) p->accum -= p->rocks, p->rocks = 0;
         else                    p->rocks -= p->accum, p->accum = 0;
      } /* Can we nullify? */

      P_PLAYER(c, P_OTHER_PLAYER(p->number))->rocks += p->accum;

      #if USE_NETWORK
         /* Send rocks to network player */
         if(P_NETWORK(c)) {
            if(pnet_send_rocks(c, p, p->accum) < 0) p_window_set_message(c->window, pnet_get_error());
         }
      #endif /* Network? */

      if(!c->tutorial) {
         /* Update the score */
         if(p->accum > 0) p->score += (p->accum - 1) * 10;
      }

      /* Proceed to the state where we wait to receive a pile of rocks */
      p->accum = 0;
      p_game_set_state(p, P_STATE_ROCKS, P_FALL_TIME);

   }

   /* Send board to network player */
   p_game_send_board(c, p);

}


static void p_game_advance_state_matching_2(pconfig *c, pplayer *p) {

   /* Accumulate some more rocks against the opponent */
   /* Clear off all the explosions from previous matching state */
   p->accum += p_manip_clear_explosions(p->field);

   /* Proceed to collapse state resulting from the erased matches */
   p_game_set_state(p, P_STATE_MATCH_FALL, P_FALL_TIME);

   /* Send board to network player */
   p_game_send_board(c, p);

}


static void p_game_advance_state_loser_drain(pconfig *c, pplayer *p) {

   pfield *f;
   int j;
   int k;

   /* Check for which columns are already cleared */
   f = p->field;
   k = 0;
   j = c->fieldwidth - 1;;
   while(j && !k) {
      if(!P_IS_CLEAR(p_field_get(f, j, c->fieldheight - 1))) k++;
      j--;
   }
   
   /* Is there anything left to drain? */
   if(k > 0) {
   
      /* Drain a random column */
      k = 0;
      while(k == 0) {
         j = rand() % c->fieldwidth;
         if(!P_IS_CLEAR(p_field_get(f, j, c->fieldheight - 1))) {
            /* This column may be drained */
            p_field_set(f, j, c->fieldheight - 1, P_CLEAR);
            p_field_redraw_point(f, j, c->fieldheight - 1);
            k++;
         }
      }

      /* Cause a collapse */
      p_manip_collapse(f);
      p_game_set_drop_time(p, P_DRAIN_TIME);

   } else {
   
      /* Nothing left; switch to final state */
      p_game_set_state(p, P_STATE_LOSER, P_LONG_TIME);

      #if USE_SOUND
         p_sound_start(c->sound, P_MUSIC_PRELUDE);
      #endif /* Sound? */
   
   }

}



/***     The Advance Function    ***/



int p_game_advance(pconfig *c, pplayer *p) {
/* p_game_advance 

   Advances the game for the given player */

   int i;
   int redraw = P_REDRAW_NONE;
   struct timeval dummy;

   i = p->number;
   
   /* See if player's state time has elapsed? */
   if(P_STATE_READY(p, dummy)) {
      /* Current game time setup */
      p_game_advance_game_time(c);

      /* Player (and timer) will need redrawing */
      redraw = redraw | P_REDRAW_PLAYER(i);

      switch(p->state) {

         /*    Network control state
         
            Receive network packets to control this player.
            
         */
         case P_STATE_NETWORK:
            p_game_advance_state_network(c, p);
            redraw = redraw | P_REDRAW_NEXT;
            break;
         
         
         /*    Playing or falling
               
            These are playable states, with a block falling until
            it hits something.
            
         */
         case P_STATE_FALLING:
            p_game_advance_state_falling(c, p);
            break;

         case P_STATE_AIDROP:
            p_game_advance_state_aidrop(c, p);
            break;

         case P_STATE_PLAYING:
         case P_STATE_AIFALL:
            p_game_advance_state_playing(c, p);
            break;


         /*    Idle state

            Select and display next tile, determine if a game over
            condition exists; if so, then set opposing player to
            winner and current player to loser.  
            
            State transits to P_STATE_PLAYING, except in game over
            conditions.  
            
            AI control is asserted here.
            
         */
         case P_STATE_IDLE:
            p_game_advance_state_idle(c, p);
            redraw = redraw | P_REDRAW_NEXT;
            break;


         /*    Falling objects
         
            This state is selected when objects are falling.
            This state transits to a matching state or rocks
            generation state, depending. 
            
         */
         case P_STATE_ROCKS_FALL:
         case P_STATE_MATCH_FALL:
            p_game_advance_state_objects_fall(c, p);
            break;


         /*    Rocks generation
         
            This state is activated when a piece has fallen
            completely, and the player's rocks need to be
            thrown into the mix.  This should occur after
            all matching has occurred. This will likely
            give way to at least one collapse state.
            
         */
         case P_STATE_ROCKS:
            p_game_advance_state_rocks(c, p);
            break;


         /*    Matching state
         
            Blocks are being matched.  Always yields to the second
            matching state if any matches were found. Otherwise,
            yields to the rock state.
            
         */
         case P_STATE_MATCHING:
            p_game_advance_state_matching(c, p);
            break;


         /*    Second matching state
            
            In this state, explosion "stars" are replaced by clear
            tiles, and a process of collapsing is begun.  If we are
            in this state, then there must have been some matches.
            
         */
         case P_STATE_MATCHING_2:
            p_game_advance_state_matching_2(c, p);
            break;


         case P_STATE_LOSER_:
            p_game_advance_state_loser_drain(c, p);
            break;


         default:
            p_field_redraw_all(p->field);
            p_game_set_drop_time(p, P_LONG_TIME);
            break;

      }

   }

   /* Return the current redraw flag */
   return(redraw);

}


int p_game_advance_all(pconfig *c) {
/* p_game_advance_all

   Advances both players */

   int redraw = P_REDRAW_NONE;      /* Clear the redraw flag */
   struct timeval tsub;             /* Temporary time value */

   /* Get current gametime and determine if timer needs a redraw */
   p_game_time(&tsub);
   TV_SUB(tsub, c->timerupdate);
   if(tsub.tv_sec >= 1) {
      /* Please redraw the timer */
      redraw = redraw | P_REDRAW_TIMER;
      p_game_time(&c->timerupdate);
   }

   /* Whose turn is it to be advanced first? */
   if(c->whose_turn == 0) {
      c->whose_turn = 1;
      redraw = p_game_advance(c, P_PLAYER_0(c)) | redraw;
      redraw = p_game_advance(c, P_PLAYER_1(c)) | redraw;
   } else {
      c->whose_turn = 0;
      redraw = p_game_advance(c, P_PLAYER_1(c)) | redraw;
      redraw = p_game_advance(c, P_PLAYER_0(c)) | redraw;
   }

   /* Return the redraw flag */   
   return(redraw);

}


void p_game_while_paused(pconfig *c) {
/* p_game_while_paused */

   #if USE_NETWORK
      if(P_NETWORK(c)) {
         if(pnet_recv_for_unpause(c) < 0) p_window_set_message(c->window, pnet_get_error());
      }
   #endif /* network? */

}



/***     End Game and Activate Pause      ***/



void p_game_end_(pconfig *c) {
/* p_game_end_ */

   p_game_set_new_state(P_PLAYER_0(c), P_STATE_DISABLED);
   p_game_set_new_state(P_PLAYER_1(c), P_STATE_DISABLED);
   
}


void p_game_end(pconfig *c) {
/* p_game_end */

   #if USE_NETWORK
      if(P_NETWORK(c)) {
         if(pnet_send_end(c) < 0) p_window_set_message(c->window, pnet_get_error());
      }
   #endif /* network? */

   p_game_end_(c);
   P_PLAYER_0(c)->wins = 0;
   P_PLAYER_1(c)->wins = 0;

   /* Stop sound playback? */
   #if USE_SOUND
      if(c->sound != NULL && c->sound->selection != P_MUSIC_PRELUDE) p_sound_start(c->sound, P_MUSIC_PRELUDE);
   #endif /* Sound? */
   
}


void p_game_pause_(pconfig *c, int paused) {
/* p_game_pause_ */

   struct timeval tv;

   if(P_GAME_IDLE(c)) return;
   if((paused && c->paused) || (!paused && !c->paused)) return;
   
   if(!paused) {
      /* Game was paused; resume */
      c->paused = 0;
      c->needredraw = 1;
      p_window_paint(c, c->window, P_REDRAW_ALL);
      p_window_set_status(c->window, NULL);
      /* Calculate amount of time game was paused; add to t1 */
      p_game_time(&tv);
      TV_SUB(tv, c->t2);
      TV_ADD(c->t1, tv);
      p_game_resync_timers(c);
   } else {
      /* Game is being paused */
      c->paused = 1;
      c->needredraw = 1;
      p_window_paint(c, c->window, P_REDRAW_ALL);
      p_window_set_status(c->window, "  P A U S E D  ");
      p_game_time(&c->t2);
   }

}


void p_game_pause(pconfig *c, int paused) {
/* p_game_pause */

   if(P_GAME_IDLE(c)) return;
   if((paused && c->paused) || (!paused && !c->paused)) return;
   
   p_game_pause_(c, paused);

   #if USE_NETWORK
   if(P_NETWORK(c)) {
      /* Need to tell opponent to (un)pause */
      if(pnet_send_pause(c, paused) < 0) p_window_set_message(c->window, pnet_get_error());
   }
   #endif /* network? */

   /* Pause sound playback? */
   #if USE_SOUND
      if(paused) p_sound_pause(c->sound);
      else p_sound_unpause(c->sound);
   #endif /* Sound? */
   
}


