/*************************************************************************
*									 *
*	 YAP Prolog 							 *
*									 *
*	Yap Prolog was developed at NCCUP - Universidade do Porto	 *
*									 *
* Copyright L.Damas, V. Santos Costa and Universidade do Porto 1985--	 *
*									 *
**************************************************************************
*									 *
* File:		stdpreds.c						 *
* comments:	quick saver/loader					 *
*									 *
* Last rev:     $Date: 2011-08-29$,$Author: vsc $			 *
* $Log: not supported by cvs2svn $					 *
*									 *
*************************************************************************/

#include <SWI-Stream.h>
#include <Yap.h>

static size_t save_bytes(IOSTREAM *stream, void *ptr, size_t sz)
{
  return Sfwrite(ptr, sz, 1, stream);
}

static size_t save_pointer(IOSTREAM *stream, void *ptr)
{
  void *p = ptr;
  return save_bytes(stream, &p, sizeof(void *));
}

static size_t save_uint(IOSTREAM *stream, UInt val)
{
  UInt v = val;
  return save_bytes(stream, &v, sizeof(UInt));
}

static size_t save_int(IOSTREAM *stream, Int val)
{
  Int v = val;
  return save_bytes(stream, &v, sizeof(Int));
}

static size_t save_atom(IOSTREAM *stream, Atom at)
{
  if (IsWideAtom(at)) {
    UInt sz = wcslen(RepAtom(at)->WStrOfAE);
    CHECK(save_tag(stream, QLF_WIDE_ATOM));
    CHECK(save_uint(stream, sz));
    return save_bytes(stream, RepAtom(at)->WStrOfAE, (sz+1)*sizeof(wchar_t));
  } else {
    CHECK(save_tag(stream, QLF_ATOM));
    return save_bytes(stream, RepAtom(at)->StrOfAE, (sz+1)*sizeof(char));
  }
}

static size_t save_Arity(IOSTREAM *stream, Int a)
{
  return save_uint(stream, a);
}

static size_t save_CellPtoHeap(IOSTREAM *stream, CELL *ptr)
{
  return save_pointer(stream, ptr);
}

static size_t save_ConstantTerm(IOSTREAM *stream, Term t)
{
  if (IsIntTerm(t)) {
    CHECK(save_int_tag(stream));
    return save_int(stream, IntOfTerm(t));
  }
  CHECK(save_atom_tag(stream));
  return save_atom(stream, AtomOfTerm(t));
}

static size_t save_DoubleInCode(IOSTREAM *stream, Term t)
{
  return save_pointer(stream, (void *)RepAppl(t));
}

static size_t save_DBGroundTerm(IOSTREAM *stream, Term t)
{
  return save_term(stream, t);
}

static size_t save_Func(IOSTREAM *stream, Functor f)
{
  CHECK(save_atom(stream, NameOfFunctor(f)));
  return save_Arity(stream, ArityOfFunctor(f));
}

static size_t save_ExternalFunction(IOSTREAM *stream, Functor f)
{
  Yap_Error(INTERNAL_ERROR, TermNil, "trying to save an ExternalFunction");
  return 0;
}

static size_t save_IntegerInCode(IOSTREAM *stream, Term t)
{
  return save_pointer(stream, (void *)RepAppl(t));
}

static size_t save_PtoLUIndex(IOSTREAM *stream, Term t)
{
  Yap_Error(INTERNAL_ERROR, TermNil, "trying to save PtoLUIndex");
  return 0;
}

static size_t save_PtoOp(IOSTREAM *stream, yamop *l)
{
  return save_pointer(stream, (void *)l);
}

static size_t save_PtoLUClause(IOSTREAM *stream, struct logic_upd_clause *t)
{
  Yap_Error(INTERNAL_ERROR, TermNil, "trying to save PtoLUIndex");
  return 0;
}

static size_t save_ConstantTerm(IOSTREAM *stream, Term t)
{
  return save_atom(stream, AtomOfTerm(t));
}

static size_t save_BlobTermInCode(IOSTREAM *stream, Term t)
{
  return save_pointer(stream, (void *)RepAppl(t));
}

static size_t save_Opcode(IOSTREAM *stream, OPCODE op)
{
  return save_int(stream, Yap_op_from_opcode(op));
}

#ifdef YAPOR
static size_t save_OrArg(IOSTREAM *stream, unsigned int i)
{
  return save_uint(stream, i);
}
#endif /* YAPOR */

#ifdef YAPOR
static size_t save_PtoPred(IOSTREAM *stream, struct pred_entry *ap)
{
  if (ap->ModuleOfPred) {
    CHECK(save_atom(stream, AtomOfTerm(ap->ModuleOfPred)));
  } else {
    CHECK(save_atom(stream, AtomProlog));
  }
  if (ap->ArityOfPE) {
    CHECK(save_int(stream, ap->ArityOfPE));
    return save_atom(stream, NameOfFunctor(ap->FunctorOfPred));
  } else {
    CHECK(save_int(stream, 0));
    return save_atom(stream, (Atom)(ap->FunctorOfPred));
  }
}

static size_t save_Constant(IOSTREAM *stream, Term t)
{
  if (IsIntTerm(t)) {
    CHECK(save_int_tag(stream));
    return save_int(stream, IntOfTerm(t));
  }
  CHECK(save_atom_tag(stream));
  return save_atom(stream, AtomOfTerm(t));
}


#ifdef TABLING
static size_t save_TabEntry(IOSTREAM *stream, struct table_entry *ap)
{
  return save_pointer(stream, NULL);
}
#endif

#if PRECOMPUTE_REGADDRESS
#define arg_from_x(I)		(((CELL *)(I))-XREGS)
#else
#define arg_from_x(I)		(I)
#endif /* PRECOMPUTE_REGADDRESS */

static size_t save_X(IOSTREAM *stream, XREG reg)
{
  return save_int(stream, arg_from_x(reg));
}

static size_t save_Y(IOSTREAM *stream, YREG reg)
{
  return save_int(stream, reg);
}

static size_t
save_code(IOSTREAM *stream, yamop *pc, yamop *max) {
#include "saveclause.h"
  if (max && max > pc) {
    return save_bytes(stream, pc, max);
  }
  return 1;
}

static size_t
save_lu_clause(IOSTREAM *stream, LogUpdate *cl) {
  CHECK(save_new_clause_marker(stream));
  CHECK(save_term(stream, cl->ClSource));
  return save_code(stream, cl->ClCode, cl->ClSource);
}

static size_t
save_dynamic_clause(IOSTREAM *stream, DynamicUpdate *cl) {
  CHECK(save_new_clause_marker(stream));
  CHECK(save_term(stream, cl->ClSource));
  return save_code(stream, cl->ClCode, cl->ClSource);
}

static size_t
save_static_clause(IOSTREAM *stream, LogUpdate *cl) {
  CHECK(save_new_clause_marker(stream));
  if (!(cl->ClFlags & FactMask)) {
    CHECK(save_term(stream, cl->usc.ClSource));
    return save_code(stream, cl->ClCode, cl->usc.ClSource);
  } else {
    return save_code(stream, cl->ClCode, NULL);
  }
}

static size_t
save_mega_clause(IOSTREAM *stream, LogUpdate *cl) {
  UInt i;
  yamop *ptr;
  UInt ncls = cl->ClPred->cs.p_code.NOfClauses;

  for (i = 0, ptr = cl->ClCode; i < ncls; i++) {
    yamop *nextptr = (yamop *)((char *)ptr + cl->ClItemSize);
    CHECK(save_new_clause_marker(stream));
    CHECK(save_code(stream, ptr, nextptr));
    ptr = nextptr;
  }
  return 1;
}

static size_t
save_clauses(IOSTREAM *stream, PredEntry *pp) {
  yamop        *FirstC, *LastC;

  FirstC = pp->cs.p_code.FirstClause;
  LastC = pp->cs.p_code.LastClause;
  if (FirstC == NULL && LastC == NULL) {
    return save_end_clauses_marker(stream);
  }
  if (pp->PredFlags & LogUpdatePredFlag) {
    LogUpdClause *cl = ClauseCodeToLogUpdClause(First);

    while (cl != NULL) {
      CHECK(save_lu_clause(stream, cl));
      cl = cl->ClNext;
    }
  } else if (pp->PredFlags & MegaClausePredFlag) {
    MegaClause *cl = ClauseCodeToMegaClause(First);

    CHECK(save_mega_clause(stream, cl));
  } else if (pp->PredFlags & DynamicPredFlag) {
    yamop *cl = First;

    do {
      CHECK(save_dynamic_clause(stream, ClauseCodeToDynamicClause(cl)));
      if (cl == Last) return;
      cl = NextDynamicClause(cl);
    } while (TRUE);
  } else {
    StaticClause *cl = ClauseCodeToStaticClause(First);

    do {
      CHECK(save_static_clause(stream, cl));
      if (cl->ClCode == Last) return;
      cl = cl->ClNext;
    } while (TRUE);
  }
  return save_end_clauses_marker(stream);
}

}

static size_t
save_pred(IOSTREAM *stream, PredEntry *ap) {
  CHECK(save_atom(stream, ap->NameOfPE));
  CHECK(save_uint(stream, ap->ArityOfPE));
  CHECK(save_uint(stream, ap->PredFlags));
  CHECK(save_uint(stream, ap->cs.p_code.NOfClauses));
  return save_clauses(stream, ap);
}

static size_t
save_module(IOSTREAM *stream, Term mod) {
  PredEntry *ap = Yap_ModulePred(mod);
  while (ap) {
    CHECK(save_pred(stream, ap));
    ap = ap->NextPredOfModule;
  }
}

