//
// $Id: BufferingInStream.h,v 1.13 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(BUFFERINGINSTREAM_OL_GUARD)
#define BUFFERINGINSTREAM_OL_GUARD

#include <ObjectiveLib/LayeredInStream.h>

/**
 * A layered in stream that filters its data through a buffer. This concrete class channels
 * all read data through a buffer, which in some cases can dramatically improve
 * performance, for example when the layer below the buffering stream performs a very
 * time-consuming task.
 *
 * @sa OLBufferingOutStream
 *
 * @ingroup Streams
 */
@interface OLBufferingInStream : OLLayeredInStream
{
@protected
    /**
     * The buffer
     */
    uint8_t*        buffer;

    /**
     * The current position from which to read in the buffer
     */
    unsigned        position;

    /**
     * The current number of bytes in the buffer
     */
    unsigned        currentCount;

    /**
     * The capacity of the buffer
     */
    unsigned        capacity;
}

/**
 * Create and return a new stream. The new stream uses @a underStream as the next lower
 * layer in the stream stack and uses the default buffer size of 32K.
 *
 * @param underStream the stream from which this stream gets its data
 * @return a reference to the newly created stream
 */
+ (id) streamWithInStream: (OLInStream*)underStream;

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the stream. The new stream uses @a underStream as the next lower
 * layer in the stream stack and uses the default buffer size of 32K.
 *
 * @param underStream the stream from which this stream gets its data
 * @return a reference to this stream
 */
- (id) initWithInStream: (OLInStream*)underStream;

/**
 * Initialize the stream. The new stream uses @a underStream as the next lower
 * layer in the stream stack and uses the buffer size of @a size.
 *
 * @param underStream the stream from which this stream gets its data
 * @param size the size of the buffer
 * @return a reference to this stream
 */
- (id) initWithInStream: (OLInStream*)underStream bufferSize: (unsigned)size;
#if !defined(OL_NO_OPENSTEP)
- (void) dealloc;
#endif
/* @} */
#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources;
#endif

/**
 * Read a byte from the stream. This is an optimization of reading a byte that
 * can be very fast when the buffer is not empty.
 *
 * @exception OLInputOutputException if there is an error reading from the stream
 * @exception OLEndOfStreamException if the end of the stream has been reached
 * @return the byte read
 */
- (uint8_t) readByte;

/**
 * Read a sequence of bytes from the stream. Bytes are read from the buffer and the
 * buffer is filled as needed. If the buffer is empty and no bytes can be read from
 * the next lower stream in the stack, then this message returns UINT_MAX.
 *
 * @exception OLInputOutputException if there is an error reading from the stream
 * @param dest the address to which the bytes should be read
 * @param max the maximum number bytes that should be read
 * @return the number of bytes read or UINT_MAX if no bytes could be read
 */
- (unsigned) readBytes: (uint8_t*)dest count: (unsigned)max;

@end

#endif
