// Copyright 2000-2018 JetBrains s.r.o.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php.config.library;

import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.extensions.AbstractExtensionPointBean;
import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.AtomicNotNullLazyValue;
import com.intellij.openapi.vfs.VfsUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.util.xmlb.annotations.Attribute;
import org.jetbrains.annotations.NotNull;

import java.net.URL;
import java.util.stream.Stream;

/**
 * Allows to add a custom library root or to override bundled PHP Runtime.
 *
 * @see PhpLibraryRootProvider
 */
public class PhpLibraryRoot extends AbstractExtensionPointBean {

  private static final Logger LOG = Logger.getInstance(PhpLibraryRoot.class);

  public static final ExtensionPointName<PhpLibraryRoot> EP_NAME = ExtensionPointName.create("com.jetbrains.php.libraryRoot");

  /**
   * This attribute specifies a path to a library root in resources.
   */
  @Attribute("path")
  public String path;

  /**
   * This attribute states that a library root is intended to override bundled PHP Runtime.
   */
  @Attribute("runtime")
  public boolean runtime;

  /**
   * This attribute specifies a qualified name of a custom implementation of the {@link PhpLibraryRootProvider} interface.
   * If it is specified, other attributes are ignored.
   */
  @Attribute("implementation")
  public String implementationClass;

  private final AtomicNotNullLazyValue<PhpLibraryRootProvider> myInstanceHolder = new AtomicNotNullLazyValue<PhpLibraryRootProvider>() {
    @NotNull
    @Override
    protected PhpLibraryRootProvider compute() {
      if (implementationClass == null) {
        return new MyPhpLibraryRootProvider();
      }
      try {
        return instantiate(implementationClass, ApplicationManager.getApplication().getPicoContainer());
      }
      catch (final ClassNotFoundException e) {
        throw new RuntimeException(e);
      }
    }
  };

  @NotNull
  public PhpLibraryRootProvider getProvider() {
    return myInstanceHolder.getValue();
  }

  private class MyPhpLibraryRootProvider implements PhpLibraryRootProvider {

    /**
     * {@inheritDoc}
     */
    @Override
    @NotNull
    public Stream<VirtualFile> getLibraryRoots(@NotNull final Project project) {
      URL url = getLoaderForClass().getResource(path);
      if (url == null) {
        url = PhpLibraryRoot.class.getResource(path);
      }
      if (url != null) {
        final VirtualFile root = VfsUtil.findFileByURL(url);
        if (root != null) {
          if (root.isDirectory()) {
            return Stream.of(root);
          }
          LOG.error("Library root must be a directory", path);
          return Stream.empty();
        }
      }
      LOG.error("Please provide either a valid path or an appropriate implementation", path);
      return Stream.empty();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isRuntime() {
      return runtime;
    }
  }
}
