/**
 * @file libcomprex/archive.h Archive API
 * 
 * $Id: archive.h,v 1.22 2003/01/02 02:49:38 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _LIBCOMPREX_ARCHIVE_H_
#define _LIBCOMPREX_ARCHIVE_H_

#ifdef __cplusplus
extern "C" {
#endif

typedef struct _CxArchive CxArchive;  /**< An archive. */

#include <libcomprex/directory.h>
#include <libcomprex/file.h>
#include <libcomprex/fp.h>
#include <libcomprex/module.h>
#include <libcomprex/types.h>

/**
 * @struct _CxArchive archive.h libcomprex/archive.h
 *
 * A representation of an archive of files.
 */
struct _CxArchive
{
	CxModule      *module;      /**< The module.                     */
	CxArchiveType  type;        /**< The archive type.               */
	CxAccessMode   accessMode;  /**< The file access mode.           */

	CxArchive     *parent;      /**< The parent archive.             */
	CxFP          *fp;          /**< The associated file pointer.    */

	char *name;                 /**< The base filename.              */

	/**
	 * The full path to the archive. This may include a non-physical path
	 * (FTP site, archive, etc.) in the path.
	 */
	char  *path;

	/**
	 * The full physical path to the archive. If the file is non-local,
	 * this will be a temporary filename.
	 */
	char  *physPath;

	/** Non-zero if the archive is stored locally. */
	char isLocal;

	unsigned int fileSize;      /**< The size of the archive file.          */
	unsigned int archiveSize;   /**< The total size of all contents.        */

	size_t fileCount;           /**< Total number of files in the archive. */

	CxDirectory *root;          /**< The root directory in the archive.     */

	void *moduleData;           /**< Module-specific data.                  */

	unsigned int refCount;      /**< The reference count.                   */

	/**
	 * The archive specific callback that will be called upon the extraction
	 * of each file within the archive
	 */
	void (*ex_callback) (CxArchive *, CxFile *, size_t onFile, size_t fileCount);
};


/**************************************************************************/
/** @name Structure (De)allocation Functions                              */
/**************************************************************************/
/*@{*/

/**
 * Creates a new CxArchive structure.
 *
 * @return A new, empty CxArchive structure.
 *
 * @see cxDestroyArchive()
 */
CxArchive *cxNewArchive(void);

/**
 * Destroys a CxArchive structure.
 *
 * @param archive The structure to destroy.
 *
 * @see cxNewArchive()
 */
void cxDestroyArchive(CxArchive *archive);

/*@}*/

/**************************************************************************/
/** @name Attribute Modification Functions                                */
/**************************************************************************/
/*@{*/

/**
 * Sets the archive's asssociated module.
 *
 * This should really only be used by libcomprex.
 *
 * @param archive The archive.
 * @param module  The associated module.
 *
 * @see cxGetArchiveModule()
 */
void cxSetArchiveModule(CxArchive *archive, CxModule *module);

/**
 * Sets the archive type.
 *
 * @param archive The archive.
 * @param type    The archive type.
 *
 * @see cxGetArchiveType()
 */
void cxSetArchiveType(CxArchive *archive, CxArchiveType type);

/**
 * Sets the archive's file access mode.
 *
 * @param archive The archive.
 * @param mode    The access mode.
 *
 * @see cxGetArchiveAccessMode()
 */
void cxSetArchiveAccessMode(CxArchive *archive, CxAccessMode mode);

/**
 * Sets the archive's physical filename.
 *
 * @param archive  The archive.
 * @param filename The physical filename.
 * 
 * @see cxGetArchiveFileName()
 */
void cxSetArchiveFileName(CxArchive *archive, const char *filename);

/**
 * Sets the full path to the archive.
 *
 * This may include such non-physical elements as FTP sites, archives,
 * etc. in the path.
 *
 * @param archive The archive.
 * @param path    The path to the archive.
 *
 * @see cxSetArchivePath()
 * @see cxGetArchivePhysicalPath()
 * @see cxSetArchivePhysicalPath()
 */
void cxSetArchivePath(CxArchive *archive, const char *path);

/**
 * Sets the full physical path to the archive.
 *
 * If the archive is non-local, this should be a temporary file.
 *
 * @param archive The archive.
 * @param path    The physical path to the file.
 *
 * @see cxGetArchivePhysicalPath()
 * @see cxGetArchivePath()
 * @see cxSetArchivePath()
 */
void cxSetArchivePhysicalPath(CxArchive *archive, const char *path);

/**
 * Sets the archive's physical file size.
 * 
 * @param archive  The archive.
 * @param fileSize The physical size of the archive.
 *
 * @see cxGetArchiveFileSize()
 */
void cxSetArchiveFileSize(CxArchive *archive, unsigned int fileSize);

/**
 * Sets the total size of the archive's contents.
 *
 * @param archive     The archive.
 * @param archiveSize The total size of the archive's contents.
 *
 * @see cxGetArchiveSize()
 */
void cxSetArchiveSize(CxArchive *archive, unsigned int archiveSize);

/**
 * Sets whether or not the archive is stored locally.
 *
 * If the archive is located online, or within another archive, this
 * should be 0. Otherwise, this should be 1.
 *
 * @param archive The archive.
 * @param isLocal 1 if the archive is stored locally; 0 otherwise.
 *
 * @see cxIsArchiveLocal()
 */
void cxSetArchiveLocal(CxArchive *archive, char isLocal);

/**
 * Sets the archive specific extraction callback.
 *
 * @param archive  The archive.
 * @param callback The callback function.
 */
void cxSetArchiveExtractCallback(CxArchive *archive,
								 void (*callback)(CxArchive *, CxFile *,
												  size_t, size_t));

/*@}*/


/**************************************************************************/
/** @name Attribute Retrieval Functions                                   */
/**************************************************************************/
/*@{*/

/**
 * Returns the archive's associated module.
 *
 * @param archive The archive.
 *
 * @return The archive's associated module.
 *
 * @see cxSetArchiveModule()
 */
CxModule *cxGetArchiveModule(CxArchive *archive);

/**
 * Returns the archive type.
 *
 * @param archive The archive.
 *
 * @return The archive's type.
 *
 * @see cxSetArchiveType()
 */
CxArchiveType cxGetArchiveType(CxArchive *archive);

/**
 * Returns the archive's file access mode.
 *
 * @param archive The archive.
 * 
 * @return The archive's file access mode.
 *
 * @see cxSetArchiveAccessMode()
 */
CxAccessMode cxGetArchiveAccessMode(CxArchive *archive);

/**
 * Returns the archive's physical filename.
 *
 * @param archive The archive.
 *
 * @return The archive's physical filename.
 *
 * @see cxSetArchiveFileName()
 */
const char *cxGetArchiveFileName(CxArchive *archive);

/**
 * Returns the full path to the specified archive.
 *
 * This may include such non-physical elements as FTP sites, archives,
 * etc. in the path.
 *
 * @param archive The archive.
 *
 * @return The path to the archive.
 *
 * @see cxSetArchivePath()
 * @see cxGetArchivePhysicalPath()
 * @see cxSetArchivePhysicalPath()
 */
const char *cxGetArchivePath(CxArchive *archive);

/**
 * Returns the full physical path to the file.
 *
 * If the archive is non-local, this will be a temporary file.
 *
 * @param archive The archive.
 *
 * @return The physical path to the file.
 *
 * @see cxSetArchivePhysicalPath()
 * @see cxGetArchivePath()
 * @see cxSetArchivePath()
 */
const char *cxGetArchivePhysicalPath(CxArchive *archive);

/**
 * Returns the total number of files in the archive.
 *
 * @param archive The archive.
 *
 * @return The total number of files in the archive.
 */
size_t cxGetArchiveFileCount(CxArchive *archive);

/**
 * Returns the archive's physical file size.
 *
 * @param archive The archive.
 *
 * @return The archive's physical file size.
 *
 * @see cxSetArchiveFileSize()
 */
unsigned int cxGetArchiveFileSize(CxArchive *archive);

/**
 * Returns the total size of the archive's contents.
 *
 * @param archive The archive.
 *
 * @return The total size of the archive's contents.
 *
 * @see cxSetArchiveSize()
 */
unsigned int cxGetArchiveSize(CxArchive *archive);

/**
 * Returns whether or not the archive is stored locally.
 *
 * If the archive is located online, or within another archive,
 * this will be 0. Otherwise, this will be 1.
 *
 * @param archive The archive.
 *
 * @return 1 if the archive is stored locally; 0 otherwise.
 *
 * @see cxSetArchiveLocal()
 */
char cxIsArchiveLocal(CxArchive *archive);

/**
 * Returns the root directory in the archive.
 *
 * @param archive The archive.
 *
 * @return The archive's root directory.
 */
CxDirectory *cxGetArchiveRoot(CxArchive *archive);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _LIBCOMPREX_ARCHIVE_H_ */

