import logging

from multiqc.base_module import BaseMultiqcModule, ModuleNoSamplesFound

from .relatedness2 import Relatedness2Mixin
from .tstv_by_count import TsTvByCountMixin
from .tstv_by_qual import TsTvByQualMixin
from .tstv_summary import TsTvSummaryMixin

log = logging.getLogger(__name__)


class MultiqcModule(BaseMultiqcModule, Relatedness2Mixin, TsTvByCountMixin, TsTvByQualMixin, TsTvSummaryMixin):
    """
    #### Important General Note

    - Depending on the size and density of the variant data (vcf),
      some of the stat files generated by vcftools can be very large.
      If you find that some of your input files are missing, increase
      the [config.log_filesize_limit](https://docs.seqera.io/multiqc/usage/troubleshooting#big-log-files)
      so that the large file(s) will not be skipped by MultiQC. Note, however,
      that this might make MultiQC very slow!

    This module parses the outputs from VCFTools' various commands:

    #### Implemented

    - `relatedness2`
      - Plots a heatmap of pairwise sample relatedness.
      - Not to be confused with the similarly-named command `relatedness`
    - `TsTv-by-count`
      - Plots the transition to transversion ratio as a function of
        alternative allele count (using only bi-allelic SNPs).
    - `TsTv-by-qual`
      - Plots the transition to transversion ratio as a function of SNP
        quality threshold (using only bi-allelic SNPs).
    - `TsTv-summary`
      - Plots a bargraph of the summary counts of each type of transition and
        transversion SNPs.

    #### To do

    VCFTools has a number of outputs not yet supported in MultiQC which
    would be good to add. Please check GitHub if you'd like these added
    or (better still), would like to contribute!

    - basic stats
    - relatedness
    - freq
    - depth
    - [Everything else](https://vcftools.github.io/man_latest.html)

    """

    def __init__(self):
        super(MultiqcModule, self).__init__(
            name="VCFTools",
            anchor="vcftools",
            href="https://vcftools.github.io",
            info="Program to analyse and reporting on VCF files.",
            doi="10.1093/bioinformatics/btr330",
        )

        n = dict()
        n["relatedness2"] = self.parse_relatedness2()

        n["tstv_by_count"] = self.parse_tstv_by_count()
        if n["tstv_by_count"] > 0:
            log.info(f"Found {n['tstv_by_count']} TsTv.count reports")

        n["tstv_by_qual"] = self.parse_tstv_by_qual()
        if n["tstv_by_qual"] > 0:
            log.info(f"Found {n['tstv_by_qual']} TsTv.qual reports")

        n["tstv_summary"] = self.parse_tstv_summary()
        if n["tstv_summary"] > 0:
            log.info(f"Found {n['tstv_summary']} TsTv.summary reports")

        if sum(n.values()) == 0:
            raise ModuleNoSamplesFound
