//
// $Id: FileInStream.h,v 1.17 2007/04/15 23:32:57 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(FILEINSTREAM_OL_GUARD)
#define FILEINSTREAM_OL_GUARD

#include <ObjectiveLib/ConnectedInStream.h>
#include <ObjectiveLib/Types.h>

/**
 * @class OLFileInStream FileInStream.h ObjectiveLib/FileInStream.h
 *
 * A connected stream that reads its data from a file. This is a concrete connected stream
 * that is connected to a file.
 *
 * @sa OLFileOutStream
 *
 * @ingroup Streams
 */
@interface OLFileInStream : OLConnectedInStream
{
@protected
    /**
     * The file descriptor of the file from which data is read
     */
    ol_fd fd;
}

/**
 * Create and return a new file in stream. A new stream is created with the
 * given path.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @exception OLInputOutputException if a file cannot be opened with the given file name
 * @param path the path 
 * @return a reference to theh newly created stream
 */
+ (id) streamWithPath: (const char*)path;

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the stream. The stream will be created with the given file descriptor and
 * all new data will be read starting from the file's current position. The stream will
 * take ownership of the file descriptor and will assume that the stream is responsible
 * for closing the file descriptor.
 *
 * @param fileDesc the file descriptor of the file to which to write
 * @return a reference to this stream
 */
- (id) initWithFileDescriptor: (ol_fd)fileDesc;

/**
 * Initialize the stream. The stream will be created with the given file name.
 *
 * @exception OLInputOutputException if a file cannot be opened with the given file name
 * @param path the path to the file
 * @return a reference to this stream
 */
- (id) initWithPath: (const char*)path;

#if !defined(OL_NO_OPENSTEP)
- (void) dealloc;
#endif
/* @} */

- (void) close;

/**
 * Return the file descriptor associated with the file stream.
 *
 * @note The file stream always owns the file descriptor, so one should never close a
 * file descriptor that is already associated with a file stream.
 *
 * @return the file descriptor
 */
- (ol_fd) fileDescriptor;

#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources;
#endif

/**
 * Read a sequence of bytes from the stream. Bytes up to the amount of @a max are
 * read from the file. The number of bytes read is returned except in the case where
 * the end of the file has been reached. Rather than raising an exception in this
 * case, this method returns UINT_MAX.
 *
 * @exception OLInputOutputException if there is an error reading from the stream
 * @param buffer the address to which the bytes should be read
 * @param max the maximum number bytes that should be read
 * @return the number of bytes read or UINT_MAX if no bytes could be read
 */
- (unsigned) readBytes: (uint8_t*)buffer count: (unsigned)max;

@end

#endif
