/*
 * AppleRecords - A DAAP client
 * Copyright (C) 2004  Chris Davies <c.davies@cdavies.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

package org.cdavies.applerecords;

import javax.swing.*;
import javax.swing.border.*;
import java.awt.*;
import java.util.*;

import org.cdavies.applerecords.status.*;

/**
 * 
 * This panel forms the status control at the bottom of an instance of
 * AppleRecords, basically a text area for displaying status information,
 * and a cute little 'information' icon in bottom right, which is about
 * the most artistic that I get.
 * <p>
 * There are two different types of status message. Timed status messages
 * appear in the status area until they are replaced by a newer message
 * or until a set timeout set by the STATUS_DELAY private member of this
 * class. Permenant status messages remain until replaced by a newer
 * message.
 * 
 * 
 * @author       Chris Davies (c.davies@cdavies.org)
 * @version      0.5 21/06/2004
 *
 */

public class AppleRecordsStatusPanel extends JPanel {
	
	private JLabel _statusLabel;
	private java.util.Timer _statusTimer;
	private StatusTimerTask _task = null;
	
	private static final long STATUS_DELAY = 3000;
	
	/**
	 *
	 * What may come as a (hideous) suprise to many will be
	 * to learn this panel is actually the top level panel
	 * for the entire project. I just discovered that myself,
	 * and to be honest I was a little horrified. I'm sure I had
	 * a good reason for this, but I'm damned if I can think of 
	 * it at the moment.
	 * <p>
	 * That aside, this method contructs a new Status Panel, which
	 * consists of the Status pane at the bottom, and the rest of 
	 * the content at the top (go figure). It uses the standard 
	 * resource locator for this project to load the information
	 * icon.
	 *
	 * @param	contents	The rest of the AppleRecords application contents panel.
	 * @see	ResourceLocator
	 *
	 */
	
	public AppleRecordsStatusPanel(JPanel contents) {
		
		super();
		
		ResourceLocator _loc = new ResourceLocator();
		
		_statusTimer = new java.util.Timer(true);
		
		setLayout(new GridBagLayout());
		
		/* Construct and add the status display */
		_statusLabel = new JLabel("", new ImageIcon(_loc.loadImageResource("information")), JLabel.RIGHT);
		_statusLabel.setHorizontalAlignment(JLabel.RIGHT);
		_statusLabel.setHorizontalTextPosition(JLabel.LEFT);
		_statusLabel.setBorder(new EtchedBorder());
		
		GridBagConstraints _cons;
		
		_cons = new GridBagConstraints();
		_cons.gridx = 0;
		_cons.gridy = 0;
		_cons.weightx = 1;
		_cons.weighty = 1;
		_cons.fill = GridBagConstraints.BOTH;
		
		add(contents, _cons);
		
		_cons = new GridBagConstraints();
		_cons.gridx = 0;
		_cons.gridy = 1;
		_cons.weightx = 1;
		_cons.weighty = 0;
		_cons.fill = GridBagConstraints.HORIZONTAL;
		
		add(_statusLabel, _cons);
		
		setTimedText("Welcome To AppleRecords by Chris Davies");
		
	}
	
	private synchronized void _cancelTask() {
		
		if (_task != null) {
			
			_task.cancel();
			_task = null;
			
		}
		
	}
	
	/**
	 *
	 * Sets a permanamt status message to display in the status
	 * display. This method and its kin have a slight bug in that
	 * they totally ignore font metrics so they may display messages
	 * far to long for the status area, which results in untidy
	 * multiline status displays.
	 * <p>
	 * This will be fixed.
	 *
	 * @param	text	The text to display in the status area
	 *
	 */
	
	public synchronized void setText(String text) {
		
		_cancelTask();
		_statusLabel.setText("<html><i>" + text + "</i></html>");
		
	}
	
	/**
	 *
	 * Ditto on the bug for <code>setText</code>.
	 * <p>
	 * This method sets a text message in the status display that disappears
	 * after a set period of time. This is implemented via a <code>TimerTask</code>
	 * which blanks the status area after the given time has elapsed.
	 * <p>
	 * Of course, this can still be overwritten by setting a later
	 * status message.
	 *
	 * @param	text	The text to display in the status area
	 * @see	org.cdavies.applerecords.status.StatusTimerTask
	 *
	 */
	
	public synchronized void setTimedText(String text) {
		
		_cancelTask();
		_task = new StatusTimerTask(this);
		
		_statusTimer.schedule(_task, STATUS_DELAY);
		_statusLabel.setText("<html><i>" + text + "</i></html>");
		
	}
	
	
}
