#.rst:
# liblcf-config
# -------
#
# Exports the native liblcf headers and library.
#
# Imported Targets
# ^^^^^^^^^^^^^^^^
#
# This module defines the following :prop_tgt:`IMPORTED` targets:
#
# ``liblcf::liblcf``
#   The ``liblcf`` library, if found.
#
# Result Variables
# ^^^^^^^^^^^^^^^^
#
# This module will set the following variables in your project:
#
# ``LIBLCF_INCLUDE_DIRS``
#   where to find liblcf headers.
# ``LIBLCF_LIBRARIES``
#   the libraries to link against to use liblcf.
# ``LIBLCF_FOUND``
#   true if the liblcf headers and libraries were found.
# ``LIBLCF_VERSION_STRING``
#   version of liblcf

# Compute the installation prefix relative to this file.
get_filename_component(_IMPORT_PREFIX "${CMAKE_CURRENT_LIST_FILE}" PATH)
# Use original install prefix when loaded through a
# cross-prefix symbolic link such as /lib -> /usr/lib.
get_filename_component(_realCurr "${_IMPORT_PREFIX}" REALPATH)
get_filename_component(_realOrig "/usr/pkg/lib/cmake/liblcf" REALPATH)
if(_realCurr STREQUAL _realOrig)
	set(_IMPORT_PREFIX "${_realOrig}")
endif()
unset(_realOrig)
unset(_realCurr)
# peel layers until above library directory
get_filename_component(_IMPORT_PREFIX "${_IMPORT_PREFIX}" PATH)
get_filename_component(_IMPORT_PREFIX "${_IMPORT_PREFIX}" PATH)
get_filename_component(_IMPORT_PREFIX "${_IMPORT_PREFIX}" PATH)
# debian multiarch has one layer more
if(_IMPORT_PREFIX MATCHES ".+/lib$")
	get_filename_component(_IMPORT_PREFIX "${_IMPORT_PREFIX}" PATH)
endif()
if(_IMPORT_PREFIX STREQUAL "/")
	set(_IMPORT_PREFIX "")
endif()

# Set the header file directory.
set(LIBLCF_INCLUDE_DIR "${_IMPORT_PREFIX}/include"
	CACHE PATH "LCF include directory")

# Look for the library.
# Allow LIBLCF_LIBRARY to be set manually, as the location of the liblcf library
if(NOT LIBLCF_LIBRARY)
	find_library(LIBLCF_LIBRARY_RELEASE
		NAMES lcf
		PATHS ${_IMPORT_PREFIX}/lib
		DOC "LCF library (release)"
		NO_DEFAULT_PATH NO_CMAKE_FIND_ROOT_PATH)

	find_library(LIBLCF_LIBRARY_DEBUG
		NAMES lcf
		PATHS ${_IMPORT_PREFIX}/lib
		DOC "LCF library (debug)"
		NO_DEFAULT_PATH NO_CMAKE_FIND_ROOT_PATH)

	include(SelectLibraryConfigurations)
	select_library_configurations(LIBLCF)
endif()

set(LIBLCF_VERSION_STRING 0.7.0)

include(FindPackageHandleStandardArgs)
find_package_handle_standard_args(liblcf
	REQUIRED_VARS LIBLCF_LIBRARY LIBLCF_INCLUDE_DIR
	VERSION_VAR LIBLCF_VERSION_STRING)

if(LIBLCF_FOUND)
	set(LIBLCF_LIBRARIES ${LIBLCF_LIBRARY})
	set(LIBLCF_LINK_LIBRARIES "")
	set(LIBLCF_INCLUDE_DIRS ${LIBLCF_INCLUDE_DIR})

	if(NOT TARGET liblcf::liblcf)
		add_library(liblcf::liblcf UNKNOWN IMPORTED)

		if(1)
			find_package(ICU COMPONENTS i18n uc data REQUIRED QUIET)
			list(APPEND LIBLCF_LIBRARIES ${ICU_LIBRARIES})
			list(APPEND LIBLCF_LINK_LIBRARIES ICU::i18n ICU::uc ICU::data)
		endif()
		if(1)
			find_package(expat CONFIG QUIET)
			if(expat_FOUND)
				list(APPEND LIBLCF_LINK_LIBRARIES expat::expat)
			else()
				# Fallback to old expat detection
				find_package(EXPAT REQUIRED QUIET)
				list(APPEND LIBLCF_LINK_LIBRARIES EXPAT::EXPAT)
			endif()
			list(APPEND LIBLCF_LIBRARIES ${EXPAT_LIBRARIES})
		endif()

		set_target_properties(liblcf::liblcf PROPERTIES
			IMPORTED_LINK_INTERFACE_LANGUAGES "CXX"
			INTERFACE_INCLUDE_DIRECTORIES "${LIBLCF_INCLUDE_DIRS}"
			INTERFACE_LINK_LIBRARIES "${LIBLCF_LINK_LIBRARIES}")

		if(LIBLCF_LIBRARY_RELEASE)
			set_property(TARGET liblcf::liblcf APPEND PROPERTY
				IMPORTED_CONFIGURATIONS RELEASE)
			set_target_properties(liblcf::liblcf PROPERTIES
				IMPORTED_LOCATION_RELEASE "${LIBLCF_LIBRARY_RELEASE}")
		endif()

		if(LIBLCF_LIBRARY_DEBUG)
			set_property(TARGET liblcf::liblcf APPEND PROPERTY
				IMPORTED_CONFIGURATIONS DEBUG)
			set_target_properties(liblcf::liblcf PROPERTIES
				IMPORTED_LOCATION_DEBUG "${LIBLCF_LIBRARY_DEBUG}")
		endif()

		if(NOT LIBLCF_LIBRARY_RELEASE AND NOT LIBLCF_LIBRARY_DEBUG)
			set_property(TARGET liblcf::liblcf APPEND PROPERTY
				IMPORTED_LOCATION "${LIBLCF_LIBRARY}")
		endif()

		set_property(TARGET liblcf::liblcf APPEND PROPERTY
			COMPILE_FEATURES cxx_std_14)

		mark_as_advanced(LIBLCF_LINK_LIBRARIES)
	endif()
endif()

mark_as_advanced(LIBLCF_INCLUDE_DIR LIBLCF_LIBRARY)
